/*
 * SOURCE:  bios.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: low level bios routines
 *
 * UPDATES: 08/15/1991 - major rewrite
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

struct InputEvent _inputevent;

struct InputEvent *InputEventAddr( void ) {

    return &_inputevent;
}

/*
   Set Input Interrupt Routine
   Internal Service
*/
unsigned int ( *SetHook( unsigned int ( *Function )(), int num ))() {

    unsigned int ( *OldFunction )() = NULL;
    unsigned int (**funcptr)();

    funcptr = &(_inputevent.wait_input);

    OldFunction = funcptr[num];
    funcptr[num] = Function;

    return OldFunction;
}


/*
;   Set Actual Page
;   Internal Sevice
*/
unsigned char SetActualPage( unsigned char page ) {

    unsigned char oldpage;

    oldpage = _inputevent.actual_page;
    _inputevent.actual_page = page;

    return oldpage;
}



/*
 *    Name: ToggleBlink
 *  Return: void
 * Purpose: Interrupt 10h Service 10h Function 3
 *          This service changes the bit used to determine whether the most
 *          significant bit of a character attribute intensifies or blinks
 *          the display. BL contains the intensified bit value. When this
 *          value equals 1, the adapter enables blinking.
 *
 *
 * (c)M.Schollmeyer
 */
void ToggleBlink( unsigned char blink ) {

    _asm {
        mov     ax, 1003h
        mov     bl, blink
        int     10h
    }
}


/*   Set Cursor Size
;   Interrupt 10h Service 1
*/
void SetCursorSize( unsigned char start, unsigned char end ) {

    _asm {
        mov     ah, 01h
        mov     cl, end
        mov     ch, start
        int     10h
    }
}


/*
;   Clear Cursor
;   Interrupt 10h Sevice 1
*/
void ClearCursor( void ) {

    _asm {
        mov     ah, 1h
        mov     ch, 10h
        xor     cl, cl
        int     10h
    }
}


/*
;   Set Cursor Position
;   Interrupt 10h Sevice 2
*/
void SetCursorPos( int col, int row ) {

    _asm {
        mov     ah, 2h
        push    seg _inputevent
        pop     es
        mov     bh, es:_inputevent.actual_page
        mov     dx, col
        mov     cx, row
        mov     dh, cl
        int     10h
    }
}



/*
;   Set Visual Page
;   Interrupt 10h Service 05h
*/
void SetVisualPage( int page ) {

    _asm {
        mov     ax, page
        mov     ah, 5h
        int     10h
    }
}

/*
;   Set Overscan Register
;   Interrupt 10h Service 10h Function 1
;
*/
void SetOverscan( unsigned char color ) {

    _asm {
        mov     ax, 1001h
        mov     bh, color
        int     10h
    }
}

/*
;   Read Overscan Register
;   Interrupt 10h Service 10h Function 8
;
*/
unsigned char GetOverscan( void ) {

    _asm {
        mov     ax, 1008h
        int     10h
        mov     al, bh
    }
}


/*
;   Delay
;   Interrupt 15h Service 86h
*/
int Delay( unsigned long micros ) {

    int error;

    _asm {
        mov     ah, 86h
        mov     dx, word ptr micros
        mov     cx, word ptr micros+2
        int     15h
        mov     error, 1
        jnc     ok
        mov     error, 0
ok:
        nop
    }
    return error;
}


/*
;   Read Keyboard Character
;   Interrupt 16h Sevice 0
*/
unsigned int ReadKeybrd( void ) {

    unsigned int key;

    _asm {
        mov     ah, 00h
        int     16h
        mov     key, ax
    }
    return key;
}

/*
;   Check Keyboard
;   Interrupt 16h Sevice 1
*/
unsigned int CheckKeybrd( void ) {

    unsigned int key;

    _asm {
        mov     ah, 01h
        int     16h

        jnz     done
        xor     ax, ax
done:
        mov     key, ax
    }
    return key;
}

/*
    Scan Keyboard Buffer for 'key'
 */
BOOL ScanKeybrd( unsigned int key ) {

    unsigned int _far *bptr, *eptr;

    FP_SEG(bptr) = 0;
    FP_OFF(bptr) = 0x041a;
    FP_OFF(bptr) = 0x0400 + *bptr;

    FP_SEG(eptr) = 0;
    FP_OFF(eptr) = 0x041c;
    FP_OFF(eptr) = 0x0400 + *eptr;

    while( bptr != eptr ) {
        if( FP_OFF(bptr) == 0x0449 )
            FP_OFF(bptr) = 0x041e;
        if( *bptr == key )
            return TRUE;
        ++bptr;
    }
    return FALSE;
}

/*
;   Set Repeat Rate
;   Interrupt 16h Sevice 3
*/
void SetKeyRate( int delay, int rate ) {

    _asm {
        mov     ax, rate
        and     ax, 03h             ; rate must be 0...3
        mov     bh, al
        mov     ax, delay
        and     ax, 1fh             ; delay must be 0...1bh
        mov     bl, al
        mov     ah,03h
        mov     al,05h
        int     16h                 ; call service function
    }
}



/*   Mouse Functions
*/
static unsigned char ismouseshow   = FALSE; /* TRUE if mouse is visible */
                        /*                FALSE if mouse is hidden */



int ResetMouse( void ) {

    unsigned int i;

    _asm {
        xor     ax,ax
        int     33h
        cmp     ax, 0
        and     ax,bx
        push    seg _inputevent
        pop     es
        mov     es:_inputevent.mousestatus, ax
    }

    i = (VideoBase->ScreenWidth<<3)-1;
    _asm {
        mov     ax, 7
        mov     cx, 0
        mov     dx, i
        int     33h
    }
    i = (VideoBase->ScreenHeight<<3)-1;
    _asm {
        mov     ax, 8
        mov     cx, 0
        mov     dx, i
        int     33h
    }

    ismouseshow = FALSE;
    return _inputevent.mousestatus;
}



void ShowMouse( void ) {

    if( _inputevent.mousestatus )
        _asm {
            mov ax, 1
            int 33h
        }

}


void HideMouse( void ) {

    if( _inputevent.mousestatus )
        _asm {
            mov         ax, 2
            int         33h
        }

}


unsigned int GetMouse( void ) {

    unsigned int mousepos;

    if( _inputevent.mousestatus )
        _asm {
            mov     ax, 03h     ; call sevice function
            int     33h
            mov     ax, cx
            mov     cl, 3
            shr     ax, cl
            mov     ah, al
            mov     cl, 3
            shr     dx, cl
            mov     al,dl
            mov     mousepos, ax
        }
    return mousepos;
}



/*
;   Set Mouse Pointer Position
;   Interrupt 33h Sevice 04h
*/
void SetMouse( int x, int y ) {

    if( _inputevent.mousestatus )
        _asm {
            mov     ax, y
            mov     cl, 3
            shl     ax, cl
            mov     dx, ax
            mov     ax, x
            mov     cl, 3
            shl     ax, cl
            mov     cx, ax

            mov     ax, 04h     ; call sevice function
            int     33h
        }
}


void SetPointerPage( int page ) {

    if( _inputevent.mousestatus )
        _asm {
            mov     bx, page
            mov     ax, 1dh     ; call sevice function
            int     33h
    }
}

unsigned long GetMouseMove( void ) {

    if( _inputevent.mousestatus )
        _asm {
            mov     ax, 0bh
            int     33h
            mov     ax, cx

        }
}



void SetDoubleClick( unsigned int clicktime ) {

    _inputevent.doublepress = clicktime;
}

static char learnmode = 0;
static unsigned int learnkey = 0;       // key to toggle LEARNMODE
static unsigned int helpkey = 0;


static unsigned int macrotab[MAXMACROCHARS+1];

static unsigned int macropos = 0;
static unsigned int *macrobuf;

void SetLearnKey( unsigned int key ) {

    learnkey = key;
}

void SetHelpKey( unsigned int key ) {

    helpkey = key;
}

char GetLearnState( void ) {

    return learnmode;
}

void LaunchMacro( unsigned int *buffer, int slot ) {

    macrobuf = buffer;
    if( macrobuf == NULL )
        return;

    learnmode = (char)(MACROMODE + slot);
}


/*
 *    Name: ReadInput
 *  Return: Pointer to InputEvent structure
 * Purpose: Loops until the user acts and returns
 *          info about this action.
 *
 *
 * (c)M.Schollmeyer
 */
struct InputEvent *ReadInput( void ) {

    static unsigned int  oldmx, /* if a mouse button is pressed, the mouse */
                         oldmy; /* position is stored here, double press   */
                                /* info is only stored if the mouse is not */
                                /* moved                                   */
    static unsigned long buttontime[4] = { 0L,0L,0L };
                                /* buttontime is used to store the         */
                                /* ClockCounter if a mouse button is       */
                                /* pressed to check if it is a double      */
                                /* press                                   */

    static unsigned long lastmove      = 0L; /* ticks for last mousemove   */
                                /*              for hide mousepointer      */

    unsigned int         key,    /* to store keyboard input                */
                         button = 0, /* to store mouse buttons             */
                         mousex = 0, /* to store mouse positions           */
                         mousey = 0;
    unsigned long        clock;  /* to get ClockCounter                    */
    struct Cursor        sav;

    /* check if a macro is currently executing */
    if( learnmode >= MACROMODE ) {
        /* get keyboard info from bacro buffer */
        _inputevent.key = macrobuf[macropos];
        /* check if the macro was aborted */
        if( ScanKeybrd(BK_ESC) || (_inputevent.key == 0 ) ) {
            /* the macro is finished, either by reaching the end of   */
            /* the macro buffer (wich is null terminated) or because  */
            /* ESC is pressed                                         */
            learnmode = 0;
            macropos = 0;
            _inputevent.buttons = 0;
            _inputevent.key = learnkey;
            e_drawflags();
        } else {
            /* not yet reached end: clear mouse buttons and increment */
            /* macro buffer pointer                                   */
            _inputevent.buttons = 0;
            _inputevent.shift = macrobuf[++macropos];
            ++macropos;

            if( (_inputevent.key == helpkey) && _inputevent.helpfunc )
                (_inputevent.helpfunc)( -1 );
        }

        return &_inputevent;
    }

    _inputevent.key = 0;

    _asm {
        mov     ah, 03h                        ; get cursor status
        push    seg _inputevent
        pop     es
        mov     bh, es:_inputevent.actual_page
        int     10h
        mov     es:_inputevent.cursory, dh
        mov     es:_inputevent.cursorx, dl
    }


    /* MAIN LOOP: handle input events... */
    for ever {
        /* first, check if we should hide the mouse pointer */
        clock = GetClockCounter();
        /* if MOUSEHIDETIME elapsed AND mouse is not already hidden
           AND the mouse was not moved THEN hidemouse */
        if( clock - MOUSEHIDETIME > lastmove
            && ismouseshow
            && !GetMouseMove()  ) {
                /* clear the flag and hile mouse */
                ismouseshow = FALSE;
                HideMouse();
        }
        if( GetMouseMove() ) {
            if( !ismouseshow ) {
                ismouseshow = TRUE;
                ShowMouse();
            }
            /* set lastmove to the current ClockCounter */
            lastmove = GetClockCounter();
        }


        /* check for hook routine */
        if( _inputevent.wait_input ) {
            key = (*_inputevent.wait_input)();
            if( key ) {
                _inputevent.key = key;
                break;
            }
        }

        _asm {
            mov     ah, 01h         ; get keyboard status
            int     16h
            jnz     getkey          ; jump if nothing pressed
            jmp     getmouse
getkey:
            mov     ah, 00h         ; call read keyboard to clear...
            int     16h             ; ...key from keyboard buffer
            mov     key, ax

            mov     ah, 12h                     ; get shift status
            int     16h
            push    seg _inputevent
            pop     es
            mov     es:_inputevent.shift, ax    ; save new shift status
        }

        _inputevent.key = key;
        if( key == learnkey ) {
            /* the macro key is pressed, look if recording is on */
            if( learnmode != LEARNMODE ) {
                /* switch recording on */
                learnmode = LEARNMODE;
                macropos = 0;
                e_drawflags();
            } else {
                /* recording was on */
                macrotab[macropos] = 0;

                GetCursor( _inputevent.actual_page, &sav );
                SetCursorSize( 0, 7 );
                SetCursorPos(
                  (*_inputevent.msgfunc)( MSG_STNDRD, "Press the macro-key <F5-F8>:" )
                    , SCREEN_HEIGHT-1 );
                learnmode = 0;
                while( (key = CheckKeybrd()) == 0 )
                    if( _inputevent.wait_time )
                        (*_inputevent.wait_time)( 0 );
                key = ReadKeybrd()>>8;
                e_status( 1 );
                SetCursorSize( sav.Start, sav.End );
                SetCursorPos( sav.XPos, sav.YPos );

                if( key >= 0x3f && key <= 0x42 ) {
                    key -= 0x3f;
                    PutProfileData( PR_MACROS(key),
                        (char *)macrotab, (macropos + 2)*sizeof(int) );
                }
                macropos = 0;
                e_drawflags();
                /* ReadInput() will now return the learnkey indicating that
                   the calling routine should request for HIS macro keystroke
                   or just ignore it.
                 */
            }
            continue;
        } else {
            if( key == helpkey ) {
                if( _inputevent.helpfunc )
                    (_inputevent.helpfunc)( -1 );
                _inputevent.buttons = 0;
                _inputevent.key = 0;
            }

            if( learnmode == LEARNMODE ) {
                macrotab[macropos] = _inputevent.key;
                macrotab[++macropos] = _inputevent.shift;
                ++macropos;
                if( macropos >= MAXMACROCHARS ) {
                    macropos = 0;
                    learnmode = 0;
                    macrotab[MAXMACROCHARS] = 0;
                    DoErrorBox( 0, "Out of memory for learn buffer, sorry...\nRecording terminated." );
                    _inputevent.key = learnkey;
                }
            } else
                learnmode = 0;
            break;
        }

getmouse:
        if( _inputevent.mousestatus ) {
            _asm {
                mov         ax, 03h         ; get mouse status
                int         33h
                and         bx, 111b
                mov     button, bx
                mov     mousex, cx
                mov     mousey, dx
            }

            if( button != (_inputevent.buttons & 0x07) )
                break;
        }

        if( _inputevent.wait_time )
            (*_inputevent.wait_time)( 0 );

    } /* end of MAIN LOOP */

    if( _inputevent.mousestatus ) {
        mousex = mousex / 8;
        mousey = mousey / 8;

        if( button ) {

            if( !ismouseshow ) {
                ismouseshow = TRUE;
                ShowMouse();
                lastmove = GetClockCounter();
            }

            clock = GetClockCounter();
            if( buttontime[ (button & 0x07) >> 1 ]
                + _inputevent.doublepress > clock
                && oldmx == mousex && oldmy == mousey ) {
                    button |= button<<3;
            }
            buttontime[ (button & 0x07) >> 1 ] = clock;
            oldmx = mousex;
            oldmy = mousey;
        }

        _inputevent.mousex = mousex;
        _inputevent.mousey = mousey;
        _inputevent.buttons = button;
    }

    return &_inputevent;
}


/*
 *    Name: GetInput
 *  Return: Pointer to InputEvent structure
 * Purpose: Fills the fileds of the InputEvent structure with input
 *          values and returns immediately.
 *
 *
 * (c)M.Schollmeyer
 */
struct InputEvent *GetInput( void ) {

    _asm {
        push    seg _inputevent
        pop     es
        mov     es:_inputevent.key, 00h

        mov     ah, 01h             ; get keyboard status
        int     16h
        jz      getshift            ; jump if nothing pressed
        mov     es:_inputevent.key,ax  ; save key
        mov     ah, 00h             ; call read keyboard to clear...
        int     16h                 ; ...key from keyboard buffer
getshift:
        mov     ah, 12h             ; get shift status
        int     16h
        mov     es:_inputevent.shift, ax            ; save new shift status
    }

    if( _inputevent.mousestatus )
        _asm {
            mov     ax, 03h         ; get mouse status
            int     33h
            and     bx, 07h         ; bx = bx & 111b
            push    seg _inputevent
            pop     es
            mov     es:_inputevent.buttons, bx  ; save mouse buttons
            mov     ax, cx
            mov     cl, 3
            sar     ax, cl
            mov     es:_inputevent.mousex, ax
            mov     cl, 3
            shr     dx, cl
            mov     es:_inputevent.mousey, dx
        }

    _asm {
        mov     ah, 03h
        push    seg _inputevent
        pop     es
        mov     bh, es:_inputevent.actual_page
        int     10h
        mov     es:_inputevent.cursory, dh
        mov     es:_inputevent.cursorx, dl
    }

    if( _inputevent.wait_time )
        (*_inputevent.wait_time)( 0 );

    return &_inputevent;
}


/*
 *    Name: GetCursor
 *  Return: void
 * Purpose: Fills a structure with cursor position and shape
 *
 *
 *
 * (c)M.Schollmeyer
 */
void GetCursor( char page, struct Cursor *c ) {

    unsigned char start, end, x, y;

    if( page == -1 )
        page = _inputevent.actual_page;

    _asm {
        mov     ah, 03h
        mov     bh, page
        int     10h
        mov     bx, offset c
        mov     start, ch
        mov     end, cl
        mov     y, dh
        mov     x, dl
    }
    c->Start = start;
    c->End = end;
    c->XPos = x;
    c->YPos = y;
}


/*
 *    Name: GetClockCounter
 *  Return: value of the clock counter
 * Purpose: Reads the value of the clock counter using
 *          interrupt 0x1a service 0
 *
 *
 * (c)M.Schollmeyer
 */
unsigned long GetClockCounter( void ) {

    _asm {

        xor     ah, ah
        int     1ah
        mov     ax, dx
        mov     dx, cx
    }
}

/* end of file bios.c */
