/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains command string parsing functions.
 *
 *	This command string interpreter facility is a hybrid between
 *	the RSX command string interpreter and the VMS command language
 *	interpreter. The syntax of an acceptable command is:
 *
 *	verb [/switch [=value(s)] ... ]
 *	     param [=value(s)] [/switch [=value(s)] ... ]
 *	     [,param ...] ...
 *
 *	where "value(s)" can be a single switch value or multiple
 *	values. For multiple values, the syntax "(value1,value2,...)"
 *	or "value1:value2:..." are equivalent. Quoted strings delimited
 *	by double quotes (") are permitted. Quotes within quoted strings
 *	must be represented by two adjacent double quote characters. In the
 *	absence of quoted entities, all verbs, parameters, switches and
 *	switch values are converted to upper case prior to being passed
 *	on to the application using the facility. Parameter values and
 *	switch values may contain matched sets of parenthesis (unless the
 *	very first character of a switch value is a left parenthesis, in
 *	which case it is interpreted as a list of values).
 *
 *	This is quite similar to DCL syntax. The command language
 *	interface used internally to parse the command, however, is more
 *	like RSX's CSI$1 and CSI$2 facilities than VMS CLI facilities.
 *	The application interfaces to this facility consist of declarations
 *	that form descriptor blocks for verbs, switches and switch and
 *	parameter values. Convenient macros for defining these are in the
 *	"csi.h" header file (macros "csivb", "csisw", "csisv" and "csiend").
 *	Parsing of individual parameters are done under direct program
 *	control, so different syntaxes can be used for differing contexts.
 *	Enough information is available to the application to correctly
 *	identify missing switches, illegal switch value syntax, etc. in the
 *	command string. Very limited error checking is done by the command
 *	interpreter itself.
 *
 *	You may ask: Why not use VMS's command language facility?
 *	Answer: Because it allows only one command context to be accessed
 *	        at one time. Using it for subordinate functions would
 *	        interfere with obtaining additional input from the
 *	        main command line used to invoke the program. Having to
 *	        write my own command string interpreter was a pain in the
 *	        ---, but necessary because of this limitation. However,
 *	        it does provide a transportable facility that can be used
 *	        on non-VMS systems, if needed.
 */

#include "types.h"
#include "csi.h"
#include <ctype.h>

#include "strng.p"

#define STATE_INDEF    0
#define STATE_PARAM    1
#define STATE_SWITCH   2
#define STATE_SWVAL    3
#define STATE_SWVALIST 4
#define STATE_STOP     5

#define TYPE_WHITE     0
#define TYPE_ALPHANUM  1
#define TYPE_PARAMSEP  2
#define TYPE_LPAREN    3
#define TYPE_RPAREN    4
#define TYPE_SLASH     5
#define TYPE_SWSEP     6
#define TYPE_OTHER     7

char Scratch_String[256];

struct Head_Desc {
	struct Param_Block **Cur_Param_Head;
	struct Switch_Block **Cur_Switch_Head;
	struct Swval_Block **Cur_Swval_Head;
};

/*
 *	Routine Do_Command performs an initial parse on the
 *	command line (much like RSX's CSI$1), identifies
 *	the verb, then dispatches control to the user
 *	supplied routine.
 */

int Do_Command (Command_Line, Verb_Tab, User_Arg, CSI_Block)
char *Command_Line;
struct Verb_Table *Verb_Tab;
unsigned long User_Arg;
struct CSI_Status_Block *CSI_Block;
{
	auto   struct CSI_Status_Block *CSI_Ptr;
	auto   struct Verb_Table *Verb_Ptr;
	auto   struct Param_Block *Param_Ptr;
	auto   struct Switch_Block *Switch_Ptr;
	auto   struct Swval_Block *Swval_Ptr;
	auto   char *Ptr;
	auto   unsigned int Index;
	auto   int Status, Match;
	extern int Parse_Command();
	msgcode CSI_NULLCOMMAND, CSI_MULTCOMMAND, CSI_ILLCOMMAND, CSI_AMBCOMMAND;

	CSI_Ptr = CSI_Block;
	CSI_Ptr->Flags = 0;
	CSI_Ptr->CSI_Mask = 0;
	if ((Status = Parse_Command (Command_Line, CSI_Ptr)) != 0) {
		Status = 0;
		Param_Ptr = CSI_Ptr->Param_Addr[0];
		Ptr = &Param_Ptr->Param_Value[0];
		if (*Ptr == '\0')
			Message (CSI_NULLCOMMAND);
		else {
			if (Param_Ptr->Link != 0)
				Message (CSI_MULTCOMMAND, Ptr, 1);
			for (Verb_Ptr = Verb_Tab; Verb_Ptr->Verb_Name != 0; Verb_Ptr++)
			if ((Match = Compare_Keyword_M (Ptr, Verb_Ptr->Verb_Name, Verb_Ptr->Min_Verb_Len,
						        MAX_VERB_LEN, 0)) != 0)
				break;
			if (Verb_Ptr->Verb_Name == 0)
				Message (CSI_ILLCOMMAND, Ptr, 1);
			else if (Match == 2)
				Message (CSI_AMBCOMMAND, Ptr, 1);
			else {
				CSI_Ptr->Verb = Verb_Ptr;
				for (Index = CSI_Ptr->N_Params; Index < 10; Index++)
					CSI_Ptr->Param_Addr[Index] = 0;
				for (Index = 0; Index < 10; Index++)
					CSI_Ptr->Param_Loc[Index] = CSI_Ptr->Param_Addr[Index];
				Status = (*Verb_Ptr->Func) (User_Arg, CSI_Ptr);
			}
		}
	}
/*
 *	Release memory taken up by the parse blocks:
 */
	for (Index = 0; Index < CSI_Ptr->N_Params; Index++) {
		while ((Param_Ptr = CSI_Ptr->Param_Addr[Index]) != 0) {
			while ((Switch_Ptr = Param_Ptr->Switch) != 0) {
				while ((Swval_Ptr = Switch_Ptr->Swval) != 0) {
					Switch_Ptr->Swval = Swval_Ptr->Link;
					Mem_Free (Swval_Ptr);
				}
				Param_Ptr->Switch = Switch_Ptr->Link;
				Mem_Free (Switch_Ptr);
			}
			CSI_Ptr->Param_Addr[Index] = Param_Ptr->Link;
			Mem_Free (Param_Ptr);
		}
	}
	return (Status);
}

/*
 *	Routine 'Parse_Command' parses the input command line,
 *	placing all components of the command line into a series
 *	of data structures. At this time, the parsing is not
 *	under the control of the user-supplied verb tables; this
 *	is a syntax-only parse:
 */

int Parse_Command (Command_Line, CSI_Block)
char *Command_Line;
struct CSI_Status_Block *CSI_Block;
{
	auto   struct CSI_Status_Block *CSI_Ptr;
	auto   char *In_Ptr, *Out_Ptr, *Temp_In, *Temp_Out;
	auto   unsigned long Status;
	auto   unsigned int State, Prev_State, C_Type, P_Count;
	auto   unsigned short Length;
	auto   char c;
	static struct Head_Desc Heads;
	extern unsigned long Copy_Quoted_String();
	msgcode CSI_NORMAL, CSI_TOOMANYPARAM, CSI_CMDTOOLONG, CSI_MISSPAREN,
		CSI_EXTRAPAREN;
/*
 *	Some in-line code macros:
 */
#define Init_String \
    Out_Ptr = &Scratch_String[0]; \
    *Out_Ptr = '\0'; \
    Length = sizeof (Scratch_String) - 1

#define Append_Paren \
    for (; P_Count > 0; P_Count--) { \
	if (Length == 0) { \
	    Status = CSI_CMDTOOLONG; \
	    State = STATE_STOP; \
	    break; \
	} \
	*Out_Ptr++ = ')'; \
	*Out_Ptr = '\0'; \
	Length--; \
    }

	CSI_Ptr = CSI_Block;
	Init_String;
	In_Ptr = Command_Line;
	while (isspace(*In_Ptr))
		In_Ptr++;

	CSI_Ptr->N_Params = 1;
	Heads.Cur_Param_Head = &CSI_Ptr->Param_Addr[0];
	Heads.Cur_Switch_Head = 0;
	Heads.Cur_Swval_Head = 0;

	State = STATE_PARAM;
	Prev_State = STATE_INDEF;
	P_Count = 0;
	while (State != STATE_STOP && (c = *In_Ptr) != '\0') {
		if (isspace(c))
			C_Type = TYPE_WHITE;
		else if (isalpha(c) || isdigit(c))
			C_Type = TYPE_ALPHANUM;
		else if (c == '/')
			C_Type = TYPE_SLASH;
		else if (c == ',' || c == '+')
			C_Type = TYPE_PARAMSEP;
		else if (c == '(')
			C_Type = TYPE_LPAREN;
		else if (c == ')')
			C_Type = TYPE_RPAREN;
		else if (c == ':' || c == '=')
			C_Type = TYPE_SWSEP;
		else
			C_Type = TYPE_OTHER;

		switch (State) {

		case STATE_INDEF:
			switch (C_Type) {

			case TYPE_WHITE:
				In_Ptr++;
				break;

			case TYPE_PARAMSEP:
			case TYPE_SLASH:
				if (P_Count > 0) {
					Message (CSI_MISSPAREN, Scratch_String, 1);
					Append_Paren;
				}
				Change_CSI_State (Prev_State, &Heads);
				Init_String;
				for (In_Ptr++; isspace(*In_Ptr); In_Ptr++)
					;
				State = (C_Type == TYPE_SLASH) ? STATE_SWITCH : STATE_PARAM;
				break;

			case TYPE_SWSEP:
				Change_CSI_State (Prev_State, &Heads);
				Init_String;
				for (In_Ptr++; isspace(*In_Ptr); In_Ptr++)
					;
				if ((c = *In_Ptr) == '/' || c == ',' || c == '+' || c == ':' || c == '=') {
					Prev_State = STATE_SWVAL;
					State = STATE_INDEF;
				} else if (c == '(') {
					P_Count = 1;
					In_Ptr++;
					State = STATE_SWVALIST;
				} else
					State = STATE_SWVAL;
				break;

			default:
				Change_CSI_State (Prev_State, &Heads);
				Init_String;
				if (CSI_Ptr->N_Params >= 10) {
					State = STATE_STOP;
					Status = CSI_TOOMANYPARAM;
				} else {
					Heads.Cur_Param_Head = &CSI_Ptr->Param_Addr[CSI_Ptr->N_Params];
					Heads.Cur_Switch_Head = 0;
					Heads.Cur_Swval_Head = 0;
					CSI_Ptr->N_Params++;
					State = STATE_PARAM;
				}
			}
			break;

		case STATE_SWVAL:
		case STATE_PARAM:
			if (C_Type == TYPE_SLASH || (P_Count == 0 && (C_Type == TYPE_WHITE ||
			    C_Type == TYPE_PARAMSEP || C_Type == TYPE_SWSEP))) {
				Prev_State = State;
				State = STATE_INDEF;
			} else if (C_Type == TYPE_WHITE) {
				In_Ptr++;
			} else {
				if (C_Type == TYPE_LPAREN)
					P_Count++;
				else if (C_Type == TYPE_RPAREN) {
					if (P_Count == 0) {
						Message (CSI_EXTRAPAREN, Scratch_String, 1);
						In_Ptr++;
						break;
					} else
						P_Count--;
				}
				In_Ptr++;
				goto Copy_Char;
			}
			break;

		case STATE_SWITCH:
			if (C_Type == TYPE_ALPHANUM) {
				In_Ptr++;
				goto Copy_Char;
			} else {
				Prev_State = State;
				State = STATE_INDEF;
			}
			break;

		case STATE_SWVALIST:
			switch (C_Type) {

			case TYPE_WHITE:
				In_Ptr++;
				break;

			case TYPE_PARAMSEP:
				In_Ptr++;
				if (c == ',' && P_Count == 1) {
					Change_CSI_State (STATE_SWVAL, &Heads);
					Init_String;
					break;
				}
				goto Copy_Char;

			case TYPE_SLASH:
				Prev_State = State;
				State = STATE_INDEF;
				break;

			case TYPE_RPAREN:
				In_Ptr++;
				if (--P_Count == 0) {
					Prev_State = State;
					State = STATE_INDEF;
					break;
				}
				goto Copy_Char;

			case TYPE_LPAREN:
				P_Count++;

			default:
				In_Ptr++;
				goto Copy_Char;
			}
			break;

Copy_Char:		if (c == '"' || c == '\'') {
				if (Length < 2)
					Status = CSI_CMDTOOLONG;
				else {
					Temp_In = &In_Ptr[-1];
					Temp_Out = Out_Ptr;
					Status = Copy_Quoted_String (&Temp_In, &Temp_Out, Length);
					Length -= Temp_Out - Out_Ptr;
					In_Ptr = Temp_In;
					Out_Ptr = Temp_Out;
				}
			} else if (Length == 0)
				Status = CSI_CMDTOOLONG;
			else {
				*Out_Ptr++ = _toupper(c);
				*Out_Ptr = '\0';
				Length--;
				Status = CSI_NORMAL;
			}
			if (Status == CSI_CMDTOOLONG)
				State = STATE_STOP;
			else if (Status != CSI_NORMAL)
				Message (Status, Scratch_String, 1);
		}
	}
/*
 *	Check for an unterminated switch value list or missing
 *	right parenthesis in parameter or switch value:
 */
	if (P_Count > 0) {
		Message (CSI_MISSPAREN, Scratch_String, 1);
		Append_Paren;
	}
/*
 *	Check for abnormal parse termination; if all ok,
 *	terminate any unfinished data structure.
 */
	if (State == STATE_STOP) {	/* Fatal error */
		Message (Status, Scratch_String, 1);
		return (0);
	} else
		Change_CSI_State ((State == STATE_INDEF) ? Prev_State : State, &Heads);
	return (1);
}
#undef Init_String
#undef Append_Paren

Change_CSI_State (State, Heads)
int State;
struct Head_Desc *Heads;
{
	auto   struct Param_Block *Param_Ptr;
	auto   struct Switch_Block *Switch_Ptr;
	auto   struct Swval_Block *Swval_Ptr;
	extern struct Param_Block *Create_Param_Block();
	extern struct Switch_Block *Create_Switch_Block();
	extern struct Swval_Block *Create_Swval_Block();

	if (State == STATE_PARAM) {
		Param_Ptr = Create_Param_Block (Heads->Cur_Param_Head, Scratch_String);
		Heads->Cur_Switch_Head = &Param_Ptr->Switch;
		Heads->Cur_Swval_Head = &Param_Ptr->Prmval;
	} else if (State == STATE_SWITCH) {
		Switch_Ptr = Create_Switch_Block (Heads->Cur_Switch_Head, Scratch_String, 0);
		Heads->Cur_Swval_Head = &Switch_Ptr->Swval;
	} else if (State == STATE_SWVAL || State == STATE_SWVALIST)
		Swval_Ptr = Create_Swval_Block (Heads->Cur_Swval_Head, Scratch_String);
}

struct Param_Block *Create_Param_Block (List_Head, Str)
struct Param_Block **List_Head;
char *Str;
{
	auto   struct Param_Block *Ptr;
	extern char *Mem_Alloc();
	extern int strlen();

	Ptr = (struct Param_Block *) Mem_Alloc (sizeof (struct Param_Block) + strlen (Str) + 1);
	Ptr->Link = 0;
	Ptr->Switch = 0;
	Ptr->Prmval = 0;
	strcpy (Ptr->Param_Value, Str);
	Link_CSI_Block (Ptr, List_Head);
	return (Ptr);
}

struct Switch_Block *Create_Switch_Block (List_Head, Str, Negate)
struct Switch_Block **List_Head;
char *Str;
int Negate;
{
	auto   struct Switch_Block *Ptr;
	auto   int Length;
	extern char *Mem_Alloc();
	extern int strlen();
	msgcode CSI_NULLSWITCH;

	if ((Length = strlen (Str) + 1) == 1)
		Message (CSI_NULLSWITCH);
	if (Negate != 0)
		Length += 2;
	Ptr = (struct Switch_Block *) Mem_Alloc (sizeof (struct Switch_Block) + Length);
	Ptr->Link = 0;
	Ptr->Swval = 0;
	Ptr->Defaulted = 0;
	if (Negate == 0)
		strcpy (Ptr->Switch_Name_Value, Str);
	else {
		strcpy (Ptr->Switch_Name_Value, "NO");
		strcat (Ptr->Switch_Name_Value, Str);
	}
	Link_CSI_Block (Ptr, List_Head);
	return (Ptr);
}

struct Swval_Block *Create_Swval_Block (List_Head, Str)
struct Swval_Block **List_Head;
char *Str;
{
	auto   struct Swval_Block *Ptr;
	extern char *Mem_Alloc();
	extern int strlen();

	Ptr = (struct Swval_Block *) Mem_Alloc (sizeof (struct Swval_Block) + strlen (Str) + 1);
	Ptr->Link = 0;
	Ptr->Defaulted = 0;
	strcpy (Ptr->Swval_Value, Str);
	Link_CSI_Block (Ptr, List_Head);
	return (Ptr);
}

Link_CSI_Block (Block_Ptr, List_Head)
char *Block_Ptr, **List_Head;
{
	auto   char **List_Ptr;

	for (List_Ptr = List_Head; *List_Ptr != 0; List_Ptr = *List_Ptr)
		;
	*List_Ptr = Block_Ptr;
}

/*
 *	Routine Get_Parameter retrieves the next parameter in the
 *	command line for the specified parameter number. A return
 *	value of zero means an error occurred (but a parameter will
 *	have been returned, possibly incomplete), a return value of
 *	+1 means a parameter was returned and no errors occurred,
 *	and a return value of -1 means that no parameter was
 *	returned (end of list).
 *
 *	Note that parsing of the command in the semantic context of
 *	the user-supplied tables is not done until a parameter is
 *	actually accessed. All errors encountered are treated as
 *	non-fatal and recovered from in a reasonable manner.
 */

int Get_Parameter (Param_Number, Param_Addr, Param_Length, Value_List, Switch_List, CSI_Ptr)
unsigned int Param_Number;
char *Param_Addr;
unsigned short Param_Length;
struct Value_Table *Value_List;
struct Switch_Table *Switch_List;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Param_Block *Param_Ptr;
	auto   struct Verb_Table *Verb_Ptr;
	auto   struct Switch_Block *Switch_Ptr;
	auto   struct Switch_Table *Switch_Tab;
	auto   char *Out_Ptr, *In_Ptr;
	auto   int Match, Error_Count;
	auto   unsigned short Length;
	extern struct Switch_Block *Create_Switch_Block();
	extern int Get_Switch_Value();
	msgcode CSI_LONGPARAM, CSI_ILLSWITCH, CSI_AMBSWITCH, CSI_ILLSWVAL,
		CSI_ILLPRMVAL, CSI_LONGSWVAL, CSI_EXTRASWVAL, CSI_LONGPRMVAL,
		CSI_EXTRAPRMVAL;

	Error_Count = 0;
	Verb_Ptr = CSI_Ptr->Verb;
/*
 *	Check for invalid parameter or end of parameter list:
 */
	if (Param_Number >= CSI_Ptr->N_Params ||
	    (Param_Ptr = CSI_Ptr->Param_Loc[Param_Number]) == 0)
		return (-1);
/*
 *	Copy the parameter value to the user's storage area,
 *	if supplied:
 */
	if ((Out_Ptr = Param_Addr) != 0) {
		Length = Param_Length - 1;
		for (In_Ptr = &Param_Ptr->Param_Value[0]; *In_Ptr != '\0'; )
		if (Length == 0) {
			Message (CSI_LONGPARAM, Param_Ptr->Param_Value, 1);
			Error_Count++;
			break;
		} else {
			*Out_Ptr++ = *In_Ptr++;
			Length--;
		}
		*Out_Ptr = '\0';
	}
/*
 *	Supply any default parameter values that may be required:
 */
	if (Value_List != 0)
		Supply_Switch_Value (Value_List, &Param_Ptr->Prmval);
/*
 *	For any switches that are default, add the appropiate entries to the
 *	parse blocks; also supply any default switch values:
 */
	if (Switch_List != 0) {
		for (Switch_Tab = Switch_List; Switch_Tab->Switch_Name != 0; Switch_Tab++) {
			for (Switch_Ptr = Param_Ptr->Switch; Switch_Ptr != 0; Switch_Ptr = Switch_Ptr->Link)
			if (Switch_Ptr->Switch_Name_Value[0] != '\0' &&
			    Compare_Keyword_M (Switch_Ptr->Switch_Name_Value, Switch_Tab->Switch_Name,
					       Switch_Tab->Min_Switch_Len, MAX_SWITCH_LEN,
					       Switch_Tab->Flags & NEGATABLE) != 0)
				break;
			if (Switch_Ptr == 0 && (Switch_Tab->Flags & (DEFAULT | DEF_NEGATE)) != 0) {
				Switch_Ptr = Create_Switch_Block (&Param_Ptr->Switch,
								  Switch_Tab->Switch_Name,
								  Switch_Tab->Flags & DEF_NEGATE);
				Switch_Ptr->Defaulted = 1;
			}
			if (Switch_Ptr != 0 && Switch_Tab->Values != 0)
				Supply_Switch_Value (Switch_Tab->Values, &Switch_Ptr->Swval);
		}
	}
/*
 *	Place parameter value information into the user's output buffers:
 */
	if (Value_List != 0) {
		if (Get_Switch_Value (Param_Ptr->Prmval, Value_List, CSI_LONGPRMVAL, CSI_EXTRAPRMVAL) == 0)
			Error_Count++;
	} else if (Param_Ptr->Prmval != 0)
		Message (CSI_ILLPRMVAL, Param_Ptr->Param_Value, 1);
/*
 *	Place switch and switch value information into the user's
 *	output buffers:
 */
	for (Switch_Ptr = Param_Ptr->Switch; Switch_Ptr != 0; Switch_Ptr = Switch_Ptr->Link)
	if (Switch_Ptr->Switch_Name_Value[0] != '\0') {
		if ((Switch_Tab = Switch_List) == 0) {
			Message (CSI_ILLSWITCH, Switch_Ptr->Switch_Name_Value, 1);
			Error_Count++;
		} else {
			for (; Switch_Tab->Switch_Name != 0; Switch_Tab++)
			if ((Match = Compare_Keyword_M (Switch_Ptr->Switch_Name_Value,
						        Switch_Tab->Switch_Name,
						        Switch_Tab->Min_Switch_Len, MAX_SWITCH_LEN,
						        Switch_Tab->Flags & NEGATABLE)) != 0)
				break;
			if (Switch_Tab->Switch_Name == 0) {
				Message (CSI_ILLSWITCH, Switch_Ptr->Switch_Name_Value, 1);
				Error_Count++;
			} else if (Match == 2 || Match == -2) {
				Message (CSI_AMBSWITCH, Switch_Ptr->Switch_Name_Value, 1);
				Error_Count++;
			} else {
				Set_CSI_Mask (Switch_Tab, &CSI_Ptr->CSI_Mask);
				Set_CSI_Mask (Switch_Tab, Verb_Ptr->Mask_Addr);
				if (Match < 0)
					Set_CSI_Mask (Switch_Tab, Verb_Ptr->Neg_Mask_Addr);
				if (Switch_Ptr->Defaulted != 0)
					Set_CSI_Mask (Switch_Tab, Verb_Ptr->Def_Mask_Addr);
				if (Switch_Tab->Values != 0) {
					if (Get_Switch_Value (Switch_Ptr->Swval, Switch_Tab->Values,
							      CSI_LONGSWVAL, CSI_EXTRASWVAL) == 0)
						Error_Count++;
				} else if (Switch_Ptr->Swval != 0)
					Message (CSI_ILLSWVAL, Switch_Ptr->Switch_Name_Value, 1);
			}
		}
	}
	CSI_Ptr->Param_Loc[Param_Number] = Param_Ptr->Link;
	return ((Error_Count == 0) ? 1 : 0);
}

Supply_Switch_Value (Switch_Values, Value_Head)
struct Switch_Value_Table *Switch_Values;
struct Swval_Block **Value_Head;
{
	auto   struct Swval_Block *Prev_Swval, *Swval_Ptr, *New_Swval;
	auto   struct Switch_Value_Table *Value_Tab;
	auto   int Length;
	extern char *Mem_Alloc();
	extern int strlen();
/*
 *	For each expected switch value, create an entry if it does not already
 *	exist. Also provide any default value(s), if specified:
 */
	for (Value_Tab = Switch_Values, Prev_Swval = (struct Swval_Block *) Value_Head;
			Value_Tab->Value_Addr != 0; Value_Tab++, Prev_Swval = Prev_Swval->Link)
	if ((Swval_Ptr = Prev_Swval->Link) == 0) {
		Length = (Value_Tab->Default_Value == 0) ? 1 : strlen (Value_Tab->Default_Value) + 1;
		New_Swval = (struct Swval_Block *) Mem_Alloc (sizeof (struct Swval_Block) + Length);
		New_Swval->Link = 0;
		New_Swval->Defaulted = 1;
		if (Value_Tab->Default_Value == 0)
			New_Swval->Swval_Value[0] = '\0';
		else
			strcpy (New_Swval->Swval_Value, Value_Tab->Default_Value);
		Prev_Swval->Link = New_Swval;
	} else if (Swval_Ptr->Swval_Value[0] == '\0' && Value_Tab->Default_Value != 0) {
		New_Swval = (struct Swval_Block *) Mem_Alloc (sizeof (struct Swval_Block) +
							      strlen (Value_Tab->Default_Value) + 1);
		New_Swval->Link = Swval_Ptr->Link;
		New_Swval->Defaulted = 1;
		strcpy (New_Swval->Swval_Value, Value_Tab->Default_Value);
		Mem_Free (Swval_Ptr);
		Prev_Swval->Link = New_Swval;
	}
}

Set_CSI_Mask (Switch_Tab, Mask_Addr)
struct Switch_Table *Switch_Tab;
unsigned int *Mask_Addr;
{
	if (Mask_Addr != 0) {
		if ((Switch_Tab->Flags & CLEAR) != 0)
			*Mask_Addr &= ~Switch_Tab->Mask;
		else
			*Mask_Addr |= Switch_Tab->Mask;
	}
}

/*
 *	Routine Get_Switch_Value fills in the switch value(s) for
 *	a given switch:
 */

int Get_Switch_Value (Value_List, Value_Tab_List, Long_Error, Extra_Error)
struct Swval_Block *Value_List;
struct Switch_Value_Table *Value_Tab_List;
unsigned long Long_Error, Extra_Error;
{
	auto   struct Swval_Block *Swval_Ptr;
	auto   struct Switch_Value_Table *Value_Tab;
	auto   char *In_Ptr, *Out_Ptr;
	auto   int Error_Count;
	auto   unsigned short Length;

	Error_Count = 0;
	for (Value_Tab = Value_Tab_List, Swval_Ptr = Value_List; (Out_Ptr = Value_Tab->Value_Addr) != 0;
							Value_Tab++, Swval_Ptr = Swval_Ptr->Link) {
		Length = Value_Tab->Value_Length - 1;
		for (In_Ptr = &Swval_Ptr->Swval_Value[0]; *In_Ptr != '\0'; )
		if (Length == 0) {
			Message (Long_Error, Swval_Ptr->Swval_Value, 1);
			Error_Count++;
			break;
		} else {
			*Out_Ptr++ = *In_Ptr++;
			Length--;
		}
		*Out_Ptr = '\0';
	}
/*
 *	Check for extras:
 */
	for (; Swval_Ptr != 0; Swval_Ptr = Swval_Ptr->Link)
		Message (Extra_Error, Swval_Ptr->Swval_Value, 1);
	return ((Error_Count == 0) ? 1 : 0);
}

/*
 *	Routine Copy_Quoted_String outputs the contents of a quoted
 *	string, stripping out the quote in the process:
 */

unsigned long Copy_Quoted_String (In, Out, Out_Length)
char **In, **Out;
unsigned short Out_Length;
{
	auto   char *In_Ptr, *Out_Ptr, *End_Ptr;
	auto   unsigned long Status;
	auto   char c;
	msgcode CSI_NORMAL, CSI_MISSQUOTE, CSI_CMDTOOLONG;

	In_Ptr = *In;
	Out_Ptr = *Out;
	End_Ptr = &Out_Ptr[Out_Length-1];
	c = *In_Ptr;
	Status = CSI_NORMAL;
	while (*++In_Ptr != '\0') {
		if (*In_Ptr == c) {
			if (*(In_Ptr+1) != c)
				break;
			In_Ptr++;
		}
		if (Out_Ptr < End_Ptr)
			*Out_Ptr++ = *In_Ptr;
		else
			Status = CSI_CMDTOOLONG;
	}
	if (*In_Ptr == c)
		In_Ptr++;
	else if (Status == CSI_NORMAL)
		Status = CSI_MISSQUOTE;
	*Out_Ptr = '\0';
	*In = In_Ptr;
	*Out = Out_Ptr;
	return (Status);
}

/*
 *	Routine 'Get_Parameter_Count' returns the number of parameter
 *	values for a particular parameter number. This always returns
 *	the number of parameters remaining (that is, how many calls to
 *	Get_Parameter are required to finish the list).
 */

unsigned int Get_Parameter_Count (Param_Number, CSI_Ptr)
unsigned int Param_Number;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Param_Block *Param_Ptr;
	auto   unsigned int Count;

	Count = 0;
	if (Param_Number < CSI_Ptr->N_Params)
		for (Param_Ptr = CSI_Ptr->Param_Loc[Param_Number]; Param_Ptr != 0; Param_Ptr = Param_Ptr->Link)
			Count++;
	return (Count);
}

Dump_CSI (CSI_Block)
struct CSI_Status_Block *CSI_Block;
{
	auto   struct CSI_Status_Block *CSI_Ptr;
	auto   struct Param_Block *Param_Ptr;
	auto   struct Switch_Block *Switch_Ptr;
	auto   struct Swval_Block *Swval_Ptr;
	auto   int Index;

	CSI_Ptr = CSI_Block;
	for (Index = 0; Index < CSI_Ptr->N_Params; Index++) {
		printf ("Parameter %d:\n", Index);
		for (Param_Ptr = CSI_Ptr->Param_Addr[Index]; Param_Ptr != 0; Param_Ptr = Param_Ptr->Link) {
			printf ("  Parameter value: \"%s\"\n", Param_Ptr->Param_Value);
			for (Swval_Ptr = Param_Ptr->Prmval; Swval_Ptr != 0; Swval_Ptr = Swval_Ptr->Link)
				printf ("    Parameter sub-value: \"%s\"\n", Swval_Ptr->Swval_Value);
			for (Switch_Ptr = Param_Ptr->Switch; Switch_Ptr != 0; Switch_Ptr = Switch_Ptr->Link) {
				printf ("    Switch \"%s\"\n", Switch_Ptr->Switch_Name_Value);
				for (Swval_Ptr = Switch_Ptr->Swval; Swval_Ptr != 0; Swval_Ptr = Swval_Ptr->Link)
					printf ("      Switch value: \"%s\"\n", Swval_Ptr->Swval_Value);
			}
		}
	}
}
