IMPLEMENTATION MODULE ansivdu;

(* Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   Implements the InitANSI procedure that initializes the generic VDU routines
   and parameters used in DVItoVDU.
   DVItoVDU assumes text lines start at 1 and increase down the screen; the
   same numbering scheme is used by ANSI standard terminals.  When updating the
   window (via ShowChar/ShowRectangle calls), DVItoVDU assumes the top left
   pixel on the screen is (0,0).  We need to add 1 to both DVItoVDU coordinates
   to get the correct position on the ANSI screen, since the top left position
   is actually (1,1).
*)

FROM vduinterface IMPORT
   (* VAR *)
   DVIstatusl, windowstatusl, messagel, commandl, bottoml,
   windowh, windowv, windowwd, windowht,
   TeXtoASCII,
   StartText, MoveToTextLine, ClearTextLine, ClearScreen,
   StartGraphics, LoadFont, ShowChar, ShowRectangle,
   ResetVDU;

FROM screenio IMPORT
   Write, WriteCard, WriteString;

CONST
   ESC = 33C;

VAR
   cursrow, curscol : CARDINAL;   (* ANSIShowChar remembers cursor location *)

(******************************************************************************)

PROCEDURE InitANSI;

(* For ANSI VDUs a screen pixel has to be an entire character position.
   The dialogue region will be the top 4 lines.
   The window region will be lines 5 to 24; it is 20 pixels high by 80
   pixels wide.
*)

BEGIN
DVIstatusl    := 1;
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := 24;
(* (windowh,windowv) uses DVItoVDU's coordinate scheme in which the top left
   pixel is at (0,0) *)
windowh  := 0;
windowv  := 4;
windowwd := 80;   (* changed to 132 if VDU = VT100132 *)
windowht := 20;

StartText      := ANSIStartText;
MoveToTextLine := ANSIMoveToTextLine;
ClearTextLine  := ANSIClearTextLine;
ClearScreen    := ANSIClearScreen;
StartGraphics  := ANSIStartGraphics;
LoadFont       := ANSILoadFont;
ShowChar       := ANSIShowChar;
ShowRectangle  := ANSIShowRectangle;
ResetVDU       := ANSIResetVDU;
END InitANSI;

(******************************************************************************)

PROCEDURE ANSIStartText;

BEGIN
(* do nothing; text and graphics the same *)
END ANSIStartText;

(******************************************************************************)

PROCEDURE ANSIMoveToTextLine (line : CARDINAL);

(* Move cursor to start of given line. *)

BEGIN
MoveAbs(line,1);
END ANSIMoveToTextLine;

(******************************************************************************)

PROCEDURE MoveAbs (row, col : CARDINAL);

(* Move cursor to given screen position. *)

BEGIN
Write(ESC); Write('[');
WriteCard(row);
Write(';');
WriteCard(col);
Write('H');
END MoveAbs;

(******************************************************************************)

PROCEDURE ANSIClearTextLine (line : CARDINAL);

(* Erase given line; note that DVItoVDU does not assume anything about the
   cursor location at the end of this routine.
*)

BEGIN
MoveAbs(line,1);
Write(ESC);
WriteString('[K');   (* erase to end of line *)
END ANSIClearTextLine;

(******************************************************************************)

PROCEDURE ANSIClearScreen;

BEGIN
Write(ESC);
WriteString('[2J');   (* erase entire screen *)
END ANSIClearScreen;

(******************************************************************************)

PROCEDURE ANSIStartGraphics;

(* ANSI VDUs make no distinction between text and graphics.
   All we do is reset the current cursor position to some undefined state for
   use in the next ANSIShowChar call.
*)

BEGIN
cursrow := 0;
END ANSIStartGraphics;

(******************************************************************************)

PROCEDURE ANSILoadFont (fontname : ARRAY OF CHAR;
                        fontsize : CARDINAL;
                        mag, hscale, vscale : REAL);

BEGIN
(* only one character size available on most ANSI VDUs, so do nothing *)
END ANSILoadFont;

(******************************************************************************)

PROCEDURE ANSIShowChar (screenh, screenv : CARDINAL;
                        ch : CHAR);

(* Show the given Terse character (mapped to ASCII) at the given position.
   We remember the cursor position (cursrow,curscol) so we can reduce the
   output bytes needed to position the next Terse character.
   StartGraphics resets the position to an undefined state (cursrow = 0).
   We also reset when the cursor reaches the right edge (= windowwd) to
   avoid possibility of any auto wrap.
*)

VAR amount : CARDINAL;

BEGIN
(* first translate DVItoVDU coordinates into actual screen location *)
INC(screenh);
INC(screenv);
IF cursrow = screenv THEN
   (* The cursor is on the same line as in previous ShowChar call so we only
      need to move left or right, and probably just a small amount (if at all).
   *)
   IF screenh = curscol THEN       (* cursor in correct location *)
      INC(curscol);                (* cursor will move right when ch written *)
   ELSIF screenh < curscol THEN    (* move cursor left *)
      amount := curscol - screenh;
      Write(ESC); Write('[');
      IF amount > 1 THEN           (* default is 1 col *)
         WriteCard(amount);
         DEC(curscol,amount-1);    (* no need to do this if amount = 1 *)
      END;
      Write('D');
   ELSE                            (* move cursor right *)
      amount := screenh - curscol;
      Write(ESC); Write('[');
      IF amount > 1 THEN WriteCard(amount) END;   (* default is 1 col *)
      INC(curscol,amount+1);
      Write('C');
   END;
ELSE                               (* cursrow undefined or ch on a new line *)
   MoveAbs(screenv,screenh);
   cursrow := screenv;
   curscol := screenh + 1;         (* cursor will move right when ch written *)
END;
IF screenh = CARDINAL(windowwd) THEN   (* ch will be written at right edge *)
   cursrow := 0;                       (* so avoid auto wrap next time around *)
END;
Write(TeXtoASCII[ch]);
END ANSIShowChar;

(******************************************************************************)

PROCEDURE ANSIShowRectangle (screenh, screenv,           (* top left pixel *)
                             width, height : CARDINAL;   (* size of rectangle *)
                             ch : CHAR);                 (* black pixel *)

(* Display the given rectangle using the given black pixel character.
   DVItoVDU ensures the entire rectangle is visible.
*)

VAR i, j : CARDINAL;

BEGIN
(* first convert TeX ch to ASCII *)
ch := TeXtoASCII[ch];
INC(screenh);
FOR i := 1 TO height DO   (* adding 1 to screenv here! *)
   (* move cursor to start of next row *)
   MoveAbs(screenv+i,screenh);
   FOR j := 1 TO width DO
      Write(ch);
   END;
END;
END ANSIShowRectangle;

(******************************************************************************)

PROCEDURE ANSIResetVDU;

BEGIN
(* no need to do anything *)
END ANSIResetVDU;

(******************************************************************************)

BEGIN
END ansivdu.
