IMPLEMENTATION MODULE DCLInterface;

(* Author:         Andrew Trevorrow
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   June, 1985

   Description:
   Gets the DVItoVDU command line and extracts the DVI file parameter
   and qualifier values according to DVITOVDU.CLD.
   Extraction is done using the VMS CLI routines.

   Revised:
   July, 1985 (to be consistent with TeX and IMPRINT from Kellerman & Smith)
 - /FONT_DIRECTORY now specifies a list of string values.
 - /XSIZE and /YSIZE are specified as dimensions with units in
   IN, CM, MM, PC, PT or PX (corresponding to the new DVItoVDU commands).
   Their values are converted to the nearest pixel and exported via
   paperwd and paperht.

   September, 1986
 - /FONT_DIRECTORY is back to a single value!
 - /DUMMY_FONT is assumed to reside in /FONT_DIRECTORY.

   November, 1987 (while at The Open University)
 - Added /TFM_DIRECTORY qualifier.

   June--August, 1988 (while at Aston University)
 - Added /PSPREFIX qualifier.
 - Added /HOFFSET and /VOFFSET qualifiers.
*)

FROM CommandLanguageInterface IMPORT
   CLI$PRESENT, CLI$GET_VALUE;

FROM VMS IMPORT
   SYS$TRNLOG;
   (* DANGER??? TRNLOG is obsolete under version 4; System Routines manual
      recommends TRNLNM, but its bloody hideous and doesn't have a foreign
      definition module yet.
   *)

FROM Conversions IMPORT
   StringToCard, StringToReal, Done;

FROM ScreenIO IMPORT
   Write, WriteString, WriteLn, Halt;

CONST
   NULL = 0C;             (* SYSDEP: terminates a string *)

VAR
   value : stringvalue;   (* temporary string *)

(******************************************************************************)

PROCEDURE GetDVIFile;

(* Get DVI file name from command line *)

VAR i, status : CARDINAL;

BEGIN
DVIname := '';
status := CLI$GET_VALUE('FILESPEC',DVIname);   (* CLD ensures it is there *)
i := HIGH(DVIname);
WHILE (i > 0) AND (DVIname[i] = ' ') DO        (* remove trailing blanks *)
   DVIname[i] := NULL;                         (* SYSDEP: pad with NULLs *)
   DEC(i);
END;
IF DVIname[i] = ':' THEN                       (* translate logical name *)
   IF Translate(DVIname,i) THEN
      (* do nothing more in either case *)
   END;
ELSE
   INC(i);   (* = LEN(DVIname) *)
   IF NOT ExplicitExt(DVIname) THEN            (* append .DVI *)
      IF i + 3 <= HIGH(DVIname) THEN
         DVIname[i]   := '.';
         DVIname[i+1] := 'D';
         DVIname[i+2] := 'V';
         DVIname[i+3] := 'I';
      ELSE   (* user has given a mighty long file spec! *)
         WriteString('DVI file=');
         WriteString(DVIname); WriteLn;
         WriteString('File specification is too long!'); WriteLn;
         Halt(2);
      END;
   END;
END;
(* bad DVIname will be detected upon open in main module *)
END GetDVIFile;

(******************************************************************************)

PROCEDURE Translate (VAR logname : ARRAY OF CHAR;       (* in/out *)
                         lastpos : CARDINAL)            (* position of colon *)
                    : BOOLEAN;

(* SYSDEP: lastpos in logname should be a colon.
   Return TRUE if given logname can be translated and return the
   equivalence name in logname.
   If no translation, return FALSE and don't alter logname.
*)

VAR translen, dummy, i,
    trncount : CARDINAL;   (* avoid infinite recursive translation *)
    dsbmask  : BITSET;     (* disable search bit mask *)

BEGIN
logname[lastpos] := NULL;  (* remove colon from logical name *)
dsbmask  := {};            (* search all tables from process up to system *)
translen := 0;             (* necessary! *)
IF SYS$TRNLOG(logname,translen,logname,dummy,dummy,dsbmask) = 1 THEN
   FOR i := translen TO lastpos DO
      logname[i] := NULL;                     (* SYSDEP: pad with NULLs *)
   END;
   (* try further translations until we fail, or too many translations *)
   trncount := 1;
   lastpos  := translen;
   translen := 0;
   WHILE (trncount < 10) AND
         (SYS$TRNLOG(logname,translen,logname,dummy,dummy,dsbmask) = 1) DO
      FOR i := translen TO lastpos DO
         logname[i] := NULL;                  (* SYSDEP: pad with NULLs *)
      END;
      INC(trncount);
      lastpos  := translen;
      translen := 0;
   END;
   RETURN TRUE;
ELSE                       (* no initial translation; restore colon *)
   logname[lastpos] := ':';
   RETURN FALSE;
END;
END Translate;

(******************************************************************************)

PROCEDURE ExplicitExt (fname : ARRAY OF CHAR) : BOOLEAN;

(* SYSDEP: VAX/VMS files have an extension of the form ".xxx", also known as
   the file type.  If given file specification contains an extension then
   TRUE is returned, otherwise FALSE.
*)

VAR pos : CARDINAL;   ch : CHAR;

BEGIN
pos := LEN(fname);
WHILE pos > 0 DO       (* search backwards looking for . or : or ] *)
   DEC(pos);
   ch := fname[pos];
   IF ch = '.' THEN
      RETURN TRUE
   ELSIF (ch = ':') OR (ch = ']') THEN   (* don't need to look further *)
      RETURN FALSE
   END;
END;
RETURN FALSE;
END ExplicitExt;

(******************************************************************************)

PROCEDURE GetCardinal (qualifier : ARRAY OF CHAR;
                       VAR n : CARDINAL);

(* Check if qualifier is present.  If so, then get value, check it is
   a positive integer, and return via n.
*)

VAR i, status : CARDINAL;

BEGIN
IF ODD(CLI$PRESENT(qualifier)) THEN             (* CLD ensures it has a value *)
   status := CLI$GET_VALUE(qualifier,value);
   i := HIGH(value);
   WHILE (i > 0) AND (value[i] = ' ') DO        (* remove trailing blanks *)
      value[i] := NULL;                         (* SYSDEP: pad with NULLs *)
      DEC(i);
   END;
   IF value[i] = ':' THEN                       (* translate logical name *)
      IF Translate(value,i) THEN
         (* do nothing more in either case *)
      END;
   END;
   n := StringToCard(value);
   IF Done() AND (n > 0) THEN
      (* return *)
   ELSE
      WriteString('Bad /');
      WriteString(qualifier);
      WriteString(' value! (=');
      WriteString(value); Write(')'); WriteLn;
      WriteString('Specify a positive integer.'); WriteLn;
      Halt(2);
   END;
ELSE
   n := 0;                                      (* qualifier not present *)
END;
END GetCardinal;

(******************************************************************************)

PROCEDURE GetPosDimension (qualifier : ARRAY OF CHAR; VAR pixels : CARDINAL);

(* Check if qualifier is present.  If so, then get value, check it is
   a valid positive dimension, convert and return via pixels.
   A valid positive dimension consists of a positive integer or real number
   followed by a two-letter unit: IN, CM, MM, PC, PT or PX (or in lowercase).
*)

VAR
   i, status : CARDINAL;
   r : REAL;
   ch1, ch2 : CHAR;
   units : (in,cm,mm,pc,pt,px);

BEGIN
IF ODD(CLI$PRESENT(qualifier)) THEN             (* CLD ensures it has a value *)
   status := CLI$GET_VALUE(qualifier,value);
   i := HIGH(value);
   WHILE (i > 0) AND (value[i] = ' ') DO        (* remove trailing blanks *)
      value[i] := NULL;                         (* SYSDEP: pad with NULLs *)
      DEC(i);
   END;
   IF value[i] = ':' THEN                       (* translate logical name *)
      IF Translate(value,i) THEN
         (* do nothing more in either case *)
      END;
   END;
   IF i = 0 THEN i := 1 END;
   (* extract units *)
   IF    (Cap(value[i-1]) = 'I') AND (Cap(value[i]) = 'N') THEN
      units := in;
   ELSIF (Cap(value[i-1]) = 'C') AND (Cap(value[i]) = 'M') THEN
      units := cm;
   ELSIF (Cap(value[i-1]) = 'M') AND (Cap(value[i]) = 'M') THEN
      units := mm;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'C') THEN
      units := pc;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'T') THEN
      units := pt;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'X') THEN
      units := px;
   ELSE
      WriteString('Bad units in /');
      WriteString(qualifier);
      WriteString(' dimension! (=');
      WriteString(value); Write(')'); WriteLn;
      WriteString('Last two letters should be IN, CM, MM, PC, PT or PX.');
      WriteLn;
      Halt(2);
   END;
   ch1 := value[i-1];             (* remember letters in units *)
   ch2 := value[i];
   value[i]   := NULL;            (* remove units *)
   value[i-1] := NULL;
   r := StringToReal(value);
   IF Done() AND (r > 0.0) THEN   (* convert r to pixels *)
      CASE units OF
         in : pixels := TRUNC(r * FLOAT(resolution) + 0.5) |
         cm : pixels := TRUNC((r / 2.54) * FLOAT(resolution) + 0.5) |
         mm : pixels := TRUNC((r / 25.4) * FLOAT(resolution) + 0.5) |
         pt : pixels := TRUNC((r / 72.27) * FLOAT(resolution) + 0.5) |
         pc : pixels := TRUNC((r / 72.27) * 12.0 * FLOAT(resolution) + 0.5) |
         px : pixels := TRUNC(r + 0.5)
      END;
   ELSE
      value[i-1] := ch1;          (* restore units *)
      value[i]   := ch2;
      WriteString('Bad /');
      WriteString(qualifier);
      WriteString(' value! (=');
      WriteString(value); Write(')'); WriteLn;
      WriteString('Specify a positive dimension.');
      WriteLn;
      Halt(2);
   END;
ELSE
   pixels := 0;                   (* qualifier not present *)
END;
END GetPosDimension;

(******************************************************************************)

PROCEDURE GetDimension (qualifier : ARRAY OF CHAR; VAR pixels : INTEGER);

(* Check if qualifier is present.  If so, then get value, check it is
   a valid dimension, convert and return via pixels.
   A valid dimension consists of an integer or real number (possibly negative)
   followed by a two-letter unit: IN, CM, MM, PC, PT or PX (or in lowercase).
*)

VAR
   i, status : CARDINAL;
   r : REAL;
   ch1, ch2 : CHAR;
   units : (in,cm,mm,pc,pt,px);

BEGIN
IF ODD(CLI$PRESENT(qualifier)) THEN             (* CLD ensures it has a value *)
   status := CLI$GET_VALUE(qualifier,value);
   i := HIGH(value);
   WHILE (i > 0) AND (value[i] = ' ') DO        (* remove trailing blanks *)
      value[i] := NULL;                         (* SYSDEP: pad with NULLs *)
      DEC(i);
   END;
   IF value[i] = ':' THEN                       (* translate logical name *)
      IF Translate(value,i) THEN
         (* do nothing more in either case *)
      END;
   END;
   IF i = 0 THEN i := 1 END;
   (* extract units *)
   IF    (Cap(value[i-1]) = 'I') AND (Cap(value[i]) = 'N') THEN
      units := in;
   ELSIF (Cap(value[i-1]) = 'C') AND (Cap(value[i]) = 'M') THEN
      units := cm;
   ELSIF (Cap(value[i-1]) = 'M') AND (Cap(value[i]) = 'M') THEN
      units := mm;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'C') THEN
      units := pc;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'T') THEN
      units := pt;
   ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'X') THEN
      units := px;
   ELSE
      WriteString('Bad units in /');
      WriteString(qualifier);
      WriteString(' dimension! (=');
      WriteString(value); Write(')'); WriteLn;
      WriteString('Last two letters should be IN, CM, MM, PC, PT or PX.');
      WriteLn;
      Halt(2);
   END;
   ch1 := value[i-1];             (* remember letters in units *)
   ch2 := value[i];
   value[i]   := NULL;            (* remove units *)
   value[i-1] := NULL;
   r := StringToReal(value);
   IF Done() THEN                 (* convert r to pixels *)
      CASE units OF
         in : pixels := TRUNC(ABS(r) * FLOAT(resolution) + 0.5);
      |  cm : pixels := TRUNC((ABS(r)/2.54) * FLOAT(resolution) + 0.5);
      |  mm : pixels := TRUNC((ABS(r)/25.4) * FLOAT(resolution) + 0.5);
      |  pt : pixels := TRUNC((ABS(r)/72.27) * FLOAT(resolution) + 0.5);
      |  pc : pixels := TRUNC((ABS(r)/72.27) * 12.0 * FLOAT(resolution) + 0.5);
      |  px : pixels := TRUNC(ABS(r) + 0.5);
      END;
      IF r < 0.0 THEN pixels := -pixels END;
   ELSE
      value[i-1] := ch1;          (* restore units *)
      value[i]   := ch2;
      WriteString('Bad /');
      WriteString(qualifier);
      WriteString(' value! (=');
      WriteString(value); Write(')'); WriteLn;
      WriteString('Specify a valid dimension.');
      WriteLn;
      Halt(2);
   END;
ELSE
   pixels := 0;                   (* qualifier not present *)
END;
END GetDimension;

(******************************************************************************)

PROCEDURE Cap (ch : CHAR) : CHAR;

(* Hamburg's CAP is stupid - do my own. *)

BEGIN
IF (ch < 'a') OR (ch > 'z') THEN
   RETURN ch;
ELSE
   RETURN CHR(ORD(ch) - 32);
END;
END Cap;

(******************************************************************************)

PROCEDURE GetString (qualifier : ARRAY OF CHAR;
                     VAR s     : ARRAY OF CHAR);

(* Check if qualifier is present.  If so, then get value and return via s. *)

VAR i, status : CARDINAL;

BEGIN
IF ODD(CLI$PRESENT(qualifier)) THEN             (* CLD ensures it has a value *)
   status := CLI$GET_VALUE(qualifier,s);
   i := HIGH(s);
   WHILE (i > 0) AND (s[i] = ' ') DO            (* remove trailing blanks *)
      s[i] := NULL;                             (* SYSDEP: pad with NULLs *)
      DEC(i);
   END;
   IF s[i] = ':' THEN                           (* translate logical name *)
      IF Translate(s,i) THEN
         (* do nothing more in either case *)
      END;
   END;
ELSE
   s[0] := NULL;                                (* SYSDEP: LEN(s) will be 0 *)
END;
(* the main module will detect bad s value sooner or later *)
END GetString;

(******************************************************************************)

PROCEDURE Append (VAR s1 : ARRAY OF CHAR; s2 : ARRAY OF CHAR);

(* Append s2 to s1. *)

VAR i, j : CARDINAL;

BEGIN
i := LEN(s1);   (* SYSDEP: assumes s1 ends with NULL, unless full *)
j := 0;
WHILE (i <= HIGH(s1)) AND (j <= HIGH(s2)) AND (s2[j] <> NULL) DO
   s1[i] := s2[j];
   INC(i);
   INC(j);
END;
(* check for overflow??? *)
(* DEBUG
IF (i > HIGH(s1)) AND (j <= HIGH(s2)) AND (s2[j] <> NULL) THEN
   WriteString('No room to append '); WriteString(s2); WriteLn;
   Halt(2);
END;
GUBED *)
IF i <= HIGH(s1) THEN s1[i] := NULL END;
END Append;

(******************************************************************************)

(* SYSDEP: CLD file is used to supply most qualifiers with default values. *)

BEGIN
GetDVIFile;                             (* initialize DVIname *)
GetCardinal('MAGNIFICATION',mag);       (* 0 if no /MAG override *)
GetCardinal('RESOLUTION',resolution);   (* get resolution BEFORE dimens *)
GetPosDimension('XSIZE',paperwd);
GetPosDimension('YSIZE',paperht);
GetDimension('HOFFSET',hoffset);        (* 0 if not given *)
GetDimension('VOFFSET',voffset);        (* ditto *)
GetString('VDU',vdu);
GetString('HELP_FILE',helpname);
GetString('PSPREFIX',psprefix);
GetString('TFM_DIRECTORY',tfmdir);
GetString('FONT_DIRECTORY',fontdir);
GetString('DUMMY_FONT',value);
dummyfont := fontdir;                   (* prefix dummyfont with fontdir *)
Append(dummyfont,value);
END DCLInterface.
