/* Source Installer, Copyright (c) 2005-2006 Claudio Fontana

 action.c - main actions

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "global.h"
#include "gui.h"
#include "menu.h"
#include "dialog.h"
#include "console.h"
#include "greeting.h"
#include "text.h"
#include "package_list.h"
#include "status.h"

#include "action.h"

static int action_install_aux(SRCINST_PHANDLE h, char* filename, int reconf);
static int action_find_configure(SRCINST_PHANDLE h, char* filename);
static int action_configure(SRCINST_PHANDLE h);
static void action_start(void);
static void action_end(void);

void action_add(char* filename, char* packagename) {
  SRCINST_PHANDLE h; int result;
  char* title = "Adding new package";
  SRCINST_TYPE ft;

  result = 0; h = 0;
  action_start();

  ft = srcinst_file_type(filename);
  if (ft != SRCINST_TYPE_DIR && ft != SRCINST_TYPE_FILE) {
    dialog_warning(title, "Invalid package",
		   "Not a regular file or directory:", filename);
    goto end_proc;
  }
  
  if (!(h = srcinst_new_package(packagename))) {
    dialog_warning(title, "Package already exists",
		   "Could not create already existing package:", packagename);
    goto end_proc;
  }

  if (!action_install_aux(h, filename, 1))
    goto end_proc;

  result = 1;
  
 end_proc:
  if (h && !srcinst_close_package(h)) {
    dialog_warning(title, "Error updating package information",
		   "Could not update and close the package information.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Package added successfully", 
		   "Your new package has been successfully added.", 0);
  }
}

static int action_install_aux(SRCINST_PHANDLE h, char* filename, int reconf) {
  float v;			/* autoconf version */
  SRCINST_ERR rv;
  SRCINST_MFEATS mf;

  char* title = "Installation";

  status_done(0.0, 10.0);

  if ((rv = srcinst_init_build(h, filename, 0)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Could not initialize build.", 0);
    return 0;
  }

  status_done(10.0, 12.5);

  if (reconf) {
    if ((rv = srcinst_chdir_build(h, 0)) != SRCINST_ERR_OK) {
      dialog_warning(title, "Could not change build directory", 
	     "Could not change build directory to reconfigure package.", 0);
      return 0;
    }

    if ((rv = srcinst_detect_autoconf(h, &v)) == SRCINST_ERR_MISSING) {
      status_done(12.5, 15.0);

      if (!action_find_configure(h, filename))
	return 0;
      
    } else if (rv != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv),
		     "Error detecting configuration script.", 0);
      return 0;
    }

    status_done(15.0, 20.0);
    
    if (!action_configure(h)) {
      return 0;
    }
  }
  
  status_done(20.0, 25.0);

  mf = srcinst_detect_makefile_features(h);

  if (!mf) {
    dialog_warning(title, "Makefile detection failed", 
		   "Failed to detect Makefile or Makefile features.", 0);
    return 0;
  }
  
  if (!srcinst_is_feature_supported(mf, SRCINST_FEATURE_DESTDIR) &&
      !srcinst_is_feature_supported(mf, SRCINST_FEATURE_INSTALL_ROOT)) {

    if (!srcinst_is_feature_supported(mf, SRCINST_FEATURE_PREFIX)) {
      dialog_warning(title, "Staged installation not supported",
		     "This package does not support staged installation.",
		     0);
      srcinst_free_makefile_features(mf);
      return 0;
      
    } else if (!dialog_confirm(title, "Staged installation not supported",
			       "This package does not support staged installation.\nHowever, prefix redirection could work. It is error-prone and discouraged.\n\nDo you want to try using prefix redirection?", 0, 0)) {
      srcinst_free_makefile_features(mf);
      return 0;
    }
  }

  if (state.o.source) {
    if ((rv = srcinst_install_source(h, mf, state.o.src_compress)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv),
		     "Could not install the configured source code.\n"
		     "Check console for the reasons behind the error.", 0);
      srcinst_free_makefile_features(mf);
      return 0;
    }
  }
  
  status_done(25.0, 90.0);
  
  if ((rv = srcinst_build(h, mf)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Package build processes failed.\n"
		   "Remember that you can get information for a bug report\n"
		   "quickly by right-clicking in the console window, and \n"
		   "choosing Save Log.", 0);
    srcinst_free_makefile_features(mf); return 0;
  }
  
  if (state.o.install) {
    status_done(90.0, 92.5);

    if ((rv = srcinst_test_install(h, mf, state.o.strip)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv), 
		     "Test installation failed.\n"
		     "Real installation will not be attempted.", 0);
      srcinst_free_makefile_features(mf); return 0;
    }

    if (1) {			/* check_conflicts */
      struct srcinst_string_list conflicts;
      
      if (!srcinst_get_conflict_list(h, &conflicts)) {
	dialog_warning(title, "Conflict test failed", 
		       "Could not perform the package conflict test.", 0);
	srcinst_free_makefile_features(mf); return 0;
      }
      
      if (conflicts.count > 0) {
	int i; char* conflicts_string = 0;

	for (i = 0; i < conflicts.count; i++) {
	  conflicts_string = srcinst_append(conflicts_string, "\n");
	  conflicts_string = srcinst_append(conflicts_string,
					    conflicts.strings[i]);
	}

	i = dialog_confirm(title, "Conflicts detected", "Installing this package would overwrite files of other packages!\nThe files involved are:\n", conflicts_string, "Do you really want to overwrite the already claimed files?");
	free(conflicts_string);
	
	if (!i) {
	  srcinst_free_string_list(&conflicts);
	  srcinst_free_makefile_features(mf);
	  return 0;
	}
      }
      srcinst_free_string_list(&conflicts);
    }

    status_done(92.5, 100.0);
    
    if ((rv = srcinst_install_binary(h, mf, state.o.strip)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv), 
		     "Failed to install binaries to their final location.\n"
		     "See the console for the reasons behind the error.", 0);
      srcinst_free_makefile_features(mf); return 0;
    }
  }
  
  status_done(99.0, 100.0);
  
  srcinst_free_makefile_features(mf);
  return 1;
}

static int action_configure(SRCINST_PHANDLE h) {
  struct srcinst_configure_sets sets;
  struct srcinst_string_list configure_strings;
  char *prefix = 0;

  SRCINST_ERR rv;
  int result = 0;
  char* title = "Configuration";
  
  srcinst_init_string_list(&configure_strings);
  srcinst_add_string_list(&configure_strings, "./configure");

  if (state.o.manual_conf) {
    if ((rv = srcinst_detect_configure_opt(h, &sets)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv),
		     "Failed to detect configure options.", 0);
      if (rv == SRCINST_ERR_INTR)
	goto end_proc;

    } else {
      int result2; 
      result2 = dialog_configure(&sets, &prefix, &configure_strings);
      srcinst_free_configure_opt(&sets);
      if (!result2)
	goto end_proc;
    }
  }
  
  if ((rv = srcinst_configure(h, prefix ? prefix : state.o.prefix,
			      &configure_strings)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Configuration script failed.\n"
		   "Check the console for the reasons behind the error.\n", 0);
    goto end_proc;
  }

  result = 1;

 end_proc:
  srcinst_free_string_list(&configure_strings);

  if (prefix)
    free(prefix);

  return result;
}

static int action_find_configure(SRCINST_PHANDLE h, char* filename) {
  struct srcinst_string_list scripts;
  SRCINST_ERR rv;
  int result = 0;
  char* subdir = 0;
  char* title = "Find configure script";
      
  srcinst_init_string_list(&scripts);

  if ((rv = srcinst_find_configure(h, &scripts)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Error searching for a suitable configuration script.", 0);
    goto end_proc;
  }

  if (scripts.count == 0) {
    dialog_warning(title, "No configuration script available",
   "Could not detect a suitable script or script-generating script", 0);
    goto end_proc;
  }

  if (scripts.count == 1) {
    subdir = srcinst_dirname(scripts.strings[0]);

  } else {
    int idx;
    idx = dialog_input_select_string(title, "Multiple build directories have been detected.\nThis is a sometimes a deprecated way to support different systems.\nPlease select a build directory from these choices.", &scripts);
    if (idx == -1)
      goto end_proc;

    subdir = srcinst_dirname(scripts.strings[idx]);
  }

  if ((rv = srcinst_chdir_build(h, subdir)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Could not move to the build subdirectory.", 0);
    goto end_proc;
  }

  result = 1;

 end_proc:
  if (subdir)
    free(subdir);

  srcinst_free_string_list(&scripts);
  return result;
}

void action_install(int reconf) {
  int result;
  char* title = "Install package";
  char* source;

  action_start();

  result = 0;
  
  if (!(source = srcinst_get_source_location(state.package))) {
    dialog_warning(title, "Failed to find source code", 
		   "This package source code is not reachable, \n"
		   "even though it should be available.", 0);
    goto end_proc;
  }
  
  if (!action_install_aux(state.package, source, reconf))
    goto end_proc;

  result = 1;

 end_proc:
  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package",
		   "Error trying to finalize and close the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Installation complete", "Package has been successfully installed.", 0);
  }
}

void action_remove(void) {
  int result = 0;
  SRCINST_ERR rv;
  char* title = "Remove package";
  
  action_start();

  status_done(0.0, 10.0);
  
  if (srcinst_is_installed(state.package)) {
    if (srcinst_is_source_available(state.package) &&
	srcinst_get_installed_files(state.package)->count == 0) {
      rv = srcinst_init_build(state.package,
			      srcinst_get_source_location(state.package), 0);
      
      if (rv != SRCINST_ERR_OK) {
	dialog_warning(title, srcinst_error_string(rv),
		       "Could not initialize build for source-assisted "
		       "uninstallation.", 0);
	goto end_proc;
      }
    }

    status_done(10.0, 95.0);

    if ((rv = srcinst_uninstall_binary(state.package)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv), 
		     "Could not uninstall binaries.", 0);
      goto end_proc;
    }
  }

  status_done(95.0, 98.0);
    
  if (srcinst_is_source_available(state.package)) {
    if ((rv = srcinst_uninstall_source(state.package)) != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv), 
		     "Could not remove stored source code", 0);
      goto end_proc;
    }
  }

  status_done(98.0, 100.0);
  result = 1;

 end_proc:
  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package", 
		   "Could not finalize and close the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Removal complete", "Package has been successfully removed.", 0);
  }
}

void action_uninstall(void) {
  SRCINST_ERR rv;
  int result = 0;
  char* title = "Uninstall package";

  action_start();

  status_done(0.0, 10.0);

  if (srcinst_is_source_available(state.package) && 
      srcinst_get_installed_files(state.package)->count == 0) {
    rv = srcinst_init_build(state.package, 
			    srcinst_get_source_location(state.package), 0);

    if (rv != SRCINST_ERR_OK) {
      dialog_warning(title, srcinst_error_string(rv), 
		     "Could not initialize build "
		     "for source-assisted uninstallation.", 0);
      
      goto end_proc;
    }
  }

  status_done(10.0, 98.0);

  if ((rv = srcinst_uninstall_binary(state.package)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Could not uninstall binaries.", 0);
    goto end_proc;
  }

  status_done(98.0, 100.0);
  result = 1;

 end_proc:

  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package",
		   "Could not finalize and close the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Package uninstalled", "The package has been uninstalled successfully.", 0);
  }
}

void action_upgrade(char* filename, char* packagename) {
  SRCINST_PHANDLE h; int rv;
  char* title = "Upgrade package";
  char* oldname;

  SRCINST_TYPE ft;

  rv = 0; h = 0;
  action_start();

  ft = srcinst_file_type(filename);
  if (ft != SRCINST_TYPE_DIR && ft != SRCINST_TYPE_FILE) {
    dialog_warning(title, "Invalid package",
		   "Not a regular file or directory:", filename);
    goto end_proc;
  }

  if (!(oldname = srcinst_get_name(state.package))) {
    dialog_warning(title, "Invalid old package",
		   "Could not read old package name.", 0);
    goto end_proc;
  }
  
  /* XXX old package not closed!
     srcinst_close_package(state.package); */
  
  if (!(h = srcinst_upgrade_package(packagename, oldname))) {
    dialog_warning(title, "Cannot start package upgrade",
		   "Could not start the upgrade procedure, "
		   "check that the old package is installed, "
		   "and that the new package name is valid", 0);
    goto end_proc;
  }

  if (!action_install_aux(h, filename, 1))
    goto end_proc;

  rv = 1;
  
 end_proc:
  
  if (h) {
    rv = srcinst_close_package(h) && rv;
  }
  
  status_done(100.0, 100.0);
  action_end();

  if (rv) {
    dialog_message(title, "Package upgraded successfully.",
		   "Your package has been successfully upgraded.", 0);
  } else {
    dialog_message(title, "Package _not_ upgraded.",
		   "The old package has been restored.", 0);
  }
}

void action_export_info(SRCINST_PHANDLE h, char* fname, SRCINST_EXPORT fmt) {
  SRCINST_ERR rv;
  char* title = "Export information";

  if ((rv = srcinst_export_info(h, fname, fmt)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv),
		   "Failed to export the package information.", 0);

  } else {
    dialog_message(title, "Information successfully exported", "Package information has been successfully written to", fname);
  }
}

void action_export_bin(char* fname, SRCINST_COMP fmt) {
  SRCINST_ERR rv; SRCINST_PHANDLE h;
  char* title = "Export binary package";
  h = state.package;

  action_start();
  status_done(0.0, 100.0);

  if ((rv = srcinst_export_binary(h, fname, fmt)) != SRCINST_ERR_OK) {
    dialog_warning(title, srcinst_error_string(rv), 
		   "Failed to export binary package.", 0);

  }

  status_done(100.0, 100.0);
  action_end();

  if (rv == SRCINST_ERR_OK) {
    dialog_message(title, "Binary package successfully exported", "Binary package has been successfully written as", fname);
  }
}

static void action_start(void) {
  console_clear();
  console_show();
  status_reset();
  status_show();

  state.actions_locked = 1;

  srcinst_register_stdout_callback(console_print_stdout);
  srcinst_register_stderr_callback(console_print_stderr);
  srcinst_register_spawn_callback(console_print_spawn);
  srcinst_register_beat_callback(gui_doevents);
  gui_update();
}

static void action_end(void) {
  status_hide();

  state.package = 0;
  state.actions_locked = 0;

  package_list_fill();

  srcinst_unregister_stdout_callback(console_print_stdout);
  srcinst_unregister_stderr_callback(console_print_stderr);
  srcinst_unregister_spawn_callback(console_print_spawn);
  srcinst_unregister_beat_callback(gui_doevents);
  gui_update();
}
