/* Source Installer, Copyright (c) 2005,2006,2007 Claudio Fontana

 sourceinstall.c - sourceinstall command line frontend

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#include <srcinst/srcinst.h>
#include "options.h"
#include "sourceinstall.h"

char* program_name = 0;
struct options o;

/* CONFIGUREQ param determines if configuration phase is necessary.
   It can assume values 0 = no, 1 = yes. */

int add_package(char* packagename, char* source) {
  SRCINST_PHANDLE h; int rv;
  
  if (!(h = srcinst_new_package(packagename))) {
    srcinst_warning(SRCINST_ERR_BUSY, "package already exists", packagename);
    return 0;
  }

  rv = install_package_aux(h, source, 1);
  return (srcinst_close_package(h) && rv);
}

int list_packages(char* pattern) {
  int i;
  struct srcinst_string_list list;
  SRCINST_REGEX reg;
  
  srcinst_list_packages(&list);

  if (list.count == 0) {
    return 1;
  }

  if (pattern) {
    reg = srcinst_init_regex(pattern, -1);
  }

  for (i = 0; i < list.count; i++) {
    if (!pattern || srcinst_exec_regex(reg, list.strings[i], 0)) {
      printf("%s\n", list.strings[i]);
    }
  }

  srcinst_free_string_list(&list);

  if (pattern) {
    srcinst_free_regex(reg);
  }

  return 1;
}

int check_package(char* packagename) {
  char* tmp; 
  SRCINST_PHANDLE h; 

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_INVALID, "must specify a PACKAGE_NAME", 0);
    return 0;
  }
  
  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  printf("package name: %s\n", packagename);

  if ((tmp = srcinst_get_descr(h))) {
    printf("package description: %s\n\n", tmp);
    if ((tmp = srcinst_get_descr_long(h))) {
      printf("%s\n\n", tmp);
    }
  }

  if ((tmp = srcinst_get_configure_string(h))) {
    printf("configured as: %s\n", tmp);
  }

  printf("source available: %s\n", srcinst_is_source_available(h) ? "yes" : "no");

  if ((tmp = srcinst_get_source_location(h))) {
    off_t ssize;
    printf("source location: %s\n", tmp);
    
    if ((ssize = srcinst_get_source_size(h)) != (off_t)-1) {
      char* offstr;
      offstr = srcinst_offtostr(ssize);

      printf("compressed source size (KB): %s\n", offstr);
      free(offstr);
    }

    if ((tmp = srcinst_get_build_subdir(h))) {
      printf("build subdirectory: %s\n", tmp);
    }
  }

  printf("installed: %s\n", srcinst_is_installed(h) ? "yes" : "no");

  if (srcinst_is_installed(h)) {
    off_t ssize; 
    struct srcinst_string_list* list;
    if ((ssize = srcinst_get_installed_size(h)) != (off_t)-1) {
      char* offstr;
      offstr = srcinst_offtostr(ssize);
      printf("installed size (KB): %s\n", offstr);
      free(offstr);
    }

    if ((tmp = srcinst_get_prefix(h))) {
      printf("installation prefix: %s\n", tmp);
    }

    list = srcinst_get_installed_files(h);

    if (list->count > 0) {
      int i; int install_check; /* 1 = ok, 0 = error */
      install_check = 1;

      for (i = 0; i < list->count; i++) {
	int missing;

	missing = srcinst_file_ltype(list->strings[i]) == SRCINST_TYPE_NONE;
	if (missing) {
	  printf("%s [%s]\n", list->strings[i], "MISSING");
	  install_check = 0;
	}
      }

      if (install_check)
	printf("installed files: all present and reachable.\n");

    } else {
      printf("installed files: no files installed, or no info available.\n");
    }
  }

  return srcinst_close_package(h);
}

int install_package_aux(SRCINST_PHANDLE h, char* source, int configureq) {

  SRCINST_ERR err;
  SRCINST_MFEATS mf; int rv = 0;
  
  if ((err = srcinst_init_build(h, source, o.subdir)) != SRCINST_ERR_OK) {
    srcinst_warning(err, "could not initialize build", 0);
    return 0;
  }

  if (configureq) {
    if ((err = srcinst_configure(h, o.prefix, &(o.configure_strings)))
	!= SRCINST_ERR_OK) {
      srcinst_warning(err, "configuration failed", 0);
      if (!o.force)
	return 0;
    }
  }
  
  if (!(mf = srcinst_detect_makefile_features(h))) {
    srcinst_warning(SRCINST_ERR_ACTION,
		    "failed to detect makefile features", 0);
    return 0;
  }

  if (!srcinst_is_feature_supported(mf, SRCINST_FEATURE_DESTDIR) &&
      !srcinst_is_feature_supported(mf, SRCINST_FEATURE_INSTALL_ROOT)) {

    if (!srcinst_is_feature_supported(mf, SRCINST_FEATURE_PREFIX)) {
      srcinst_warning(SRCINST_ERR_POLICY,
		      "build system does not support staged installs", 0);
      goto end_proc;

    } else if (o.force) {
      srcinst_warning(SRCINST_ERR_POLICY,
		      "build system does not support staged installs, "
		      "forcing prefix redirection", 0);
    } else {
      srcinst_warning(SRCINST_ERR_POLICY,
		      "build system does not support staged installs. Use "
		      "--force to use prefix redirection (discouraged)", 0);
      goto end_proc;
    }
  }
  
  if (o.source) {
    if ((err = srcinst_install_source(h, mf, o.src_compress))
	!= SRCINST_ERR_OK) {
      srcinst_warning(err, "source installation failed", 0);
      goto end_proc;
    }
  }

  if ((err = srcinst_build(h, mf)) != SRCINST_ERR_OK) {
    srcinst_warning(err, "compilation phase failed", 0);
    goto end_proc;
  }

  if (o.install) {

    if ((err = srcinst_test_install(h, mf, o.strip)) != SRCINST_ERR_OK) {
      srcinst_warning(err, "test installation failed", 0);
      goto end_proc;
    }

    if (1) {			/* check conflicts */
      struct srcinst_string_list conflicts;
      
      if (!srcinst_get_conflict_list(h, &conflicts)) {
	srcinst_warning(SRCINST_ERR_ACTION,
			"conflict test could not be performed", 0);
	goto end_proc;
      }
      
      if (conflicts.count > 0) {
	int i;
	srcinst_warning(SRCINST_ERR_BUSY, "installation would overwrite "
			"files claimed by other packages:", 0);
	for (i = 0; i < conflicts.count; i++) {
	  fprintf(stderr, "%s\n", conflicts.strings[i]);
	}
	
	srcinst_free_string_list(&conflicts);
	
	if (!o.force)
	  goto end_proc;
      }
    }
    
    if ((err = srcinst_install_binary(h, mf, o.strip)) != SRCINST_ERR_OK) {
      srcinst_warning(err, "binary installation failed", 0);
      goto end_proc;
    }
  }
  
  rv = 1;
  
 end_proc:
  srcinst_free_makefile_features(mf);
  return rv;
}

int install_package(char* packagename) {
  int rv; SRCINST_PHANDLE h; char* source; char* prefix;
  int configureq;

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }

  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  prefix = srcinst_get_prefix(h);

  configureq = o.configure_strings.count > 1 ||	       /* must reconfigure? */
    (prefix && strcmp(o.prefix, prefix) != 0);

  if (!(source = srcinst_get_source_location(h))) {
    srcinst_warning(SRCINST_ERR_MISSING,
		    "source code not available for", packagename);
    rv = 0;
  } else {
    rv = install_package_aux(h, source, configureq);
  }

  return (srcinst_close_package(h) && rv);
}

int uninstall_binary(char* packagename) {
  int rv = 0; SRCINST_ERR err; SRCINST_PHANDLE h;

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }

  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  if (!srcinst_is_installed(h)) {
    srcinst_warning(SRCINST_ERR_INVALID, "not an installed package",
		    packagename);

  } else if (!(rv = ((err = srcinst_uninstall_binary(h)) == SRCINST_ERR_OK))) {
    srcinst_warning(err, "binary uninstallation failed", 0);
  }

  return (srcinst_close_package(h) && rv);
}

int uninstall_source(char* packagename) {
  int rv = 0; SRCINST_ERR err; SRCINST_PHANDLE h;

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }
  
  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }
  
  if (!srcinst_is_source_available(h)) {
    srcinst_warning(SRCINST_ERR_MISSING, "source code not available", 0);
    
  } else if (!(rv = ((err = srcinst_uninstall_source(h)) == SRCINST_ERR_OK))) {
    srcinst_warning(err, "source uninstallation failed", 0);
  }
  
  return (srcinst_close_package(h) && rv);
}

int upgrade_package(char* newname, char* oldname, char* src) {
  SRCINST_PHANDLE h; int rv;
  
  h = 0; rv = 0;

  if (!(h = srcinst_upgrade_package(newname, oldname))) {
    srcinst_warning(SRCINST_ERR_ACTION, "cannot start upgrade, "
		    "check package names", 0);
    return 0;
  }

  if (!install_package_aux(h, src, 1)) {
    srcinst_warning(SRCINST_ERR_ACTION, 
		    "could not install new package version", 0);
    goto end_proc;
  }

  rv = 1;

 end_proc:
  return (srcinst_close_package(h) && rv);
}

int remove_package(char* packagename) {
  int rv = 0; SRCINST_ERR err; SRCINST_PHANDLE h;
  
  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }
  
  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  if (srcinst_is_installed(h)) {
    if (!(rv = ((err = srcinst_uninstall_binary(h)) == SRCINST_ERR_OK))) {
      srcinst_warning(err, "binary uninstallation failed", 0);
      goto end_proc;
    }
  }

  if (srcinst_is_source_available(h)) {
    if (!(rv = ((err = srcinst_uninstall_source(h)) == SRCINST_ERR_OK))) {
      srcinst_warning(err, "source uninstallation failed", 0);
    }
  }
  
 end_proc:
  return (srcinst_close_package(h) && rv);
}

int export(char* packagename, char* export_format, char* filename) {
  SRCINST_PHANDLE h; SRCINST_EXPORT fmt;
  SRCINST_ERR err; int rv = 0; 

  fmt = srcinst_str_to_export_format(export_format);
  if (fmt == SRCINST_EXPORT_UNKNOWN) {
    srcinst_warning(SRCINST_ERR_INVALID, "export format", export_format);
    return 0;
  }

  if (!packagename) {
    if (fmt == SRCINST_EXPORT_LST) {
      srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
      return 0;
    }
    h = (SRCINST_PHANDLE)0;

  } else if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  if (fmt == SRCINST_EXPORT_LST && 
      (!srcinst_is_installed(h) || !srcinst_get_installed_files(h)->count)) { 
    srcinst_warning(SRCINST_ERR_INVALID, "no installed files information available for", packagename);
    rv = 0;

  } else if (!(rv = ((err = srcinst_export_info(h, filename, fmt))
		     == SRCINST_ERR_OK))) {
    srcinst_warning(err, "failed to export package information", 0);
  }

  if (h)
    rv = (srcinst_close_package(h) && rv);

  return rv;
}

int export_binary(char* packagename, char* filename) {
  SRCINST_PHANDLE h; SRCINST_ERR err;

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }

  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  if ((err = srcinst_export_binary(h, filename, o.src_compress))
      != SRCINST_ERR_OK) {
    srcinst_warning(err, "failed to export binary package", packagename);
  }
  
  return (srcinst_close_package(h) && err == SRCINST_ERR_OK);
}

int detect_builddirs(char* packagename, char* source) {
  SRCINST_PHANDLE h; SRCINST_ERR err;
  struct srcinst_string_list scripts;

  if (packagename) {
    srcinst_warning(SRCINST_ERR_INVALID, "ignoring package name", packagename);
  }

  if (!(h = srcinst_new_package(0))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not create temp package", 0);
    return 0;
  }

  if (o.subdir) {
    srcinst_warning(SRCINST_ERR_INVALID, "ignoring -D, --subdir option", 0);
  }
  
  if ((err = srcinst_init_build(h, source, 0)) != SRCINST_ERR_OK ||
      (err = srcinst_find_configure(h, &scripts)) != SRCINST_ERR_OK) {
    srcinst_warning(err, "could not scan temporary build", 0);
    srcinst_close_package(h);
    return 0;
  }

  /* first: check configure script */

  if (scripts.count == 0) {
    /* no output. The package does not provide a configure script,
       or a configure-generating script */

  } else {
    int i;
    for (i = 0; i < scripts.count; i++) {
      float version; char* dname;
      chdir(srcinst_get_dir(SRCINST_DIR_BUILD));
      chdir(srcinst_get_packdir(h));
      dname = srcinst_dirname(scripts.strings[i]);
      printf("%s\t", dname);

      if (chdir(dname) == 0 && srcinst_detect_autoconf(h, &version) ==
	  SRCINST_ERR_OK) {
	if (version < 0.1)
	  printf("[custom]\n");
	else
	  printf("[autoconf %f]\n", version);
      } else {
	printf("[unknown]\n");
      }
      free(dname);
    }
  }

  srcinst_close_package(h);
  return 1;
}

int detect_feats(char* packagename, char* source) {
  SRCINST_PHANDLE h; float version; SRCINST_MFEATS mf;
  char configure_detected, configure_test;
  SRCINST_ERR err;

  if (packagename) {
    srcinst_warning(SRCINST_ERR_INVALID, "ignoring package name", packagename);
  }

  if (!(h = srcinst_new_package(0))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not create temp package", 0);
    return 0;
  }

  if ((err = srcinst_init_build(h, source, o.subdir)) != SRCINST_ERR_OK) {
    srcinst_warning(err, "could not initialize temporary build", 0);
    srcinst_close_package(h);
    return 0;
  }

  configure_detected = srcinst_detect_autoconf(h, &version) == SRCINST_ERR_OK;

  if (configure_detected) {
    configure_test = srcinst_configure(h, o.prefix, &(o.configure_strings)) == SRCINST_ERR_OK;
  }

  mf = srcinst_detect_makefile_features(h);

  printf("configure: ");
  if (!configure_detected) {
    printf("no\n");
  } else {
    if (version < 0.1)
      printf("custom\n");
    else
      printf("autoconf %f\n", version);
    
    printf("configure test: %s", configure_test ? "success" : "failure");
  }

  printf("\nmakefile: %s\n", mf ? "yes" : "no");

  if (mf) {
    printf("DESTDIR: %s\n",
	   srcinst_is_feature_supported(mf, SRCINST_FEATURE_DESTDIR) ?
	   "yes" : "no");
    printf("INSTALL_ROOT: %s\n",
	   srcinst_is_feature_supported(mf, SRCINST_FEATURE_INSTALL_ROOT) ?
	   "yes" : "no");
    printf("uninstall: %s\n",
	   srcinst_is_feature_supported(mf, SRCINST_FEATURE_UNINSTALL) ?
	   "yes" : "no");
    printf("install-strip: %s\n",
	   srcinst_is_feature_supported(mf, SRCINST_FEATURE_STRIP) ?
	   "yes" : "no");
  }

  if (mf)
    srcinst_free_makefile_features(mf);

  srcinst_close_package(h);
  return 1;
}

int detect_opts(char* packagename, char* source) {
  SRCINST_PHANDLE h; SRCINST_ERR err;
  struct srcinst_configure_sets option_sets; char options_detected;

  if (packagename) {
    srcinst_warning(SRCINST_ERR_INVALID, "ignoring package name", packagename);
  }

  if (!(h = srcinst_new_package(0))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not create temp package", 0);
    return 0;
  }

  if ((err = srcinst_init_build(h, source, o.subdir)) != SRCINST_ERR_OK) {
    srcinst_warning(err, "could not initialize temporary build", 0);
    srcinst_close_package(h);
    return 0;
  }

  options_detected =
    (err = srcinst_detect_configure_opt(h, &option_sets)) == SRCINST_ERR_OK;

  if (!options_detected) {
    srcinst_warning(err, "could not detect configure options", 0);
    
  } else {
    if (option_sets.packs.count > 0) {
      printf("Option class: Optional Packages\n");
      print_option_set(&option_sets.packs);
    }
    if (option_sets.feats.count > 0) {
      printf("Option class: Optional Features\n");
      print_option_set(&option_sets.feats);
    }
    if (option_sets.envs.count > 0) {
      printf("Option class: Influential Environment Variables\n");
      print_option_set(&option_sets.envs);
    }
    if (option_sets.olds.count > 0) {
      printf("Option class: Old-style (2.13) features/packages\n");
      print_option_set(&option_sets.olds);
    }
    if (option_sets.res1.count > 0) {
      printf("Option class: uncategorized\n");
      print_option_set(&option_sets.res1);
    }

    srcinst_free_configure_opt(&option_sets);
  }
  
  srcinst_close_package(h);
  return 1;
}

int describe_package(char* packagename, char* string) {
  SRCINST_PHANDLE h; int rv;

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "PACKAGE_NAME", 0);
    return 0;
  }

  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }

  if (!string || string[0] == '\0') {
    rv = srcinst_set_descr(h, 0) && srcinst_set_descr_long(h, 0);

  } else {
    char* mark;
    
    if ((mark = strchr(string, '\n'))) {
      *mark = 0;
      rv = srcinst_set_descr(h, string) && srcinst_set_descr_long(h, mark + 1);
      
    } else {
      rv = srcinst_set_descr(h, string);
    }
  }

  return (srcinst_close_package(h) && rv);
}

int rename_package(char* packagename, char* newname) {
  SRCINST_PHANDLE h; int rv; 

  if (!packagename) {
    srcinst_warning(SRCINST_ERR_MISSING, "must specify a PACKAGE_NAME", 0);
    return 0;
  }

  if (!(h = srcinst_open_package(packagename))) {
    srcinst_warning(SRCINST_ERR_ACTION, "could not open package", packagename);
    return 0;
  }
  
  rv = srcinst_rename_package(h, newname) == SRCINST_ERR_OK;
  return (srcinst_close_package(h) && rv);
}

int parse_cmdline(int argc, char* argv[]) {
  int i, j, k, selected_action;
  struct option_data* o_set[] = {
    action_list, option_list, 0
  };

  o.package = 0;		/* no package name selected */
  selected_action = -1;
  
  if (argc < 2) {
    return -1;
  }

  for (i = 1; i < argc; i++) {
    int found = 0;

    if (argv[i][0] != '-') {
      if (o.package) {
	srcinst_error(SRCINST_ERR_BUSY, "package name already specified",
		      argv[i]);
      }
      o.package = srcinst_strdup(argv[i]);
      continue;
    }
    
    for (j = 0; o_set[j]; j++) {
      for (k = 0; o_set[j][k].short_option; k++) {
	struct option_data* data = o_set[j] + k;
	if ((argv[i][1] == data->short_option && argv[i][2] == '\0') ||
	    (argv[i][1] == '-' && 
	     strcmp(argv[i] + 2, data->long_option) == 0)) {
	  found = 1;
	  
	  if (o_set[j] == action_list) {
	    if (selected_action != -1) {
	      srcinst_error(SRCINST_ERR_BUSY, "multiple actions requested",
			    data->long_option);
	    }
	    selected_action = k;
	  }

	  switch (data->chosen) {
	    
	  case 0:
	    {
	      int x;
	      for (x = 0; data->params[x]; x++) {
		i++;
		if (i >= argc || (argv[i][0] == '-' && argv[i][1] != '\0')) {
		  srcinst_error(SRCINST_ERR_MISSING, 
				"required option parameter",
				data->long_option);
		}
		data->values[x] = argv[i];
	      }
	      data->chosen = 1;
	      break;
	    }
	  case 1:
	    srcinst_error(SRCINST_ERR_BUSY, "unique option already defined",
			  data->long_option);
	    
	  case 2:
	    {
	      if (++i >= argc) {
		srcinst_error(SRCINST_ERR_MISSING, "required string for",
			      data->long_option);
	      }
	      
	      srcinst_add_string_list((struct srcinst_string_list*)
				      (data->values[0]), argv[i]);
	      break;
	    }
	  default:
	    ;
	    /* assert(0); */
	  }
	  
	  goto arg_found;
	}
      }
    }

    srcinst_error(SRCINST_ERR_INVALID, "unrecognized option:", argv[i]);

  arg_found:
    ;
  }

  if (selected_action == -1) {
    /* no action is selected */

    if (o.package) {
      /* if non-option argument is specified,
	 then provide partial backward compatibility:
	 default action is Add with a single filename argument */

      selected_action = 3;	/* warning! add must be at position 3!  */
      action_list[selected_action].values[0] = o.package;
      o.package = 0;		/* reset o.package to 'default package name' */
      return selected_action;
    }
    
    return -1;
  }

  /* some action selected */

  switch (action_list[selected_action].short_option) {
  case 'h': cmdline_help();
  case 'V': cmdline_version();
  }
  
  return selected_action;
}

void cmdline_help_usage(void) {
  printf("GNU Source Installer version %s usage:\n\n", PACKAGE_VERSION);
  printf("%s ACTION [PARAM...] [OPTION [PARAM...]]... [PACKAGE_NAME]\n\n", program_name);
  printf("%s [OPTION [PARAM...]] FILENAME\n", program_name);
  printf("    if no action is specified, add FILENAME using defaults.\n");
}

void cmdline_help(void) {
  cmdline_help_usage();
  printf("\nACTIONS:\n");
  cmdline_help_options(action_list);
  printf("\nOTHER OPTIONS:\n");
  cmdline_help_options(option_list);
  printf("\nSend bug reports to " PACKAGE_BUGREPORT " .\n");
  exit(0);
}

void cmdline_help_options(struct option_data* o) {
  int i, j, limit;
  for (i = 0; o[i].short_option; i++) {
    char* help; char line[80];
    help = o[i].help;
    memset(line, 0, sizeof(line));

    sprintf(line, "-%c, --%s", o[i].short_option, o[i].long_option);
    
    if (o[i].params) {
      for (j = 0; o[i].params[j]; j++) {
	sprintf(line + strlen(line), " %s", o[i].params[j]);
      }

      if ((j = strlen(line)) < 30) {
	memset(line + j, ' ', 30 - j); j = 30;
      }

      printf("%s", line);
      limit = 79 - j;
      
      while (strlen(help) > limit) {
	char* pos, *nextpos, *strlimit;
	pos = help; strlimit = help + limit;
	    
	while ((nextpos = strchr(pos + 1, ' ')) && nextpos < strlimit) {
	  pos = nextpos;
	}
	
	fwrite(help, 1, pos - help, stdout);
	putchar('\n');

	memset(line, ' ', 32); line[32] = '\0';
	printf("%s", line);
	limit = 79 - 32;
	help = pos + 1;
      }

      printf("%s\n", help);
    }
  }
}

void cmdline_version(void) {
  printf("GNU Source Installer version %s\n", PACKAGE_VERSION);
  printf("Copyright (c) 2005,2006,2007 Claudio Fontana\n");
  printf("This is Free Software that comes with NO WARRANTY;\n");
  printf("(see file COPYING for more details)\n");
  exit(0);
}

int cmdline_options_apply(int selected_action) {
  int action_code; struct option_data* data; int i;
  
  action_code = action_list[selected_action].short_option;

  for (i = 0; (data = option_list + i)->short_option; i++) {

    if (data->chosen == 1) {
      switch (data->short_option) {
      case 'S':
	o.source = 0; break;
      case 'z':
	if ((o.src_compress = srcinst_str_to_compression_format(data->values[0])) == SRCINST_COMP_UNKNOWN)
	  srcinst_error(SRCINST_ERR_INVALID, "compression format",
			data->values[0]);
	break;
      case 't':
	o.strip = 1; break;
   /* case 'C':
      o.configure_strings is already set during option parsing; */
      case 'p':
	o.prefix = data->values[0]; break;
      case 'D':
	o.subdir = data->values[0]; break;
      case 'f':
	o.force = 1; break;
      case 'U':
	o.user = data->values[0]; break;
      case 'P':
	o.pass = data->values[0]; break;
      case 'q':
	o.quiet = 1; break;
      case 'Q':
	o.quiet = 0; break;
      }
    }
  }

  srcinst_log_stdout(!o.quiet);
  srcinst_log_stderr(1);
  srcinst_log_spawn(!o.quiet);

  data = action_list + selected_action;

  switch (data->short_option) {
  case 'r':
    return remove_package(o.package);
    
  case 'a':
    return add_package(o.package, data->values[0]);
    
  case 'l':
    return list_packages(o.package);
    
  case 'c':
    return check_package(o.package);
    
  case 'i':
    return install_package(o.package);

  case 'u':
    return uninstall_binary(o.package);

  case 'g':
    return upgrade_package(o.package, data->values[0], data->values[1]);

  case 'd':
    return describe_package(o.package, data->values[0]);

  case 'n':
    return rename_package(o.package, data->values[0]);

  case 'X':
    return export(o.package, "xml", data->values[0]);

  case 'T':
    return export(o.package, "txt", data->values[0]);

  case 'L':
    return export(o.package, "lst", data->values[0]);

  case 'k':
    return export_binary(o.package, data->values[0]);

  case '1':
    return detect_builddirs(o.package, data->values[0]);

  case '2':
    return detect_feats(o.package, data->values[0]);

  case '3':
    return detect_opts(o.package, data->values[0]);
  }
  return 0;
}

int main(int argc, char* argv[]) {
  int rv, todo;

  if (!srcinst_init(SRCINST_FLAG_LOAD_ONDEMAND)) {
    srcinst_error(SRCINST_ERR_CORE, "could not initialize libsrcinst", 0);
  }
  atexit(srcinst_fini);
  program_name = (argv[0] ? srcinst_basename(argv[0]) : "sourceinstall");

  init_options(&o);
  
  if ((todo = parse_cmdline(argc, argv)) == -1) {
    srcinst_error(SRCINST_ERR_ACTION, "nothing to do. Try -h or --help", 0);
  }
  
  if (!(rv = cmdline_options_apply(todo))) {
    srcinst_error(SRCINST_ERR_ACTION, "failure", 0);
  }

  return 0;
}

void print_option_set(struct srcinst_configure_options* set) {
  int i;
  for (i = 0; i < set->count; i++) {
    printf("Option: %s", set->options[i].key);
    if (set->options[i].param) {
      printf("\t%s", set->options[i].param);
    }
    printf("\n");
    if (set->options[i].comment) {
      printf("Comment: %s", set->options[i].comment);
    }
    printf("\n");
  }
}

