/* Source Installer, Copyright (c) 2005,2006,2007 Claudio Fontana

 srcexport.c - export functions

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "package_info.h"
#include "actions.h"
#include "srcexport.h"

/* XML */

static SRCINST_ERR _export_info_xml_single(struct _package_info *, FILE *);
static SRCINST_ERR _export_info_xml_all(FILE *);
static int _export_info_xml_open(FILE *);
static int _export_info_xml_package(struct _package_info *, FILE *);
static char *_cdata(char *str);
static int _export_info_xml_close(FILE *);

/* TXT */

static SRCINST_ERR _export_info_txt_single(struct _package_info *, FILE *);
static SRCINST_ERR _export_info_txt_all(FILE *);
static int _export_info_txt_package(struct _package_info *, FILE *);

/* LST */

static SRCINST_ERR _export_info_lst_single(struct _package_info *, FILE *);

/* BINARY */
static int _export_binary_package_aux(struct _package_info *info,
				      char *filename, char *mirror,
				      SRCINST_COMP format);


SRCINST_ERR _action_export_info(struct _package_info *info, char *filename,
				SRCINST_EXPORT format)
{

    SRCINST_ERR rv;
    FILE *f;

    if (!
	(f =
	 (strcmp(filename, "-") == 0) ? stdout : fopen(filename, "w"))) {
	return SRCINST_ERR_ACTION;
    }

    switch (format) {
    case SRCINST_EXPORT_XML:
	rv = info ? _export_info_xml_single(info,
					    f) : _export_info_xml_all(f);
	break;

    case SRCINST_EXPORT_TXT:
	rv = info ? _export_info_txt_single(info,
					    f) : _export_info_txt_all(f);
	break;

    case SRCINST_EXPORT_LST:
	rv = (!info || !info->installed
	      || info->installed_files.count ==
	      0) ? SRCINST_ERR_INVALID : _export_info_lst_single(info, f);
	break;

    default:
	rv = SRCINST_ERR_INVALID;
    }

    if (rv == SRCINST_ERR_OK && ferror(f))
	rv = SRCINST_ERR_ACTION;

    if (f != stdout) {
	if (fclose(f) == EOF && rv == SRCINST_ERR_OK)
	    rv = SRCINST_ERR_ACTION;
    }

    return rv;
}

/* XML */


static SRCINST_ERR _export_info_xml_single(struct _package_info *info,
					   FILE * f)
{
    if (!(_export_info_xml_open(f)))
	return SRCINST_ERR_ACTION;

    if (!(_export_info_xml_package(info, f)))
	return SRCINST_ERR_ACTION;

    if (!(_export_info_xml_close(f)))
	return SRCINST_ERR_ACTION;

    return SRCINST_ERR_OK;
}

static SRCINST_ERR _export_info_xml_all(FILE * f)
{
    struct _package_element *this;

    if (!(_export_info_xml_open(f)))
	return SRCINST_ERR_ACTION;

    for (this = _srcinst_state.packages.first; this; this = this->next) {
	if (!(_export_info_xml_package(this->info, f)))
	    return SRCINST_ERR_ACTION;
    }

    if (!(_export_info_xml_close(f)))
	return SRCINST_ERR_ACTION;

    return SRCINST_ERR_OK;
}

static int _export_info_xml_open(FILE * f)
{
    fprintf(f, "%s\n",
	    "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\" ?>");
    fprintf(f, "%s\n",
	    "<!DOCTYPE package_set PUBLIC \"-//GNU//DTD Source Installer package set//EN\" \"http://www.gnu.org/software/sourceinstall/package_set.dtd\">");
    if (fprintf(f, "%s\n", "<package_set>") < 0)
	return 0;

    return 1;
}

static int _export_info_xml_package(struct _package_info *info, FILE * f)
{
    int i;

    if (!info->loaded) {
	if (!_load_package_info(info))
	    return 0;
    }

    fprintf(f, "<package name=\"_%s\">\n", info->name);

    for (i = 0; i < SRCINST_PACKAGE_INFO_NKEYS; i++) {
	struct srcinst_parsed_data *data;
	char *tmp, *lk;
	data = _keywords_package_info + i;
	lk = srcinst_lower(data->key);

	switch (i) {
	case SRCINST_IKEY_I:
	    if (info->installed && info->installed_files.count > 0) {
		int j;
		struct srcinst_string_list *l;
		l = &(info->installed_files);
		fprintf(f, "<installed_files>\n");

		for (j = 0; j < l->count; j++) {
		    tmp = _cdata(l->strings[j]);
		    fprintf(f, "<file>%s</file>\n", tmp);
		    free(tmp);
		}

		fprintf(f, "</installed_files>\n");
	    }
	    break;

	case SRCINST_IKEY_D:
	    if (info->long_description) {
		tmp = _cdata(info->long_description);
		fprintf(f, "<long_description>\n");
		fprintf(f, "%s\n", tmp);
		fprintf(f, "</long_description>\n");
		free(tmp);
	    }
	    break;

	case SRCINST_IKEY_SOURCE_AVAILABLE:
	    fprintf(f, "<%s>%s</%s>\n",
		    lk, info->source_location ? "yes" : "no", lk);
	    break;

	default:

	    switch (data->format[1]) {

	    case '%':
		/* nop */
		break;

	    case 's':
		if (*((char **) data->variable)) {
		    tmp = _cdata(*((char **) (data->variable)));
		    fprintf(f, "<%s>%s</%s>\n", lk, tmp, lk);
		    free(tmp);
		}
		break;

	    case 'd':
		{
		    char b;
		    b = *((char *) (data->variable));
		    fprintf(f, "<%s>%s</%s>\n", lk, b ? "yes" : "no", lk);
		    break;
		}

	    default:
		/* OFF_FMT */
		{
		    off_t value;
		    value = *((off_t *) data->variable);
		    if (value != (off_t) - 1)
			fprintf(f, "<%s>" OFF_FMT "</%s>\n", lk, value,
				lk);
		}
	    }
	}
	free(lk);
    }

    fprintf(f, "</package>\n");

    if (ferror(f))
	return 0;

    return 1;
}

#define CDATA_OPEN "<![CDATA["
#define CDATA_CLOSE "]]>"
#define CDATA_REP "]]]]><![CDATA[>"

static char *_cdata(char *str)
{
    char *cdata, *body;
    body = srcinst_replace(str, CDATA_CLOSE, CDATA_REP);
    cdata = srcinst_strjoin(CDATA_OPEN, body, CDATA_CLOSE, 0);
    free(body);
    return cdata;
}

#undef CDATA_OPEN
#undef CDATA_CLOSE
#undef CDATA_REP

static int _export_info_xml_close(FILE * f)
{
    if (fprintf(f, "%s\n", "</package_set>") < 0)
	return 0;

    return 1;
}


/* TXT */


static int _export_info_txt_package(struct _package_info *info, FILE * f)
{

    if (!info->loaded) {
	if (!_load_package_info(info))
	    return 0;
    }

    fprintf(f, "PACKAGE %s\n", info->name);
    fprintf(f, "----------------------------------------\n");

    if (info->description) {
	fprintf(f, "%s\n\n", info->description);
	if (info->long_description)
	    fprintf(f, "%s\n\n", info->long_description);
    }

    if (info->configured) {
	fprintf(f, "configured as: %s\n", info->configured);
    }

    fprintf(f, "source available: %s\n",
	    info->source_location ? "yes" : "no");

    if (info->source_location) {
	fprintf(f, "source location: %s\n", info->source_location);

	if (info->source_size != (off_t) - 1)
	    fprintf(f, "compressed source size (KB): " OFF_FMT "\n",
		    info->source_size);

	if (info->build_subdir)
	    fprintf(f, "build subdirectory: %s\n", info->build_subdir);
    }

    fprintf(f, "installed: %s\n", info->installed ? "yes" : "no");

    if (info->installed) {

	if (info->installed_size != (off_t) - 1)
	    fprintf(f, "installed size (KB): " OFF_FMT "\n",
		    info->installed_size);

	if (info->prefix)
	    fprintf(f, "installation prefix: %s\n", info->prefix);

	if (info->installed_files.count > 0) {
	    int i;
	    fprintf(f, "\ninstalled files and directories:\n");

	    for (i = 0; i < info->installed_files.count; i++) {
		fprintf(f, "%s\n", info->installed_files.strings[i]);
	    }
	} else {
	    fprintf(f, "no info about installed files available\n");
	}
    }

    if (fprintf(f, "\n----------------------------------------\n") < 0)
	return 0;

    return 1;
}

static SRCINST_ERR _export_info_txt_single(struct _package_info *info,
					   FILE * f)
{

    if (!_export_info_txt_package(info, f))
	return SRCINST_ERR_ACTION;

    return SRCINST_ERR_OK;
}

static SRCINST_ERR _export_info_txt_all(FILE * f)
{
    struct _package_element *this;

    for (this = _srcinst_state.packages.first; this; this = this->next) {
	if (!_export_info_txt_package(this->info, f))
	    return SRCINST_ERR_ACTION;
    }

    return SRCINST_ERR_OK;
}

/* LST */

static SRCINST_ERR _export_info_lst_single(struct _package_info *info,
					   FILE * f)
{
    int i;

    if (!info->loaded) {
	if (!_load_package_info(info))
	    return SRCINST_ERR_ACTION;
    }

    for (i = 0; i < info->installed_files.count; i++) {
	if (fprintf(f, "%s\n", info->installed_files.strings[i]) < 0)
	    return SRCINST_ERR_ACTION;
    }

    return SRCINST_ERR_OK;
}

/* BINARY */

SRCINST_ERR _action_export_binary(struct _package_info * info,
				  char *filename, SRCINST_COMP format)
{
    char *mirror, *oldcwd;
    SRCINST_ERR rv;

    if (!info->installed || !info->installed_files.count)
	return SRCINST_ERR_INVALID;

    if (!(filename = srcinst_file_normalize(filename))) {
	return SRCINST_ERR_ACTION;
    }

    rv = SRCINST_ERR_ACTION;

    oldcwd = srcinst_save_cwd();
    mirror = srcinst_strjoin(info->name, "-bin", 0);

    if (chdir(srcinst_get_dir(SRCINST_DIR_BUILD)) != 0)
	goto end_proc;

    if (!srcinst_spawn_wait(PRG_mkdir, mirror, 0))
	goto end_proc;

    if (_export_binary_package_aux(info, filename, mirror, format))
	rv = SRCINST_ERR_OK;

  end_proc:
    free(mirror);
    free(filename);
    srcinst_load_cwd(oldcwd);
    _clear_builddirs();
    return rv;
}

static int _export_binary_package_aux(struct _package_info *info,
				      char *filename, char *mirror,
				      SRCINST_COMP format)
{
    int i;

    if (chdir(mirror) != 0)
	return 0;

    /* recreate installation in mirror directory */

    for (i = 0; i < info->installed_files.count; i++) {
	char *this;
	SRCINST_TYPE ft;

	ft = srcinst_file_ltype(this = info->installed_files.strings[i]);

	if (ft == SRCINST_TYPE_DIR || ft == SRCINST_TYPE_FILE) {
	    uid_t user;
	    gid_t group;
	    mode_t perm;
	    if (!srcinst_file_owner(this, &user, &group) ||
		!srcinst_file_perm(this, &perm) ||
		(ft == SRCINST_TYPE_DIR &&
		 !srcinst_spawn_wait(PRG_mkdir, this + 1, 0)) ||
		(ft == SRCINST_TYPE_FILE &&
		 !srcinst_spawn_wait(PRG_cp, this, this + 1, 0)) ||
		chmod(this + 1, perm) != 0 ||
		chown(this + 1, user, group) != 0)
		return 0;

	} else if (ft == SRCINST_TYPE_LINK) {
	    char *target;
	    int rv;
	    if (!(target = srcinst_file_readlink(this)))
		return 0;

	    rv = symlink(target, this + 1);
	    free(target);

	    if (rv != 0)
		return 0;
	}
    }

    {
	/* create binary archive */
	char *tmp, *prg, *ext;
	int rv;
	tmp = srcinst_strjoin("../", mirror, ".tar", 0);
	rv = srcinst_spawn_wait(PRG_tar, "-cf", tmp, ".", 0);
	free(tmp);
	if (!rv || chdir("..") != 0)
	    return 0;
	prg =
	    format == SRCINST_COMP_GZ ? PRG_gzip :
	    format == SRCINST_COMP_BZ2 ? PRG_bzip2 :
	    format == SRCINST_COMP_Z ? PRG_compress : PRG_gzip;

	if (!(ext = srcinst_compression_format_to_ext(format)))
	    return 0;

	tmp = srcinst_strjoin(mirror, ".tar", 0);
	if (!srcinst_spawn_wait(prg, "-f", tmp, 0)) {
	    free(tmp);
	    return 0;
	}

	tmp = srcinst_append(tmp, ext);

	/* now move binary package to target */

	rv = srcinst_spawn_wait(PRG_mv, tmp, filename, 0);
	free(tmp);
	if (!rv)
	    return 0;
    }

    return 1;
}
