/* Source Installer, Copyright (c) 2005 Claudio Fontana

 actions.c - actions implementation

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "package_info.h"
#include "actions.h"

static int _configure(char** cflags);
/* static int _make_clean(char* makefile); */
static int _make_uninstall(struct _package_info* info, char* makefile);
static int _internal_uninstall(struct _package_info* info);
static int _get_relative_dependencies(struct srcinst_string_list* dep, struct _package_info* info);
static int _count_slashes(char* path);
static int _compare_dir_depth(char** path1, char** path2);
static int _compare_strcmp(char** str1, char** str2);

static int _make(char* makefile);
static int _make_install(struct _package_info* info, char* makefile, int strip, int force);
static char* _get_makefile(void);
static char* _keep_source(struct _package_info* info, SRCINST_COMP format);

static char* _unpack(char* filename);
static int _delete_source(char* filename);
static int _is_valid_package(char* filename);

static int _find_files_aux(struct srcinst_string_list* files, char* dirname, char** good_files);

static char* _access_string(char** set, int i);

static void _init_makefile_features(struct _makefile_features* mf);
static int _test_makefile_features(struct _makefile_features* mf, char* mfile);

static int _test_makefile_features_rec(struct _makefile_features* mf, char* mfile);

static void _init_makefile_features(struct _makefile_features* mf) {
  mf->features[SRCINST_FEATURE_DESTDIR].pattern = "\\$[({]?DESTDIR[)}]?";
  mf->features[SRCINST_FEATURE_DESTDIR].supported = 0;
  mf->features[SRCINST_FEATURE_INSTALL_ROOT].pattern = "\\$[({]?INSTALL_ROOT[)}]?";
  mf->features[SRCINST_FEATURE_INSTALL_ROOT].supported = 0;
  mf->features[SRCINST_FEATURE_UNINSTALL].pattern = "uninstall *:";
  mf->features[SRCINST_FEATURE_UNINSTALL].supported = 0;
  mf->features[SRCINST_FEATURE_STRIP].pattern = "install-strip *:";
  mf->features[SRCINST_FEATURE_STRIP].supported = 0;
  mf->features[SRCINST_FEATURE_DESCR].pattern = "description *:";
  mf->features[SRCINST_FEATURE_DESCR].supported = 0;
  mf->features[SRCINST_FEATURE_LONG_DESCR].pattern = "description-long *:";
  mf->features[SRCINST_FEATURE_LONG_DESCR].supported = 0;
}

static int _test_makefile_features(struct _makefile_features* mf, char* mfile) {
  int i; FILE* m; off_t m_size;
  char* buffer;
  
  _init_makefile_features(mf);

  if ((m_size = srcinst_file_size(mfile)) == (off_t) -1) {
    return 0;
  }

  buffer = srcinst_malloc(m_size + 1);

  if (!(m = fopen(mfile, "r")))
    { free(buffer); return 0; }
    
  if ((fread(buffer, 1, m_size, m)) != m_size)
    { free(buffer); fclose(m); return 0; }

  fclose(m); buffer[m_size] = 0;

  for (i = 0; i < SRCINST_FEATURE_N; i++) {
    SRCINST_REGEX preg;
    preg = srcinst_init_regex(mf->features[i].pattern, -1);
    mf->features[i].supported = srcinst_exec_regex(preg, buffer, 0);
    srcinst_free_regex(preg);
  }
  
  free(buffer);
  return 1;
}

static int _test_makefile_features_rec(struct _makefile_features* mf, char* makefile) {

  if (!(_test_makefile_features(mf, makefile)))
    return 0;

  if (!mf->features[SRCINST_FEATURE_DESTDIR].supported &&
      !mf->features[SRCINST_FEATURE_INSTALL_ROOT].supported) {
    struct srcinst_string_list makefiles;
    char* makefile_names[2];

    srcinst_init_string_list(&makefiles);
    makefile_names[0] = makefile; makefile_names[1] = 0;
    
    if (_find_files_aux(&makefiles, ".", makefile_names)) {
      struct _makefile_features mf2; int i;

      for (i = 0; i < makefiles.count; i++) {
	if (!_test_makefile_features(&mf2, makefiles.strings[i]))
	  break;
	if (mf2.features[SRCINST_FEATURE_DESTDIR].supported)
	  { mf->features[SRCINST_FEATURE_DESTDIR].supported = 1; break; }
	if (mf2.features[SRCINST_FEATURE_INSTALL_ROOT].supported)
	  { mf->features[SRCINST_FEATURE_INSTALL_ROOT].supported = 1; break; }
      }
    }
    srcinst_free_string_list(&makefiles);
  }
  return 1;
}

/* prepare source in the build directory for later use.
   Current working directory is set to the build subdir of the package.
   If 0 is passed as SUBDIR, no changes to the existing subdir are made.
 */

SRCINST_ERR _action_init_build(struct _package_info* info, char* filename, char* subdir) {
  int ft; SRCINST_ERR rv;
  char* fullname, *basename;

  if (!(fullname = srcinst_file_normalize(filename))) {
    /* the indicated source FILENAME is unreachable */
    return SRCINST_ERR_MISSING;
  }
  
  _srcinst_state.dirs[SRCINST_DIR_RUNBUILD] = srcinst_save_cwd();
  basename = base_name(fullname);
  ft = srcinst_file_type(fullname);
  
  if (info->packdir) {
    free(info->packdir); info->packdir = 0;
  }
  
  /* validate source filename */
  
  if (ft != SRCINST_TYPE_FILE && ft != SRCINST_TYPE_DIR) {
    rv = SRCINST_ERR_INVALID; goto end_proc;
  }

  if (!srcinst_file_access(fullname, "r")) {
    rv = SRCINST_ERR_PERM; goto end_proc;
  }
  
  if (ft == SRCINST_TYPE_FILE && !_is_valid_package(fullname)) {
    rv = SRCINST_ERR_INVALID; goto end_proc;
  }
  
  /* prepare build */

  if (chdir(_srcinst_state.dirs[SRCINST_DIR_BUILD]) != 0) {
    rv = SRCINST_ERR_CORE; goto end_proc;
  }
  
  if (!srcinst_spawn_wait(PRG_cp, "-R", fullname,
			  _srcinst_state.dirs[SRCINST_DIR_BUILD], 0)) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }
  
  if (!(info->packdir = 
	ft == SRCINST_TYPE_DIR ? strdup(basename) :
	_unpack(basename)
	)) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }
  
  if (!info->name) {
  /* set default package name, ensuring uniqueness */

    info->name = srcinst_strdup(info->packdir);
    while (_search_package_list(&_srcinst_state.packages, info->name)) {
      info->name = srcinst_append(info->name, "_new");
    }
  }
  
  if (chdir(info->packdir) != 0) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }
  
  if (subdir) {
    if (info->build_subdir)
      free(info->build_subdir);

    info->build_subdir = srcinst_strdup(subdir);
  }
  
  if (info->build_subdir && chdir(info->build_subdir) != 0) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }
  
  rv = SRCINST_ERR_OK;
  info->build_ready = 1;
  
 end_proc:
  
  if (fullname) free(fullname);
  
  if (rv != SRCINST_ERR_OK) {
    srcinst_load_cwd(_srcinst_state.dirs[SRCINST_DIR_RUNBUILD]);
    _srcinst_state.dirs[SRCINST_DIR_RUNBUILD] = 0;
  }
  return rv;
}

SRCINST_ERR _action_chdir_build(struct _package_info* info, char* subdir) {
  SRCINST_ERR rv; char* target;

  if (!_check_build_ready(info))
    return SRCINST_ERR_INVALID;
  
  target = srcinst_strjoin(_srcinst_state.dirs[SRCINST_DIR_BUILD], "/",
			   info->packdir, "/", subdir ? subdir : "", 0);
  if (chdir(target) != 0) {
    rv = SRCINST_ERR_ACTION;
    goto end_proc;
  }
  
  if (info->build_subdir) {
    free(info->build_subdir);
  }

  info->build_subdir = subdir ? srcinst_strdup(subdir) : 0;

  rv = SRCINST_ERR_OK;

 end_proc:
  free(target);
  return rv;
}

SRCINST_ERR _action_fini_build(struct _package_info* info) {

  if (!info->build_ready)
    return SRCINST_ERR_INVALID;

  srcinst_load_cwd(_srcinst_state.dirs[SRCINST_DIR_RUNBUILD]);
  _srcinst_state.dirs[SRCINST_DIR_RUNBUILD] = 0;
  
  _clear_builddirs();
  if (info->packdir) {
    free(info->packdir); info->packdir = 0;
  }
  info->build_ready = 0;

  return SRCINST_ERR_OK;
}

SRCINST_ERR _action_install_binary(struct _package_info* info, int strip, int force) {
  char* makefile; SRCINST_ERR rv;
  
  if (!_check_build_ready(info))
    return SRCINST_ERR_INVALID;

  if (!(makefile = _get_makefile())) {
    return SRCINST_ERR_MISSING;
  }

  if (!_make(makefile)) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }

  if (!_make_install(info, makefile, strip, force)) {
    rv = SRCINST_ERR_ACTION; goto end_proc;
  }

  info->installed = 1;
  rv = SRCINST_ERR_OK;
  info->onclose |= SRCINST_ONCLOSE_UPDATE;

 end_proc:
  free(makefile);
  return rv;
}

SRCINST_ERR _action_install_source(struct _package_info* info, SRCINST_COMP format) {
  char* previous_source, *tmp;

  if (!_check_build_ready(info))
    return SRCINST_ERR_INVALID;

  previous_source = info->source_location;
  if (!(tmp = _keep_source(info, format)))
    return SRCINST_ERR_ACTION;

  info->source_location = tmp;
  info->source_size = srcinst_file_size(tmp);
  
  if (previous_source) {
    if (strcmp(info->source_location, previous_source) != 0) {
      _delete_source(previous_source);
    }
    free(previous_source);
  }

  info->onclose |= SRCINST_ONCLOSE_UPDATE;
  
  return SRCINST_ERR_OK;
}

static char* _keep_source(struct _package_info* info, SRCINST_COMP fmt) {
  char* source_location;
  char* tarname, *zname, *cmd;
  source_location = zname = 0;

  if (chdir(_srcinst_state.dirs[SRCINST_DIR_BUILD]) != 0) {
    return 0;
  }
  
  tarname = srcinst_strjoin(info->name, ".tar", 0);
  if (!srcinst_spawn_wait(PRG_tar, "-cf", tarname, info->packdir, 0))
    goto end_proc;

  switch (fmt) {
  case SRCINST_COMP_BZ2: cmd = PRG_bzip2; break;
  case SRCINST_COMP_Z: cmd = PRG_compress; break;
  case SRCINST_COMP_GZ: cmd = PRG_gzip; break;
  default:
    goto end_proc;
  }

  if (!srcinst_spawn_wait(cmd, "-f", tarname, 0))
    goto end_proc;

  zname = srcinst_strjoin(tarname, srcinst_compression_format_to_ext(fmt), 0);

  if (!srcinst_spawn_wait(PRG_mv, zname, _srcinst_state.dirs[SRCINST_DIR_SRC], 0))
    goto end_proc;

  if (chdir(info->packdir) != 0)
    goto end_proc;
    
  if (info->build_subdir && chdir(info->build_subdir) != 0)
    goto end_proc;

  source_location = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_SRC], zname);
  
 end_proc:
  if (zname)
    free(zname);
  if (tarname)
    free(tarname);

  return source_location;
}

SRCINST_ERR _action_uninstall_binary(struct _package_info* info) {
  int rv = 1;

  if (!info->installed)
    return SRCINST_ERR_INVALID;
  
  if (info->source_location) {
    char* makefile;

    if (!_check_build_ready(info))
      return SRCINST_ERR_INVALID;
  
    if (!(makefile = _get_makefile()))
      rv = 0;
    else {
      rv = _make_uninstall(info, makefile);
      free(makefile);
    }
  }

  if (!rv && _srcinst_state.sigint_seen) {
    return SRCINST_ERR_INTR;
  }

  if (info->installed_files.count > 0) {
    rv = _internal_uninstall(info);
    if (!rv && _srcinst_state.sigint_seen) {
      /* User interrupt detected, but at this point if the pack is conforming
	 it is probably too late. 
	 However, missing files will be detected and one should reinstall. */
      return SRCINST_ERR_INTR;
    }
  }

  if (rv) {
    info->installed = 0;
    info->installed_size = (off_t)-1;
    srcinst_free_string_list(&(info->installed_files));
    info->onclose |= SRCINST_ONCLOSE_UPDATE;
    return SRCINST_ERR_OK;
  }

  return SRCINST_ERR_ACTION;
}

/* warning: DEP string_list gets the (char**) ->strings allocated,
   but the single strings are mirrored from INFO. */

static int _get_relative_dependencies(struct srcinst_string_list* dep,
				      struct _package_info* info) {
  int i; size_t n;
  struct _package_list* packages;
  n = info->installed_files.count;
  
  packages = &_srcinst_state.packages;
  srcinst_mirror_string_list(dep, &(info->installed_files));
  
  for (i = 0; i < dep->count; i++) {
    SRCINST_TYPE ft;
    ft = srcinst_file_ltype(dep->strings[i]);
    
    if (ft != SRCINST_TYPE_FILE && ft != SRCINST_TYPE_LINK &&
	ft != SRCINST_TYPE_DIR) {
      /* only existing files, links and dirs are considered for dependencies.
         Mark element for removal */
      dep->strings[i] = 0; n--;

    } else {
      struct _package_element* this;
      int found = 0; 
      
      for (this = packages->first; this; this = this->next) {
	struct srcinst_string_list* this_list;
	/* of course I depend on myself, so skip this->info == info */
	if (this->info == info)
	  continue;

	if (!this->info->loaded) {
	  if (!_load_package_info(this->info)) {
	    free(dep->strings); dep->strings = 0; return 0;
	  }
	}
	
	if ((this_list = &(this->info->installed_files))->count > 0) {
	  if (srcinst_lsearch(this_list->strings, this_list->count, dep->strings[i]) != -1) {
	    found = 1;
	    break;
	  }
	}
      }

      if (!found) {
	/* there is no dependency for this file entry.
	   Mark element for removal */
	dep->strings[i] = 0; n--;
      }
    }
  }
  
  {
    char** new_strings; int i, j;
    new_strings = srcinst_malloc(sizeof(char*) * n);
    
    for (i = 0, j = 0; i < dep->count; i++) {
      if (dep->strings[i]) {
	new_strings[j++] = dep->strings[i];
      }
    }
    
    free(dep->strings);
    dep->strings = new_strings;
    dep->count = n;
  }
  
  return 1;
}

static int _count_slashes(char* path) {
  int count = 0;
  while ((path = strchr(path, '/'))) {
    count++; path++;
  }
  return count;
}

static int _compare_dir_depth(char** path1, char** path2) {
  int count1, count2;

  count1 = _count_slashes(*path1); count2 = _count_slashes(*path2);

  return count1 == count2 ? strcmp(*path1, *path2) : count1 - count2;
}

static int _compare_strcmp(char** str1, char** str2) {
  return strcmp(*str1, *str2);
}

static int _internal_uninstall(struct _package_info* info) {
  struct srcinst_string_list dep; int i;
  int result = 0;

  /* get a list of all file and link dependencies for the current package.
     DEP is mirrored, not deep copied. Only (char**)->strings is alloc-ed,
     and only on success. */
  if (!_get_relative_dependencies(&dep, info)) {
    return 0;
  }

  /* sorted, please */
  srcinst_qsort(dep.strings, dep.count, &_compare_strcmp);
    
  for (i = 0; i < info->installed_files.count; i++) {
    char* filename; SRCINST_TYPE ft;
    filename = info->installed_files.strings[i];
    ft = srcinst_file_ltype(filename);

    if ((ft == SRCINST_TYPE_FILE || ft == SRCINST_TYPE_LINK) &&
	srcinst_bsearch(dep.strings, dep.count, filename,
			&_access_string) == -1) {
	    
      if (!srcinst_spawn_wait(PRG_rm, "-f", filename, 0)) {
	goto end_proc;
      }
      
      *filename = '\0'; /* schedule for removal from the string list */
    }
  }

  srcinst_purge_string_list(&(info->installed_files), 0);

  /* now check empty directories. First order them by path length, so we
     check contained dirs before the container using inverse `for' loop. */

  srcinst_qsort(info->installed_files.strings, info->installed_files.count,
		&_compare_dir_depth);

  for (i = info->installed_files.count - 1; i >= 0; i--) {
    char* filename;
    filename = info->installed_files.strings[i];

    if (srcinst_file_ltype(filename) == SRCINST_TYPE_DIR) {
      struct srcinst_string_list list;
      if (!srcinst_glob_string_list(&list, filename, "*"))
	goto end_proc;
      
      if (list.count == 0) {
	/* remove empty directory */
	if (!srcinst_spawn_wait(PRG_rmdir, filename, 0))
	  goto end_proc;
	*filename = '\0';	/* schedule dir for removal from string list */
      }
      srcinst_free_string_list(&list);
    }
  }

  srcinst_purge_string_list(&(info->installed_files), 0);
  result = 1;

 end_proc:
  free(dep.strings);
  return result;
}

SRCINST_ERR _action_uninstall_source(struct _package_info* info) {
  SRCINST_ERR rv = SRCINST_ERR_OK;

  if (!info->source_location) {
    return SRCINST_ERR_INVALID;
  }
  
  if (_delete_source(info->source_location)) {
    info->source_size = (off_t)-1;
    free(info->source_location); info->source_location = 0;
    info->onclose |= SRCINST_ONCLOSE_UPDATE;
  }
  
  return rv;
}

static int _is_valid_package(char* fn) {
  char* ext, *fn2;
  int fmt; int rv = 0;
  
  if (*(ext = srcinst_file_ext(fn))) {
    if ((fmt = srcinst_ext_to_compression_format(ext)) != SRCINST_COMP_UNKNOWN) {
      if (fmt == SRCINST_COMP_TGZ || fmt == SRCINST_COMP_TBZ2)
	return 1;
      
      fn2 = srcinst_strdup(fn);
      *(srcinst_file_ext(fn2)) = '\0';

      if (*(ext = srcinst_file_ext(fn2))) {
	if ((fmt = srcinst_ext_to_archive_format(ext)) != SRCINST_ARC_UNKNOWN) {
	  rv = 1;
	}
      }
      
      free(fn2);

    } else if (srcinst_ext_to_archive_format(ext) != SRCINST_ARC_UNKNOWN) {
      /* uncompressed archive */
      rv = 1;
    }
  }

  return rv;
}

static int _delete_source(char* fn) {
  return srcinst_spawn_wait(PRG_rm, "-f", fn, 0);
}
 
static int _make_uninstall(struct _package_info* info, char* makefile) {

  if (!srcinst_spawn_wait(PRG_make, "-f", makefile, "uninstall", 0))
    return 0;

  /* remove from the list of installed files those that were successfully
     removed by make uninstall */

  srcinst_purge_string_list(&(info->installed_files), 1);
  return 1;
}

/* allocs return value string on the stack. Returns (char*)0 on failure */

static char* _get_makefile(void) {
  char* filenames[] = { "makefile", "Makefile", "GNUmakefile", 0 };
  char* extensions[] = { "", ".unix", ".UNIX", ".Unix", ".linux", ".LINUX", ".Linux", ".cvs", ".CVS", 0 };
  
  char** fp, **ep;

  for (ep = extensions; *ep; ep++) {
    for (fp = filenames; *fp; fp++) {
      char* fn;
      fn = srcinst_strjoin(*fp, *ep, 0);
      if (srcinst_file_type(fn) == SRCINST_TYPE_FILE &&
	  srcinst_file_access(fn, "r")) {
	return fn;
      }
      free(fn);
    }
  }
  
  return 0;
}

static char* _unpack(char* filename) {
  SRCINST_COMP format;
  char* rep, *ext, *cmd, *arg1, *packdir;

  rep = 0;
  filename = srcinst_strdup(filename);
  ext = srcinst_file_ext(filename);

  format = srcinst_ext_to_compression_format(ext);
  
  if (format != SRCINST_COMP_UNKNOWN) {

    switch (format) {
    case SRCINST_COMP_GZ:
      cmd = PRG_gunzip; break;
      
    case SRCINST_COMP_BZ2:
      cmd = PRG_bunzip2; break;
      
    case SRCINST_COMP_Z:
      cmd = PRG_gunzip; break;

    case SRCINST_COMP_TGZ:
      rep = ".tar"; cmd = PRG_gunzip; break;
      
    case SRCINST_COMP_TBZ2:
      rep = ".tar"; cmd = PRG_bunzip2; break;

    default:
      free(filename);
      return 0;			/* unknown compression format */
    }
    
    if (!srcinst_spawn_wait(cmd, "-f", filename, 0)) {
      return 0;
    }
    
    if (rep) {
      strcpy(ext, rep);

    } else {
      *ext = 0;
    }
  }

  ext = srcinst_file_ext(filename);

  switch (srcinst_ext_to_archive_format(ext)) {
    
  case SRCINST_ARC_TAR:
    cmd = PRG_tar; arg1 = "-xvf"; break;
    
  case SRCINST_ARC_SHAR:
    cmd = PRG_sh; arg1 = 0; break;
    
  case SRCINST_ARC_ZIP:
    cmd = PRG_unzip; arg1 = "-o"; break;

  default:
    cmd = PRG_tar; arg1 = "-xvf"; break;
  }

  packdir = srcinst_spawn_wait_expect("([^/ ]+)/", cmd, arg1 ? arg1 : filename,
				      arg1 ? filename : 0, 0);
  free(filename);
    
  return packdir;
}

/* decide what to do based on current state */

int _update_package(struct _package_list* l, struct _package_element* e) {
  
  /* update package information on the disk based on information stored
     in the INFO structure */
  
  int rv; 
  struct _package_info* info;
  
  info = e->info;

  if (!info->source_location && !info->installed) {
    /* we have to remove the package. First from the filesystem, if it is
       persistant */

    rv = _unlink_package_info(e->info); /* from the filesystem */
    _remove_package_list(l, e);         /* then from memory */

  } else {
    rv = _save_package_info(e->info);
  }

  return rv;
}

SRCINST_ERR _action_configure(struct _package_info* info, char* prefix, struct srcinst_string_list* argv) {
  struct srcinst_string_list tmp; char* str;
  
  if (!_check_build_ready(info))
    return SRCINST_ERR_INVALID;
  
  if (!_detect_configure())
    return SRCINST_ERR_MISSING;

  srcinst_copy_string_list(&tmp, argv);
  str = srcinst_strjoin("--prefix=", prefix, 0);
  srcinst_add_string_list(&tmp, str);
  free(str);
  
  srcinst_add_string_list(&tmp, 0); /* add a temporary (char*)0 at the end */

  if (!_configure(tmp.strings)) {
    srcinst_free_string_list(&tmp);
    return SRCINST_ERR_ACTION;
  }
  
  if (info->configured)
    free(info->configured);
  if (info->prefix)
    free(info->prefix);
  
  tmp.count--;			/* hack to quickly strip the last (char*)0 */
      
  info->configured = srcinst_collapse_string_list(&tmp, " ");
  info->prefix = srcinst_strdup(prefix);
  srcinst_free_string_list(&tmp);

  /* reconfiguring in the temporary build is not enough to require a package
     update. The installation that eventually follows does.
     info->onclose |= SRCINST_ONCLOSE_UPDATE;
  */
  
  return SRCINST_ERR_OK;
}

SRCINST_ERR _action_rename_package(struct _package_element* e, char* newname) {
  char* oldname; char* new_l;

  if (strcmp(newname, e->info->name) == 0)
    return SRCINST_ERR_OK;

  new_l = 0;
    
  if (e->info->source_location) {
    new_l = srcinst_replace(e->info->source_location, e->info->name, newname);
    if (!srcinst_spawn_wait(PRG_mv, e->info->source_location, new_l, 0)) {
      free(new_l); return SRCINST_ERR_ACTION;
    }
    free(e->info->source_location);
    e->info->source_location = srcinst_strdup(new_l);
    free(new_l);
  }

  oldname = e->info->name;
  e->info->name = newname;

  if (!_save_package_info(e->info)) {
    e->info->name = oldname;		/* fallback to a safe state */
    return SRCINST_ERR_ACTION;
  }

  e->info->name = oldname;
  _unlink_package_info(e->info);	  /* remove old from the filesystem */
  
  e->info->name = srcinst_strdup(newname);
  _detach_package_list(&_srcinst_state.packages, e);
  _add_package_element(&_srcinst_state.packages, e);
  
  return SRCINST_ERR_OK;
}

int _detect_configure(void) {
  SRCINST_TYPE ft; int i;
  char* autogen_scripts[] = { 
    "./autogen.sh", "./autogen", "./bootstrap.sh", "./bootstrap", 0
  };
  
  if ((ft = srcinst_file_type("./configure")) == SRCINST_TYPE_FILE)
    return 1;

  for (i = 0; autogen_scripts[i]; i++) {
    if ((ft = srcinst_file_type(autogen_scripts[i])) == SRCINST_TYPE_FILE) {
      /* found a generation script */
      int executable;
      executable = srcinst_file_access(autogen_scripts[i], "x");
      if (srcinst_spawn_wait(executable ? autogen_scripts[i] : PRG_sh,
			     executable ? 0 : autogen_scripts[i], 0)) {
	if ((ft = srcinst_file_type("./configure")) == SRCINST_TYPE_FILE) {
	  return 1;
	}
      }
    }
  }
  return 0;
}

/* find all files whose names are present in GOOD_FILES (0 terminated),
   and which are 
   in directory DIRNAME or one of its subdirectories (recurs).
   Place all these files in the FILELIST string list. */

static int _find_files_aux(struct srcinst_string_list* filelist, char* dirname, char** good_files) {
  struct srcinst_string_list dirlist; int i, rv;

  for (i = 0; good_files[i]; i++) {
    char* fullname;
    fullname = srcinst_fnjoin(dirname, good_files[i]);
    if (srcinst_file_type(fullname) == SRCINST_TYPE_FILE) {
      srcinst_add_string_list(filelist, fullname);
      free(fullname); break;
    }
    free(fullname);
  }
  
  rv = 1;

  srcinst_init_string_list(&dirlist);
  if (!srcinst_glob_string_list(&dirlist, dirname, "*"))
    return 0;
  
  for (i = 0; i < dirlist.count; i++) {
    char* newdir;
    newdir = srcinst_fnjoin(dirname, dirlist.strings[i]);
    if (srcinst_file_type(newdir) == SRCINST_TYPE_DIR) {
      free(dirlist.strings[i]); dirlist.strings[i] = 0;
      rv = _find_files_aux(filelist, newdir, good_files);
    }
    free(newdir);
    if (!rv) break;
  }
  
  srcinst_free_string_list(&dirlist);
  return rv;
}

SRCINST_ERR _action_find_configure(struct _package_info* i, struct srcinst_string_list* scripts) {
  char* good_scripts[] = { 
    "configure", "autogen.sh", "autogen", "bootstrap.sh", "bootstrap", 0
  };

  if (!_check_build_ready(i))
    return SRCINST_ERR_INVALID;

  srcinst_init_string_list(scripts);
  if (!_find_files_aux(scripts, ".", good_scripts)) {
    srcinst_free_string_list(scripts);
    return SRCINST_ERR_ACTION;
  }

  return SRCINST_ERR_OK;
}

SRCINST_ERR _action_detect_autoconf(struct _package_info* i, float* v) {
  FILE* f; SRCINST_REGEX preg; SRCINST_MATCH m;
  char buffer[SRCINST_BUFSIZE]; size_t size; *v = 0.0;
  
  if (!_check_build_ready(i))
    return SRCINST_ERR_INVALID;

  if (!_detect_configure())
    return SRCINST_ERR_MISSING;

  if (!(f = fopen("./configure", "r")))
    return SRCINST_ERR_MISSING;
  
  size = fread(buffer, 1, sizeof(buffer) - 1, f);
  fclose(f);

  if (size == (size_t)0)
    return SRCINST_ERR_ACTION;
  
  buffer[size] = '\0';
  preg = srcinst_init_regex("autoconf[^[:digit:]]+([[:digit:]]+\\.[[:digit:]]+)", REG_EXTENDED | REG_ICASE | REG_NEWLINE);
  m = srcinst_init_match(2);

  if (srcinst_exec_regex(preg, buffer, m)) {
    sscanf(buffer + srcinst_get_match_offset(m, 1), "%f", v);
  }
  
  srcinst_free_match(m); srcinst_free_regex(preg);
  return SRCINST_ERR_OK;
}

struct _makefile_features* _action_detect_makefile_features(struct _package_info* i) {
  struct _makefile_features* mf;
  char* makefile;

  if (!_check_build_ready(i))
    return 0;

  if (!(makefile = _get_makefile()))
    return 0;

  mf = srcinst_malloc(sizeof(struct _makefile_features));

  if (!_test_makefile_features_rec(mf, makefile))
    { free(mf); mf = 0; }

  free(makefile);
  return mf;
}

static int _make(char* makefile) {
  return srcinst_spawn_wait(PRG_make, "-f", makefile, 0);
}

static int _make_install(struct _package_info* info, char* makefile, int strip, int force) {
  char* destdir_variable, *install_string;
  int descr_supported, descr_long_supported;
  int destdir_supported, install_root_supported, strip_supported;

  struct _makefile_features mf;

  if (!(_test_makefile_features_rec(&mf, makefile)))
    return 0;
  
  descr_supported = mf.features[SRCINST_FEATURE_DESCR].supported;
  if (descr_supported && !info->description)
    info->description = srcinst_spawn_wait_expect_all(PRG_make, "-f", makefile, "description", 0);

  descr_long_supported = mf.features[SRCINST_FEATURE_LONG_DESCR].supported;
  if (descr_long_supported && !info->long_description)
    info->long_description = srcinst_spawn_wait_expect_all(PRG_make, "-f", makefile, "description-long", 0);
  
  destdir_supported = mf.features[SRCINST_FEATURE_DESTDIR].supported;
  install_root_supported = mf.features[SRCINST_FEATURE_INSTALL_ROOT].supported;
  strip_supported = mf.features[SRCINST_FEATURE_STRIP].supported;

  install_string = (strip_supported && strip ? "install-strip" : "install");

  if (destdir_supported) {
    destdir_variable = "DESTDIR";

  } else if (install_root_supported) {
    destdir_variable = "INSTALL_ROOT";

  } else if (force) {
    destdir_variable = 0;
    
  } else {
    srcinst_warning(SRCINST_ERR_ACTION, "DESTDIR nor INSTALL_ROOT supported, and not forced", 0);
    return 0;
  }

  if (destdir_variable) {
    char* destdir_arg, *pattern; int rv;
    struct srcinst_string_list installed_files;
    destdir_arg = srcinst_strjoin(destdir_variable, "=",
				  _srcinst_state.dirs[SRCINST_DIR_DEST], 0);
    
    /* do a test install in destdir */
    rv = srcinst_spawn_wait(PRG_make, "-f", makefile, install_string,
			    destdir_arg, 0);
    free(destdir_arg);

    if (!rv) {
      return 0;
    }

    /* fetch installed file list */
    pattern = srcinst_strjoin("^(", _srcinst_state.dirs[SRCINST_DIR_DEST], ")/", 0);

    rv = srcinst_spawn_wait_expect_strip(pattern, &installed_files, PRG_find,
					 _srcinst_state.dirs[SRCINST_DIR_DEST], 0);
    free(pattern);
    if (!rv) {
      return 0;
    }

    srcinst_free_string_list(&(info->installed_files));
    srcinst_mirror_string_list(&(info->installed_files), &installed_files);
    free(installed_files.strings);

    /* calc installation disk space usage in KB (approx) */
           
    pattern = srcinst_spawn_wait_expect("([[:digit:]]+)", PRG_du, "-b", "-s",
					_srcinst_state.dirs[SRCINST_DIR_DEST], 0);
    
    if (pattern) {
      sscanf(pattern, OFF_FMT, &(info->installed_size));
    } else {
      info->installed_size = -1;
    }
  }

  /* now do the real install */
  return srcinst_spawn_wait(PRG_make, "-f", makefile, install_string, 0);
}

/*
static int _make_clean(char* makefile) {
  return srcinst_spawn_wait(PRG_make, "-f", makefile, "clean", 0);
}
*/

static int _configure(char** cflags) {
  char* makefile;

  if (!srcinst_spawn_waitve(cflags[0], cflags))
    return 0;
  
  /* check to ensure that a makefile was generated */
  if (!(makefile = _get_makefile()))
    return 0;

  free(makefile);
  return 1;
}

int _clear_builddirs(void) {

  if (!srcinst_spawn_wait(PRG_rm, "-fr", _srcinst_state.dirs[SRCINST_DIR_BUILD], 0))
    return 0;

  if (!srcinst_spawn_wait(PRG_rm, "-fr", _srcinst_state.dirs[SRCINST_DIR_DEST], 0))
    return 0;

  return srcinst_check_dirs();
}

int _check_build_ready(struct _package_info* info) {
  if (info->build_ready)
    return 1;

  if (!info->source_location)
    return 0;

  return _action_init_build(info, info->source_location, 0) == SRCINST_ERR_OK;
}

static char* _access_string(char** set, int i) {
  return set[i];
}
