/*
 * java2html.cc
 *
 * java2html
 * generates html code from Java code
 * Copyright (C) 1999  Lorenzo Bettini, lorenzo.bettini@penteres.it
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* 
   "Robert J. Clark" <clark@klgroup.com>
   added -input -output and -title options 

   "Jari Korva" <jari.korva@iki.fi>
   added -css option

   (see CREDITS and mails.txt)
*/

#include <stdio.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>

#include "version.h"
#include "java2html.h"
#include "colors.h"
#include "tags.h"
#include "keys.h"
#include "textgen.h"
#include "decorators.h"
#include "generators.h"
#include "messages.h"

/* global symbols */

char *inputFileName ; /* what we're reading  */
static char *docTitle;
ostream* sout ;

int tabSpaces = 0 ;     // space to substitue to tabs

extern int yylex() ;
extern int yyrestart(FILE *in) ;
extern int parseTags() ;

int main( int argc, char * argv[] ) {
  FILE *in = NULL;
  int entire_doc = 0 ; // we want a real html doc
  int otherArgs ;
  char *argString ;
  short verbose = 0 ;
  char *cssUrl = NULL ;
  int use_css = 0 ; // Use CSS instead of font-tags
  
  /* initialization of global symbols */
  inputFileName = NULL ;
  sout = NULL;
  docTitle = NULL;
  
  
  if ( argc == 2 &&
       (strcmp( argv[1], "-help" ) == 0 ||
        strcmp( argv[1], "?" ) == 0 ) ) {
    cerr << "Syntax : java2html [-input infile] [-output outfile] [-doc] [-css [<cssurl>]] [-tab=#] [-title \"Document Title\"]" << endl ;
    cerr << "  -doc   : " <<
      "create a complete html doc (with <HTML>, <BODY>...\n";
    cerr << "  -tab=# : " <<
      "substitute a tab char with # spaces" << endl ;
    cerr << "  -css : Use CSS instead of font-tags in output" << endl <<
            "         <cssurl> is the URl for the CSS document" << endl;
    cerr << "  -title \"Title of HTML Document\" : \n" << 
            "           Text to use for the title of the HTML document (-doc forced)." << endl;
    cerr << "  -verbose : verbose execution" << endl ;
    cerr << "  If -input is not specified, stdin is assumed" <<
      endl;
    cerr << "  If -output is not specified, stdout is assumed"
         << endl;
    exit(0);
  }

  
  /*
   * Parse any command line options
   */
  otherArgs = 1;
  char *tabString = 0;
  
  while ( otherArgs < argc ) {
    argString = argv[ otherArgs ] ;
    if ( ( strcmp( argString, "-doc" ) == 0 ) ) {
      entire_doc = 1;
    }
    else if ( strncmp( argString, "-tab", 4 ) == 0 &&
              (tabString = strchr( argString, '=' )) !=
              0 ) {
      tabSpaces = atoi( &tabString[1] ) ;
    }
    else if (strncmp( argString, "-input", 6 ) == 0 ) {
      otherArgs++;
      if (otherArgs >= argc) {
        cerr << "No value supplied for -input option, option ignored" << endl;
        continue;
      }
      inputFileName = strdup( argv[otherArgs] ) ;
    }
    else if (strncmp( argString, "-output", 7 ) == 0) {
      otherArgs++;
      if (otherArgs >= argc) {
        cerr << "No value supplied for -output option, option ignored" << endl;
        continue;
      }
      sout = new ofstream(argv[otherArgs]) ;
      if ( ! sout ) {
        cerr << "Error in creating " <<
          argv[otherArgs] << " for output" << endl ;
        exit(1) ;
      }
    }
    else if (strncmp( argString, "-css", 4 ) == 0) {
      use_css = 1 ;
      if ( entire_doc ) {
        otherArgs++;
        if (otherArgs >= argc) {
          cerr << "No value supplied for -css option, option ignored" << endl;
          continue;
        }
        cssUrl = strdup (argv[otherArgs]);
      }
    }
    else if (strncmp( argString, "-title", 6 ) == 0) {
      otherArgs++;
      if (otherArgs >= argc) {
        cerr << "No value supplied for -title option, option ignored" << endl;
        continue;
      }
      docTitle = strdup (argv[otherArgs]);
      entire_doc = 1 ;
    }
    else if (strncmp( argString, "-verbose", 8 ) == 0) {
      otherArgs++;
      verbose = 1 ;
    }
    else {
      cerr << "Warning: unknown option \"" << argString <<
        "\"" << endl    ;
    }
    ++otherArgs ;
  }

  if ( verbose )
    setMessager( new DefaultMessages ) ;

  printMessage( JAVA2HTML_VERSION ) ;
  
  parseTags() ;

  if( use_css ) {
    createGeneratorsForCSS() ;
  }
  else {
    createGenerators() ;
  }
  
  if ( inputFileName ) {
      in = freopen (inputFileName, "r", stdin);
      if (!in) {
        cerr << "Error in opening " << inputFileName
             << " for input" << endl ;
        exit(1) ;
      }
  }

  /*
   * Use default values for any options not provided
   */
  if (sout == NULL) {
    sout = &cout;
  }
  if (in == NULL) {
    ; /* Well stdin already points to stdin so, .... */
  }
  if (docTitle == NULL) {
    docTitle = inputFileName; /* inputFileName may also be NULL,
                                 this is OK. */
  }
  
  if ( entire_doc ) {
    print_top( cssUrl );
  }

  printMessage( "translating Java code... ", cerr ) ;

  generateln( "<pre>" ) ;
  generateln( "<tt>" ) ;
  yylex() ;
  generateln( "</tt>" ) ;
  generateln( "</pre>" ) ;

  printMessage( "done !", cerr ) ;
  
  if ( entire_doc )
    print_bottom() ;
  
  return (0 );
}

void print_top( char *cssUrl ) {
  if( cssUrl == NULL ) {
    generateln( "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">" ) ;
  }
  else {
    generateln( "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\"");
    generateln( "    \"http://www.w3.org/TR/REC-html40/strict.dtd\">");
  }
  generateln( "<html>" ) ;
  generateln( "<head>" ) ;
  generateln( "<meta http-equiv=\"Content-Type\"" ) ;
  generateln( "content=\"text/html; charset=iso-8859-1\">" ) ;
  generate( "<meta name=\"GENERATOR\" content=\"" ) ;
  generate( JAVA2HTML_VERSION ) ;
  generate( "\nby Lorenzo Bettini, lorenzo.bettini@penteres.it" ) ;
  generateln( "\">" ) ;
  generate( "<title>" ) ;
  generate( ( docTitle ? docTitle : 
              ( inputFileName ? inputFileName : "Java source file" ) ) ) ;
  generateln( "</title>" ) ;
  if( cssUrl != NULL ) {
    generate( "<link rel=\"stylesheet\" href=\"" );
    generate( cssUrl );
    generateln( "\" type=\"text/css\">");
  }
  generateln( "</head>" ) ;
  if( cssUrl == NULL ) {
    generate( "<body bgcolor=\"#FFFFFF\" text=\"#000000\" link=\"#0000EE\" " ) ;
    generateln( "vlink=\"#551A8B\" alink=\"#FF0000\">" ) ;
  }
  else {
    generateln( "<body>" ) ;
  }
}

void print_bottom() {
  generateln( "</body>" ) ;
  generateln( "</html>" ) ;
}

void generate( const char *s ) {
  GlobalGenerator->generate(s) ;
}

void generateln( const char *s ) {
  GlobalGenerator->generateln(s) ;
}

void generateNewLine() {
  generateln( "" ) ;
}

void generateTab() {
  if ( tabSpaces )
    for ( register int i = 0 ; i < tabSpaces ; ++i )
      generate( SPACE_CHAR ) ;
  else
    generate( "\t" ) ;
}

void startComment( const char *s )
{
  CommentGenerator->beginText(s) ;
}

void endComment( const char *s )
{
  CommentGenerator->endText(s) ;
}

void generateComment( const char *s ) {
  CommentGenerator->generateEntire(s) ;
}

void startString( const char *s )
{
  StringGenerator->beginText(s) ;
}

void endString( const char *s )
{
  StringGenerator->endText(s) ;
}

void generateString( const char *s ) {
  StringGenerator->generateEntire(s) ;
}

void generateKeyWord( const char *s ) {
  KeywordGenerator->generateEntire(s) ;
}

void generateBaseType( const char *s ) {
  TypeGenerator->generateEntire(s) ;
}

void generateNumber( const char *s ) {
  NumberGenerator->generateEntire(s) ;
}

void startTAG( const char *tag, const char *attr, const char *val ) {
  (*sout) << "<" << tag ;
  if ( attr && val )
    (*sout) << " " << attr << "=" << val ;
  (*sout) << ">" ;
}

void endTAG( const char *tag ) {
  (*sout) << "</" << tag << ">" ;
}

void startColor( const char *color ) {
  startTAG( FONT_TAG, COLOR_TAG, color ) ;
}

void endColor() {
  endTAG( FONT_TAG ) ;
}
