/* 
   XGDrawingEngine.m

   Copyright (C) 1999 Free Software Foundation, Inc.

   Author:  Adam Fedor <fedor@gnu.org>
   Date: Jan 1999
   
   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "config.h"
#include "XGDrawingEngine.h"

#include "gnustep/xgps/XGContext.h"
#include <gnustep/xgps/XGContextPrivate.h>
#include "gnustep/xgps/XGGState.h"
#include <AppKit/NSAffineTransform.h>
#include <AppKit/PSOperators.h>
#include <AppKit/GSFontInfo.h>
#include <Foundation/NSDebug.h>
#include <Foundation/NSUserDefaults.h>

#define	CUR_CONTEXT	((XGContext*)[XGContext currentContext])

/* Initialize AppKit backend */
BOOL 
initialize_gnustep_backend(void)
{                           
  NSDebugLog(@"Initializing GNUstep GUI X/GPS backend.\n");

  [NSGraphicsContext setDefaultContextClass: [XGContext class]];
  [GSFontEnumerator setDefaultClass: [XGFontEnumerator class]];
  [GSFontInfo setDefaultClass: [XGFontInfo class]];

  /* Creating the context automatically makes it the default one */
  [XGContext defaultContextWithInfo: nil];

  if (GSDebugSet(@"XSynchronize") == YES)
    XSynchronize([XGContext currentXDisplay], True);

  [NSFontManager sharedFontManager];
  return YES;
}

/* Common graphics functions */
void
NSHighlightRect(NSRect aRect)       
{
  XGContext	*ctxt = CUR_CONTEXT;
  float		x, y, w, h;

  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect);
  h = NSHeight(aRect);

  DPScompositerect(ctxt, x, y, w, h, NSCompositeHighlight);
  [[[ctxt focusView] window] flushWindow];
}

void
NSRectFill(NSRect aRect)
{
  float x, y, w, h;

  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect);
  h = NSHeight(aRect);

  DPSrectfill(CUR_CONTEXT, x, y, w, h);
}

void
NSRectFillList(const NSRect *rects, int count)
{
  int i;
  
  for (i = 0; i < count; i++)
    NSRectFill(rects[i]);
}

void
NSRectFillListWithGrays(const NSRect *rects,const float *grays,int count) 
{
}

void
NSDrawButton(const NSRect aRect, const NSRect clipRect)
{
  float x, y, w, h, v;
  XGContext	*ctxt;
  XGGState	*current;

  if (!NSIsEmptyRect(clipRect) && NSIntersectsRect(aRect, clipRect) == NO)
    return;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

/* The lines drawn should be offset by half a linewidth inwards so
   that the entire line gets drawn inside the rectangle. */
  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect) - 1.0;
  h = NSHeight(aRect) - 1.0;
  v = 1.0;
  
  DPSsetlinewidth(ctxt, 1.0);

  if (current->viewIsFlipped == YES)
    {
      y += h;
      h = -h;
      v = -v;
    }
  else
    y++;

  // Fill the inner rectangle in light gray
  DPSsetgray(ctxt, 0.667);
  DPSrectfill(ctxt, x, y, w, h);

  // Top and left outer line in white
  DPSsetgray(ctxt, 1.0);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, 0, h);
  DPSrlineto(ctxt, w, 0);
  DPSstroke(ctxt);

  // Bottom and right inner line in dark grey
  DPSsetgray(ctxt, 0.333);
  DPSmoveto(ctxt, x+1, y+v);
  DPSrlineto(ctxt, w-2, 0);
  DPSrlineto(ctxt, 0, h-2*v);
  DPSstroke(ctxt);

  // Bottom and right outer line in black
  DPSsetgray(ctxt, 0.0);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, w, 0);
  DPSrlineto(ctxt, 0, h);
  DPSstroke(ctxt);
}

void
NSDrawGrayBezel(const NSRect aRect, const NSRect clipRect)
{
  float		x, y, w, h, v;
  XGContext	*ctxt;
  XGGState	*current;

  if (!NSIsEmptyRect(clipRect) && NSIntersectsRect(aRect, clipRect) == NO)
    return;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

/* The lines drawn should be offset by half a linewidth inwards so
   that the entire line gets drawn inside the rectangle. */
  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect) - 1.0;
  h = NSHeight(aRect) - 1.0;
  v = 1.0;
  
  if (current->viewIsFlipped == YES)
    {
      y += h;
      h = -h;
      v = -v;
    }
  else
    y++;

  DPSsetlinewidth(ctxt, 1.0);

  // Fill the inner rectangle in light gray
  DPSsetgray(ctxt, 0.667);
  DPSrectfill(ctxt, x, y, w, h);

  // Top and left outer line in dark grey
  DPSsetgray(ctxt, 0.333);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, 0, h);
  DPSrlineto(ctxt, w, 0);
  DPSstroke(ctxt);

  // Top and left inner line in black
  DPSsetgray(ctxt, 0.0);
  DPSmoveto(ctxt, x+1, y+v);
  DPSrlineto(ctxt, 0, h-2*v);
  DPSrlineto(ctxt, w-2, 0);
  DPSstroke(ctxt);

  // Bottom and right outer line in white
  DPSsetgray(ctxt, 1.0);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, w, 0);
  DPSrlineto(ctxt, 0, h-v);
  DPSstroke(ctxt);

  // Bottom and right inner line in light grey
  DPSsetgray(ctxt, 0.667);
  DPSmoveto(ctxt, x+1, y+v);
  DPSrlineto(ctxt, w-2, 0);
  DPSrlineto(ctxt, 0, h-2*v);
  DPSstroke(ctxt);
}

void
NSDrawGroove(const NSRect aRect, const NSRect clipRect)
{
  float		x, y, w, h, v;
  XGContext	*ctxt;
  XGGState	*current;

  if (!NSIsEmptyRect(clipRect) && NSIntersectsRect(aRect, clipRect) == NO)
    return;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

/* The lines drawn should be offset by half a linewidth inwards so
   that the entire line gets drawn inside the rectangle. */
  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect) - 1.0;
  h = NSHeight(aRect) - 1.0;
  v = 1.0;

  if (current->viewIsFlipped == YES)
    {
      y += h;
      h = -h;
      v = -v;
    }
  else
    y++;

  DPSsetlinewidth(ctxt, 1.0);

  // Fill the inner rectangle in light gray
  DPSsetgray(ctxt, 0.667);
  DPSrectfill(ctxt, x, y, w, h);

  // Top and left outer line in dark gray
  DPSsetgray(ctxt, 0.333);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, 0, h);
  DPSrlineto(ctxt, w, 0);
  DPSstroke(ctxt);

  // Bottom and right inner line in dark gray (too)
  DPSmoveto(ctxt, x+2, y+v);
  DPSrlineto(ctxt, w-3, 0);
  DPSrlineto(ctxt, 0, h-3*v);
  DPSstroke(ctxt);

  // Draw remaining white lines using a single white rect
  DPSsetgray(ctxt, 1.0);
  DPSrectstroke(ctxt, x+1, y-v, w, h);
}

void
NSFrameRect(const NSRect aRect)  
{
  float x, y, w, h;
  XGContext	*ctxt;
  XGGState	*current;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect);
  h = NSHeight(aRect);

  DPSsetlinewidth(ctxt, 1.0);
  DPSrectstroke(ctxt, x, y, w, h);
}

void
NSFrameRectWithWidth(const NSRect aRect, float frameWidth)
{
  float x, y, w, h;
  XGContext	*ctxt;
  XGGState	*current;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect);
  h = NSHeight(aRect);

  DPSsetlinewidth(ctxt, frameWidth);
  DPSrectstroke(ctxt, x, y, w, h);
}

void
NSRectClip(NSRect aRect)      
{
  XGContext *ctxt = CUR_CONTEXT;
  float x, y, w, h;

  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect);
  h = NSHeight(aRect);

  DPSrectclip(ctxt, x, y, w, h);
  DPSnewpath(ctxt);
}

void
NSRectClipList(const NSRect *rects, int count)
{
  int i;
  for (i=0; i<count; i++)
    NSRectClip(rects[i]);
}

void
XRRemoveClipPath(void)
{
}


//
// Optimize Drawing
//
void
NSEraseRect(NSRect aRect)
{
  XGContext	*ctxt;
  ctxt = CUR_CONTEXT;
  DPSgsave(ctxt);
  DPSsetgray(ctxt, 1.0);
  NSRectFill(aRect);
  DPSgrestore(ctxt);
}

void
NSDrawWhiteBezel(const NSRect aRect, const NSRect clipRect)
{
  float x, y, w, h, v;
  XGContext	*ctxt;
  XGGState	*current;

  if (!NSIsEmptyRect(clipRect) && NSIntersectsRect(aRect, clipRect) == NO)
    return;

  ctxt = CUR_CONTEXT;
  current = [ctxt xrCurrentGState];

/* The lines drawn should be offset by half a linewidth inwards so
   that the entire line gets drawn inside the rectangle. */
  x = NSMinX(aRect);
  y = NSMinY(aRect);
  w = NSWidth(aRect) - 1.0;
  h = NSHeight(aRect) - 1.0;
  v = 1.0;
  
  if (current->viewIsFlipped == YES)
    {
      y += h;
      h = -h;
      v = -v;
    }
  else
    y++;

  DPSsetlinewidth(ctxt, 1.0);

  // Fill the inner rectangle in white
  DPSsetgray(ctxt, 1.0);
  DPSrectfill(ctxt, x, y, w, h);

  // Top and left outer line in dark grey
  DPSsetgray(ctxt, 0.333);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, 0, h);
  DPSrlineto(ctxt, w, 0);
  DPSstroke(ctxt);

  // Top and left inner line in dark gray (too)
  DPSmoveto(ctxt, x+1, y+v);
  DPSrlineto(ctxt, 0, h-2*v);
  DPSrlineto(ctxt, w-2, 0);
  DPSstroke(ctxt);

  // Bottom and right outer line in white
  DPSsetgray(ctxt, 1.0);
  DPSmoveto(ctxt, x, y);
  DPSrlineto(ctxt, w, 0);
  DPSrlineto(ctxt, 0, h-v);
  DPSstroke(ctxt);

  // Bottom and right inner line in light grey
  DPSsetgray(ctxt, 0.667);
  DPSmoveto(ctxt, x+1, y+v);
  DPSrlineto(ctxt, w-2, 0);
  DPSrlineto(ctxt, 0, h-3*v);
  DPSstroke(ctxt);
}

void
NSDrawBezel(NSRect aRect, NSRect clipRect)
{
  NSDrawGrayBezel(aRect, clipRect);
}

//*****************************************************************************
//
// 	Draws an unfilled rectangle, clipped by clipRect, whose border
//	is defined by the parallel arrays sides and grays, both of length
//	count. Each element of sides specifies an edge of the rectangle,
//	which is drawn with a width of 1.0 using the corresponding gray level
//	from grays. If the edges array contains recurrences of the same edge,
//	each is inset within the previous edge.
//
//*****************************************************************************

NSRect
NSDrawTiledRects(NSRect boundsRect, NSRect clipRect,
  const NSRectEdge *sides, const float *grays, int count)
{
  return NSZeroRect;
}


//
// Read the Color at a Screen Position
//
NSColor*
NSReadPixel(NSPoint location)
{
  return nil;
}

//
// Text Functions
//
//
// Filter Characters Entered into a Text Object
//
unsigned short
NSEditorFilter(unsigned short theChar,
  int flags, NSStringEncoding theEncoding)
{
  return 0;
}

unsigned short
NSFieldFilter(unsigned short theChar,
 int flags, NSStringEncoding theEncoding)
{
  return 0;
}

//
// Calculate or Draw a Line of Text (in Text Object)
//
int
NSDrawALine(id self, NSLayInfo *layInfo)
{
  return 0;
}

int
NSScanALine(id self, NSLayInfo *layInfo)
{
  return 0;
}

//
// Calculate Font Ascender, Descender, and Line Height (in Text Object)
//
void
NSTextFontInfo(id fid, float *ascender, float *descender, float *lineHeight)
{}

//
// Access Text Object's Word Tables
//
NSData*
NSDataWithWordTable(const unsigned char *smartLeft,
  const unsigned char *smartRight,
  const unsigned char *charClasses,
  const NSFSM *wrapBreaks,
  int wrapBreaksCount,
  const NSFSM *clickBreaks,
  int clickBreaksCount,
  BOOL charWrap)
{
  return nil;
}

void
NSReadWordTable(NSZone *zone,
		     NSData *data,
		     unsigned char **smartLeft,
		     unsigned char **smartRight,
		     unsigned char **charClasses,
		     NSFSM **wrapBreaks,
		     int *wrapBreaksCount,
		     NSFSM **clickBreaks,
		     int *clickBreaksCount,
		     BOOL *charWrap)
{}

//
// Array Allocation Functions for Use by the NSText Class
//
NSTextChunk*
NSChunkCopy(NSTextChunk *pc, NSTextChunk *dpc)
{
  return NULL;
}

NSTextChunk*
NSChunkGrow(NSTextChunk *pc, int newUsed)
{
  return NULL;
}

NSTextChunk*
NSChunkMalloc(int growBy, int initUsed)
{
  return NULL;
}

NSTextChunk*
NSChunkRealloc(NSTextChunk *pc)
{
  return NULL;
}

NSTextChunk*
NSChunkZoneCopy(NSTextChunk *pc,
                             NSTextChunk *dpc,
                             NSZone *zone)
{
  return NULL;
}

NSTextChunk*
NSChunkZoneGrow(NSTextChunk *pc, int newUsed, NSZone *zone)
{
  return NULL;
}

NSTextChunk*
NSChunkZoneMalloc(int growBy, int initUsed, NSZone *zone)
{
  return NULL;
}

NSTextChunk*
NSChunkZoneRealloc(NSTextChunk *pc, NSZone *zone)
{
  return NULL;
}


//
// Imaging Functions
//
//
// Copy an image
//
void
NSCopyBitmapFromGState(int srcGstate, NSRect srcRect, NSRect destRect)
{
}

void
NSCopyBits(int srcGstate, NSRect srcRect, NSPoint destPoint)
{
  float x, y, w, h;

  x = NSMinX(srcRect);
  y = NSMinY(srcRect);
  w = NSWidth(srcRect);
  h = NSHeight(srcRect);

  DPScomposite(CUR_CONTEXT, x, y, w, h, srcGstate, destPoint.x, destPoint.y,
	       NSCompositeCopy);
}

//
// Play the System Beep
//
void
NSBeep(void)
{
  XBell([XGContext currentXDisplay], 50);
}

//
// Draw a Distinctive Outline around Linked Data
//
void
NSFrameLinkRect(NSRect aRect, BOOL isDestination)
{
}

float
NSLinkFrameThickness(void)
{
  return 0;
}

/* Wraps */
/* Graphics Wraps */
void 
GSWSetMatrix(GSCTXT *ctxt, float m[6])
{
  [[(XGContext *)ctxt xrCurrentGState] DPSsetmatrix: m];
}

/* Context helper wraps */
static unsigned int unique_index = 0;

unsigned int 
GSWDefineAsUserObj(GSCTXT *ctxt)
{
  [ctxt DPSsendint: ++unique_index];
  [ctxt DPSexch];
  [ctxt DPSdefineuserobject];
  return unique_index;
}

void
GSWViewIsFlipped(GSCTXT *ctxt, BOOL flipped)
{
  ([(XGContext *)ctxt xrCurrentGState])->viewIsFlipped = flipped;
}

void
GSWinitcontext(GSCTXT *ctxt, int window_number, GC xgc, Drawable drawable, 
	int xoff, int yoff)
{
  DPSsetgcdrawable(ctxt, xgc, (void *)drawable, xoff, yoff);
  DPSinitmatrix(ctxt);
  DPSinitclip(ctxt);
  [[(XGContext *)ctxt xrCurrentGState] setWindow: window_number];
}

//*****************************************************************************
//
//	X utility functions
//
//	_sendKeyCode	send a KeyCode to an X window
//	_sendKeysym		send a Keysym to an X window
//	_sendXString 	send a text string to an X window
//	_findXWindow	find lowest X window for a given global coordinate
//
//	inspired by E F Johnson's examples in Advanced X Window Applications
//	Programming, 2nd ed.
//
//*****************************************************************************

static int
_sendKeyCode(Display* display, Window window, KeyCode keycode, int state)
{
  static XKeyEvent event;	// Send a keycode to an X window
  int status;
  XEvent *xEvent;

  // fill in all fields of the event structure.
  event.type        = KeyPress;
  event.display     = display;
  event.window      = window;
  event.root        = RootWindow(display, DefaultScreen(display));
  event.keycode     = keycode;
  event.state       = state;
  event.time        = CurrentTime;
  event.same_screen = True;
  event.x           = 0;
  event.y           = 0;
  event.x_root      = 0;
  event.y_root      = 0;
  event.subwindow   = (Window)None;

  xEvent = (XEvent *)&event;

  status = XSendEvent(display, window, False, KeyPressMask, xEvent);

  // if send is successful send a KeyRelease event for each KeyPress event
  if (status != 0)
    {
      event.type = KeyRelease;
      event.time = CurrentTime;

      status = XSendEvent(display, window, True, KeyReleaseMask, xEvent);
    }

  return status;
}

static int
_sendKeysym(Display* display, Window window, KeySym keysym)
{
  KeyCode	keycode;		// send a KeySym to an X window
  int		status = 0;
  int		state;

  if (keysym != NoSymbol)
    {
      if ((keysym >= XK_A) && (keysym <= XK_Z))
	state = ShiftMask;
      else
	state = 0x00;

      keycode = XKeysymToKeycode(display, keysym);

      if (keycode != 0)
	status = _sendKeyCode(display, window, keycode, state);
    }

  return status;
}

int
_sendXString(Display* display, Window window, const char* string)
{
  int	i;
  int	status = 0;

  /*
   * send a text string to another X window as a series of KeyPress
   * and KeyRelease events
   */
  for (i = 0; string[i] != '\0'; i++)
    {
      if (string[i] == '\n')
	status = _sendKeysym(display, window, XK_Return);
      else
	status = _sendKeysym(display, window, string[i]);

      if (status == 0)		// return immediately on errors
	return status;
    }

  XFlush(display);

  return status;
}

