/* 
   PXKApplication.m

   NSApplication for GNUstep GUI X/DPS Backend

   The NSApplication class manages the main event loop of
   the program, keeps track of the application's windows
   and which one is the key window.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Ovidiu Predescu <ovidiu@net-community.com>
   Date: June 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: August 1998

   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <gnustep/xdps/stdpxk.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSRunLoop.h>
#include <Foundation/NSTimer.h>
#include <Foundation/NSNotification.h>

#include <AppKit/NSWindow.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSCStringText.h>						// For definitions of  													
														// some other keys
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKApplication.h>
#include <gnustep/xdps/PXKScreen.h>

#if LIB_FOUNDATION_LIBRARY
# include <Foundation/NSPosixFileDescriptor.h>
#elif defined(NeXT_PDO)
# include <Foundation/NSFileHandle.h>
# include <Foundation/NSNotification.h>
#endif

//
// Backend structure for PXKApplication
//
typedef struct _PXKApp_struct
{
  	Window grab_window;
	Display *xDisplay;

} PXKApp_struct;

#define XGRAB (((PXKApp_struct *)be_app_reserved)->grab_window)
#define PXKDISPLAY (((PXKApp_struct *)be_app_reserved)->xDisplay)

//
// Class variables
//
static NSEvent *nullEvent;


//
//  Private functions
//
static NSEvent*
process_key_event (XEvent* xEvent, PXKDPSContext* ctxt, NSEventType eventType);

static unsigned short 
process_key_code(XEvent *xEvent,KeySym keysym,unsigned int eventModifierFlags);

static unsigned int process_modifier_flags(unsigned int state);



@implementation PXKApplication

//
// Class methods
//
+ (void)initialize
{
	if (self == [PXKApplication class])
		[self setVersion:1];							// Initial version
}

//
// Instance methods
//
- init
{
	be_app_reserved = malloc(sizeof(PXKApp_struct));	// Allocate backend
	XGRAB = 0;											// structure

	[super init];

	PXKDISPLAY = [(PXKDPSContext*)[NSDPSContext currentContext] xDisplay];
	nullEvent = [NSApplication getNullEvent];			// store a ptr to the 
														// class NULL event
	return self;
}

- (void)dealloc
{
	free(be_app_reserved);								// Deallocate backend
														// structure
	[super dealloc];
}

- (void)run
{
Display* xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];

	[super run];
  					// xxx We should close the DPS context as well?
	XCloseDisplay(xDisplay);							// Close the X Display
}

- (NSEvent *)nextEventMatchingMask:(unsigned int)mask
			 			untilDate:(NSDate *)expiration
			    		inMode:(NSString *)mode
			   			dequeue:(BOOL)flag
{
NSEvent *event;
								// _flushWindows called before and after 
								// nextEventMatchingMask to approximate the
								// original mechanism in NSApplication
	[NSWindow _flushWindows];				// flush any windows that need it

	event = [super nextEventMatchingMask:mask
			 		untilDate:expiration
			    	inMode:mode
			   		dequeue:flag];

	[NSWindow _flushWindows];				// flush any windows that need it

	return event;
}

#if !defined(LIB_FOUNDATION_LIBRARY) && !defined(NeXT_PDO)
														// Assume gnustep-base 
- (void)readyForReadingOnFileDescriptor:(int)fd
{
	[self _nextEvent];
}
#endif

- (void)_nextEvent										// get next X event	
{											
NSEvent *e = nil;
static PXKDPSContext *context;
XEvent xEvent;
NSRect new_frame;
static NSRect xFrame;
static Time timeOfLastClick = 0;			
static int clickCount = 1;
NSPoint eventLocation;
static unsigned int eventFlags;
static unsigned int last_XEvent_state = -1;
PXKWindow *w = nil;
Window xWin;
static PXKWindow *window = nil;
static Window lastXWin = None;
static int windowNumber;
NSEventType eventType;
int count;
static unsigned long timeOfLastMotion = 0;

	while ((count = XPending (PXKDISPLAY)) > 0)		// if X events are pending
	{
//	fprintf(stderr,"PXKAppliation ((XPending count = %d): \n", count);
	do	{											// loop and grab all of the 
		NSDebugLog(@"Get next XWindows event\n");	// events from the X queue
		XNextEvent (PXKDISPLAY, &xEvent);			// else do nothing
		switch (xEvent.type) 
			{										// mouse button events 
  			case ButtonPress:
				NSDebugLog(@"ButtonPress: \
						xEvent.xbutton.time %u timeOfLastClick %u \n", 
						xEvent.xbutton.time, timeOfLastClick);
										// hardwired test for a double click
										// default of 300 should be user set;
										// under NS the windowserver does this
				if(xEvent.xbutton.time < (unsigned long)(timeOfLastClick +300))
					clickCount++;
				else
					clickCount = 1;							// reset the click
				timeOfLastClick = xEvent.xbutton.time;		// count

				eventType = (xEvent.xbutton.button == Button1 ? 
								NSLeftMouseDown : NSRightMouseDown);
				if(xEvent.xbutton.state != last_XEvent_state)
					{
					eventFlags = process_modifier_flags(xEvent.xbutton.state);
					last_XEvent_state = xEvent.xbutton.state;
					}
									// if pointer is grabbed use grab window
				xWin = (XGRAB == 0) ? xEvent.xbutton.window : XGRAB;
				if(xWin != lastXWin)						
					{										// find NSWindow
					window = [PXKWindow windowForXWindow:xWin];	
					windowNumber = [window windowNumber];
					lastXWin = xWin;
					context = (PXKDPSContext*)[NSDPSContext currentContext];
					}		
				xFrame = [window xFrame];
				eventLocation = [context userPointFromXPoint: 
										NSMakePoint(xEvent.xbutton.x, 
										xFrame.size.height-xEvent.xbutton.y)];
															
				e = [NSEvent mouseEventWithType:eventType	// create NSEvent	
								location:eventLocation
								modifierFlags:eventFlags
								timestamp:(NSTimeInterval)xEvent.xbutton.time
								windowNumber:windowNumber
								context:context
								eventNumber:xEvent.xbutton.serial
								clickCount:clickCount
								pressure:1.0];
//				fprintf(stderr, 
//						"PXKApplication button press, events in queue: %d \n", 
//						[event_queue count]);
    			break;
  			case ButtonRelease:
				eventType = (xEvent.xbutton.button == Button1 ?
								NSLeftMouseUp : NSRightMouseUp);
				if(xEvent.xbutton.state != last_XEvent_state)
					{
					eventFlags = process_modifier_flags(xEvent.xbutton.state);
					last_XEvent_state = xEvent.xbutton.state;
					}
									// if pointer is grabbed use grab window
				xWin = (XGRAB == 0) ? xEvent.xbutton.window : XGRAB;			
				if(xWin != lastXWin)						
					{										// find NSWindow
					window = [PXKWindow windowForXWindow:xWin];	
					windowNumber = [window windowNumber];
					lastXWin = xWin;
					context = (PXKDPSContext*)[NSDPSContext currentContext];
					}		
				xFrame = [window xFrame];
				eventLocation = [context userPointFromXPoint: 
										NSMakePoint(xEvent.xbutton.x, 
										xFrame.size.height- xEvent.xbutton.y)];
															
				e = [NSEvent mouseEventWithType:eventType	// create NSEvent	
								location:eventLocation
								modifierFlags:eventFlags
								timestamp:(NSTimeInterval)xEvent.xbutton.time
								windowNumber:windowNumber
								context:context
								eventNumber:xEvent.xbutton.serial
								clickCount:clickCount
								pressure:1.0];
				break;

			case CirculateNotify:							// change to the 
				NSDebugLog(@"CirculateNotify\n");			// stacking order
				break;
	
			case CirculateRequest:
				NSDebugLog(@"CirculateRequest\n");
				break;
	
			case ClientMessage:								// client events
				NSDebugLog(@"ClientMessage\n");
				break;
	
			case ColormapNotify:						// colormap attribute
				NSDebugLog(@"ColormapNotify\n");		// changes
				break;
												// the window has been resized,
												// change the width and height
			case ConfigureNotify:				// to send to draw_text and
												// draw_graphics in next Expose
				NSDebugLog(@"ConfigureNotify\n");
				new_frame.origin.x = (float)xEvent.xconfigure.x;
				new_frame.origin.y = (float)xEvent.xconfigure.y;
				new_frame.size.width = (float)xEvent.xconfigure.width;
				new_frame.size.height = (float)xEvent.xconfigure.height; 
				NSDebugLog(@"New frame %f %f %f %f\n", 
							new_frame.origin.x, new_frame.origin.y,
							new_frame.size.width, new_frame.size.height);
				NSDebugLog(@"border_width %d\n", 
							xEvent.xconfigure.border_width);
				w = [PXKWindow windowForXWindow:xEvent.xconfigure.window];
				if (xEvent.xconfigure.window == [(PXKWindow *)w xWindow])
					[(PXKWindow *)w setFrameFromXFrame:new_frame];
				NSDebugLog(@"after ConfigureNotify\n");
				break;								
													// same as ConfigureNotify		
			case ConfigureRequest:					// but we get this event
				NSDebugLog(@"ConfigureRequest\n");	// before the change has 
				break;								// actually occurred 

			case CreateNotify:							// a window has been
				NSDebugLog(@"CreateNotify\n");			// created
				break;
	
			case DestroyNotify:							// a window has been
				NSDebugLog(@"DestroyNotify\n");			// destroyed
				break;

			case EnterNotify:							// when the pointer
				NSDebugLog(@"EnterNotify\n");			// enters a window
				break;
	
			case LeaveNotify:							// when the pointer 
				NSDebugLog(@"LeaveNotify\n");			// leaves a window
				break;

			case Expose:							// a portion of the window
				{									// has become visible and
				XRectangle rectangle;				// we must redisplay it
	
				NSDebugLog(@"Expose\n");
														// find the NSWindow
				w = [PXKWindow windowForXWindow:xEvent.xexpose.window];
				[w setXExposed: YES];
	
				rectangle.x = xEvent.xexpose.x;
				rectangle.y = xEvent.xexpose.y;
				rectangle.width = xEvent.xexpose.width;
				rectangle.height = xEvent.xexpose.height;
				[w addExposedRectangle:rectangle];
		
				if (xEvent.xexpose.count == 0)
					[w processExposedRectangles];
				break;
				}

			case FocusIn:								// keyboard focus
				{										// entered a window
				PXKWindow *wind = nil;
														// Find the NSWindow
				NSDebugLog(@"FocusIn\n");				// and make it the key 
														// window 
				wind = [PXKWindow windowForXWindow:xEvent.xexpose.window];
				[wind makeKeyWindow];
				break;
				}
	
			case FocusOut:								// keyboard focus left
				NSDebugLog(@"FocusOut\n");				// a window
				break;
	
			case GraphicsExpose:
				NSDebugLog(@"GraphicsExpose\n");
				break;
	
			case NoExpose:
				NSDebugLog(@"NoExpose\n");
				break;

			case GravityNotify:						// window is moved because
				NSDebugLog(@"GravityNotify\n");		// of a change in the size
				break;								// of its parent
	
			case KeyPress:							// a key has been pressed
				context = (PXKDPSContext*)[NSDPSContext currentContext];
				e = process_key_event (&xEvent, context, NSKeyDown);
				break;
		
			case KeyRelease:						// a key has been released
				context = (PXKDPSContext*)[NSDPSContext currentContext];
				e = process_key_event (&xEvent, context, NSKeyUp);
				break;
													 
			case KeymapNotify:						// reports the state of the
				NSDebugLog(@"KeymapNotify\n");		// keyboard when pointer or
				break;								// focus enters a window
	
			case MapNotify:							// when a window changes
				NSDebugLog(@"MapNotify\n");			// state from ummapped to
				break;								// mapped or vice versa 
		
			case UnmapNotify:
				{									// find the NSWindow and
				PXKWindow *wind;					// inform it that it's no
													// longer visible
				NSDebugLog(@"UnmapNotify\n");			
														
				wind = [PXKWindow windowForXWindow:xEvent.xunmap.window];
				[(PXKWindow *)wind setXUnmap: NO];
				break;
				}
													 
			case MapRequest:						// like MapNotify but
				NSDebugLog(@"MapRequest\n");		// occurs before the
				break;								// request is carried out
	
			case MappingNotify:						// keyboard or mouse   
				NSDebugLog(@"MappingNotify\n");		// mapping has been changed
				break;								// by another client
	
			case MotionNotify: 						// the mouse has moved
				{
				NSDebugLog(@"MotionNotify\n");
										// regulate the translation of X motion
										// events in order to avoid saturating
										// the AppKit with unneeded NSEvents
				if((!inTrackingLoop && (xEvent.xmotion.time > 
					(unsigned long)(timeOfLastMotion + 50))) 
					|| (inTrackingLoop && (count == 1)))	
				{
				timeOfLastMotion = xEvent.xmotion.time;
				if (xEvent.xbutton.button == Button1)
					eventType = NSLeftMouseDragged;
				else
					{ 
					if (xEvent.xbutton.button == Button2)
						eventType = NSRightMouseDragged;
					else
						eventType = NSMouseMoved;
					}
				if(xEvent.xbutton.state != last_XEvent_state)
					{
					eventFlags = process_modifier_flags(xEvent.xbutton.state);
					last_XEvent_state = xEvent.xbutton.state;
					}
									// if pointer is grabbed use grab window
				xWin = (XGRAB == 0) ? xEvent.xbutton.window : XGRAB;			
				if(xWin != lastXWin)						
					{										// find NSWindow
					window = [PXKWindow windowForXWindow:xWin];	
					windowNumber = [window windowNumber];
					lastXWin = xWin;
					context = (PXKDPSContext*)[NSDPSContext currentContext];
					xFrame = [window xFrame];
					}		
				eventLocation = NSMakePoint(xEvent.xbutton.x, 
										xFrame.size.height - xEvent.xbutton.y);

				e = [NSEvent mouseEventWithType:eventType	// create NSEvent	
								location:eventLocation
								modifierFlags:eventFlags
								timestamp:(NSTimeInterval)xEvent.xbutton.time
								windowNumber:windowNumber
								context:context
								eventNumber:xEvent.xbutton.serial
								clickCount:1
								pressure:1.0];
				}
//				else
//				fprintf(stderr, " PXKApplication motion holdoff \n");
				break;
				}
														
			case PropertyNotify:					// a window property has
				NSDebugLog(@"PropertyNotify\n");	// changed or been deleted
				break;
		
			case ReparentNotify:					// a client successfully
				NSDebugLog(@"ReparentNotify\n");	// reparents a window 
				NSDebugLog(@"parent offset %f %f\n", xEvent.xreparent.x,
							xEvent.xreparent.y);
				break;

			case ResizeRequest:							// another client 
				NSDebugLog(@"ResizeRequest\n");			// attempts to change
				break;									// the size of a window
	
			case SelectionClear:						// events dealing with
			case SelectionNotify:						// the selection
			case SelectionRequest:
				NSDebugLog(@"Selection*\n");
				break;

			case VisibilityNotify:						// the visibility of
				NSDebugLog(@"VisibilityNotify\n");		// a window has changed
				break;									 
															// We shouldn't get
			default:										// here unless we
				NSLog(@"Received an untrapped event\n");	// forgot to trap
				break;										// an event above
			} 										// end of event type switch  
		if(e)   
	  		[event_queue addObject: e];					// add event to queue
		e = nil;											
		count--;										// decrement number of
	  }													// X events pending in
	while (count > 0);									// the X queue				
	};
}

- (void)setupRunLoopInputSourcesForMode:(NSString*)mode
{
Display* xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];
int xEventQueueFd = XConnectionNumber (xDisplay);
NSRunLoop* currentRunLoop = [NSRunLoop currentRunLoop];

							// Invoke limitDateForMode: to setup the current 
							// mode of the run loop (the doc says that this 
							// method and acceptInputForMode:beforeDate: are 
							// the only ones that setup the current mode).
	[currentRunLoop limitDateForMode:mode];

#if defined(LIB_FOUNDATION_LIBRARY)
  {
    id fileDescriptor = [[[NSPosixFileDescriptor alloc]
									initWithFileDescriptor:xEventQueueFd]
									autorelease];

    [fileDescriptor setDelegate:self];
    [fileDescriptor monitorFileActivity:NSPosixReadableActivity];
  }
#elif defined(NeXT_PDO)
  {
    id fileDescriptor = [[[NSFileHandle alloc]
									initWithFileDescriptor:xEventQueueFd]
									autorelease];

    [[NSNotificationCenter defaultCenter] addObserver:self
									selector:@selector(activityOnFileHandle:)
									name:NSFileHandleDataAvailableNotification
									object:fileDescriptor];
    [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
									[NSArray arrayWithObject:mode]];
  }
#else
														// Assume gnustep-base
	[(id)currentRunLoop addReadDescriptor:xEventQueueFd 	
						object:self 
						forMode:mode];
#endif
}

#if LIB_FOUNDATION_LIBRARY
- (void)activity:(NSPosixFileActivities)activity
		posixFileDescriptor:(NSPosixFileDescriptor*)fileDescriptor
{
	[self _nextEvent];
}
#elif defined(NeXT_PDO)
- (void)activityOnFileHandle:(NSNotification*)notification
{
id fileDescriptor = [notification object];
id runLoopMode = [[NSRunLoop currentRunLoop] currentMode];

	[fileDescriptor waitForDataInBackgroundAndNotifyForModes:
		    						[NSArray arrayWithObject:runLoopMode]];
	[self _nextEvent];
}
#endif

- (void)setGrabXWindow:(Window)w
{
	XGRAB = w;
}

- (Window)grabXWindow
{
	return XGRAB;
}
									// TODO: correctly map key events to 
									// NSEvents. This can be made when 16-bit
									// character strings will be available.
static NSEvent*
process_key_event (XEvent* xEvent, PXKDPSContext* context,
					NSEventType eventType)
{
char buf[256];
int count;
XComposeStatus compose;
NSString *keys, *ukeys;
KeySym keysym;
NSPoint eventLocation;
unsigned short keyCode;
unsigned int eventFlags;
NSEvent *event;
NSApplication *theApp = [NSApplication sharedApplication];
PXKWindow *window;
NSRect xFrame = NSZeroRect;

	NSDebugLog(@"Process key event");

	eventFlags = process_modifier_flags(xEvent->xkey.state);
    
	count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);

	if (count > 255) 							// Make sure that the string
		buf[255] = '\0';						// is properly terminated
	else 
		{	
		if (count < 1) 
			buf[0] = '\0';
		else 
			buf[count] = '\0';
		}

	window = (PXKWindow*)[theApp keyWindow];	// Set window to be key window
	if (window)
		xFrame = [window xFrame];

	eventLocation = [context userPointFromXPoint: NSMakePoint( 
									xEvent->xbutton.x,
									xFrame.size.height - xEvent->xbutton.y)];
	NSDebugLog (@"xLocation = (%d, %d), userLocation = (%f, %f)",
					xEvent->xbutton.x, (int)(xFrame.size.height - 	
					xEvent->xbutton.y), eventLocation.x, eventLocation.y);

	keyCode = process_key_code(xEvent, keysym, eventFlags);
    
	ukeys = [NSString stringWithCString:buf];
	keys = ukeys; 				/* Stupid implementation (to be replaced) */

	event = [NSEvent keyEventWithType:eventType
					 location:eventLocation
					 modifierFlags:eventFlags
					 timestamp:(NSTimeInterval)xEvent->xkey.time
					 windowNumber:[window windowNumber]
					 context:context
					 characters:keys
					 charactersIgnoringModifiers:ukeys
					 isARepeat:NO
					 keyCode:keyCode];

	return event;
}

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, 
				 unsigned int eventModifierFlags)
{
unsigned short keyCode = 0;

	if ((keysym == XK_Return) || (keysym == XK_KP_Enter) 
			|| (keysym == XK_Linefeed))
		{												// do nothing for now 
		keyCode = 0x0d;
		} 
	else 
		if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R)) 
			{			// The shift button is depressed.  This should have 
						// already been detected in the process_modifier_flags 
			}			// function.  Therefore, nothing is done here...
															  
	if ((keysym >= XK_F1) && (keysym <= XK_F35)) 			// if a function
		{													// key was pressed
		eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
	
		switch(xEvent->xkey.keycode) 
			{
			case XK_F1: keyCode = NSF1FunctionKey; break;
			case XK_F2: keyCode = NSF2FunctionKey; break;
			case XK_F3: keyCode = NSF3FunctionKey; break;
			case XK_F4: keyCode = NSF4FunctionKey; break;
			case XK_F5: keyCode = NSF5FunctionKey; break;
			case XK_F6: keyCode = NSF6FunctionKey; break;
			case XK_F7: keyCode = NSF7FunctionKey; break;
			case XK_F8: keyCode = NSF8FunctionKey; break;
			case XK_F9: keyCode = NSF9FunctionKey; break;
			case XK_F10: keyCode = NSF10FunctionKey; break;
			case XK_F11: keyCode = NSF11FunctionKey; break;
			case XK_F12: keyCode = NSF12FunctionKey; break;
			case XK_F13: keyCode = NSF13FunctionKey; break;
			case XK_F14: keyCode = NSF14FunctionKey; break;
			case XK_F15: keyCode = NSF15FunctionKey; break;
			case XK_F16: keyCode = NSF16FunctionKey; break;
			case XK_F17: keyCode = NSF17FunctionKey; break;
			case XK_F18: keyCode = NSF18FunctionKey; break;
			case XK_F19: keyCode = NSF19FunctionKey; break;
			case XK_F20: keyCode = NSF20FunctionKey; break;
			case XK_F21: keyCode = NSF21FunctionKey; break;
			case XK_F22: keyCode = NSF22FunctionKey; break;
			case XK_F23: keyCode = NSF23FunctionKey; break;
			case XK_F24: keyCode = NSF24FunctionKey; break;
			case XK_F25: keyCode = NSF25FunctionKey; break;
			case XK_F26: keyCode = NSF26FunctionKey; break;
			case XK_F27: keyCode = NSF27FunctionKey; break;
			case XK_F28: keyCode = NSF28FunctionKey; break;
			case XK_F29: keyCode = NSF29FunctionKey; break;
			case XK_F30: keyCode = NSF30FunctionKey; break;
			case XK_F31: keyCode = NSF31FunctionKey; break;
			case XK_F32: keyCode = NSF32FunctionKey; break;
			case XK_F33: keyCode = NSF33FunctionKey; break;
			case XK_F34: keyCode = NSF34FunctionKey; break;
			case XK_F35: keyCode = NSF35FunctionKey; break;
		
			default: 											// do nothing 
			}
		} 
	else 
		{
		if (keysym == XK_BackSpace) {
			keyCode = NSBackspaceKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Delete) {
			keyCode = NSDeleteFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Home) {
			keyCode = NSHomeFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Left) {
			keyCode = NSLeftArrowFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Up) {
			keyCode = NSUpArrowFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Right) {
			keyCode = NSRightArrowFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Down) {
			keyCode = NSDownArrowFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Prior) {
			keyCode = NSPrevFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#ifndef NeXT
		} else if (keysym == XK_Page_Up) {
			keyCode = NSPageUpFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
		} else if (keysym == XK_Next) {
			keyCode = NSNextFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#ifndef NeXT
		} else if (keysym == XK_Page_Down) {
			keyCode = NSPageDownFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
		} else if (keysym == XK_End) {
			keyCode = NSEndFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Begin) {
			keyCode = NSBeginFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Select) {
			keyCode = NSSelectFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Print) {
			keyCode = NSPrintFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Execute) {
			keyCode = NSExecuteFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Insert) {
			keyCode = NSInsertFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Undo) {
			keyCode = NSUndoFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Redo) {
			keyCode = NSRedoFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Menu) {
			keyCode = NSMenuFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Find) {
			keyCode = NSFindFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Help) {
			keyCode = NSHelpFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Break) {
			keyCode = NSBreakFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;    
		} else if (keysym == XK_Mode_switch) {
			keyCode = NSModeSwitchFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#ifndef NeXT
		} else if (keysym == XK_Sys_Req) {
			keyCode = NSSysReqFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
		} else if (keysym == XK_Scroll_Lock) {
			keyCode = NSScrollLockFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Pause) {
			keyCode = NSPauseFunctionKey;		
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if (keysym == XK_Clear) {
			keyCode = NSClearDisplayFunctionKey;
			eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
		} else if ((keysym == XK_Shift_L) || (keysym == XK_Shift_R)) {
			eventModifierFlags = eventModifierFlags |
		NSFunctionKeyMask |
		NSShiftKeyMask; 
		} else if ((keysym == XK_Control_L) || (keysym == XK_Control_R)) {
			eventModifierFlags = eventModifierFlags |
		NSFunctionKeyMask |
		NSControlKeyMask; 
		} else if (keysym == XK_Alt_R) {
			eventModifierFlags = eventModifierFlags |
		NSFunctionKeyMask |
		NSAlternateKeyMask; 
		} else if (keysym == XK_Alt_L) {
			eventModifierFlags = eventModifierFlags |
		NSFunctionKeyMask |
		NSCommandKeyMask; 
		} else if (keysym == XK_Tab) {
			keyCode = 0x09;
		}
		}
															// If the key press 
	if ((keysym >= XK_KP_Space) && (keysym <= XK_KP_9)) 	// originated from
		{													// the key pad
		eventModifierFlags = eventModifierFlags | NSNumericPadKeyMask;

		switch(keysym) 
			{
			case XK_KP_F1:        keyCode = NSF1FunctionKey;         break;
			case XK_KP_F2:        keyCode = NSF2FunctionKey;         break;
			case XK_KP_F3:        keyCode = NSF3FunctionKey;         break;
			case XK_KP_F4:        keyCode = NSF4FunctionKey;         break;
#ifndef NeXT
			case XK_KP_Home:      keyCode = NSHomeFunctionKey;       break;
			case XK_KP_Left:      keyCode = NSLeftArrowFunctionKey;  break;
			case XK_KP_Up:        keyCode = NSUpArrowFunctionKey;    break;
			case XK_KP_Right:     keyCode = NSRightArrowFunctionKey; break;
			case XK_KP_Down:      keyCode = NSDownArrowFunctionKey;  break;
			case XK_KP_Prior:     keyCode = NSPrevFunctionKey;       break;
//			case XK_KP_Page_Up:   keyCode = NSPageUpFunctionKey;     break;
			case XK_KP_Next:      keyCode = NSNextFunctionKey;       break;
//			case XK_KP_Page_Down: keyCode = NSPageDownFunctionKey;   break;
			case XK_KP_End:       keyCode = NSEndFunctionKey;        break;
			case XK_KP_Begin:     keyCode = NSBeginFunctionKey;      break;
			case XK_KP_Insert:    keyCode = NSInsertFunctionKey;     break;
			case XK_KP_Delete:    keyCode = NSDeleteFunctionKey;     break;
#endif
			default:  break;  /* Nothing to do */
    		}
  		} 												// End of keypad stuff 

	if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9)) ||
			((keysym >= XK_space) && (keysym <= XK_asciitilde)))
		{	
		}													// Not processed 

	return keyCode;
}
												// process_modifier_flags() 
static unsigned int								// determines which modifier
process_modifier_flags(unsigned int state)		// keys (Command, Control,
{												// Shift,  and so forth) were
unsigned int eventModifierFlags = 0; 			// held down while the event			
												// occured. eventModifierFlags
	if (state & ControlMask)					// is set accordingly.
		eventModifierFlags = eventModifierFlags | NSControlKeyMask;

	if (state & ShiftMask)
		eventModifierFlags = eventModifierFlags | NSShiftKeyMask;

	if (state & Mod1Mask)
		{
		eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
		NSDebugLog (@"setButtonModifierFlags(): Mod1Mask\n");
		}

	if (state & Mod2Mask) 
		{
		eventModifierFlags = eventModifierFlags | NSCommandKeyMask; 
		NSDebugLog (@"setButtonModifierFlags(): Mod2Mask\n");
		}

	if (state & Mod3Mask) 
		{
		eventModifierFlags = eventModifierFlags | NSAlphaShiftKeyMask;
		NSDebugLog (@"setButtonModifierFlags(): Mod3Mask\n");
		}

	if (state & Mod4Mask) 
		{
		eventModifierFlags = eventModifierFlags | NSHelpKeyMask; 
		NSDebugLog (@"setButtonModifierFlags(): Mod4Mask\n");
		}

	if (state & Mod5Mask) 
		{
		eventModifierFlags = eventModifierFlags | NSControlKeyMask; 
		NSDebugLog (@"setButtonModifierFlags(): Mod5Mask\n");
		}

	return eventModifierFlags;
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [PXKApplication class])
    return [super class];
  return [self class];
}

@end
