/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <GrFFT_vCC.h>
#include <gr_fft.h>
#include <math.h>

static const int N_OUTPUTS = 1;

GrFFT_vCC::GrFFT_vCC (int fft_size, bool forward, bool window)
  : VrSigProc (N_OUTPUTS, fft_size * sizeof (VrComplex), fft_size * sizeof (VrComplex)),
    d_fft_size (fft_size), d_window (forward && window), d_winvals (0)
{
  d_fft = new gr_fft_complex (d_fft_size, forward);

  if (d_window){
    d_winvals = new float[d_fft_size];
    for (int i = 0; i < d_fft_size; i++)
      d_winvals[i] = 0.54 - 0.46 * cos (2 * M_PI / d_fft_size * i); // hamming window
  }
}

GrFFT_vCC::~GrFFT_vCC ()
{
  delete d_fft;
  delete [] d_winvals;
}

int 
GrFFT_vCC::work (VrSampleRange output, void *ao[],
		 VrSampleRange inputs[], void *ai[])
{
  assert (numberInputs == 1);

  VrComplex *in  = ((VrComplex **) ai)[0];
  VrComplex *out = ((VrComplex **) ao)[0];

  unsigned int	input_data_size  = d_fft_size * sizeof (VrComplex);
  unsigned int	output_data_size = d_fft_size * sizeof (VrComplex);

  unsigned int count = 0;

  while (count++ < output.size){

    // copy input into optimally aligned buffer

    if (d_window){	
      VrComplex *dst = d_fft->get_inbuf ();
      for (int i = 0; i < d_fft_size; i++) 	// apply window
	dst[i] = in[i] * d_winvals[i];
    }
    else 
      memcpy (d_fft->get_inbuf (), in, input_data_size);

    // compute the fft
    d_fft->execute ();

    // copy result to our output
    memcpy (out, d_fft->get_outbuf(), output_data_size);

    in  += d_fft_size;
    out += d_fft_size;
  }

  return output.size;
}
