/*
     This file is part of GNUnet.
     (C) 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/fs_event_handler.h
 * @brief Main event handler for file-sharing
 * @author Christian Grothoff
 */
#include "common.h"


struct SearchResult;


/**
 * Context we keep for a search tab.
 */
struct SearchTab
{
  /**
   * This is a doubly-linked list.
   */
  struct SearchTab *next;

  /**
   * This is a doubly-linked list.
   */
  struct SearchTab *prev;

  /**
   * Set in case this is an inner search, otherwise NULL.
   */
  struct SearchResult *parent;

  /**
   * Handle for this search with FS library.
   */
  struct GNUNET_FS_SearchContext *sc;

  char *query_txt;

  GtkBuilder *builder;

  GtkWidget *frame;

  GtkWidget *tab_label;

  GtkWidget *close_button;

  GtkWidget *clear_button;

  GtkWidget *play_button;

  GtkWidget *pause_button;

  GtkLabel *label;

  GtkTreeStore *ts;

  unsigned int num_results;

};


struct DownloadEntry
{

  /**
   * Download entry of the parent (for recursive downloads),
   * NULL if we are either a top-level download (from URI,
   * from opened directory, orphaned from search or direct
   * search result).
   */
  struct DownloadEntry *pde;

  /**
   * Associated search result, or NULL if we don't belong
   * to a search directly (download entry).
   */
  struct SearchResult *sr;

  /**
   * FS handle to control the download.
   */
  struct GNUNET_FS_DownloadContext *dc;

  /**
   * URI for the download.
   */
  struct GNUNET_FS_Uri *uri;

  /**
   * Meta data for the download.
   */
  struct GNUNET_CONTAINER_MetaData *meta;

  /**
   * Where in the tree view is this download being displayed.
   */
  GtkTreeRowReference *rr;

  /**
   * Tree store where we are stored.
   */
  GtkTreeStore *ts;
  
  /**
   * Tab where this download is currently on display.
   */
  struct SearchTab *tab;

  /**
   * Has the download completed (or errored)?
   */
  int is_done;

};




/**
 * Setup a new top-level entry in the URI tab.  If necessary, create
 * the URI tab first.
 *
 * @param iter set to the new entry
 * @param srp set to search result 
 * @param meta metadata for the new entry
 * @param uri URI for the new entry
 * @return NULL on error, otherwise search tab with the new entry
 */
struct SearchTab *
GNUNET_GTK_add_to_uri_tab (GtkTreeIter *iter,
			   struct SearchResult **sr,
			   const struct GNUNET_CONTAINER_MetaData *meta,
			   const struct GNUNET_FS_Uri *uri);



/**
 * Add a search result to the given search tab.
 *
 * @param tab search tab to extend
 * @param iter set to position where search result is added
 * @param parent_rr reference to parent entry in search tab
 * @param uri uri to add
 * @param meta metadata of the entry
 * @param result associated FS search result (can be NULL)
 * @param applicability_rank how relevant is the result
 * @return entry for the search result
 */
struct SearchResult *
GNUNET_GTK_add_search_result (struct SearchTab *tab,
			      GtkTreeIter *iter,
			      GtkTreeRowReference *parent_rr,
			      const struct GNUNET_FS_Uri *uri,
			      const struct GNUNET_CONTAINER_MetaData *meta,
			      struct GNUNET_FS_SearchResult *result,
			      uint32_t applicability_rank);


/**
 * Notification of FS to a client about the progress of an 
 * operation.  Callbacks of this type will be used for uploads,
 * downloads and searches.  Some of the arguments depend a bit 
 * in their meaning on the context in which the callback is used.
 *
 * @param cls closure
 * @param info details about the event, specifying the event type
 *        and various bits about the event
 * @return client-context (for the next progress call
 *         for this operation; should be set to NULL for
 *         SUSPEND and STOPPED events).  The value returned
 *         will be passed to future callbacks in the respective
 *         field in the GNUNET_FS_ProgressInfo struct.
 */
void* GNUNET_GTK_fs_event_handler (void *cls,
				   const struct GNUNET_FS_ProgressInfo *info);


/* end of fs_event_handler.h */
