/*
     This file is part of GNUnet.
     (C) 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/common.c
 * @brief Common functions used in various places
 * @author Christian Grothoff
 */
#include "common.h"

/**
 * Add meta data to list store.
 *
 * @param cls closure (the GtkListStore)
 * @param plugin_name name of the plugin that produced this value;
 *        special values can be used (i.e. '<zlib>' for zlib being
 *        used in the main libextractor library and yielding
 *        meta data).
 * @param type libextractor-type describing the meta data
 * @param format basic format information about data 
 * @param data_mime_type mime-type of data (not of the original file);
 *        can be NULL (if mime-type is not known)
 * @param data actual meta-data found
 * @param data_len number of bytes in data
 * @return 0 to continue (always)
 */ 
int
GNUNET_GTK_add_meta_data_to_list_store (void *cls,
					const char *plugin_name,
					enum EXTRACTOR_MetaType type,
					enum EXTRACTOR_MetaFormat format,
					const char *data_mime_type,
					const char *data,
					size_t data_len)
{
  GtkListStore *ls = GTK_LIST_STORE (cls);

  if ( (format == EXTRACTOR_METAFORMAT_UTF8) ||
       (format == EXTRACTOR_METAFORMAT_C_STRING) )
    gtk_list_store_insert_with_values (ls,
				       NULL,
				       G_MAXINT,
				       0, type,
				       1, format,
				       2, EXTRACTOR_metatype_to_string (type),
				       3, data,
				       -1);
  return 0;
}


/**
 * Convert the year from the spin button to an expiration
 * time (on midnight, January 1st of that year).
 */
struct GNUNET_TIME_Absolute 
GNUNET_GTK_get_expiration_time (GtkSpinButton *spin)
{
  struct GNUNET_TIME_Absolute ret;
  int year;
  time_t tp;
  struct tm t;

  year = gtk_spin_button_get_value_as_int (spin);
  memset (&t, 0, sizeof (t));
  if (year < 1900)
    {
      GNUNET_break (0);
      return GNUNET_TIME_absolute_get (); /* now */
    }
  t.tm_year = year - 1900;
  t.tm_mday = 1;
  t.tm_mon = 1;
  t.tm_wday = 1;
  t.tm_yday = 1;
  tp = mktime (&t);
  GNUNET_break (tp != (time_t) -1);
  ret.abs_value = tp * 1000LL; /* seconds to ms */
  GNUNET_break (GNUNET_TIME_absolute_get ().abs_value < ret.abs_value);
  return ret;
}


void
GNUNET_GTK_setup_expiration_year_adjustment (GtkBuilder *builder)
{
  GtkAdjustment *aj;
  struct tm *t;
  int year;
  time_t tp;

  tp = time (NULL);
  t = gmtime (&tp);
  if (t == NULL)
    year = 0;
  else
    year = t->tm_year + 1900;
  aj = GTK_ADJUSTMENT (gtk_builder_get_object (builder,
					       "expiration_year_adjustment"));
  gtk_adjustment_set_value (aj, year + 2);
  gtk_adjustment_set_lower (aj, year + 1);
}


GdkPixbuf *
GNUNET_GTK_get_thumbnail_from_meta_data (const struct GNUNET_CONTAINER_MetaData *meta)
{
  GdkPixbuf *pixbuf;
  GdkPixbufLoader *loader;
  size_t ts;
  unsigned char *thumb;

  thumb = NULL;
  ts = GNUNET_CONTAINER_meta_data_get_thumbnail (meta, &thumb);
  if (ts == 0)
    return NULL;
  loader = gdk_pixbuf_loader_new ();
  gdk_pixbuf_loader_write (loader, (const guchar *) thumb, ts, NULL);
  pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
  gdk_pixbuf_loader_close (loader, NULL);
  if (pixbuf != NULL)
    g_object_ref (pixbuf);
  g_object_unref (loader);
  GNUNET_free (thumb);
  return pixbuf;
}


/**
 * mmap the given file and run the GNUNET_FS_directory_list_contents
 * function on it.
 */
void
GNUNET_GTK_mmap_and_scan (const char *filename,
			  GNUNET_FS_DirectoryEntryProcessor dep,
			  void *dep_cls)
{
  struct GNUNET_DISK_FileHandle *fh;
  struct GNUNET_DISK_MapHandle *mh;
  uint64_t fsize;
  void * ddata;

  if (GNUNET_OK !=
      GNUNET_DISK_file_size (filename,
			     &fsize,
			     GNUNET_YES))
    {
      GNUNET_break (0);
      return;
    }
  fh = GNUNET_DISK_file_open (filename,
			      GNUNET_DISK_OPEN_READ,
			      GNUNET_DISK_PERM_NONE);
  if (fh == NULL)
    {
      GNUNET_log_strerror_file (GNUNET_ERROR_TYPE_ERROR, "open", filename);
      return;
    }
  ddata = GNUNET_DISK_file_map (fh,
				&mh,
				GNUNET_DISK_MAP_TYPE_READ,
				(size_t) fsize);
  if (ddata == NULL)
    {
      GNUNET_log_strerror_file (GNUNET_ERROR_TYPE_ERROR, "mmap", filename);
      GNUNET_break (GNUNET_OK ==
		    GNUNET_DISK_file_close (fh));
      return;
    }
  if (GNUNET_SYSERR ==
      GNUNET_FS_directory_list_contents ((size_t) fsize,
					 ddata,
					 0,
					 dep, dep_cls))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
		  _("Selected file `%s' is not a GNUnet directory!\n"),
		  filename);
    }
  GNUNET_break (GNUNET_OK ==
		GNUNET_DISK_file_unmap (mh));
  GNUNET_break (GNUNET_OK ==
		GNUNET_DISK_file_close (fh));
}


/* end of common.c */
