/*
     This file is part of GNUnet
     (C) 2008 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/about/chat.c
 * @author Christian Grothoff
 * @author Igor Wronsky
 * @author Nathan Evans
 */

#include "platform.h"
#include <GNUnet/gnunet_util_pseudonym.h>
#include <GNUnet/gnunet_chat_lib.h>
#include "gnunetgtk_common.h"

struct GNUNET_CHAT_safe_write_struct
{
  GtkWidget *text_view;
  const char *message;
  const char *sender;
};

struct GNUNET_CHAT_safe_nick_write_struct
{
  GtkListStore *model;
  GdkPixbuf *icon;
  const struct GNUNET_MetaData *meta;
  const char *nick;
  GNUNET_HashCode pid;
};

struct GNUNET_CHAT_gui_chat_client
{
  struct GNUNET_CHAT_gui_chat_client *next;
  GtkWidget *send_button;
  GtkWidget *text_view;
  GtkListStore *nick_model;
  GtkWidget *chatFrame;
  GtkWidget *chat_entry;
  GtkWidget *tab_label;
  GladeXML *chatXML;
  GladeXML *labelXML;
  struct GNUNET_CHAT_Room *room;
  GNUNET_HashCode mypid;
};

/**
 * For nicknames in chat view.
 */
enum
{
  CHAT_NICKNAME = 0,
  CHAT_METADATA,
  CHAT_ICON,
  CHAT_PID,
  CHAT_NUM,
};


static struct GNUNET_Mutex *lock;

static struct GNUNET_GE_Context *chat_ectx;

static struct GNUNET_GC_Configuration *chat_cfg;

static struct GNUNET_CHAT_gui_chat_client *list;

static void *
write_save_call (void *arg)
{
  struct GNUNET_CHAT_safe_write_struct *cls = arg;
  GtkTextBuffer *buffer;
  char *message_buf;
  int message_buf_size;

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (cls->text_view));
  message_buf_size =
    snprintf (NULL, 0, "<%s>: %s\n", cls->sender, cls->message);
  message_buf = GNUNET_malloc (message_buf_size + 1);
  GNUNET_snprintf (message_buf,
                   message_buf_size + 1,
                   "<%s>: %s\n", cls->sender, cls->message);
  gtk_text_buffer_insert_at_cursor (buffer, message_buf, message_buf_size);
  GNUNET_free (message_buf);
  return NULL;
}

/**
 * Safe call to add a nick to the chat room view
 */
static void *
add_nick_save_call (void *arg)
{
  struct GNUNET_CHAT_safe_nick_write_struct *cls = arg;
  GtkListStore *model;
  GtkTreeIter iter;
  GNUNET_HashCode *pid;

  model = GTK_LIST_STORE (cls->model);
  gtk_list_store_append (model, &iter);
  pid = GNUNET_malloc (sizeof (GNUNET_HashCode));
  *pid = cls->pid;
  gtk_list_store_set (model,
                      &iter,
                      CHAT_PID, pid,
                      CHAT_METADATA,
                      GNUNET_meta_data_duplicate (cls->meta), CHAT_ICON,
                      cls->icon, CHAT_NICKNAME, cls->nick, -1);
  return NULL;
}

/**
 * Safe call to remove a nick from the chat room view
 */
static void *
remove_nick_save_call (void *arg)
{
  struct GNUNET_CHAT_safe_nick_write_struct *cls = arg;
  GtkListStore *model;
  GtkTreeIter iter;
  GNUNET_HashCode *pid;
  struct GNUNET_MetaData *meta;

  model = GTK_LIST_STORE (cls->model);
  /* find and remove existing entry */
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
    {
      do
        {
          pid = NULL;
          gtk_tree_model_get (GTK_TREE_MODEL (model),
                              &iter,
                              CHAT_PID, &pid, CHAT_METADATA, &meta, -1);
          if ((pid != NULL) &&
              (0 == memcmp (pid, &cls->pid, sizeof (GNUNET_HashCode))))
            {
              GNUNET_meta_data_destroy (meta);
              GNUNET_free (pid);
              gtk_list_store_remove (model, &iter);
              return NULL;      /* done! */
            }
        }
      while (gtk_tree_model_iter_next (GTK_TREE_MODEL (model), &iter));
    }
  return NULL;
}


/**
 * This method is called when the user clicks on
 * the "CANCEL (X)" button in the TAB of the
 * chat notebook.
 */
void
on_closeChatButton_clicked_chat (GtkWidget * chatPage,
                                 GtkWidget * closeButton)
{
  struct GNUNET_CHAT_gui_chat_client *pos;
  struct GNUNET_CHAT_gui_chat_client *prev;
  GtkNotebook *chatnotebook;
  int index;
  int i;

  prev = NULL;
  pos = list;
  while (pos != NULL)
    {
      if (pos->tab_label == chatPage)
        break;
      prev = pos;
      pos = pos->next;
    }
  if (prev == NULL)
    list = pos->next;
  else
    prev->next = pos;
  GNUNET_GE_ASSERT (NULL, pos != NULL);
  GNUNET_GTK_run_with_save_calls
    ((GNUNET_ThreadMainFunction) & GNUNET_CHAT_leave_room, pos->room);

  chatnotebook =
    GTK_NOTEBOOK (glade_xml_get_widget
                  (GNUNET_GTK_get_main_glade_XML (), "chatnotebook"));
  /* remove page from notebook */
  index = -1;
  for (i = gtk_notebook_get_n_pages (chatnotebook) - 1; i >= 0; i--)
    if (pos->chatFrame == gtk_notebook_get_nth_page (chatnotebook, i))
      index = i;
  GNUNET_GE_BREAK (NULL, index != -1);
  gtk_notebook_remove_page (chatnotebook, index);
  UNREF (pos->chatXML);
  UNREF (pos->labelXML);
  GNUNET_free (pos);
}



/**
 * A message was sent in the chat to us.
 *
 * @param senderNick what is the nickname of the sender? (maybe NULL)
 * @param message the message (maybe NULL, especially if confirmation
 *        is requested before delivery; the protocol will ensure
 *        that this function is called again with the full message
 *        if a confirmation is transmitted; if the message is NULL,
 *        the user is merely asked if engaging in the exchange is ok
 * @param room in which room was the message received?
 * @param options options for the message
 * @return GNUNET_OK to accept the message now, GNUNET_NO to
 *         accept (but user is away), GNUNET_SYSERR to signal denied delivery
 */
static int
receive_callback (void *cls,
                  struct GNUNET_CHAT_Room *room,
                  const GNUNET_HashCode * sender,
                  const struct GNUNET_MetaData *member_info,
                  const char *message, GNUNET_CHAT_MSG_OPTIONS options)
{
  struct GNUNET_CHAT_gui_chat_client *client = cls;
  struct GNUNET_CHAT_safe_write_struct writearg;
  char *sndr;

  sndr =
    sender == NULL ? _("anonymous") : GNUNET_pseudonym_id_to_name (chat_ectx,
                                                                   chat_cfg,
                                                                   sender);
  writearg.text_view = client->text_view;
  writearg.message = message;
  writearg.sender = sndr;
  GNUNET_GTK_save_call (&write_save_call, &writearg);
  GNUNET_free_non_null (sndr);
  return GNUNET_OK;
}

static int
member_list_callback (void *cls,
                      const struct GNUNET_MetaData *member_info,
                      const GNUNET_RSA_PublicKey * pkey,
                      GNUNET_CHAT_MSG_OPTIONS opt)
{
  struct GNUNET_CHAT_gui_chat_client *client = cls;
  struct GNUNET_CHAT_safe_nick_write_struct writearg;
  char *nick;
  char *path;
  char *filename;

  GNUNET_hash (pkey, sizeof (GNUNET_RSA_PublicKey), &writearg.pid);
  nick = GNUNET_pseudonym_id_to_name (chat_ectx, chat_cfg, &writearg.pid);
  writearg.model = client->nick_model;
  writearg.icon = NULL;
  if (0 == memcmp (&writearg.pid, &client->mypid, sizeof (GNUNET_HashCode)))
    {
      path = GNUNET_get_installation_path (GNUNET_IPK_SELF_PREFIX);
      filename =
        GNUNET_malloc (strlen (path) +
                       strlen ("share/gnunet-gtk/self.png") + 1);
      strcpy (filename, path);
      GNUNET_free (path);
      strcat (filename, "share/gnunet-gtk/self.png");
      writearg.icon = gdk_pixbuf_new_from_file (filename, NULL);
      GNUNET_free (filename);
    }
  writearg.meta = member_info;
  writearg.nick = nick;
  if (member_info != NULL)
    GNUNET_GTK_save_call (&add_nick_save_call, &writearg);
  else
    GNUNET_GTK_save_call (&remove_nick_save_call, &writearg);
  GNUNET_free (nick);
  if (writearg.icon != NULL)
    g_object_unref (writearg.icon);
  return GNUNET_OK;
}

void
on_chat_frame_send_button_click_event_chat (GtkWidget * widget, gpointer data)
{
  unsigned int seq;
  const char *message;
  struct GNUNET_CHAT_gui_chat_client *pos;

  GNUNET_mutex_lock (lock);
  pos = list;
  while ((pos != NULL) && (pos->send_button != widget) &&
         (pos->chat_entry != widget))
    pos = pos->next;
  GNUNET_GE_ASSERT (NULL, pos != NULL);

  message = (const char *) gtk_entry_get_text (GTK_ENTRY (pos->chat_entry));
  if (strlen (message) > 0)
    {
      GNUNET_CHAT_send_message (pos->room,
                                message, GNUNET_CHAT_MSG_OPTION_NONE, NULL,
                                &seq);
      gtk_entry_set_text (GTK_ENTRY (pos->chat_entry), "");
    }
  GNUNET_mutex_unlock (lock);
}

static void
create_chat_room_tab (const char *room_name, const char *nick)
{
  GtkWidget *chatnotebook;
  GtkLabel *label;
  GtkTreeView *treeview;
  struct GNUNET_CHAT_gui_chat_client *client;
  struct GNUNET_MetaData *meta;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  int col;

  chatnotebook =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "chatnotebook");

  client = GNUNET_malloc (sizeof (struct GNUNET_CHAT_gui_chat_client));
  client->chatXML =
    glade_xml_new (GNUNET_GTK_get_glade_filename (),
                   "chatFrame", PACKAGE_NAME);
  GNUNET_GTK_connect_glade_with_plugins (client->chatXML);
  client->chatFrame =
    GNUNET_GTK_extract_main_widget_from_window (client->chatXML, "chatFrame");
  client->labelXML =
    glade_xml_new (GNUNET_GTK_get_glade_filename (),
                   "chatTabLabelWindow", PACKAGE_NAME);
  GNUNET_GTK_connect_glade_with_plugins (client->labelXML);
  client->tab_label
    = GNUNET_GTK_extract_main_widget_from_window (client->labelXML,
                                                  "chatTabLabelWindow");
  label = GTK_LABEL (glade_xml_get_widget (client->labelXML, "chatTabLabel"));
  gtk_label_set (label, room_name);
  gtk_notebook_append_page (GTK_NOTEBOOK (chatnotebook),
                            client->chatFrame, client->tab_label);
  gtk_widget_show (chatnotebook);

  client->nick_model = gtk_list_store_new (CHAT_NUM, G_TYPE_STRING,     /* nickname */
                                           G_TYPE_POINTER,      /* metadata */
                                           GDK_TYPE_PIXBUF,     /* icon */
                                           G_TYPE_POINTER);     /* pid */
  client->send_button =
    glade_xml_get_widget (client->chatXML, "chatSendButton");
  client->text_view = glade_xml_get_widget (client->chatXML, "chatLogViewer");
  client->chat_entry =
    glade_xml_get_widget (client->chatXML, "chatLineTextEntry");
  treeview =
    GTK_TREE_VIEW (glade_xml_get_widget
                   (client->chatXML, "roomMembersTreeView"));
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (client->nick_model));

  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (treeview,
                                                     -1,
                                                     _("Nickname"),
                                                     renderer,
                                                     "text", CHAT_NICKNAME,
                                                     NULL);
  column = gtk_tree_view_get_column (treeview, col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, CHAT_NICKNAME);
  gtk_tree_view_column_set_min_width (column, 0);

  renderer = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_insert_column_with_attributes (treeview,
                                                     -1,
                                                     "",
                                                     renderer,
                                                     "pixbuf", CHAT_ICON,
                                                     NULL);
  column = gtk_tree_view_get_column (treeview, col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, FALSE);
  gtk_tree_view_column_set_reorderable (column, FALSE);

  GNUNET_mutex_lock (lock);
  client->next = list;
  list = client;
  GNUNET_mutex_unlock (lock);

  meta = GNUNET_meta_data_create ();
  GNUNET_meta_data_insert (meta, EXTRACTOR_TITLE, nick);
  client->room =
    GNUNET_CHAT_join_room (chat_ectx, chat_cfg, nick,
                           meta,
                           room_name,
                           -1,
                           &receive_callback, client, &member_list_callback,
                           client, NULL, NULL, &client->mypid);
  GNUNET_meta_data_destroy (meta);
}

void
on_chat_room_name_button_click_event_chat (GtkWidget * widget, gpointer data)
{
  GtkEntry *room_entry;
  GtkEntry *nick_entry;
  const char *room_text;
  const char *nick_text;

  room_entry =
    GTK_ENTRY (glade_xml_get_widget
               (GNUNET_GTK_get_main_glade_XML (), "chatRoomNameEntry"));
  nick_entry =
    GTK_ENTRY (glade_xml_get_widget
               (GNUNET_GTK_get_main_glade_XML (), "chatRoomMonikerEntry"));
  room_text = (const char *) gtk_entry_get_text (room_entry);
  nick_text = (const char *) gtk_entry_get_text (nick_entry);
  create_chat_room_tab (room_text, nick_text);
  gtk_entry_set_text (nick_entry, "");
  gtk_entry_set_text (room_entry, "");
}

void
init_chat (struct GNUNET_GE_Context *ectx,
           struct GNUNET_GC_Configuration *cfg)
{
  GtkWidget *tab;

  chat_ectx = ectx;
  chat_cfg = cfg;
  lock = GNUNET_mutex_create (GNUNET_NO);
  tab =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "chatnotebookvbox");
  gtk_widget_show (tab);
}

/* end of chat.c */
