#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2009 Free Software Foundation
#
# FILE:
# GRDataMapper.py
#
# DESCRIPTION:
# Helper class for use by GRLayout
#
# NOTES:
#
# HISTORY:
#

import GRExceptions
from gnue.common.apps import GDebug
from gnue.common.formatting import GDataFormatter
from gnue.common.external.fixedpoint import FixedPoint
import types, string
import sys

if sys.hexversion >= 0x02040000:
    import decimal

#
# Class used internally by GRDataMapper
# Classes other than GRDataMapper should not
# even care that this exists...
#
# There is one instance of this for EACH GRSection
#
# TODO: should this functionality be moved to the
# TODO: GRSection class since there's a 1:1 relation?
#
class GRDataMapperSection:
  def __init__(self, name, source, parent):
    self.parent = parent
    self.name = name
    self.source = source
    self.sections = []
    self.fields = {}
    self.summaries = {}
    self.formulas = {}
    self.formulaObjects = {}
    self._fieldsOld = {}
    self.toplevel = 0   # Is this the first section to use a datasource
    self.initial = 1    # Are we in an initial run in which we have no history?
    self.grouping = 0   # Is this a "grouping" section?
    self._lastSection = None
    self.datasource = None
    self._resultsets = []

    self._clearOnNextSet = 0

    self._summMap = {'sum': self._summ_sum,
                     'count': self._summ_count,
                     'avg': self._summ_avg,
                     'min': self._summ_min,
                     'max': self._summ_max }

    if parent != None:
      parent.addChildSection(self)


  #
  #  Are we or are any of our descendants
  #  the specified object
  #
  def isAncestorOf (self, mapper):
    if self == mapper:
      return 1
    else:
      rv = 0
      for section in self.sections:
        rv = rv or section.isAncestorOf(mapper)
        if rv:
          break
      return rv

  #
  #  Add a field
  #
  def addField(self, name):
    self.fields[name] = ""

  #
  #  Add a summary
  #
  def addSummary(self, function, key):
    try:
      self.summaries[key][function] = None
    except KeyError:
      self.summaries[key] = {function:None}

    # If this function relies on other
    # functions, add those as well (e.g.,
    # average depends on sum and count)
    try:
      for func in _summMultiMapping[function]:
        self.addSummary(self, func, key)
    except KeyError:
      pass

  #
  #  Add a formula
  #
  def addFormula(self, name, object):
    self.formulas[name] = None
    self.formulaObjects[name] = object



  #
  #  Get a field's current value
  #
  def getField(self, name, format):
    return GDataFormatter.applyFormatting(self.fields[name], format)

  #
  #  Get a summary's current value
  #
  def getSummary(self, name, function, format):
    val = self.summaries[name][function]
    if val is None:
      return ""
    else:
      return "%s" % val

  #
  #  Get a formula's current value
  #
  def getFormula(self, name, format):
    val = self.formulas[name]
    if val is None:
      return ""
    else:
      return GDataFormatter.applyFormatting("%s" % val, format)

  #
  #  Add a section
  #
  def addChildSection(self, mapper):
    self.sections.append(mapper)


  #
  # Zero out the summary data
  #
  def clearSummaries(self):
    for key in self.summaries.keys():
      for function in self.summaries[key].keys():
        self.summaries[key][function] = None
    self._clearOnNextSet = 0


  #
  # Will the next recordset cause our section to change?
  #
  def _precheckNextFields(self, recordset):
    if not recordset:
      return 0
    changed = 0
    for field in self.fields.keys():
      val = recordset.getField(field)
      assert gDebug(10, 'self=%s,%s'%(self,self.name))
      if ( not self.grouping or \
           self.fields[field] != val ):
        changed = 1
        assert gDebug(10, "Field %s changed after nextRecord (%s,%s,%s,%s)" % (field, self.initial, self.grouping, self.fields[field], val))
        break
    assert gDebug(10, "After _precheckNextFields, changed=%s" % changed)
    return changed


  def _dsResultSetActivated (self, event):
    assert gDebug (5, 'GRDataMapper._dsResultSetActivated on %s' % self.name)
    self._resultsets.insert (0, event.resultSet)


  ##
  ## Summary calculation functions
  ##

  # Summary function: "count"
  def _summ_count(self, key, value):
    if value is not None:
      try:
        self.summaries[key]['count'] += 1
      except TypeError:
        self.summaries[key]['count'] = 1

  # Summary function: "sum"
  def _summ_sum(self, key, value):
    if value is not None:
      if type(value) in _numericTypes:
        try:
          self.summaries[key]['sum'] += FixedPoint(value)
        except TypeError:
          self.summaries[key]['sum'] = FixedPoint(value)
      else:
        raise "Attempting to 'sum' a non-numeric %s" % (key)

  # Summary function: "average"
  def _summ_avg(self, key, value):
    if type(value) in _numericTypes:
      self.summaries[key]['average'] = \
          self.summaries[key]['sum'] / self.summaries[key]['count']
    else:
      raise "Attempting to 'average' a non-numeric %s" % (key)

  # Summary function: "min"
  def _summ_min(self, key, value):
    if value is not None:
      existing = self.summaries[key]['min']
      if existing is None or value < existing:
        self.summaries[key]['min'] = value

  # Summary function: "max"
  def _summ_max(self, key, value):
    if value is not None:
      existing = self.summaries[key]['min']
      if existing is None or value > existing:
        self.summaries[key]['min'] = value



###########################################
#
#
###########################################
class GRDataMapper(GRDataMapperSection):

  def __init__(self, sources):
    GRDataMapperSection.__init__(self, "", "", None)
    self.sources = sources
    self.sectionMap = {}
    self.sourceMap = {}

  def addSection(self, section, source, parentSection):
    if self.sectionMap.has_key (section):
      raise GRExceptions.SectionHasDuplicateName, \
         'Section "%s" is defined multiple times' % section

    if parentSection == None or not self.sectionMap.has_key(parentSection):
      parentMapper = self
    else:
      parentMapper = self.sectionMap[parentSection]

    if self.sourceMap.has_key(source) and source != None and\
       not self.sourceMap[source][0].isAncestorOf(parentMapper):
      raise GRExceptions.SourceMappedToSiblingSections, \
         "Section '%s' attempts to use source '%s', but does not descend from '%s'" % \
               (section, source, self.sourceMap[source][0].name)

    mapper = GRDataMapperSection(section, source, parentMapper)

    self.sectionMap[section] = mapper

    if not self.sourceMap.has_key(source):
      assert gDebug(5,"Setting section %s as source controller" % self.name)
      self.sourceMap[source] = [mapper]
      mapper.toplevel = 1
      if source != None:
        mapper.datasource = self.sources.getDataSource(source)
        mapper.datasource.registerEventListeners ({
          'dsResultSetActivated': mapper._dsResultSetActivated})
    else:
      self.sourceMap[source][-1].grouping = 1
      self.sourceMap[source].append(mapper)

    assert gDebug (10, "sourceMap=%s" % self.sourceMap)

  def addFieldToSection(self, section, field, bound=1):
    self.sectionMap[section].addField(field)
    self.sources.getDataSource(self.sectionMap[section].source) \
          .referenceField(field)

  def addSummaryToSection(self, function, section, field, formula):
    self.sectionMap[section].addSummary(function, (field, formula))
    if not field is None:
      self.sources.getDataSource(self.sectionMap[section].source) \
            .referenceField(field)

  def addFormulaToSection(self, section, name, object):
    self.sectionMap[section].addFormula(name, object)


  #
  #
  #
  def getFirstRecord(self, source):

    if source == None:
      assert gDebug (4, 'Skipping ResultSet creation for empty source')
      return None

    for s in self.sourceMap[source]:
      s.initial = 1

    controlSection = self.sourceMap[source][0]

    # Only load a new resultset if this is not a child source
    # as child sources were queries by the master
    if controlSection.parent == None or controlSection.parent.source == "":
      assert gDebug (4, 'Creating ResultSet for source %s' % source)
      controlSection.resultset = \
          self.sources.getDataSource(source).createResultSet(readOnly=1)
      if controlSection.resultset.nextRecord():
        controlSection.__nextRecord = controlSection.resultset.current
      else:
        controlSection.__nextRecord = None
    else:
      assert gDebug(4, 'Getting pre-created ResultSet for source "%s"; parent=%s' \
             % (source, controlSection.parent))
      controlSection.resultset = controlSection._resultsets.pop()
#      resultset = self.sourceMap[controlSection.parent.source].resultset.current.
      if controlSection.resultset.nextRecord():
        controlSection.__nextRecord = controlSection.resultset.current
      else:
        controlSection.__nextRecord = None

    return self.getNextRecord(source)


  # Returns a string containing first section to change.
  # If None, then this datasource is all used up.
  def getNextRecord(self, source):

    assert gDebug (6, 'Getting next record for source %s' % source)
    if  source == None:
      assert gDebug (6, 'No next record to return for source %s' % source)
      return (None, None)

    controlSection = self.sourceMap[source][0]

    if not controlSection.__nextRecord:
      return (None, None)
    else:

      recordset = controlSection.__nextRecord
      if controlSection.resultset.nextRecord():
        controlSection.__nextRecord = controlSection.resultset.current
      else:
        controlSection.__nextRecord = None

      firstSection = None
      nextSection = None

      # Apply the recordset to each section we control,
      for s in self.sourceMap[source]:

        # Save the new field values in the data handler
        for field in s.fields.keys():
          s.fields[field] = recordset.getField(field)


      # determining the first section to change.
      for s in self.sourceMap[source]:

        # Reset running summary counts if necessary...

        changed = s._precheckNextFields(recordset)

        # What is the current section?
        if firstSection == None and changed:
          assert gDebug(10, "After next record, first changed section is %s" % s.name)
          firstSection = s.name

        #   and run formula triggers
        for formula in s.formulas.keys():
          if s.formulaObjects[formula] != None:
            s.formulas[formula] = s.formulaObjects[formula].processTrigger('On-Process')

        # ... and save any new summary data
        if s._clearOnNextSet:
          s.clearSummaries()

        for key in s.summaries.keys():
          field, formula = key
          for function in _summFunctions:
            if s.summaries[key].has_key(function):
              if not field is None:
                s._summMap[function](key, recordset.getField(field))
              elif (not formula is None) and s.formulas.has_key(formula):
                s._summMap[function](key, s.formulas[formula])

        # What will the next section be?
        if (nextSection is None and \
            controlSection.__nextRecord and \
            s._precheckNextFields(controlSection.__nextRecord)):
          nextSection = s.name


      # Save this in case
      controlSection._lastSection = firstSection
      return (firstSection, nextSection)


#
# Contains all valid "summary" functions
# and the order they need to be calculated
# (e.g., count and sum need to happen before
# average)

_summFunctions = ('count','sum','min','max','avg','_accum',)

#
# Contains all "summary" functions that
# depend on other record keeping. (e.g.,
# 'averages' need the total and the count
# tracked.)
#
_summMultiMapping = { 'avg': ('sum','count'),
                    }

# Python types that are numeric
_numericTypes = [types.FloatType, types.IntType, types.LongType]
if sys.hexversion >= 0x02040000:
    _numericTypes.append(decimal.Decimal)
