/*
 * RunBenchmark.cs - Implementation of the "RunBenchmark" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.RunBenchmark"/> class executes a
/// particular benchmark and determines its score.</para>
/// </summary>
public class RunBenchmark
{
	// Internal state.
	private IBenchmark benchmark;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.RunBenchmark"/>
	/// instance.</para>
	/// </summary>
	///
	/// <param name="benchmark">
	/// <para>The benchmark to be executed.</para>
	/// </param>
	public RunBenchmark(IBenchmark benchmark)
	{
		this.benchmark = benchmark;
	}

	/// <summary>
	/// <para>Get the name of the benchmark.</para>
	/// </summary>
	///
	/// <value>
	/// <para>A <see cref="T:System.String"/> that contains the name
	/// of the benchmark.</para>
	/// </value>
	public String Name
	{
		get
		{
			return benchmark.Name;
		}
	}

	/// <summary>
	/// <para>Determine the number of runs per second that the benchmark
	/// is capable of executing.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>The number of runs per second for the benchmark.</para>
	/// </returns>
	private int RunsPerSec()
	{
		int count;

		// Initialize the benchmark.
		if(!benchmark.Initialize(0))
		{
			return 0;
		}

		// Start the stop-watch.
		StopWatch stopWatch = new StopWatch();
		stopWatch.Start();

		// Run the benchmark multiple times until a second is passed.
		count = 0;
		while(stopWatch.Elapsed.Ticks < TimeSpan.TicksPerSecond)
		{
			benchmark.Run();
			++count;
		}

		// Stop the stop-watch.
		stopWatch.Stop();

		// Clean up temporary state used by the benchmark.
		benchmark.CleanUp();

		// Return the number of runs per second.
		return count;
	}

	/// <summary>
	/// <para>Run the benchmark for a number of seconds, where the
	/// number of runs per second has already been computed.</para>
	/// </summary>
	///
	/// <param name="runsPerSec">
	/// <para>The number of runs per second.</para>
	/// </param>
	///
	/// <returns>
	/// <para>The elapsed time for running the benchmark.</para>
	/// </returns>
	private TimeSpan Run(int runsPerSec)
	{
		int count;

		// Initialize the benchmark.
		if(!benchmark.Initialize(0))
		{
			return TimeSpan.FromTicks(0);
		}

		// Start the stop-watch.
		StopWatch stopWatch = new StopWatch();
		stopWatch.Start();

		// Run the benchmark for up to 3 seconds.
		count = runsPerSec * 3;
		while(count > 0)
		{
			--count;
			benchmark.Run();
		}

		// Stop the stop-watch.
		stopWatch.Stop();

		// Clean up temporary state used by the benchmark.
		benchmark.CleanUp();

		// Return the elapsed time.
		return stopWatch.Elapsed;
	}

	/// <summary>
	/// <para>Determine the test score for this benchmark.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the score value, or -1 if the benchmark
	/// ran too fast for its score to be measured.</para>
	/// </returns>
	public int GetScore()
	{
		int runs = RunsPerSec();
		if(runs == 0)
		{
			return -1;
		}
		TimeSpan elapsed = Run(runs);
		int ms = (int)(elapsed.Ticks / TimeSpan.TicksPerMillisecond);
		if(ms != 0)
		{
			return ((runs * 3 * benchmark.Magnification) / ms);
		}
		else
		{
			return -1;
		}
	}

} // class RunBenchmark

} // namespace PNetMark
