/*
 * MethodBenchmark.cs - Implementation of the "MethodBenchmark" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.MethodBenchmark"/> class implements
/// the <see cref="T:PNetMark.IBenchmark"/> interface, and runs number
/// of method tests.</para>
/// </summary>
public class MethodBenchmark : IBenchmark
{
	// Internal state.
	public int numIterations;
	public int depthCount;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.MethodBenchmark"/>
	/// instance.</para>
	/// </summary>
	public MethodBenchmark()
	{
		numIterations = 100;
		depthCount = 0;
	}

	/// <summary>
	/// <para>Initialize the benchmark to run on this system.</para>
	/// </summary>
	///
	/// <param name="size">
	/// <para>A value that indicates the size of the benchmark to run.
	/// This should be zero to use the default value.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns <see langword="true"/> if the benchmark could be
	/// initialized, or <see langword="false"/> otherwise.</para>
	/// </returns>
	public bool Initialize(int size)
	{
		if(size != 0)
		{
			numIterations = size;
		}
		return true;
	}

	/// <summary>
	/// <para>Get the name of the benchmark.</para>
	/// </summary>
	///
	/// <value>
	/// <para>A <see cref="T:System.String"/> that contains the name
	/// of the benchmark.</para>
	/// </value>
	public String Name
	{
		get
		{
			return "Method";
		}
	}

	/// <summary>
	/// <para>Get the magnification factor for test scoring.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The magnification factor.</para>
	/// </value>
	public int Magnification
	{
		get
		{
			return 3728;
		}
	}

	/// <summary>
	/// <para>Run the benchmark.</para>
	/// </summary>
	public void Run()
	{
		int iter, sum, temp1, temp2;
		sum = 0;
		for(iter = 0; iter < numIterations; ++iter)
		{
			temp1 = ArithmeticSeries(iter);
			temp2 = NotInlineableSeries(iter);
			if(temp1 > temp2)
			{
				sum += temp2;
			}
			else
			{
				sum += temp1;
			}
		}
	}

	/// <summary>
	/// <para>Compute an arithmetic series.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the result of the series.</para>
	/// </returns>
	public int ArithmeticSeries(int iter)
	{
		if(iter == 0)
		{
			return 0;
		}
		else
		{
			return iter + ArithmeticSeries(iter - 1);
		}
	}

	/// <summary>
	/// <para>Compute an arithmetic series, and attempt to
	/// disable JIT inlining.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the result of the series.</para>
	/// </returns>
	public int NotInlineableSeries(int iter)
	{
		int result = iter;
		++depthCount;
		if(iter == 0)
		{
			return result;
		}
		else if((iter & 1) != 0)
		{
			result += NotInlineableSeries(iter - 1);
		}
		else
		{
			result += 1 + NotInlineableSeries(iter - 1);
		}
		return result;
	}

	/// <summary>
	/// <para>Clean up after the benchmark has been run.</para>
	/// </summary>
	public void CleanUp()
	{
		// Nothing needs to be done here.
	}

} // class MethodBenchmark

} // namespace PNetMark
