/*
 * Timeout.cs - Timeout handling object.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

/// <summary>
/// <para>The <see cref="T:Curses.Timeout"/> class manages the
/// installation and firing of timeouts within curses applications.</para>
///
/// <para>The programmer can observe timeouts in two ways.  The first is
/// to inherit this class and override the <c>OnFired</c> method.  The
/// second is to add an event handler to the <c>Fired</c> event.</para>
/// </summary>
public class Timeout
{
	// Internal state.
	private DateTime nextFireTime;
	private TimeSpan recurTimeSpan;
	private Timeout nextTimeout;
	private Timeout prevTimeout;
	private bool onList;
	private static Timeout timeoutList = null;

	/// <summary>
	/// <para>Create a new <see cref="T:Curses.Timeout"/> instance
	/// that is set to fire once after a certain period of time.</para>
	/// </summary>
	///
	/// <param name="fireTime">
	/// <para>The time to wait before the timeout fires.</para>
	/// </param>
	public Timeout(TimeSpan fireTime)
			{
				nextFireTime = DateTime.UtcNow + fireTime;
				recurTimeSpan = new TimeSpan(-1L);
				Insert();
			}

	/// <summary>
	/// <para>Create a new <see cref="T:Curses.Timeout"/> instance
	/// that is set to fire after a certain period of time, and
	/// then to keep firing periodically.</para>
	/// </summary>
	///
	/// <param name="fireTime">
	/// <para>The time to wait before the timeout fires for the
	/// first time.</para>
	/// </param>
	///
	/// <param name="recurTime">
	/// <para>The time to wait between timeout firings after the
	/// timeout fires for the first time.</para>
	/// </param>
	public Timeout(TimeSpan fireTime, TimeSpan recurTime)
			{
				nextFireTime = DateTime.UtcNow + fireTime;
				recurTimeSpan = recurTime;
				Insert();
			}

	/// <summary>
	/// <para>Insert this object into the timeout event queue.</para>
	/// </summary>
	private void Insert()
			{
				Timeout timeout = timeoutList;
				Timeout last = null;
				while(timeout != null && nextFireTime >= timeout.nextFireTime)
				{
					last = timeout;
					timeout = timeout.nextTimeout;
				}
				prevTimeout = last;
				nextTimeout = timeout;
				if(timeout != null)
				{
					timeout.prevTimeout = this;
				}
				if(last != null)
				{
					last.nextTimeout = this;
				}
				else
				{
					timeoutList = this;
				}
				onList = true;
			}

	/// <summary>
	/// <para>Determine the next time that a timeout should fire.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns the number of ticks, since 1 Jan 0001, for the
	/// time to fire the timeout.  Returns -1 if no timeouts are
	/// currently installed.</para>
	/// </value>
	internal static long TimeToFire
			{
				get
				{
					if(timeoutList != null)
					{
						return timeoutList.nextFireTime.Ticks;
					}
					else
					{
						return -1L;
					}
				}
			}

	/// <summary>
	/// <para>Fire any pending timeouts.</para>
	/// </summary>
	internal static void FirePending()
			{
				DateTime now = DateTime.UtcNow;
				while(timeoutList != null &&
					  now >= timeoutList.nextFireTime)
				{
					timeoutList.Fire();
				}
			}

	/// <summary>
	/// <para>Fire this timeout, and re-install it at a new
	/// position within the timeout list.</para>
	/// </summary>
	private void Fire()
			{
				// Remove the timeout from the list.
				if(nextTimeout != null)
				{
					nextTimeout.prevTimeout = prevTimeout;
				}
				if(prevTimeout != null)
				{
					prevTimeout.nextTimeout = nextTimeout;
				}
				else
				{
					timeoutList = nextTimeout;
				}
				onList = false;

				// Let the application know that the timeout has fired.
				OnFired(this, null);

				// Add the timeout back onto the list if it wasn't removed.
				if(recurTimeSpan.Ticks != -1L)
				{
					nextFireTime += recurTimeSpan;
					Insert();
				}
			}

	/// <summary>
	/// <para>Remove this timeout from the list permanently.</para>
	/// </summary>
	public void Remove()
			{
				if(onList)
				{
					if(nextTimeout != null)
					{
						nextTimeout.prevTimeout = prevTimeout;
					}
					if(prevTimeout != null)
					{
						prevTimeout.nextTimeout = nextTimeout;
					}
					else
					{
						timeoutList = nextTimeout;
					}
				}
				recurTimeSpan = new TimeSpan(-1L);
			}

	/// <summary>
	/// <para>Handle the firing of a timeout.</para>
	/// </summary>
	///
	/// <param name="sender">
	/// <para>The object that sent the fired indication.  Usually
	/// the timeout object itself.</para>
	/// </param>
	///
	/// <param name="args">
	/// <para>The arguments for the event.  Usually
	/// <see langword="null"/>.</para>
	/// </para>
	protected virtual void OnFired(Object sender, EventArgs args)
			{
				if(Fired != null)
				{
					Fired(sender, args);
				}
			}

	/// <summary>
	/// <para>Event that is signalled when the timeout fires.</para>
	/// </summary>
	public event TimeoutFiredEventHandler Fired;

} // class Timeout

} // namespace Curses
