/*****************************************************************************\
 * Copyright 2005, 2006 Niels Lohmann, Christian Gierds, Dennis Reinert      *
 *                                                                           *
 * This file is part of BPEL2oWFN.                                           *
 *                                                                           *
 * BPEL2oWFN is free software; you can redistribute it and/or modify it      *
 * under the terms of the GNU General Public License as published by the     *
 * Free Software Foundation; either version 2 of the License, or (at your    *
 * option) any later version.                                                *
 *                                                                           *
 * BPEL2oWFN is distributed in the hope that it will be useful, but WITHOUT  *
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  *
 * more details.                                                             *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with BPEL2oWFN; if not, write to the Free Software Foundation, Inc., 51   *
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.                      *
\*****************************************************************************/

%{
/*!
 * \file bpel-lexic.c
 *
 * \brief BPEL lexic (implementation)
 *
 * This file defines and implements the lexic of BPEL. It consists mainly of
 * three types of terminal symbols: BPEL keywords as the are defined in the
 * BPEL4WS 1.1 specification, "arbitrary" strings for attribute names and
 * values, and special symbols as XML brackets, whitespace etc.
 *
 * The lexer has two additional "start conditions":
 *  - ATTRIBUTE to avoid lexical errors while reading an attribute
 *  - COMMENT   to merge all XML-comments
 * 
 * \author  
 *          - responsible Niels Lohmann <nlohmann@informatik.hu-berlin.de>
 *          - last changes of \$Author: nlohmann $
 *          
 * \date
 *          - created 2005-11-10
 *          - last changed: \$Date: 2006/01/09 08:38:09 $
 * 
 * \note    This file is part of the tool BPEL2oWFN and was created during the
 *          project "Tools4BPEL" at the Humboldt-Universitt zu Berlin. See
 *          http://www.informatik.hu-berlin.de/top/forschung/projekte/tools4bpel
 *          for details.
 *
 * \note    This file was created using Flex 2.5.4 reading file bpel-lexic.ll.
 *          See http://www.gnu.org/software/flex for details.
 *
 * \version \$Revision: 1.24 $
 *
 * \todo
 *          - add rules to ignored everything non-BPEL
 *          - add a more elegant way to handle XSD-namespaces
 */
%}




 /* flex options */
%option noyywrap
%option yylineno
%option nodefault
%option debug



%{

#include <cstring>


// generated by Kimwitu++
#include "bpel-kc-k.h"       // phylum definitions
#include "bpel-kc-yystype.h" // data types for tokens and non-terminals


// generated by Bison
#include "bpel-syntax.h" // list of all tokens used



extern int yyerror(const char *msg);



/// Store the current start condition of the lexer to return safely after
/// comments.
int currentView;

// two additional views for attributes and comments
#define ATTRIBUTE 1 ///< start condition to allow lexing attributes
#define COMMENT 2   ///< start condition to allow lexing comments
#define XMLHEADER 3 ///< start condition to allow lexing xml headers



%}



namestart		[A-Za-z\200-\377_]
namechar		[A-Za-z\200-\377_0-9.\-:]
name			{namestart}{namechar}*
esc			"&#"[0-9]+";"|"&#x"[0-9a-fA-F]+";"
string			\"([^"]|{esc})*\"
comment			([^-]|"-"[^-])*
xmlheader		([^?]|"-"[^?])*
bpwsns			"bpws:"|"bpel:"


 /* start conditions of the lexer */
%s ATTRIBUTE
%s COMMENT
%s XMLHEADER



%%



 /* comments */
"!--"				{ currentView = YY_START; BEGIN(COMMENT); }
<COMMENT>{comment}		{ /* skip */ }
<COMMENT>"-->"[ \t\r\n]*"<"	{ BEGIN(currentView); }

"?xml"				{ currentView = YY_START; BEGIN(XMLHEADER); }
<XMLHEADER>{xmlheader}		{ /* skip */ }
<XMLHEADER>"?>"[ \t\r\n]*"<"	{ BEGIN(currentView); }


 /* attributes */
<ATTRIBUTE>{name}	{ yylval.yt_casestring = kc::mkcasestring(yytext);
                          return X_NAME; }
<ATTRIBUTE>{string}	{ std::string stringwoquotes = std::string(yytext).substr(1, strlen(yytext)-2);
                          yylval.yt_casestring = kc::mkcasestring(stringwoquotes.c_str());
                          return X_STRING; }
<ATTRIBUTE>"="		{ return X_EQUALS; }


 /* XML-elements */
"<"				{ return X_OPEN; }
"/"				{ return X_SLASH; }
">"				{ BEGIN(INITIAL); return X_CLOSE; }
">"[ \t\r\n]*"<"		{ BEGIN(INITIAL); return X_NEXT; }


 /* names of BPEL-elements */
{bpwsns}?"assign"		{ BEGIN(ATTRIBUTE); return K_ASSIGN; }
{bpwsns}?"case"			{ BEGIN(ATTRIBUTE); return K_CASE; }
{bpwsns}?"catch"		{ BEGIN(ATTRIBUTE); return K_CATCH; }
{bpwsns}?"catchAll"		{ BEGIN(ATTRIBUTE); return K_CATCHALL; }
{bpwsns}?"compensate"		{ BEGIN(ATTRIBUTE); return K_COMPENSATE; }
{bpwsns}?"compensationHandler"	{ return K_COMPENSATIONHANDLER; }
{bpwsns}?"copy"			{ return K_COPY; }
{bpwsns}?"correlation"		{ BEGIN(ATTRIBUTE); return K_CORRELATION; }
{bpwsns}?"correlations"		{ return K_CORRELATIONS; }
{bpwsns}?"correlationSet"	{ BEGIN(ATTRIBUTE); return K_CORRELATIONSET; }
{bpwsns}?"correlationSets"	{ return K_CORRELATIONSETS; }
{bpwsns}?"empty"		{ BEGIN(ATTRIBUTE); return K_EMPTY; }
{bpwsns}?"eventHandlers"	{ return K_EVENTHANDLERS; }
{bpwsns}?"faultHandlers"	{ return K_FAULTHANDLERS; }
{bpwsns}?"flow"			{ BEGIN(ATTRIBUTE); return K_FLOW; }
{bpwsns}?"from"			{ BEGIN(ATTRIBUTE); return K_FROM; }
{bpwsns}?"import"		{ BEGIN(ATTRIBUTE); return K_IMPORT; }
{bpwsns}?"invoke"		{ BEGIN(ATTRIBUTE); return K_INVOKE; }
{bpwsns}?"link"			{ BEGIN(ATTRIBUTE); return K_LINK; }
{bpwsns}?"links"		{ BEGIN(ATTRIBUTE); return K_LINKS; }
{bpwsns}?"onAlarm"		{ BEGIN(ATTRIBUTE); return K_ONALARM; }
{bpwsns}?"onMessage"		{ BEGIN(ATTRIBUTE); return K_ONMESSAGE; }
{bpwsns}?"otherwise"		{ return K_OTHERWISE; }
{bpwsns}?"partner"		{ BEGIN(ATTRIBUTE); return K_PARTNER; }
{bpwsns}?"partnerLink"		{ BEGIN(ATTRIBUTE); return K_PARTNERLINK; }
{bpwsns}?"partnerLinks"		{ return K_PARTNERLINKS; }
{bpwsns}?"partners"		{ return K_PARTNERS; }
{bpwsns}?"pick"			{ BEGIN(ATTRIBUTE); return K_PICK; }
{bpwsns}?"process"		{ BEGIN(ATTRIBUTE); return K_PROCESS; }
{bpwsns}?"receive"		{ BEGIN(ATTRIBUTE); return K_RECEIVE; }
{bpwsns}?"reply"		{ BEGIN(ATTRIBUTE); return K_REPLY; }
{bpwsns}?"scope"		{ BEGIN(ATTRIBUTE); return K_SCOPE; }
{bpwsns}?"sequence"		{ BEGIN(ATTRIBUTE); return K_SEQUENCE; }
{bpwsns}?"source"		{ BEGIN(ATTRIBUTE); return K_SOURCE; }
{bpwsns}?"switch"		{ BEGIN(ATTRIBUTE); return K_SWITCH; }
{bpwsns}?"target"		{ BEGIN(ATTRIBUTE); return K_TARGET; }
{bpwsns}?"terminate"		{ BEGIN(ATTRIBUTE); return K_TERMINATE; }
{bpwsns}?"throw"		{ BEGIN(ATTRIBUTE); return K_THROW; }
{bpwsns}?"to"			{ BEGIN(ATTRIBUTE); return K_TO; }
{bpwsns}?"variable"		{ BEGIN(ATTRIBUTE); return K_VARIABLE; }
{bpwsns}?"variables"		{ return K_VARIABLES; }
{bpwsns}?"wait"			{ BEGIN(ATTRIBUTE); return K_WAIT; }
{bpwsns}?"while"		{ BEGIN(ATTRIBUTE); return K_WHILE; }

 /* white space */
[ \t\r\n]*			{ /* skip white space */ }

{name}				{ yylval.yt_casestring = kc::mkcasestring(yytext);
                                  return X_NAME; }

 /* end of input file */
<<EOF>>				{ return EOF; }


 /* anything else */
.				{ yyerror("lexical error"); }
