// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"
#include <cc++/strchar.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

Fifo fifo;

static Trunk *getTrunk(char *str)
{
	if(!str)
		return NULL;

	return driver->getTrunkPort(atoi(str));
}

bool Fifo::waitPid(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	if(!trunk)
		return false;

	if(!argv[2])
		return false;

	trunk->tgi.pid = atoi(argv[2]);
	return true;
}

bool Fifo::setSymbol(char **argv)
{
	Script::Symbol *sym;
	Trunk *trunk = getTrunk(argv[1]);
	if(!trunk)
		return false;

	if(!argv[3] || !argv[2])
		return false;

	trunk->enterMutex();
	sym = trunk->getLocal(argv[2], trunk->getSymbolSize());
	if(sym)
	{
		if(!sym->flags.readonly)
		{
			snprintf(sym->data, sym->flags.size + 1, "%s",  argv[3]);
			if(sym->flags.commit)
				trunk->commit(sym);
		}
		
	}
	trunk->leaveMutex();
//	trunk->setSymbol(argv[2], trunk->getSymbolSize());
//	trunk->setSymbol(argv[2], argv[3]);
	return true;
}

bool Fifo::reload(char **argv)
{
	Module *mod = Module::cmdFirst;
	int argc = 1;
	bool rts = true;

	if(!argv[1])
	{
		while(mod)
		{
			mod->reload();
			mod = mod->cmdNext;
		}
		return true;
	}

	while(argv[argc])
	{
		mod = getModule(MODULE_FIFO, argv[argc++]);
		if(mod)
			mod->reload();
		else
			rts = false;
	}
	return rts;
}

bool Fifo::submit(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	char buffer[PIPE_BUF / 2];
	TrunkEvent event;
	int argc = 3;
	unsigned seq;
	int len = 0;

	if(!trunk || !argv[2])
		return false;

	seq = atoi(argv[2]);
	while(argv[argc] && len < sizeof(buffer))
	{
		if(argc > 3)
			buffer[len++] = '&';
		strncpy(buffer + len, argv[argc++], sizeof(buffer) - len);
		buffer[sizeof(buffer) - 1] = 0;
		len = strlen(buffer);
	}
	event.id = TRUNK_SERVICE_LOOKUP;
	event.parm.lookup.seq = seq;
	event.parm.lookup.result = true;
	event.parm.lookup.data = buffer;
	return trunk->postEvent(&event);
}

bool Fifo::setLimit(char **argv)
{
	Mixer *mixer = driver->getMixer(atoi(argv[1]));
	Conference *conf;
	int grp = 0;
	int argc = 2;
	bool rts = true;

	if(!mixer || !argv[2])
		return false;

	while(argv[argc] && rts)
	{
		conf = mixer->getConference(grp++);
		if(!conf)
			return false;
		rts = conf->setConference(atoi(argv[argc++]));
	}
	return rts;
}

bool Fifo::setMixer(char **argv)
{
	Mixer *mixer = driver->getMixer(atoi(argv[1]));
	if(!mixer || !argv[2] || !argv[3])
		return false;

	return mixer->setMixer(atoi(argv[2]), atoi(argv[3]));
}

bool Fifo::ringScript(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk || !argv[2])
		return false;

	event.id = TRUNK_RING_START;
	event.parm.argv = &argv[2];

	return trunk->postEvent(&event);
}

bool Fifo::redirectScript(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk || !argv[2])
		return false;

	event.id = TRUNK_RING_REDIRECT;
	event.parm.argv = &argv[2];

	return trunk->postEvent(&event);
}

bool Fifo::reqScript(char **argv)
{
	TrunkGroup *grp;
	char *exp = argv[1];
	
	if(!exp || !argv[2])
		return false;

	grp = getGroup(argv[2]);
	if(!grp || !argv[3])
		return false;

	request(grp, &argv[3], atoi(exp));
	return true;
}

bool Fifo::startScript(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	TrunkGroup *group = getGroup(argv[1]);	
	TrunkEvent event;
	int port = driver->getTrunkCount();
	
	if((!trunk && !group) || !argv[2])
		return false;

	if(trunk && !group)
	{
		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = &argv[2];
		return trunk->postEvent(&event);
	}
	if(!group)
		return false;

	while(port--)
	{
		if(driver->getTrunkGroup(port) != group)
			continue;

		trunk = driver->getTrunkPort(port);
		if(!trunk)
			continue;

		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = &argv[2];
		if(trunk->postEvent(&event))
			return true;
	}
	return false;
}

bool Fifo::postKey(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	char *digits = argv[2];
	TrunkEvent event;
	bool rtn = true;

	if(!trunk || !argv[2])
		return false;

	while(digits && rtn)
	{
		event.id = TRUNK_DTMF_KEYUP;
		event.parm.dtmf.duration = 40;
		event.parm.dtmf.e1 = event.parm.dtmf.e2 = 0;
		switch(*digits)
		{
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			event.parm.dtmf.digit = *digits - '0';
			rtn = trunk->postEvent(&event);
			break;
		case '*':
			event.parm.dtmf.digit = 10;
			rtn = trunk->postEvent(&event);
			break;
		case '#':
			event.parm.dtmf.digit = 11;
			rtn = trunk->postEvent(&event);
			break; 
		default:
			rtn = false;
		}
		if(*(++digits))
#ifdef	COMMON_THREAD_SLEEP
			Thread::sleep(60);
#else
			ccxx_sleep(60);
#endif
	}
	return rtn;
}

bool Fifo::exitPid(char **argv)
{
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk)
		return false;

	event.id = TRUNK_EXIT_SHELL;
	if(argv[2])
		event.parm.status = atoi(argv[2]);
	else
		event.parm.status = 0;

	return trunk->postEvent(&event);
}

bool Fifo::hangupLine(char **argv)
{
	TrunkGroup *group = getGroup(argv[1]);
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;
	int port;

	if(trunk && !group)
	{
		event.id = TRUNK_STOP_DISCONNECT;
		return trunk->postEvent(&event);
	}

	if(!group)
		return false;

	for(port = 0; port < driver->getTrunkCount(); ++port)
	{
		if(driver->getTrunkGroup(port) != group)
			continue;

		trunk = driver->getTrunkPort(port);
		if(!trunk)
			continue;

		event.id = TRUNK_STOP_DISCONNECT;
		trunk->postEvent(&event);
	}
	return true;
}

bool Fifo::busyLine(char **argv)
{
	TrunkGroup *group = getGroup(argv[1]);
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;
	int port;

	if(trunk && !group)
	{
		event.id = TRUNK_MAKE_BUSY;
		return trunk->postEvent(&event);
	}
	if(!group)
		return false;

	for(port = 0; port < driver->getTrunkCount(); ++port)
	{
		if(driver->getTrunkGroup(port) != group)
			continue;

		trunk = driver->getTrunkPort(port);
		if(!trunk)
			continue;

		event.id = TRUNK_MAKE_BUSY;
		trunk->postEvent(&event);
	}
	return true;
}

bool Fifo::setSpan(char **argv)
{
	TrunkEvent event;
	Trunk *trunk;
	unsigned span;
	unsigned port;

	const char *cspan = argv[1];
	const char *mode = argv[2];

	if(!cspan || !mode)
		return false;

	span = atoi(cspan);
	if(!stricmp(mode, "busy"))
		event.id = TRUNK_MAKE_BUSY;
	else if(!stricmp(mode, "idle") || !stricmp(mode, "up"))
		event.id = TRUNK_MAKE_IDLE;
	else if(!stricmp(mode, "stop") || !stricmp(mode, "down"))
		event.id = TRUNK_MAKE_STANDBY;
	else
		return false;

	return driver->spanEvent(span, &event);
}


bool Fifo::setCard(char **argv)
{
        TrunkEvent event;
        Trunk *trunk;
        unsigned card;
        unsigned port;

        const char *ccard = argv[1];
        const char *mode = argv[2];

        if(!ccard || !mode)
                return false;

        card = atoi(ccard);
        if(!stricmp(mode, "busy"))
                event.id = TRUNK_MAKE_BUSY;
        else if(!stricmp(mode, "idle") || !stricmp(mode, "up"))
                event.id = TRUNK_MAKE_IDLE;
        else if(!stricmp(mode, "stop") || !stricmp(mode, "down"))
                event.id = TRUNK_MAKE_STANDBY;
        else
                return false;

        return driver->cardEvent(card, &event);
}

bool Fifo::idleLine(char **argv)
{
	TrunkGroup *group = getGroup(argv[1]);
	Trunk *trunk = getTrunk(argv[1]);
	TrunkEvent event;
	int port;

	if(trunk && !group)
	{
		event.id = TRUNK_MAKE_IDLE;
		return trunk->postEvent(&event);
	}

	if(!group)
		return false;

	for(port = 0; port < driver->getTrunkCount(); ++port)
	{
		if(driver->getTrunkGroup(port) != group)
			continue;

		trunk = driver->getTrunkPort(port);
		if(!trunk)
			continue;

		event.id = TRUNK_MAKE_IDLE;
		trunk->postEvent(&event);
	}
	return true;
}

bool Fifo::setSchedule(char **argv)
{
	if(argv[1])
	{
		strcpy(schedule, argv[1]);
		if(!stricmp(schedule, "*"))
			scheduler.altSchedule(NULL);
		else
			scheduler.altSchedule(schedule);
	}
	else
		scheduler.altSchedule(NULL);
	return true;
}
		
bool Fifo::command(const char *cmd, ostream *fd)
{
	Module *next;
	char buffer[PIPE_BUF / 2];
	bool rts = false;
	char *args[65];
	int argc = 0;
	char **argv = args;
	char *arg;
	char *sp;
	char *ptr;
	const char *token = keyserver.getToken();
	int tlen = strlen(token);

	if(!fd)
		fd = &slog;

	strncpy(buffer, cmd, sizeof(buffer) - 1);
	buffer[sizeof(buffer) - 1] = 0;

	enterMutex();
	slog(Slog::levelDebug) << "fifo: cmd=" << cmd << endl;
	if(strstr(buffer, token))
	{
		ptr = buffer;
		while(isspace(*ptr))
			++ptr;
	
		ptr = strtok_r(ptr, "\n", &sp);	
		while(NULL != (sp = strstr(ptr, token)))
		{
			argv[argc++] = ptr;
			*sp = 0;
			ptr = sp + tlen;
		}
		while(isspace(*ptr))
			++ptr;
		if(*ptr)
			argv[argc++] = ptr;
	}
	else
	{
		argv[argc++] = strtok_r(buffer, " \t\n\r", &sp);
		while(argc < 64)
		{
			arg = strtok_r(NULL, " \t\n\r", &sp);
			if(!arg)
				break;
			argv[argc++] = arg;
		}
	}
	argv[argc] = NULL;

	if(!argv[0])
	{
		leaveMutex();
		return false;
	}

	if(!*argv[0])
	{
		leaveMutex();
		return false;
	}

	next = Module::cmdFirst;

	while(next && !rts)
	{
		rts = next->command(argv, fd);
		next = next->cmdNext;
	}

	if(!stricmp(argv[0], "down"))
	{
		if(argv[1])
			snprintf(service, sizeof(service), "down::%s", argv[1]);
		else
			raise(SIGINT);
		rts = true;
	}
	else if(!stricmp(argv[0], "up"))
		service[0] = 0;
	else if(!stricmp(argv[0], "restart"))
	{
		restart_server = true;
		raise(SIGINT);
		rts = true;
	}
	else if(!stricmp(argv[0], "compile"))
	{
		driver->getImage();
		rts = true;
	}		
	else if(!stricmp(argv[0], "wait"))
		rts = waitPid(argv);
	else if(!stricmp(argv[0], "exit"))
		rts = exitPid(argv);
	else if(!stricmp(argv[0], "set"))
		rts = setSymbol(argv);
	else if(!stricmp(argv[0], "submit"))
		rts = submit(argv);
	else if(!stricmp(argv[0], "debug"))
		rts = debug->debugFifo(argv);
	else if(!stricmp(argv[0], "ring"))
		rts = ringScript(argv);
	else if(!stricmp(argv[0], "redirect"))
		rts = redirectScript(argv);
	else if(!stricmp(argv[0], "busy"))
		rts = busyLine(argv);
	else if(!stricmp(argv[0], "idle"))
		rts = idleLine(argv);
	else if(!stricmp(argv[0], "span"))
		rts = setSpan(argv);
	else if(!stricmp(argv[0], "card"))
		rts = setCard(argv);
	else if(!stricmp(argv[0], "start"))
		rts = startScript(argv);	
	else if(!stricmp(argv[0], "request"))
		rts = reqScript(argv);
	else if(!stricmp(argv[0], "disconnect") || !stricmp(argv[0], "hangup"))
		rts = hangupLine(argv);
	else if(!stricmp(argv[0], "post") || !stricmp(argv[0], "key"))
		rts = postKey(argv);
	else if(!stricmp(argv[0], "schedule"))
		rts = setSchedule(argv);
	else if(!stricmp(argv[0], "mixer"))
		rts = setMixer(argv);
	else if(!stricmp(argv[0], "limit"))
		rts = setLimit(argv);
	else if(!stricmp(argv[0], "reload"))
		rts = reload(argv);
	leaveMutex();
	return rts;
}


#ifdef	CCXX_NAMESPACES
};
#endif
