/* -*- coding: cp852 -*-

   Internet address conversion functions
   comes from Watt-32 project

*/
/*!\file netaddr.c
 *
 *  Provide some more BSD address functions.
 *  Gisle Vanem, Oct 12, 1995
 *
 *  inet_network() is Copyright (c) 1983, 1993
 *  The Regents of the University of California.  All rights reserved.
 *
 *  \version v0.0
 *    \date Jan 11, 1991 \author E. Engelke
 *
 *  \version v0.2
 *    \date Apr 10, 1995 \author G. Vanem, function priv_addr()
 *
 *  \version v0.3
 *    \date Nov 12, 2003 \author G. Vanem, some functions moved from now
 *                       obsolete udp_nds.c.
 */

#include <string.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include "basmac.h"

/**
 * Check if 'str' is simply an ip address.
 * Accepts 'a.b.c.d' or '[a.b.c.d]' forms.
 *  \retval TRUE if 'str' is an IPv4 address.
 */
bool isaddr (const char *str)
{
  int ch;

  while ((ch = *str++) != 0)
  {
    if (isdigit(ch))
       continue;
    if (ch == '.' || ch == ' ' || ch == '[' || ch == ']')
       continue;
    return (FALSE);
  }
  return (TRUE);
}

/**
 * Check if 'str' is a dotless ip address.
 * Accept "[ 0-9\[\]]+". Doesn't accept octal base.
 *  \retval TRUE if 'str' is a dotless address.
 */
bool isaddr_dotless (const char *str, dword *ip)
{
  char  buf[10] = { 0 };
  int   ch, i = 0;
  dword addr;

  while ((ch = *str++) != '\0')
  {
    if (ch == '.' || i == sizeof(buf))
       return (FALSE);

    if (isdigit(ch))
    {
      buf[i++] = ch;
      continue;
    }
    if (ch == ' ' || ch == '[' || ch == ']')
       continue;
    return (FALSE);
  }

  buf[i] = '\0';
  addr = atol (buf);
  if (addr == 0)
     return (FALSE);

  if ((addr % 256) == 0)         /* LSB must be non-zero */
     return (FALSE);

  if (((addr >> 24) % 256) == 0) /* MSB must be non-zero */
     return (FALSE);

  if (ip)
     *ip = addr;
  return (TRUE);
}

/**
 * Converts [a.b.c.d] or a.b.c.d to 32 bit IPv4 address.
 *  \retval 0 on error (safer than -1)
 *  \retval IPv4 address (host order)
 */
dword aton (const char *str)
{
  dword ip = 0;
  int   i;
  char *s = (char*)str;

  if (*s == '[')
     ++s;

  for (i = 24; i >= 0; i -= 8)
  {
    int cur = atoi (s);

    ip |= (dword)(cur & 0xFF) << i;
    if (!i)
       return (ip);

    s = strchr (s, '.');
    if (!s)
       return (0);      /* return 0 on error */
    s++;
  }
  return (0);           /* cannot happen ??  */
}

/**
 * Convert a string into an IP-address.
 *  \retval address on \b host order.
 *  \retval 0 string isn't a (dotless) address.
 */
dword _inet_addr (const char *s)
{
  dword addr = 0;

  if (isaddr(s))
     return aton (s);
  if (isaddr_dotless(s,&addr))
     return (addr);
  return (0);
}

dword inet_addr (const char *addr)
{
  dword ip = htonl (_inet_addr(addr));

  if (ip)
     return (ip);
  return 0;
}

/**
 * Convert an IP-address 'ip' into a string.
 *  \retval 's'.
 *  \note 'ip' is on \b host order.
 */
char *_inet_ntoa (char *s, dword ip)
{
  static char buf[2][20];
  static int idx = 0;

  if (!s)
  {
    s = buf [idx];
    idx ^= 1;
  }
  itoa ((int)(ip >> 24), s, 10);
  strcat (s, ".");
  itoa ((int)(ip >> 16) & 0xFF, strchr(s,'\0'), 10);
  strcat (s, ".");
  itoa ((int)(ip >> 8) & 0xFF, strchr(s,'\0'), 10);
  strcat (s, ".");
  itoa ((int)(ip & 0xFF), strchr(s,'\0'), 10);
  return (s);
}

char * inet_ntoa (struct in_addr addr)
{
  static char buf [4][20];   /* use max 4 at a time */
  static int  idx = 0;
  char       *rc  = buf [idx++];

  idx &= 3;
  return _inet_ntoa (rc, ntohl(addr.s_addr));
}
