#include <bios.h>
#include "mx.h"



/******************************************************************************
                              INTERNAL DEFINITIONS
 ******************************************************************************/


/* timer beat in miliseconds */
#define _MX_BEAT                       100


/* pointer status */
#define _POINTER_IDLE                  0
#define _POINTER_MOVING                1
#define _POINTER_STOPPED               2


/******************************************************************************
                                INTERNAL MACROS
 ******************************************************************************/


#define _WDW(W)                        ((_window *)W)
#define _DLG(D)                        ((_dialog *)D)


#define _UNCLIP_SCREEN()\
   set_clip(screen, 0, 0, VIRTUAL_W, VIRTUAL_H)


/* calculates the intersection of two rectangles */
#define _INTERSECT_RECTS(A, B, R)\
   (R).x1 = MAX((A).x1, (B).x1);\
   (R).y1 = MAX((A).y1, (B).y1);\
   (R).x2 = MIN((A).x2, (B).x2);\
   (R).y2 = MIN((A).y2, (B).y2)


/* checks if two rectangles overlap */
#define _RECTS_OVERLAP(A, B)\
   (!(((B).x1 > (A).x2) ||\
      ((B).y1 > (A).y2) ||\
      ((B).x2 < (A).x1) ||\
      ((B).y2 < (A).y1)))


/* checks if a point is into the given rectangle */
#define _POINT_IN_RECT(R, X, Y)\
   ((X >= (R).x1) &&\
    (X <= (R).x2) &&\
    (Y >= (R).y1) &&\
    (Y <= (R).y2))


/* compares the two rectangles */
#define _RECTS_DIFFER(A, B)\
   (((A).x1 != (B).x1) ||\
    ((A).y1 != (B).y1) ||\
    ((A).x2 != (B).x2) ||\
    ((A).y2 != (B).y2))


/* setups the window info */
#define _SETUP_WINDOW_INFO(WIN, WINFO)\
   WINFO.window = (Window)WIN;\
   WINFO.x1 = WIN->position.x1;\
   WINFO.y1 = WIN->position.y1;\
   WINFO.x2 = WIN->position.x2;\
   WINFO.y2 = WIN->position.y2;\
   WINFO.border_width = WIN->border_width;\
   WINFO.extension = WIN->extension


/* executes the event handler of a window */
#define _CALL_WINDOW_HANDLER(WIN, EVNT) {\
   WindowInfo winfo;\
   _SETUP_WINDOW_INFO(WIN, winfo);\
   WIN->event_handler(&winfo, (Event *)&(EVNT));\
}


/* executes the expose event */
#define _DO_EXPOSE(WIN, EXP, CNT)\
   EXP.type = WindowExpose;\
   EXP.count = CNT;\
   set_clip(screen, EXP.x1, EXP.y1, EXP.x2, EXP.y2);\
   _CALL_WINDOW_HANDLER(WIN, EXP)


/* executes the event handler of a dialog */
#define _CALL_DIALOG_HANDLER(DLG, EVNT)\
   DLG->event_handler((Dialog)DLG, (Event *)&(EVNT))


/* fills a button event structure with specified data */
#define _SETUP_BUTTON_EVENT(E, TYPE, WINX, WINY, WPTR)\
   E.type = TYPE;\
   E.root_x = _mouse_x;\
   E.root_y = _mouse_y;\
   E.window_x = E.root_x - WINX;\
   E.window_y = E.root_y - WINY;\
   E.pointer_child = (Window)WPTR;\
   E.button = _mouse_b;\
   E.clicks = _mouse_clicks;\
   E.time = _button_timer


/* fills a pointer event with specified data */
#define _SETUP_POINTER_EVENT(E, TYPE, WINX, WINY, WPTR, WCRS)\
   E.type = TYPE;\
   E.root_x = _mouse_x;\
   E.root_y = _mouse_y;\
   E.window_x = E.root_x - WINX;\
   E.window_y = E.root_y - WINY;\
   E.pointer_child = (Window)WPTR;\
   E.crossed_child = (Window)WCRS;


/* if a window has grabbed events then all events are reported
   to it independantly of the pointer position */
#define _PROPAGATION_CHILD(WIN)\
   ((WIN->grab_child) ? (WIN->grab_child) : (WIN->pointer_child))


/* checks if a window has grabbed events */
#define _WINDOW_GRABBED_EVENTS(WIN)\
   ((WIN->parent) && (WIN->parent->grab_child == WIN))


/******************************************************************************
                              INTERNAL STRUCTURES
 ******************************************************************************/


typedef struct _cursor {
   BITMAP *bitmap;
   int x;
   int y;
} _cursor;


typedef struct _surface {
   struct _surface *next;
   struct _surface *prev;
   struct _region *region;
   int x1;
   int y1;
   int x2;
   int y2;
} _surface;


typedef struct _region {
   _surface *first;
   _surface *last;
   Rect bounds;
   int count;
} _region;


typedef struct _window {
   struct _window *parent;
   struct _window *lower;
   struct _window *higher;
   struct _window *bottom_child;
   struct _window *top_child;
   struct _window *grab_child;
   struct _window *pointer_child;
   struct _window *dialog_prev;
   struct _window *dialog_next;
   struct _dialog *dialog;
   int x;
   int y;
   int width;
   int height;
   int border_width;
   Rect position;
   Rect viewport;
   Rect clipping;
   Rect clipview;
   _region back_region;
   _region fore_region;
   _cursor *cursor;
   int processed_events;
   int propagated_events;
   void *extension;
   void (*event_handler)(WindowInfo *, Event *);
   Bool is_visible;
   Bool is_drawn;
} _window;


typedef struct _dialog {
   _window *first;
   _window *last;
   _window *grab_window;
   _window *pointer_window;
   _cursor *cursor;
   void (*event_handler)(Dialog, Event *);
   Bool is_drawn;
   Bool is_running;
   Bool to_update;
   Bool to_destroy;
} _dialog;


/******************************************************************************
                              INTERNAL VARIABLES
 ******************************************************************************/


static Bool _lib_init = False;
static void *pointer_sem = Null;
static _cursor *_current_cursor = Null;
static _cursor *_new_cursor = Null;


static int _mouse_x = 0;
static int _mouse_y = 0;
static int _mouse_b = 0;
static int _init_x = 0;
static int _init_y = 0;
static int _old_mouse_x = 0;
static int _old_mouse_y = 0;
static int _old_mouse_b = 0;
static int _mouse_clicks = 0;
static _window *_keyboard_window = Null;


/* lock stuff from here */
static void _dummy(){}
static void _dummy_end();


static int _button_timer = 0;
static int _clicks = 0;
static int _click_timer = 0;
static int _click_timeout = 500;
static int _pointer_motion_status = _POINTER_IDLE;
static int _pointer_motion_timer = 0;
static int _pointer_motion_timeout = 500;


/******************************************************************************
                              INTERNAL FUNCTIONS
 ******************************************************************************/


/* timer */
static void _mx_timer()
{
   /* count time since last button press */
   _button_timer += _MX_BEAT;
   /* catch clicks reset */
   _click_timer += _MX_BEAT;
   if (_click_timer >= _click_timeout) {
      _click_timer = 0;
      _clicks = 0;
   }
   /* handle pointer motion stop trace */
   if (_pointer_motion_status == _POINTER_MOVING) {
      _pointer_motion_timer += _MX_BEAT;
      if (_pointer_motion_timer >= _pointer_motion_timeout) {
         _pointer_motion_timer = 0;
         _pointer_motion_status = _POINTER_STOPPED;
      }
   }
}


/* locked up to here */
static void _dummy_end(){}


/* hides the mouse pointer */
static void _hide_mouse(void *sem)
{
   if (!pointer_sem) {
      pointer_sem = sem;
      show_mouse(Null);
   }
}


/* shows the mouse pointer */
static void _show_mouse(void *sem)
{
   if (pointer_sem == sem) {
      pointer_sem = Null;
      show_mouse(screen);
   }
}


/* adds a surface to the region */
static _surface *_add_surface(_region *rg, _surface *f, int x1, int y1, int x2, int y2)
{
   _surface *s = calloc(1, sizeof(_surface));

   if (s) {
      s->x1 = x1;
      s->y1 = y1;
      s->x2 = x2;
      s->y2 = y2;
      if ((!f) || (!rg->count)) {
         rg->first = s;
         rg->last = s;
      }
      else {
         s->prev = f;
         s->next = f->next;
         if (f->next) f->next->prev = s; else rg->last = s;
         f->next = s;
      }
      s->region = rg;
      rg->count++;
   }
   return s;
}


/* frees a surface */
static void _free_surface(_surface *s)
{
   if (s->prev) s->prev->next = s->next; else s->region->first = s->next;
   if (s->next) s->next->prev = s->prev; else s->region->last = s->prev;
   s->region->count--;
   free(s);
}


/* destroys a region list */
static void _destroy_region_list(_region *rg)
{
   while (rg->first) {
      rg->last = rg->first->next;
      free(rg->first);
      rg->first = rg->last;
   }
   rg->first = Null;
   rg->last = Null;
   rg->count = 0;
}


/* copies a region */
static void _copy_region(_region *src, _region *dst)
{
   _surface *s = src->first;

   _destroy_region_list(dst);
   dst->bounds = src->bounds;
   while (s) {
      _add_surface(dst, dst->last, s->x1, s->y1, s->x2, s->y2);
      s = s->next;
   }
}


/* copies a region clipped to the given rectangle */
static void _copy_region_clipped(_region *src, _region *dst, Rect *clip)
{
   _surface *s = src->first;
   Rect r;

   _destroy_region_list(dst);
   if _RECTS_OVERLAP(src->bounds, *clip) {
      dst->bounds = *clip;
      while (s) {
         if _RECTS_OVERLAP(*s, *clip) {
            _INTERSECT_RECTS(*s, *clip, r);
            _add_surface(dst, dst->last, r.x1, r.y1, r.x2, r.y2);
         }
         s = s->next;
      }
   }
}


/* subtracts a rectangle from a region */
static void _subtract_rect(_region *rg, Rect *rr)
{
   _surface *s;
   _surface *t, *n;
   int max_y1;
   int min_y2;

   if _RECTS_OVERLAP(rg->bounds, *rr) {
      s = rg->first;
      while (s) {
         n = s->next;
         if _RECTS_OVERLAP(*s, *rr) {
            t = s;
            max_y1 = MAX(s->y1, rr->y1);
            min_y2 = MIN(s->y2, rr->y2);
            /* y-banding division */
            if (s->y1 < rr->y1)
               t = _add_surface(rg, t, s->x1, s->y1, s->x2, rr->y1 - 1);
            if (s->x1 < rr->x1)
               t = _add_surface(rg, t, s->x1, max_y1, rr->x1 - 1, min_y2);
            if (rr->x2 < s->x2)
               t = _add_surface(rg, t, rr->x2 + 1, max_y1, s->x2, min_y2);
            if (rr->y2 < s->y2)
               t = _add_surface(rg, t, s->x1, rr->y2 + 1, s->x2, s->y2);
            _free_surface(s);
         }
         s = n;
      }
   }
}


/* optimizes a region */
static void _optimize_region(_region *rg)
{
   _surface *s = rg->first;
   _surface *f, *n;

   while (s) {
      n = s->next;
      f = n;
      while (f) {
         if ((s->x1 == f->x1) && (s->x2 == f->x2)) {
            if (s->y2 == f->y1 - 1) {
               f->y1 = s->y1;
               _free_surface(s);
            }
            else
            if (s->y1 == f->y2 + 1) {
               f->y2 = s->y2;
               _free_surface(s);
            }
         }
         else
         if ((s->y1 == f->y1) && (s->y2 == f->y2)) {
            if (s->x2 == f->x1 - 1) {
               f->x1 = s->x1;
               _free_surface(s);
            }
            else
            if (s->x1 == f->x2 + 1) {
               f->x2 = s->x2;
               _free_surface(s);
            }
         }
         f = f->next;
      }
      s = n;
   }
}


/* checks if the given x, y point lies into the region */
static Bool _check_xy_in_region(_region *rg, int x, int y)
{
   Bool retval = False;
   _surface *s;

   if _POINT_IN_RECT(rg->bounds, x, y) {
      s = rg->first;
      while (s) {
         if _POINT_IN_RECT(*s, x, y) {
            retval = True;
            s = Null;
         }
         else s = s->next;
      }
   }
   return retval;
}


/* returns the child from the given pos */
static _window *_get_child_from_pos(_window *par, int *pos)
{
   _window *retval = Null;
   _window *tmp;
   int count;

   if (*pos < 0) {
      count = -1;
      tmp = par->bottom_child;
      while (tmp) {
         if (count == *pos) {
            retval = tmp;
            tmp = Null;
         }
         else {
            count--;
            if ((!tmp->higher) && (count == *pos)) {
               *pos = -*pos;
               retval = tmp;
               tmp = Null;
            }
            else tmp = tmp->higher;
         }
      }
   }
   else {
      count = 1;
      tmp = par->top_child;
      while (tmp) {
         if (count == *pos) {
            retval = tmp;
            tmp = Null;
         }
         else {
            count++;
            if ((!tmp->lower) && (count == *pos)) {
               *pos = -*pos;
               retval = tmp;
               tmp = Null;
            }
            else tmp = tmp->lower;
         }
      }
   }
   return retval;
}


/* calculates the geometry of a window tree */
static void _calc_geometry(_window *win)
{
   _window *tmp;

   if (win->parent) {
      win->position.x1 = win->x + win->parent->viewport.x1;
      win->position.y1 = win->y + win->parent->viewport.y1;
      win->position.x2 = win->position.x1 + win->width - 1;
      win->position.y2 = win->position.y1 + win->height - 1;
      if ((win->parent->is_visible) &&
         _RECTS_OVERLAP(win->position, win->parent->clipview)) {
         _INTERSECT_RECTS(win->position, win->parent->clipview, win->clipping);
         win->is_visible = True;
      }
      else
         win->is_visible = False;
   }
   else {
      win->x = 0;
      win->y = 0;
      win->width = SCREEN_W;
      win->height = SCREEN_H;
      win->position.x1 = 0;
      win->position.y1 = 0;
      win->position.x2 = win->width - 1;
      win->position.y2 = win->height - 1;
      win->clipping = win->position;
      win->is_visible = True;
   }
   win->viewport.x1 = win->position.x1 + win->border_width;
   win->viewport.y1 = win->position.y1 + win->border_width;
   win->viewport.x2 = win->position.x2 - win->border_width;
   win->viewport.y2 = win->position.y2 - win->border_width;
   _INTERSECT_RECTS(win->viewport, win->clipping, win->clipview);
   tmp = win->top_child;
   while (tmp) {
      _calc_geometry(tmp);
      tmp = tmp->lower;
   }
}


/* subtracts the given window list from the given region */
static void _subtract_windows_list(_region *rg, _window *win)
{
   while (win) {
      if ((win->is_visible) &&
         _RECTS_OVERLAP(rg->bounds, win->clipping))
         _subtract_rect(rg, &win->clipping);
      win = win->higher;
   }
}


/* calculates the surfaces of a single window */
static void _calc_window_surfaces(_window *win)
{
   _destroy_region_list(&win->fore_region);
   _destroy_region_list(&win->back_region);
   if (win->is_visible) {
      /* calculate foreground region */
      if (win->parent) {
         _copy_region_clipped(&win->parent->fore_region, &win->fore_region,
            &win->clipping);
      }
      else {
         win->fore_region.bounds = win->clipping;
         _add_surface(&win->fore_region, win->fore_region.last,
            win->clipping.x1,
            win->clipping.y1,
            win->clipping.x2,
            win->clipping.y2);
      }
      if (win->higher) _subtract_windows_list(&win->fore_region, win->higher);
      _optimize_region(&win->fore_region);
      /* calculate background region */
      _copy_region(&win->fore_region, &win->back_region);
      if (win->bottom_child) _subtract_windows_list(&win->back_region,
         win->bottom_child);
      _optimize_region(&win->back_region);
   }
}


/* calculates the surfaces of the window tree */
static void _calc_surfaces(_window *win)
{
   _window *tmp;

   _calc_window_surfaces(win);
   tmp = win->top_child;
   while (tmp) {
      _calc_surfaces(tmp);
      tmp = tmp->lower;
   }
}


/* subtracts the given rectangle from the window tree */
static void _subtract_rect_from_windows(_window *win, Rect *rr)
{
   _window *tmp;

   _subtract_rect(&win->fore_region, rr);
   _subtract_rect(&win->back_region, rr);
   _optimize_region(&win->fore_region);
   _optimize_region(&win->back_region);
   tmp = win->top_child;
   while (tmp) {
      if ((tmp->is_visible) && _RECTS_OVERLAP(tmp->clipping, *rr))
      _subtract_rect_from_windows(tmp, rr);
      tmp = tmp->lower;
   }
}


/* initializes the window rendered */
static void _init_render_window(_window *win)
{
   _window *tmp;

   _calc_geometry(win);
   _calc_surfaces(win);
   if (win->is_visible) {
      if (win->parent) {
         _subtract_rect(&win->parent->back_region, &win->clipping);
         _optimize_region(&win->parent->back_region);
         tmp = win->lower;
         while (tmp) {
            if ((tmp->is_visible) && _RECTS_OVERLAP(win->clipping, tmp->clipping))
               _subtract_rect_from_windows(tmp, &win->clipping);
            tmp = tmp->lower;
         }
      }
      RefreshWindow((Window)win);
   }
}


/* recursively makes the window undrawable */
static void _undraw_window(_window *win)
{
   _window *tmp;

   win->is_visible = False;
   win->is_drawn = False;
   _destroy_region_list(&win->fore_region);
   _destroy_region_list(&win->back_region);
   tmp = win->top_child;
   while (tmp) {
      _undraw_window(tmp);
      tmp = tmp->lower;
   }
}


/* returns the times the given rectangle intersects with the
   given list of surfaces */
static inline int _get_intersect_count(_surface *s, Rect *rr)
{
   int count = 0;

   while (s) {
      if _RECTS_OVERLAP(*s, *rr) count++;
      s = s->next;
   }
   return count;
}


/* redraws a window clipped to the given rectangle */
static void _redraw_clipped_by_rect(_window *win, Rect *rr)
{
   Bool retval = win->is_visible;
   ExposeEvent expose;
   _surface *s;
   int count;

   if ((screen) && (retval)) {
      _hide_mouse(win);
      /* show window */
      s = win->back_region.first;
      count = _get_intersect_count(s, rr);
      while (s) {
         if _RECTS_OVERLAP(*s, *rr) {
            _INTERSECT_RECTS(*s, *rr, expose);
            _DO_EXPOSE(win, expose, count);
            count--;
         }
         s = s->next;
      }
      _show_mouse(win);
      _UNCLIP_SCREEN();
   }
}


/* redraws a window tree clipped to the given rectangle */
static void _redraw_tree_clipped_by_rect(_window *win, Rect *rr)
{
   _window *tmp;

   tmp = win->top_child;
   while (tmp) {
      if ((tmp->is_visible) && _RECTS_OVERLAP(tmp->clipping, *rr))
         _redraw_tree_clipped_by_rect(tmp, rr);
      tmp = tmp->lower;
   }
   _redraw_clipped_by_rect(win, rr);
}


/* when a window is removed the windows under it are exposed */
static void _unrender_window(_window *win, _window *par, _window *lower)
{
   _window *tmp;

    /* destroy visibility */
   _undraw_window(win);
    /* update background */
    tmp = lower;
    while (tmp) {
       if ((tmp->is_visible) &&
          _RECTS_OVERLAP(win->clipping, tmp->clipping)) {
          _calc_surfaces(tmp);
          _redraw_tree_clipped_by_rect(tmp, &win->clipping);
       }
       tmp = tmp->lower;
    }
    _calc_window_surfaces(par);
    _redraw_clipped_by_rect(par, &win->clipping);
}


/* inserts the given window under the given parent */
static Bool _insert_window(_window *win, _window *par, int pos)
{
   Bool retval = False;
   _window *tmp;

   if (!win->parent) {
      if (!par->top_child) {
         par->bottom_child = win;
         par->top_child = win;
         win->parent = par;
         retval = True;
      }
      else if (pos != 0) {
         tmp = _get_child_from_pos(par, &pos);
         if (tmp) {
           if (pos < 0) {
              /* join before */
              win->lower = tmp->lower;
              win->higher = tmp;
              if (tmp->lower) tmp->lower->higher = win; else par->bottom_child = win;
              tmp->lower = win;
           }
           else {
              /* join after */
              win->lower = tmp;
              win->higher = tmp->higher;
              if (tmp->higher) tmp->higher->lower = win; else par->top_child = win;
              tmp->higher = win;
           }
           win->parent = par;
           retval = True;
         }
      }
   }
   return retval;
}


/* unlinks a window from the window tree */
static Bool _remove_window(_window *win)
{
   Bool retval = False;

   if (win->parent) {
      if (win->lower)
         win->lower->higher = win->higher;
      else
         win->parent->bottom_child = win->higher;
      if (win->higher)
         win->higher->lower = win->lower;
      else
         win->parent->top_child = win->lower;
      win->parent = Null;
      win->lower = Null;
      win->higher = Null;
      retval = True;
   }
   return retval;
}


/* changes the zorder of the window */
static int _switch_z_order(_window *win, int pos)
{
   int retval = GetWindowZOrder((Window)win, pos);
   _window *par = win->parent;
   Bool b = False;

   if (pos != retval) {
      _remove_window(win);
      b = _insert_window(win, par, pos);
   }
   if (!b) {
      _insert_window(win, par, retval);
      retval = False;
   }
   return retval;
}


/* destroys each window into the tree */
static void _destroy_window(_window *win)
{
   _window *tmp = win->top_child;

   while (tmp) {
     _destroy_window(tmp);
     tmp = tmp->lower;
   }
   free(win);
}


/* releases events for the given window recursing downwards the window
   tree */
static void _release_events_downwards(_window *win)
{
   if (win->grab_child) {
      _release_events_downwards(win->grab_child);
      if (win->grab_child->dialog) {
        win->grab_child->dialog->grab_window = Null;
        win->grab_child->dialog->to_update = True;
      }
      win->grab_child = Null;
   }
}


/* releases events for the given window recursing upwards the window
   tree */
static Bool _release_events_upwards(_window *win)
{
   Bool retval = True;

   if (win->parent) {
      retval = _release_events_upwards(win->parent);
     if (retval) {
        if (win->parent->grab_child == win) {
           win->parent->grab_child = Null;
           if (win->dialog) {
              win->dialog->grab_window = Null;
              win->dialog->to_update = True;
           }
        }
        else
           retval = False;
     }
   }
   return retval;
}


/* goes through the window tree to setup the 'pointer_child' field */
static void _setup_pointer_children(_window *win)
{
   win->pointer_child = (_window *)GetXYChild((Window)win, _mouse_x, _mouse_y);
   if (win->pointer_child) _setup_pointer_children(win->pointer_child);
}


/* adds a dialog to a window */
static Bool _add_window_to_dialog(_dialog *dlg, _window *win)
{
   Bool retval = False;

   if (!win->dialog) {
      if (!dlg->first) {
         dlg->first = win;
         dlg->last = win;
      }
      else {
         win->dialog_prev = dlg->last;
         dlg->last->dialog_next = win;
         dlg->last = win;
      }
      win->dialog = dlg;
      retval = True;
   }
   return retval;
}


/* removes a window from a dialog; the window must be the top dialog
   window */
static Bool _remove_window_from_dialog(_window *win)
{
   Bool retval = False;

   if (win->dialog) {
      if (win->dialog->grab_window == win)
         ReleaseEvents((Window)win->dialog->grab_window);
      if (win->dialog_prev)
         win->dialog_prev->dialog_next = win->dialog_next;
      else
         win->dialog->first = win->dialog_next;
      if (win->dialog_next)
         win->dialog_next->dialog_prev = win->dialog_prev;
      else
         win->dialog->last = win->dialog_prev;
      win->dialog_prev = Null;
      win->dialog_next = Null;
      win->dialog = Null;
      retval = True;
   }
   return retval;
}


/* destroys a dialog */
static void _destroy_dialog(_dialog **dlg)
{
   _window *tmp = (*dlg)->last, *t;

   if ((*dlg)->grab_window) ReleaseEvents((Window)(*dlg)->grab_window);
   while (tmp) {
      t = tmp->dialog_prev;
      tmp->dialog_prev = Null;
      tmp->dialog = Null;
      tmp = t;
   }
   free(*dlg);
   *dlg = Null;
}


/* renders all windows of a dialog */
static void _init_render_dialog(_dialog *dlg)
{
  _window *tmp = dlg->last;

  while (tmp) {
     if (!tmp->is_drawn) _init_render_window(tmp);
     if ((tmp->is_visible) &&
        (_check_xy_in_region(&tmp->fore_region, _mouse_x, mouse_y)))
        _setup_pointer_children(tmp);
     tmp = tmp->dialog_prev;
  }
  dlg->is_drawn = True;
}


/* returns the dialog window under the pointer or the one that
   has grabbed events */
static _window *_get_dialog_pointer_window(_dialog *dlg)
{
   _window *retval = Null;
   _window *tmp;

   if (dlg->grab_window)
      retval = dlg->grab_window;
   else {
      tmp = dlg->last;
      while (tmp) {
         if ((tmp->is_visible) &&
            (_check_xy_in_region(&tmp->fore_region, _mouse_x, _mouse_y))) {
            retval = tmp;
            tmp = Null;
         }
         else tmp = tmp->dialog_prev;
      }
   }
   dlg->pointer_window = retval;
   return retval;
}


/* propagates the button event down the window tree */
static void _propagate_button_event(_window *win, int type)
{
   ButtonEvent ebtn;
   _window *tmp;

   /* execute event */
   if (win->processed_events & type) {
      _SETUP_BUTTON_EVENT(ebtn, type, win->position.x1, win->position.y1,
         win->pointer_child);
      _CALL_WINDOW_HANDLER(win, ebtn);
   }

   /* pass event to child */
   tmp = _PROPAGATION_CHILD(win);
   if ((tmp) && (win->propagated_events & type))
      _propagate_button_event(tmp, type);
}


/* executes a button event */
static void _do_button_event(_dialog *dlg, int type)
{
   _window *tmp;
   ButtonEvent ebtn;

   tmp = _get_dialog_pointer_window(dlg);
   if (tmp)
      _propagate_button_event(tmp, type);
   else if (dlg->event_handler) {
      _SETUP_BUTTON_EVENT(ebtn, type, 0, 0, Null);
      _CALL_DIALOG_HANDLER(dlg, ebtn);
   }
}


/* propagates a pointer event from highest to lowest to highest window */
static void _propagate_pointer_event_downwards(_window *win, int type)
{
   PointerEvent eptr;
   _window *tmp;

   if (win->cursor) _new_cursor = win->cursor;
   /* execute event */
   if (win->processed_events & type) {
      _SETUP_POINTER_EVENT(eptr, type, win->position.x1, win->position.y1,
         win->pointer_child, Null);
      _CALL_WINDOW_HANDLER(win, eptr);
   }

   /* pass event to child */
   tmp = _PROPAGATION_CHILD(win);
   if ((tmp) && (win->propagated_events & type))
      _propagate_pointer_event_downwards(tmp, type);
}


/* propagates a pointer event from lowest to highest to highest window */
static void _propagate_pointer_event_upwards(_window *win, int type)
{
   PointerEvent eptr;
   _window *tmp;

   /* pass event to child */
   tmp = _PROPAGATION_CHILD(win);
   if ((tmp) && (win->propagated_events & type))
      _propagate_pointer_event_upwards(tmp, type);

   /* execute event */
   if (win->processed_events & type) {
      _SETUP_POINTER_EVENT(eptr, type, win->position.x1, win->position.y1,
         win->pointer_child, Null);
      _CALL_WINDOW_HANDLER(win, eptr);
   }
}


/* propagates an enter event */
static void _propagate_pointer_enter(_window *win)
{
   PointerEvent eptr;

   if (win->cursor) _new_cursor = win->cursor;

   /* get child under mouse */
   win->pointer_child = (_window *)GetXYChild((Window)win, _mouse_x, _mouse_y);

   /* execute event */
   if (win->processed_events & PointerEnter) {
      _SETUP_POINTER_EVENT(eptr, PointerEnter, win->position.x1,
         win->position.y1, win->pointer_child, Null);
      _CALL_WINDOW_HANDLER(win, eptr);
   }

   /* pass event to child */
   if (win->propagated_events & PointerEnter) {

      /* if a child has grabbed events then pass the event only
         when the pointer entered the child */
      if (win->grab_child) {
         if (_check_xy_in_region(&win->grab_child->fore_region,
            _mouse_x, _mouse_y))
            _propagate_pointer_enter(win->grab_child);
      }

      /* else if there is a child under pointer propagate */
      else if (win->pointer_child)
         _propagate_pointer_enter(win->pointer_child);
   }
}


/* propagates a motion event */
static void _propagate_pointer_move(_window *win)
{
   _window *prev = win->pointer_child;
   PointerEvent eptr;

   if (win->cursor) _new_cursor = win->cursor;
   /* get child under mouse */
   win->pointer_child = (_window *)GetXYChild((Window)win, _mouse_x, _mouse_y);
   /* execute event */
   if (win->processed_events & PointerMove) {
      _SETUP_POINTER_EVENT(eptr, PointerMove, win->position.x1,
         win->position.y1, win->pointer_child, prev);
      _CALL_WINDOW_HANDLER(win, eptr);
   }

   /* if some child has grabbed events */
   if (win->grab_child) {
      /* pointer switched child */
      if (prev != win->pointer_child) {
         if ((win->propagated_events & PointerLeave) &&
             (prev == win->grab_child))
            _propagate_pointer_event_upwards(win->grab_child, PointerLeave);
         else
         if ((win->propagated_events & PointerEnter) &&
             (win->pointer_child == win->grab_child))
            _propagate_pointer_enter(win->grab_child);
      }

      /* pointer has not switched child */
      else if (win->propagated_events & PointerMove)
            _propagate_pointer_move(win->grab_child);
   }

   /* no grab child */
   else {
      /* pointer switched child */
      if (prev != win->pointer_child) {
         if ((win->propagated_events & PointerLeave) && (prev))
            _propagate_pointer_event_upwards(prev, PointerLeave);
         if ((win->propagated_events & PointerEnter) && (win->pointer_child))
            _propagate_pointer_enter(win->pointer_child);
      }

      /* pointer is in the same child */
      else if ((win->propagated_events & PointerMove) && (prev))
            _propagate_pointer_move(win->pointer_child);
   }
}


/* sets the given mouse pointer */
static void _set_cursor(_cursor *cursor)
{
   if (cursor) {
      if (_current_cursor != cursor) {
         _current_cursor = cursor;
         show_mouse(Null);
         set_mouse_sprite(cursor->bitmap);
         set_mouse_sprite_focus(cursor->x, cursor->y);
         show_mouse(screen);
      }
   }
   else {
      set_mouse_sprite(Null);
   }
}


/* executes a pointer event */
static void _do_pointer_event(_dialog *dlg, int type)
{
   PointerEvent eptr;
   _window *tmp;
   _window *prev = dlg->pointer_window;

   _new_cursor = _current_cursor;
   tmp = _get_dialog_pointer_window(dlg);
   switch (type) {
      case PointerStart:
      case PointerStop:
         if (tmp)
            _propagate_pointer_event_downwards(tmp, type);
         else {
            if (dlg->cursor) _new_cursor = dlg->cursor;
             if (dlg->event_handler) {
                _SETUP_POINTER_EVENT(eptr, type, 0, 0, Null, Null);
                _CALL_DIALOG_HANDLER(dlg, eptr);
             }
         }
         break;
      case PointerMove:
         if (prev != dlg->pointer_window) {
            if (prev) _propagate_pointer_event_upwards(prev, PointerLeave);
            if (dlg->pointer_window)
               _propagate_pointer_enter(dlg->pointer_window);
         }
         else if (prev)
            _propagate_pointer_move(prev);
         else {
            if (dlg->cursor) _new_cursor = dlg->cursor;
            if (dlg->event_handler) {
               _SETUP_POINTER_EVENT(eptr, type, 0, 0, Null, Null);
               _CALL_DIALOG_HANDLER(dlg, eptr);
            }
         }
         break;
   }
   /* set cursor on pointer event */
   if (_new_cursor != _current_cursor) _set_cursor(_new_cursor);
}


/******************************************************************************
                                  FUNCTIONS
 ******************************************************************************/


/* initializes the library */
void InitMX()
{
   if (!_lib_init) {
      _lib_init = True;
      srandom(biostime(0, 0));
      allegro_init();
      install_keyboard();
      install_timer();
      install_mouse();
      LOCK_FUNCTION(_dummy);
      install_int(_mx_timer, _MX_BEAT);
   }
}


/* inserts a window under the given parent */
Bool InsertWindow(Window window, Window parent, int pos)
{
   Bool retval = False;

   retval = _insert_window(_WDW(window), _WDW(parent), pos);
   if ((retval) && (_WDW(parent)->is_drawn)) _init_render_window(_WDW(window));
   return retval;
}


/* removes a window from its parent */
Bool RemoveWindow(Window window)
{
   _window *lower = _WDW(window)->lower;
   _window *par = _WDW(window)->parent;
   Bool retval = False;

   if (par) {
      if (par->grab_child == _WDW(window)) ReleaseEvents(window);
      retval = _remove_window(_WDW(window));
      if (retval) {
         if (_WDW(window)->dialog) _remove_window_from_dialog(_WDW(window));
         _unrender_window(_WDW(window), par, lower);
      }
   }
   return retval;
}


/* creates a window */
Window CreateWindow(
   Window parent,
   void *extension,
   void (*event_handler)(WindowInfo *, Event *),
   int x,
   int y,
   int width,
   int height,
   int border_width,
   int processed_events,
   int propagated_events)
{
   _window *win = calloc(1, sizeof(_window));

   if (win) {
      win->extension = extension;
      win->event_handler = event_handler;
      win->x = x;
      win->y = y;
      win->width = width;
      win->height = height;
      win->border_width = border_width;
      win->processed_events = processed_events;
      win->propagated_events = propagated_events;
      if (parent) InsertWindow((Window)win, parent, 1);
   }
   return (Window)win;
}


/* creates a root window */
Window CreateRootWindow(
   void *extension,
   void (*event_handler)(WindowInfo *, Event *))
{
   return CreateWindow(Null, extension, event_handler, 0, 0, 320, 200, 0,
      0xFFFFFFFF, 0xFFFFFFFF);
}


/* destroys a window */
void DestroyWindow(Window *window)
{
   RemoveWindow(*window);
   _destroy_window((_window *)&(*window));
   window = Null;
}


/* refreshes the whole window tree with the current geometry */
Bool RefreshWindow(Window window)
{
   Bool retval = _WDW(window)->is_visible;
   ExposeEvent expose;
   _window *tmp;
   _surface *s;
   int count;

   if ((screen) && (retval)) {
      _hide_mouse(_WDW(window));
      /* show children */
      tmp = _WDW(window)->top_child;
      while (tmp) {
         RefreshWindow((Window)tmp);
         tmp = tmp->lower;
      }
      /* show window */
      s = _WDW(window)->back_region.first;
      count = _WDW(window)->back_region.count;
      while (s) {
         expose.x1 = s->x1;
         expose.y1 = s->y1;
         expose.x2 = s->x2;
         expose.y2 = s->y2;
         _DO_EXPOSE(_WDW(window), expose, count);
         s = s->next;
         count--;
      }
      _WDW(window)->is_drawn = True;
      _show_mouse(_WDW(window));
      _UNCLIP_SCREEN();
   }
   return retval;
}


/* paints only the window without its children */
Bool PaintWindow(Window window)
{
   Bool retval = _WDW(window)->is_visible;
   ExposeEvent expose;
   _surface *s;
   int count;

   if ((screen) && (retval)) {
      _hide_mouse(_WDW(window));
      /* show window */
      s = _WDW(window)->back_region.first;
      count = _WDW(window)->back_region.count;
      while (s) {
         expose.x1 = s->x1;
         expose.y1 = s->y1;
         expose.x2 = s->x2;
         expose.y2 = s->y2;
         _DO_EXPOSE(_WDW(window), expose, count);
         s = s->next;
         count--;
      }
      _WDW(window)->is_drawn = True;
      _show_mouse(_WDW(window));
      _UNCLIP_SCREEN();
   }
   return retval;
}


/* applies the given drawing function into the window */
Bool DoWindowDrawing(Window window, Bool overwrite_children,
   void (*paint_function)(WindowInfo *, ExposeEvent *))
{
   WindowInfo winfo;
   _region *rg = (overwrite_children) ?
      (&_WDW(window)->fore_region) :
      (&_WDW(window)->back_region);
   Bool retval = _WDW(window)->is_visible & rg->count;
   ExposeEvent expose;
   _surface *s;
   int count;

   if ((screen) && (retval)) {
      _hide_mouse(_WDW(window));
      s = rg->first;
      count = rg->count;
      while (s) {
         _SETUP_WINDOW_INFO(_WDW(window), winfo);
         expose.type = WindowExpose;
         expose.count = count;
         expose.x1 = s->x1;
         expose.y1 = s->y1;
         expose.x2 = s->x2;
         expose.y2 = s->y2;
         set_clip(screen, expose.x1, expose.y1, expose.x2, expose.y2);
         paint_function(&winfo, &expose);
         s = s->next;
         count--;
      }
      _show_mouse(_WDW(window));
      _UNCLIP_SCREEN();
   }
   return retval;
}


/* translates coordinates from one window to another;
   if one of the windows is not given then the coordinates
   related to it are considered global */
void TranslateCoords(Window src, Window dst, int x, int y,
   int *outx, int *outy)
{
   if (src) {
      x += _WDW(src)->position.x1;
      y += _WDW(src)->position.y1;
   }
   if (dst) {
      x -= _WDW(dst)->position.x1;
      y -= _WDW(dst)->position.y1;
   }
   *outx = x;
   *outy = y;
}


/* updates the geometry of a window */
void UpdateWindowGeometry(Window window)
{
   Rect prev_position = _WDW(window)->position;
   Rect prev_clipping = _WDW(window)->clipping;
   Rect prev_viewport = _WDW(window)->viewport;
   _window *tmp;

   _calc_geometry(_WDW(window));
   if _RECTS_DIFFER(prev_position, _WDW(window)->position) {
      /* redraw window */
      _calc_surfaces(_WDW(window));
      RefreshWindow(window);
      if (_WDW(window)->parent) {
         /* redraw background */
         tmp = _WDW(window)->lower;
         while (tmp) {
            if ((tmp->is_visible) &&
               (_RECTS_OVERLAP(prev_clipping, tmp->clipping) ||
               _RECTS_OVERLAP(_WDW(window)->clipping, tmp->clipping))) {
               _calc_surfaces(tmp);
               _redraw_tree_clipped_by_rect(tmp, &prev_clipping);
            }
            tmp = tmp->lower;
         }
         _calc_window_surfaces(_WDW(window)->parent);
         _redraw_clipped_by_rect(_WDW(window)->parent, &prev_clipping);
      }
   }
   else if _RECTS_DIFFER(prev_viewport, _WDW(window)->viewport) {
      /* redraw window */
      _calc_surfaces(_WDW(window));
      RefreshWindow(window);
   }
}


/* installs the root window and subwindows at the current video mode */
Bool InstallRootWindow(Window window)
{
   Bool retval = False;

   if (!_WDW(window)->parent) {
      _init_render_window(_WDW(window));
      retval = True;
   }
   return retval;
}


/* changes the depth of the given window to the given one */
Bool SetWindowZOrder(Window window, int pos)
{
   Bool retval = False;
   int old_pos;
   _window *prev_lower;
   _window *prev_higher;
   _window *tmp;

   if ((pos) && (_WDW(window)->parent)) {
      prev_lower = _WDW(window)->lower;
      prev_higher = _WDW(window)->higher;
      old_pos = _switch_z_order(_WDW(window), pos);
      if ((old_pos) && (old_pos != pos) && (_WDW(window)->is_visible) &&
         (_WDW(window)->is_drawn)) {
         /* window raised */
         if (pos < old_pos) {
            /* calculate surfaces of lower windows */
            tmp = _WDW(window)->lower;
            while (tmp != prev_higher->lower) {
               if ((tmp->is_visible) &&
                  _RECTS_OVERLAP(tmp->clipping, _WDW(window)->clipping))
                  _calc_surfaces(tmp);
               tmp = tmp->lower;
            }
            /* redraw window */
            _calc_surfaces(_WDW(window));
            RefreshWindow(window);
         }
         /* window lowered */
         else {
            /* calculate surfaces of window */
            _calc_surfaces(_WDW(window));
            /* redraw exposed windows */
            tmp = prev_lower;
            while (tmp != _WDW(window)) {
               if ((tmp->is_visible) &&
                  _RECTS_OVERLAP(tmp->clipping, _WDW(window)->clipping)) {
                  _calc_surfaces(tmp);
                  _redraw_tree_clipped_by_rect(tmp, &_WDW(window)->clipping);
               }
               tmp = tmp->lower;
            }
         }
         /* set the pointer child of this parent */
         _setup_pointer_children(_WDW(window)->parent);
         retval = True;
      }
   }
   return retval;
}


/* sets the window's cursor */
void SetWindowCursor(Window window, Cursor cursor)
{
   _window *tmp;

   _WDW(window)->cursor = (_cursor *)cursor;
   if (_check_xy_in_region(&_WDW(window)->fore_region, _mouse_x, _mouse_y)) {
       tmp = (_window *)GetXYChild(window, _mouse_x, _mouse_y);
       if ((!tmp) || (!tmp->cursor)) _set_cursor((_cursor *)cursor);
   }
}


/* sets the window's x position */
void SetWindowX(Window window, int val)
{
   _WDW(window)->x = val;
}


/* sets the window's y position */
void SetWindowY(Window window, int val)
{
   _WDW(window)->y = val;
}


/* sets the window's width */
void SetWindowWidth(Window window, int val)
{
   _WDW(window)->width = val;
}


/* sets the window's height */
void SetWindowHeight(Window window, int val)
{
   _WDW(window)->height = val;
}


/* sets the window's border width */
void SetWindowBorderWidth(Window window, int val)
{
   _WDW(window)->border_width = val;
}


/* sets the window's event flags */
void SetWindowInput(Window window, int val)
{
   _WDW(window)->processed_events = val;
}


/* sets the window's event flags */
void SetWindowPropagation(Window window, int val)
{
   _WDW(window)->propagated_events = val;
}


/* returns the window's global position */
Rect GetWindowPosition(Window window)
{
   return _WDW(window)->position;
}


/* returns window's z order */
int GetWindowZOrder(Window window, int direction)
{
   int retval = 0;
   _window *tmp;

   if ((direction) && (_WDW(window)->parent)) {
      if (direction > 0) {
         tmp = _WDW(window);
         while (tmp) {
            retval++;
            tmp = tmp->higher;
         }
      }
      else if (direction < 0) {
         tmp = _WDW(window);
         while (tmp) {
            retval--;
            tmp = tmp->lower;
         }
      }
   }
   return retval;
}


/* returns the child under the given screen coordinate x, y position */
Window GetXYChild(Window parent, int x, int y)
{
   _window *retval = Null;
   _window *tmp;

   if ((_WDW(parent)->is_visible) && _POINT_IN_RECT(_WDW(parent)->clipview, x, y)) {
      tmp = _WDW(parent)->top_child;
      while (tmp) {
         if ((tmp->is_visible) &&
            (_check_xy_in_region(&tmp->fore_region, x, y))) {
            retval = tmp;
            tmp = Null;
         }
         else tmp = tmp->lower;
      }
   }
   return (Window)retval;
}


/* returns window relative */
Window GetWindowParent(Window window)
{
   return (Window)_WDW(window)->parent;
}


/* returns window relative */
Window GetWindowHigher(Window window)
{
   return (Window)_WDW(window)->higher;
}


/* returns window relative */
Window GetWindowLower(Window window)
{
   return (Window)_WDW(window)->lower;
}


/* returns window relative */
Window GetWindowTopChild(Window window)
{
   return (Window)_WDW(window)->top_child;
}


/* returns window relative */
Window GetWindowBottomChild(Window window)
{
   return (Window)_WDW(window)->bottom_child;
}


/* returns window dialog */
Dialog GetWindowDialog(Window window)
{
   return (Dialog)_WDW(window)->dialog;
}


/* returns a window value related to its geometry */
int GetWindowX(Window window)
{
   return _WDW(window)->x;
}


/* returns a window value related to its geometry */
int GetWindowY(Window window)
{
   return _WDW(window)->y;
}


/* returns a window value related to its geometry */
int GetWindowWidth(Window window)
{
   return _WDW(window)->width;
}


/* returns a window value related to its geometry */
int GetWindowHeight(Window window)
{
   return _WDW(window)->height;
}


/* returns a window value related to its geometry */
int GetWindowBorderWidth(Window window)
{
   return _WDW(window)->border_width;
}


/* returns window's viewport */
Rect GetWindowViewport(Window window)
{
   return _WDW(window)->viewport;
}


/* returns a window value related to its geometry */
int GetWindowViewportWidth(Window window)
{
   return _WDW(window)->viewport.x2 - _WDW(window)->viewport.x1 + 1;
}


/* returns a window value related to its geometry */
int GetWindowViewportHeight(Window window)
{
   return _WDW(window)->viewport.y2 - _WDW(window)->viewport.y1 + 1;
}


/* returns window event flags */
int GetWindowInput(Window window)
{
   return _WDW(window)->processed_events;
}


/* returns window event flags */
int GetWindowPropagation(Window window)
{
   return _WDW(window)->propagated_events;
}


/* returns window visibility */
Bool GetWindowVisible(Window window)
{
   return _WDW(window)->is_visible;
}


/* creates a dialog */
Dialog CreateDialog(Cursor cursor,
   void (*event_handler)(Dialog, Event *), Window window, ...)
{
   _dialog *dlg = calloc(1, sizeof(_dialog));
   _window *tmp;
   va_list params;

   if (dlg) {
      dlg->event_handler = event_handler;
      dlg->cursor = (_cursor *)cursor;
      _add_window_to_dialog(dlg, _WDW(window));
      va_start(params, window);
      do {
         tmp = va_arg(params, _window *);
         if (tmp) _add_window_to_dialog(dlg, tmp);
      } while (tmp);
      va_end(params);
   }
   return (Dialog)dlg;
}


/* destroys a dialog */
void DestroyDialog(Dialog *dialog)
{
   if (_DLG(*dialog)->is_running)
      _DLG(*dialog)->to_destroy = True;
   else
      _destroy_dialog((_dialog **)dialog);
}


/* adds a window into a dialog */
Bool AddDialogWindow(Dialog dialog, Window window)
{
   Bool b;

   b = _add_window_to_dialog(_DLG(dialog), _WDW(window));
   if ((b) && (!_WDW(window)->is_drawn)) _init_render_window(_WDW(window));
   return b;
}


/* removes a window from a dialog - window must be dialog top */
Bool RemoveDialogWindow(Window window)
{
   return _remove_window_from_dialog(_WDW(window));
}


/* runs a dialog, creating events and dispatching them to windows */
void RunDialog(Dialog *dialog)
{
   KeyEvent ekey;

   _DLG(*dialog)->is_running = True;

   /* get I/O values */
   _mouse_x = mouse_x;
   _mouse_y = mouse_y;
   _mouse_b = mouse_b;
   _mouse_clicks = _clicks;

   /* if dialog is not drawn then render it */
   if (!_DLG(*dialog)->is_drawn) _init_render_dialog(_DLG(*dialog));

   /* catch mouse buttons */

   /* some button pressed */
   if (_mouse_b > _old_mouse_b) {
      _button_timer = 0;
      _init_x = _mouse_x;
      _init_y = _mouse_y;
      _do_button_event(_DLG(*dialog), ButtonPress);
   }

   /* the same buttons hold pressed */
   else if ((_mouse_b) && (_mouse_b == _old_mouse_b))
      _do_button_event(_DLG(*dialog), ButtonDown);

   /* some button released */
   else if (_mouse_b < _old_mouse_b) {
      _click_timer = 0;
      _clicks++;
      _mouse_clicks = _clicks;
      _do_button_event(_DLG(*dialog), ButtonRelease);
      if ((_init_x != _mouse_x) || (_init_y != _mouse_y)) 
         _old_mouse_x = _mouse_x - 1;
   }

   /* if dialog must be updated */
   if (_DLG(*dialog)->to_update) {
      _old_mouse_x = _mouse_x - 1;
      _DLG(*dialog)->to_update = False;
   }

   /* trace mouse motion */

   /* mouse moved */
   if ((_mouse_x != _old_mouse_x) || (_mouse_y != _old_mouse_y)) {
      _pointer_motion_timer = 0;
      if (_pointer_motion_status == _POINTER_IDLE) {
         _do_pointer_event(_DLG(*dialog), PointerStart);
         _pointer_motion_status = _POINTER_MOVING;
      }
      else
      if (_pointer_motion_status == _POINTER_MOVING)
         _do_pointer_event(_DLG(*dialog), PointerMove);
   }

   /* mouse is not moving -- but it may not have stopped since a
      change in mouse mickeys does not always constitute a pixel change */
   else if (_pointer_motion_status == _POINTER_STOPPED) {
      _do_pointer_event(_DLG(*dialog), PointerStop);
      _pointer_motion_status = _POINTER_IDLE;
   }

   /* handle keypresses */
   if (keypressed()) {
      ekey.type = KeyPress;
      ekey.keyread = readkey();
      if (_keyboard_window)
         _CALL_WINDOW_HANDLER(_keyboard_window, ekey)
      else if (_DLG(*dialog)->event_handler)
         _CALL_DIALOG_HANDLER(_DLG(*dialog), ekey);
   }

   /* store I/O values */
   _old_mouse_x = _mouse_x;
   _old_mouse_y = _mouse_y;
   _old_mouse_b = _mouse_b;

   _DLG(*dialog)->is_running = False;

   /* handle dialog destruction */
   if (_DLG(*dialog)->to_destroy)
      _destroy_dialog((_dialog **)dialog);
}


/* grabs events for the given window */
Bool GrabEvents(Window window)
{
   Bool retval = True;

   if (_WDW(window)->parent) {
      retval = GrabEvents((Window)_WDW(window)->parent);
     if (retval) {
        if (!_WDW(window)->parent->grab_child) {
           _WDW(window)->parent->grab_child = _WDW(window);
           if (_WDW(window)->dialog)
              _WDW(window)->dialog->grab_window = _WDW(window);
        }
        else
           retval = False;
     }
   }
   return retval;
}


/* releases events for the given window */
Bool ReleaseEvents(Window window)
{
   _release_events_downwards(_WDW(window));
   return _release_events_upwards(_WDW(window));
}


/* sets the click threshold */
void SetClickThreshold(int msecs)
{
   _click_timeout = msecs;
}


/* gets the click threshold */
int GetClickThreshold()
{
   return _click_timeout;
}


/* redirects the keyboard input to the given window */
void RedirectKeyboard(Window window)
{
   if (window) _keyboard_window = _WDW(window);
}


/* creates a cursor */
Cursor CreateCursor(BITMAP *bmp, int x, int y)
{
   _cursor *c = malloc(sizeof(_cursor));

   if (c) {
      c->bitmap = bmp;
      c->x = x;
      c->y = y;
   }
   return (Cursor)c;
}


/* destroys a cursor */
Bool DestroyCursor(Cursor *cursor)
{
   Bool retval = False;

   if (_current_cursor != *((_cursor **)cursor)) {
      show_mouse(Null);
      set_mouse_sprite(Null);
      show_mouse(screen);
      free(*(_cursor **)cursor);
      cursor = Null;
      retval = True;
   }
   return retval;
}


/* sends the given event to the given window */
void SendEventToWindow(Window window, Event *event)
{
   _CALL_WINDOW_HANDLER(_WDW(window), (*event));
}


/* checks if the given depth is valid */
Bool CheckIfValidDepth(Window window, int depth)
{
   _window *tmp = _get_child_from_pos(_WDW(window), &depth);
   if (tmp) return True;
   return False;
}
