#! /bin/sh

#
# This script recodes .po files from unix charsets to MSDOS code pages.
# 1) It checks for .po files in the po directory.
# 2) It checks if the .po files have already been recoded.
# 3) It checks for recode version 3.5 or later, if the .po files need to be recoded.
#

declare -i i iMax

# Known country codes and MSDOS code pages.
# The ISO 639 language codes have been taken from:
#   gettext.info 
# The MSDOS code pages have been taken from:
#   MS-DOS 6.22 COUNTRY.TXT file
# available from:
#   ftp://ftp.microsoft.com/peropsys/msdos/kb/q117/8/50.txt
# Please note that the ISO language codes are *NOT* the same
# as the MSDOS country codes.
LanguageCode=(ar bg  cs  da  de  de_AT el  en  en_UK en_US eo  es  et  fa  'fi'  fr  ga  he  hi  hr  hu  id  is  it  ja  ko  la  mk  nl  no  no@nynor  no@nynorsk pl  pt  pt_BR ro  ru  sh  sk  sl  sq  sr  sv  tr  zh)
DosCodePage=(864 855 852 865 850 850   869 437 850   437   852 850 850 864 850   850 850 862 000 852 852 850 861 850 932 934 850 855 850 865 865       865        852 850 850   852 855 852 852 852 852 855 850 857 936)
iMax=${#LanguageCode[@]}-1

PATH_SEPARATOR=:
PATH=$PATH:.

# A sane default for XSRC.
if [ "x$XSRC" = x ]; then
  XSRC=.
fi
 
# Find the .po files.
if [ -d "$XSRC"/po ]; then
  po_directory=po
elif [ -d "$XSRC"/i18n ]; then
  po_directory=i18n
else
  exit 2             # This package has no NLS.
fi

# Ckeck if .po files have already been recoded
# and the required programs are available.
if [ -f "$XSRC"/${po_directory}/stamp-po ]; then
# Package has been recoded.
# Check for msgfmt and xgettext.
  for program in msgfmt xgettext; do
    $program >/dev/null 2>&1
    if [ $? -eq 127 ]; then
      echo Warning: $program.exe is required for NLS but it can not be found.
      echo Configuring with --disable-nls...
      exit 1   # One of the programs is not installed
    fi
  done
  exit 0       # .po files have been already recoded and msgfmt and xgettext are available.
else
# Package has not been recoded.
# Check for msgfmt, xgettext and recode.
  for program in msgfmt xgettext recode; do
    $program >/dev/null 2>&1
    if [ $? -eq 127 ]; then
      echo Warning: $program.exe is required for NLS but it can not be found.
      echo Configuring with --disable-nls...
      exit 1   # One of the programs needed is not installed
    elif [ $program = recode ]; then
      version=`recode 2>/dev/null | sed "1 s/^.* \([1-9]\.[0-9][0-9]*\)$/\1/"`
      if [ "$version" = 3.4 ]; then
        echo Warning: recode.exe 3.5 or later is required for NLS.
        echo Configuring with --disable-nls...
        exit 1   # recode 3.5 or later is not installed
      fi
    fi
  done
fi

# Recode the .po files.
echo Recoding the .po files...
po_files=`echo "$XSRC"/${po_directory}/*.po \
          | sed -e "s%$XSRC/${po_directory}/%%g" -e "s%\\.po%%g"`
for file in ${po_files}; do
  i=0
  while [ $i -le $iMax ]; do
    if [ ${LanguageCode[$i]} = $file ]; then
      charset=`cat "$XSRC"/${po_directory}/${file}.po | \
               sed -e "/charset=/ !d" \
                   -e "s/^.*=//" -e "s/\\\\\\\\n.*$//" \
                   -e "s/^8859/ISO-8859/" -e "s/ISO-Latin/Latin/"`
      recode -vSpo "${charset}..${DosCodePage[$i]}/" "$XSRC"/${po_directory}/${file}.po
      if [ $? -eq 0 ]; then
        sed "/charset=/ s/=.*\\\\n/=cp${DosCodePage[$i]}\\\\n/" \
            "$XSRC"/${po_directory}/${file}.po > "$XSRC"/${po_directory}/${file}.tmp
        mv -f "$XSRC"/${po_directory}/${file}.tmp "$XSRC"/${po_directory}/${file}.po
      fi
      break
    fi
    i=i+1
  done
done

touch "$XSRC"/${po_directory}/stamp-po  # Do not recode the .po files any more.
exit 0
