#include "editpane.h"
#include "keyboard.h"
#include "mouse.h"
#include <pc.h>
#include <conio.h>
#include <stdlib.h>
#include <stdio.h>

#define is_ascii_char(c) ( c > 31 && c < 127 )

String illegal_chars = "\t";

editing_pane::editing_pane()
{
   adopt( page );
   page.move_to( top(), left() );
   designated_quit_key = DEFAULT_EDITPANE_QUIT_KEY;
   _found_colour = DEFAULT_FOUND_COLOUR;

   set_fill_char( fill_char().character, fill_char().colour );
   resize( ScreenCols(), ScreenRows() );
}

editing_pane::editing_pane( const char * file_name )
{
   adopt( page );
   page.move_to( top(), left() );
   designated_quit_key = DEFAULT_EDITPANE_QUIT_KEY;
   _found_colour = DEFAULT_FOUND_COLOUR;

   set_fill_char( fill_char().character, fill_char().colour );
   resize( ScreenCols(), ScreenRows() );
   read_file( file_name );
}

editing_pane::editing_pane( const string_array & model )
{
   adopt( page );
   page.move_to( top(), left() );
   designated_quit_key = DEFAULT_EDITPANE_QUIT_KEY;
   _found_colour = DEFAULT_FOUND_COLOUR;

   set_fill_char( fill_char().character, fill_char().colour );
   operator=( model );
   resize( page.width(), page.height() );
}

editing_pane & editing_pane::
operator=( const string_array & model )
{
   page = model;
   page.cursor_to( 0, 0 );
   scroll_to_cursor();
   return *this;
}



editing_pane::~editing_pane() {
   _setcursortype( _NORMALCURSOR );
   }

String editing_pane::
current_file() { return _current_file; }

bool editing_pane::
read_file( const char * file_name )
{
   bool error = page.read_file( file_name );
   if( error ) {
      _current_file = "";
      string_array empty( 0 );
      page = empty;
      text_is_dirty = false;
      return true;
      }
   else {
      page.cursor_to( 0, 0 );
      scroll_to_cursor();
      _current_file = file_name;
      text_is_dirty = false;
      return false;
      }
}

bool editing_pane::
write_file( const char * file_name )
{
   FILE * stream = fopen( file_name, "wt" );
   if( stream ) {
      page.text.write_file( stream );
      fclose( stream );
      text_is_dirty = false;
      return false;
      }
   else {
      return true;
      }
}

void editing_pane::
set_fill_char( char character, char colour )
{
   page.set_fill_char( character, colour );
   screen_pane::set_fill_char( character, colour );
}

int editing_pane::
take_control()
{
   to_top();
   draw_all_windows();
   show_cursor();

   int event = 0;
   while( event != designated_quit_key )
   {
      if( key.pressed() )
      {
         key.get();
         event = process_key( key );
      }
      else
      {
         mouse.poll();
         event = process_mouse();
      }
   }
   return event;
}



int editing_pane::
process_key( int keystroke )
{
   show_cursor();

   if( mouse.visibility() == 0 )
      mouse.hide_cursor();

   switch( keystroke )
   {
   case alt_w_key:
      write_file( current_file() );
   break;
   case up_arrow_key:
      cursor_up();
   break;
   case down_arrow_key:
      cursor_down();
   break;
   case left_arrow_key:
      cursor_left();
   break;
   case right_arrow_key:
      cursor_right();
   break;

   case page_up_key:
      int amount;
      amount = smaller_of( height() - 2, top() - page.top() );
      if( amount <= 0 )
         page.cursor_to( page.cursor().x, 0 );
      else
      {
         scroll_up( amount );
         cursor_up( amount );
      }
   break;

   case page_down_key:
      amount = smaller_of( height() - 2, page.bottom() - bottom() );
      if( amount <= 0 )
         page.cursor_to( page.cursor().x, page.height() - 1 );
      else
      {
         scroll_down( amount );
         cursor_down( amount );
      }
   break;

   case home_key:
      page.cursor_to( 0, page.cursor().y );
   break;

   case end_key:
      page.cursor_to( page.text[page.cursor().y].length(),
                 page.cursor().y );
   break;

   case delete_key:
      if( page.cursor().x < (int)page.text[page.cursor().y].length() ) {
         page.delete_at( page.cursor().x, page.cursor().y );
         text_is_dirty = true;
         }
   break;

   case backspace_key:
      if( page.cursor().x > 0 )
      {
         if( page.cursor().x <= (int)page.text[page.cursor().y].length() )
         {
            page.delete_at( page.cursor().x - 1, page.cursor().y );
            text_is_dirty = true;
         }
         cursor_left();
      }
      else
      {
         if( page.cursor().y > 0 )
         {
            page.cursor_to( page.text[page.cursor().y - 1].length(),
                            page.cursor().y - 1 );
            page.text[page.cursor().y] += page.text[page.cursor().y + 1];
            page.delete_line( page.cursor().y + 1 );
            text_is_dirty = true;
         }
      }
   break;

   case return_key:
      page.insert_line( "", page.cursor().y + 1 );
      if( page.cursor().x <
          (int)page.text[page.cursor().y].length() )
      {
         page.text[page.cursor().y + 1]
         = page.text[page.cursor().y].from( (int)page.cursor().x );
         page[page.cursor().y + 1] = page.text[page.cursor().y + 1];

         page.text[page.cursor().y]
         = page.text[page.cursor().y].before( (int)page.cursor().x );
         page[page.cursor().y] = page.text[page.cursor().y];
      }

      page.cursor_to( 0, page.cursor().y + 1 );
      text_is_dirty = true;
   break;

   default:
      char ascii_symbol = keystroke & 0xFF;
      if( ascii_symbol == '' )
         ascii_symbol = '|';

     if( illegal_chars.contains( ascii_symbol ) )
        return keystroke;
     else
     {
         page.insert_at( ascii_symbol,
                         page.cursor().x,
                         page.cursor().y );

         cursor_right();
         text_is_dirty = true;
     }
   }
   scroll_to_cursor();
   show_cursor();
   draw_all_windows();
   return keystroke;
}

int editing_pane::
process_mouse()
{
   if( mouse.has_moved() && mouse.visibility() != 0 )
      mouse.set_visibility( 0 );

   if( mouse.is_over( *this ) && mouse.left_click() )
   {
      page.cursor_to( mouse.x() - page.left(),
                 mouse.y() - page.top() );

      if( page.cursor().x > (int)page.text[page.cursor().y].length() )
         page.cursor_to( page.text[page.cursor().y].length(),
                    page.cursor().y );

      show_cursor();
   }
   return mouse.event();
}

void editing_pane::
mouse_move_while( int condition )
{
   hide_cursor();
   screen_pane::mouse_move_while( condition );
   show_cursor();
}

void editing_pane::
mouse_scroll_while( int condition, int direction )
{
   hide_cursor();
   screen_pane::mouse_scroll_while( condition, direction );
   scroll_snap_back();
   draw_all_windows();
}

void editing_pane::
scroll_snap_back()
{
   if( page.right() < right() )
      scroll_left( right() - page.right() );
   if( page.left() > left() )
      scroll_right( ( page.left() - left() ) );
   if( page.bottom() < bottom() )
      scroll_up( bottom() - page.bottom() );
   if( page.top() > top() )
      scroll_down( page.top() - top() );
}

void editing_pane::cursor_left( int amount ) {
   point cursor = page.cursor();
   page.cursor_to( cursor.x - amount, cursor.y );
}
void editing_pane::cursor_right( int amount ) {
   point cursor = page.cursor();
   page.cursor_to( cursor.x + amount, cursor.y );
}
void editing_pane::cursor_up( int amount ) {
   point cursor = page.cursor();
   page.cursor_to( cursor.x, cursor.y - amount );
}
void editing_pane::cursor_down( int amount ) {
   point cursor = page.cursor();
   page.cursor_to( cursor.x, cursor.y + amount );
}

void editing_pane::
cursor_to( int x, int y )
{
   point cursor_pos = { x, y };
   cursor_to( cursor_pos );
}

void editing_pane::
show_cursor()
{
   point text_cursor = page.cursor() + page.top_left();
   ScreenSetCursor( text_cursor.y, text_cursor.x );

   if( this -> contains( text_cursor ) )
      _setcursortype( _NORMALCURSOR );
   else
      _setcursortype( _NOCURSOR );

}

void editing_pane::
hide_cursor()
{
   _setcursortype( _NOCURSOR );
}

void editing_pane::
scroll_to_cursor()
{
   point text_pos = page.cursor() + page.top_left();

   if( text_pos.x > right() )
      scroll_right( text_pos.x - right() );
   else
   if( text_pos.x < left() )
      scroll_left( left() - ( page.cursor().x + page.left() ) );

   if( text_pos.y >= bottom() )
      scroll_down( text_pos.y - bottom() );
   else
   if( text_pos.y <= top() )
      scroll_up( top() - text_pos.y );
}

void editing_pane::
cursor_to( point pos )
{
   pos -= page.top_left();
   page.cursor_to( pos.x, pos.y );
}


bool editing_pane::
next_match( bool case_sensitive, const char * search_string, point start )
{
   if( search_string )
      _current_search_string = search_string;

   if( start == point_not_found )
      start = page.cursor();

   point found;

   if( case_sensitive == true )
      found = page.text.index( _current_search_string, start.x, start.y );
   else
      found = page.text.ci_index( _current_search_string, start.x, start.y );

   if( found != point_not_found )
   {
      page[found.y].set_colour
              ( _found_colour, found.x, _current_search_string.length() );
      found.x += _current_search_string.length();
      cursor_to( found );
      scroll_to_cursor();
      if( page.top()
        + found.y == bottom() )
         scroll_down( 3 );

      show_cursor();
      draw_all_windows();

      return true;
   }
   return false;
}


