/*
INI-File class     Header-File
-------------------------------------------------------------
(c) 1997 M.Baas
*/

#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <dir.h>
#include <unistd.h>
#include <errno.h>
#include "inifile.h"

// Possible error codes from the iniclass
static const char* nomem  = "Inifile: no memory";
static const char* iopen  = "Inifile: error opening input";
static const char* oopen  = "Inifile: error opening output";
static const char* nopen  = "Inifile: file not open";
static const char* illval = "Inifile: illegal parameter found";
//static const char* nvalue = "Inifile: no parameter value";
static const char* rname  = "Inifile: rename error";
static const char* wrerr  = "Inifile: write error";

/*
Chained list class
----------------------------------------------------------
(c) 1997 M.Baas
*/

/*----------------------------insert-----------------------------
Insert node n in the list.
---------------------------------------------------------------*/
void list::insert(node *n)
   {
	insert_last(n);
   }


/*--------------------------insert_first-------------------------
Insert node n as first node of the list.
---------------------------------------------------------------*/
void list::insert_first(node *n)
   {
	n->succ=first;
	n->pred=NULL;
	if (first!=NULL) first->pred=n;
	first=n;
	if (last==NULL) last=n;
   }


/*--------------------------insert_last--------------------------
Insert node n as last node of the list.
---------------------------------------------------------------*/
void list::insert_last(node *n)
   {
	n->succ=NULL;
	n->pred=last;
	if (last!=NULL) last->succ=n;
	last=n;
	if (first==NULL) first=n;
   }


/*--------------------------insert_after-------------------------
Insert node n after node p.

Pre: p must be part of the list!
---------------------------------------------------------------*/
void list::insert_after(node *n, node *p)
   {
	if (p==last) insert_last(n);
	else
	   {
		n->succ = p->succ;
		n->pred = p;
		(n->succ)->pred=n;
		p->succ = n;
	   };
   }


/*--------------------------delete_node--------------------------
Delete node n from the list without freeing n.

Pre: n must be part of the list!
---------------------------------------------------------------*/
void list::delete_node(node *n)
   {
	if (n==first)
	   {
		first=n->succ;
		n->succ=NULL;
// n->pred is already NULL since it's the first node
		if (first==NULL) last=NULL; else first->pred=NULL;
		return;
	   };
	if (n==last)
	   {
		last=n->pred;
		n->pred=NULL;
// n->succ is already NULL since it's the last node
		if (last==NULL) first=NULL; else last->succ=NULL;
		return;
	   };

// Node is neither first nor last node.

	(n->succ)->pred = n->pred;
	(n->pred)->succ = n->succ;
	n->succ=NULL;
	n->pred=NULL;
   }


/*--------------------------delete_list--------------------------
Delete all nodes from the list without freeing them.
---------------------------------------------------------------*/
void list::delete_list()
   {
	node *n = first;
	node *s;

// Setting succ/pred from all nodes to NULL
	while(n!=NULL)
	   {
		s=n->succ;
		n->succ=NULL;
		n->pred=NULL;
		n=s;
	   };

	first=NULL;
	last=NULL;
   };


/*----------------------------free_node--------------------------
Delete node n from the list and free it.

Pre: n must be part of the list!
---------------------------------------------------------------*/
void list::free_node(node *n)
   {
	delete_node(n);
	delete n;
   }


/*----------------------------free_list--------------------------
Delete all nodes from the list and free them.
---------------------------------------------------------------*/
void list::free_list()
   {
	node *n = first;
	node *s;

// Free all nodes...
	while(n!=NULL)
	   {
		s=n->succ;
		delete n;
		n=s;
	   };

	first=NULL;
	last=NULL;
   };


/*==================================================================
Lookup table
==================================================================*/

/*--------------------------insert_at------------------------------
Insert row r at the given index.

Pre: index must be in a valid range (0,1,...,number).
Input:  r     - Row to be inserted into the table
index - This will be the new index of r
Return: False, if there was not enough memory to create a new table
-----------------------------------------------------------------*/
bool lookupTable::insert_at(row* r,int index)
   {
// Do we have to enlarge the table?
	if (number==tableSize)
	   {
		int  newsize  = (tableSize==0)? 1 : 2*tableSize;
// Allocate the new table...
		row** newtable = new row* [newsize];
		if (newtable==NULL) return false;
// Copy the elements before index...
		for(int i=0; i<index; i++)
			newtable[i]=table[i];
// Copy the elements after index...
		for(int i=index; i<number; i++)
			newtable[i+1]=table[i];
// Insert new row
		newtable[index]=r;
		number++;
// Keep the new table and delete the old one
		delete [] table;
		table     = newtable;
		tableSize = newsize;
	   }
	else                       // There's no need for a new table...
	   {
// Copy elements after index...
		for(int i=number-1; i>=index; i--)
			table[i+1]=table[i];
// Insert new row
		table[index]=r;
		number++;
	   };

	return true;
   }


/*-------------------------------add------------------------------
Add a new row to the table.
The resulting table isn't sorted yet!

Input:  r     - Row to be inserted into the table
Return: False, if there was not enough memory to create a new table
-----------------------------------------------------------------*/
bool lookupTable::add(row* r)
   {
	return insert_at(r,number);
   }


/*-----------------------------index_of---------------------------
Search a row with identifier k.

Pre:    The table must be sorted!
Input:  k     -  Identifier to search
Output: index -  Index of first row with identifier k
or index where k should be inserted to maintain
the sorted order of the table.
n     -  Number of rows with identifier k
Return: True, if a row was found.
-----------------------------------------------------------------*/
bool lookupTable::index_of(char* k, int &index, int &n)
   {
	int l = 0;
	int r = number-1;
	int cmpres;

// Searching row with identifier k ...
	while(l<=r)
	   {
		index  = (l+r)/2;
		cmpres = strcmp(table[index]->get_identifier(),k);
// Identifiers found?
		if (cmpres==0)
		   {
// First row found?
			if (l==r)
			   {
				n=1;
				l++;
// Count the elements with identical identifiers...
				while(l<number)
				   {
					if(strcmp(table[l]->get_identifier(),k)!=0) break;
					l++;
					n++;
				   };
				return true;
			   }
			else r=index;
		   }
		else                    // Identifier not found yet
		   {
			if (cmpres<0) l=index+1; else r=index-1;
		   };
	   };

// No row found with identifier k
	index=l;
	n=0;
	return false;
   }


static int compareRows(row** r1,row** r2)
   {
	int res=strcmp((*r1)->get_identifier(),(*r2)->get_identifier());
	if (res!=0) return res;
	else
	   {
		return (*r1)->rowNumber - (*r2)->rowNumber;
	   };
   }


void lookupTable::sort()
   {
	if (number>0) qsort(table,number,sizeof(row*),compareRows);
   }


/*--------------------------- [] operator ------------------------
Return a pointer to the row at the given index.
If index is not valid the function returns NULL.
-----------------------------------------------------------------*/
row* lookupTable::operator[](int index)
   {
	if (index<0 || index>=number)
		return NULL;
	else
		return table[index];
   }


/*==================================================================
Row
==================================================================*/


/*---------------------------set_string---------------------------
Set the string that contains the actual INI file row.

Pre: s!=NULL
-----------------------------------------------------------------*/
bool row::set_string(char *s)
   {
	int len=strlen(s);

// Do we need a larger buffer?
	if (len >= bufferSize)
	   {
		free(rowBuffer);
		bufferSize=len+1;
		rowBuffer=(char*)malloc(bufferSize);
		if (rowBuffer==NULL) { bufferSize=0; return false; };
	   };
// Copy the string
	strcpy(rowBuffer,s);
	return true;
   };


/*==================================================================
parameterRow
==================================================================*/


/*---------------------------set_string---------------------------
Set the string that contains the actual INI file row which
must contain a valid parameter.

Pre: s!=NULL
s must be a valid parameter row (is_parameter(s) must return true)
-----------------------------------------------------------------*/
bool parameterRow::set_string(char *s)
   {
	use_newvalue=false;

// Copy string
	if (!row::set_string(s)) return false;

	param     = rowBuffer;
	value     = strchr(rowBuffer,'=');
	param_end = value;

// Delete whitespace at the end of the parameter value
	char *p  = value+strlen(value)-1;
	char *pf = p;
	while(isspace(*p)) p--;
	p[1]=0;
// Skip whitespace at the beginning of the parameter value
	value++;
	while(isspace(*value)) value++;

// New values larger than that need a memory allocation
	maxvaluesize_without_allocating = pf-value+1;

// Skip whitespace at the beginning of the parameter name
	while(isspace(*param)) param++;
// If param is equal to param_end here then param = ""
	if (param!=param_end)
	   {
// Skip whitespace at the end of the parameter name
		param_end--;
		while(isspace(*param_end)) param_end--;
		param_end++;
	   };

	dummy_char = *param_end;
	*param_end = 0;
	return true;
   }


/*---------------------------set_value----------------------------
Set a new parameter value.

Pre: A parameter was set (=set_string() was called)
v != NULL
Return: False, if the new value couldn't be set (no memory)
-----------------------------------------------------------------*/
bool parameterRow::set_value(char *v)
   {
	int len = strlen(v);

// Can the value be stored in the previous allocated memory?
// newvalue_size has to be -1 if there's no memory allocated!
	if (len<=newvalue_size)
	   {
		strcpy(newvalue,v);
		use_newvalue=true;
		return true;
	   };

// Can the value be stored in the original row?
	if (len<=maxvaluesize_without_allocating)
	   {
		strcpy(value,v);
		use_newvalue=false;
		return true;
	   };

// v is bigger than all the previous values, so we must allocate new memory

	free(newvalue);
	newvalue=(char*)malloc(len+1);
	if (newvalue==NULL)
	   {
		use_newvalue=false;
		return false;
	   };
	strcpy(newvalue,v);
	use_newvalue=true;
   return true;
   }


/*---------------------------is_parameter-------------------------
Tests if string s is a valid parameter.
s is valid if it contains a '='.

Pre: s!=NULL
-----------------------------------------------------------------*/
bool parameterRow::is_parameter(char *s)
   {
	if (strchr(s,'=')!=NULL) return true; else return false;
   }


/*-------------------------------write----------------------------
Writes the row to os.

Pre: A string was set with set_string()
-----------------------------------------------------------------*/
void parameterRow::write(ostream &os)
   {
	*param_end=dummy_char;

	if (use_newvalue)
	   {
		char tmp = *value;
		*value=0;
		os<<rowBuffer<<newvalue<<"\n";
		*value=tmp;
	   }
	else
	   {
		row::write(os);
	   };
	*param_end=0;
   }


/*==================================================================
sectionRow
==================================================================*/

/*---------------------------set_string---------------------------
Set the string that contains the actual INI file row which
must contain a valid section.

Pre: s!=NULL
s must be a valid section row (is_section(s) must return true)
-----------------------------------------------------------------*/
bool sectionRow::set_string(char *s)
   {
// Copy string
	if (!row::set_string(s)) return false;

	sectionName = rowBuffer+1;
	sectionEnd  = strchr(rowBuffer,']');
	*sectionEnd = 0;
	return true;
   }


/*---------------------------is_section---------------------------
Tests if string s is a valid section.

Pre: s!=NULL
-----------------------------------------------------------------*/
bool sectionRow::is_section(char *s)
   {
	if (*s!='[') return false;
	if (strchr(s,']')==NULL) return false;
	return true;
   }


/*-------------------------------write----------------------------
Writes the row to os.

Pre: A string was set with set_string()
-----------------------------------------------------------------*/
void sectionRow::write(ostream &os)
   {
	*sectionEnd=']';
	row::write(os);
	*sectionEnd=0;
   }


/*------------------------------setup-----------------------------
Initializing parameter lookup table and other stuff.

Pre: Chained list with rows must be complete.
Return: False = No memory.
-----------------------------------------------------------------*/
bool sectionRow::setup()
   {
// Clear lookup table
	params.clear();

	firstParam=lastParam=NULL;

// Start with the row after this section
	row *r = (row*)succ;

// Collect parameter definitions...
	while(r!=NULL)
	   {
// Have we reached the top of another section? then we are finished
		if (r->row_type() == row::section) break;
// Parameter?
		if (r->row_type() == row::parameter)
		   {
// Add the parameter to the lookup table
			if (!params.add(r)) { params.clear(); return false; };
// Update firstParam/lastParam
			if (firstParam==NULL) firstParam=(parameterRow*)r;
			lastParam=(parameterRow*)r;
		   };
// Next row
		r=(row*)r->succ;
	   };
// Sorting the lookup table
	params.sort();
	return true;
   }


/*--------------------------get_parameter-------------------------
Search a parameter and return its value or NULL if the parameter
doesn't exist.

Pre: The lookup table must be initialized (setup() must be called)
-----------------------------------------------------------------*/
char* sectionRow::get(char *paramname)
   {
	int index,n;

// Look for the parameter in the lookup table
	if (params.index_of(paramname,index,n))
		return ((parameterRow*)params[index])->get_value();
	else
		return NULL;
   }


/*--------------------------set-------------------------
Input:   paramname - Parameter name
value     - New value to be set
create    - True = Create parameter if it doesn't exist
lst       - List with rows to insert new parameter
Return:  0 = OK. Value changed/parameter created.
-1 = No memory to change or create value.
1 = Parameter not found (only when create=false).
-----------------------------------------------------------------*/
int sectionRow::set(char *paramname,char *value,bool create,list *lst)
   {
	int index,n;

// Look for the parameter in the lookup table
	if (params.index_of(paramname,index,n))
	   {
// Set the new value
		if (((parameterRow*)params[index])->set_value(value))
			return 0;
		else
			return -1;
	   }
	else
	   {
// Should a new parameter be created?
		if (create)
		   {
// Create a parameter definition
			char* s = (char*)malloc(strlen(paramname)+1+strlen(value)+1);
			if (s==NULL) return -1;
			strcpy(s,paramname);
			strcat(s,"=");
			strcat(s,value);

// Allocate a new parameterRow object
			parameterRow *pr = new parameterRow();
			if (pr==NULL) { free(s); return -1; };

// Set created definition
			if (!pr->set_string(s)) { free(s); delete(pr); return -1; };
			free(s);

// Insert new parameter in row list
			if (lastParam!=NULL)
			   {
				lst->insert_after(pr,lastParam);
				lastParam=pr;
				if (firstParam==NULL) firstParam=pr;
			   }
			else
			   {
				lst->insert_after(pr,this);
				firstParam=pr;
				lastParam=pr;
			   };

// Update lookup table
			params.insert_at(pr,index);
			return 0;
		   }
		else return 1;
	   };
   }


/*==================================================================
INI file
==================================================================*/

/*-------------------------------open-----------------------------
Open INI file (that is, connect a file with the object).
-----------------------------------------------------------------*/
bool INIfile::open(char *filename, bool create_flag)
   {
	errorCode=0;

// If there's already a file opened then it will be closed now.
	close();

// May we create a new file if it doesn't already exist?
	if (create_flag)
	   {
// Check if file doesn't exist and create a new file if necessary.
		if (access(filename,F_OK)!=0) return create(filename);
	   };

// Create dummy section for parameters without a section
	if (!createEmptySection()) return false;

// Expand & copy filename...
	char ename[MAXPATH];
	if (!fexpand(filename,ename))
	   {
		errorCode = nopen;
		return false;
	   };
	name=(char*)malloc(strlen(ename)+1);
	if (name==NULL)
	   {
		errorCode = nomem;
		return false;
	   };
	strcpy(name,ename);


// Read INI file
	if (!readRows()) return false;

// Initialize sections & build section lookup table...
	row *r=(row*)rows.first;
	while(r!=NULL)
	   {
// Section? then call setup()
		if (r->row_type() == row::section)
		   {
			sectionRow* sr = (sectionRow*)r;
// Call setup()
			if (!sr->setup())
			   {
				errorCode = nomem;
				return false;
			   };
// Add section to the lookup table
			if (!sections.add(sr))
			   {
				errorCode = nomem;
				return false;
			   };
// Continue after the last parameter (if there is one)
			if (sr->lastParam != NULL) r = sr->lastParam;
		   };
		if (r!=NULL) r=(row*)r->succ;
	   };
// Sort the lookup table
	sections.sort();

	ini_is_open=true;
	return true;
   }


/*------------------------------create----------------------------
Create a new INI file.
-----------------------------------------------------------------*/
bool INIfile::create(char *filename)
   {
	errorCode=0;

// If there's already a file opened then it will be closed now.
	close();

// Create dummy section for parameters without a section
	if (!createEmptySection()) return false;

// Call setup()
	sectionRow *srow=(sectionRow*)rows.first;
	if (!srow->setup())
	   {
		errorCode = nomem;
		return false;
	   };
// Add section to the lookup table
	if (!sections.add(srow))
	   {
		errorCode = nomem;
		return false;
	   };

// Expand & copy filename...
	char ename[MAXPATH];
	if (!fexpand(filename,ename))
	   {
		errorCode = nopen;
		return false;
	   };
	name=(char*)malloc(strlen(ename)+1);
	if (name==NULL)
	   {
		errorCode = nomem;
		return false;
	   };
	strcpy(name,ename);

	ini_is_open=true;
	changed=true;
	return true;
   }


/*-------------------------------close----------------------------
-----------------------------------------------------------------*/
void INIfile::close()
   {
// Write changes (errors will be ignored!)...
	flush();
// Just in case flush() failed (changes will be lost)
	changed=false;

	sections.clear();
	rows.free_list();
	free(name);
	name=NULL;
	ini_is_open=false;
	errorCode="";
   }


/*-----------------------------flush------------------------------
Writes parameter changes to disk (if there were any).

Return: False = Operation failed.

Hinweis: Falls False zurckgegeben wird ist es ist mglich,
da die Datei zerschossen wurde!
-----------------------------------------------------------------*/
bool INIfile::flush()
   {
	errorCode=0;

// No file open? then we're finished
	if (!ini_is_open) return true;

	int  name_len = strlen(name);
	char newname[name_len+1];  // keeps name of INI file backup
	strcpy(newname,name);
	newname[name_len-1]='~';   // replace last character with ~

// Were there changes? yes? then write the whole thing
	if (changed)
	   {
                              // true if we could keep an old INI file
		bool bak_exists = false;

// Is there already a INI file on disk? (or was a new file created)
		if (access(name,F_OK)==0)
		   {
// Rename the existing INI file
			errno=0;
			rename(name,newname);
			if (errno!=0)
			   {
				errorCode = rname;
				return false;
			   };
			bak_exists=true;
		   };

// Open output file
		ofstream of(name);
		if (!of.good())
		   {
                              // Restore previous version
			if (bak_exists) rename(newname,name);
			errorCode = oopen;
			return false;
		   };

// First row is the dummy object (which won't be written to disk)
		row *r = (row*)rows.first->succ;
		while(r!=NULL)
		   {
			r->write(of);
			r=(row*)r->succ;
		   };
// Was there an error while writing the file?
		if (!of.good())
		   {
			of.close();
                              // Restore previous version
			if (bak_exists) rename(newname,name);
			errorCode = wrerr;
			return false;
		   };
		changed=false;

// Delete backup
		if (bak_exists) remove(newname);
	   };
	return true;
   }


/*----------------------------get_parameter-----------------------
Returns the value of param if section.param exists, otherwise
the function returns defaultparam.
-----------------------------------------------------------------*/
char* INIfile::get(char *section, char *param, char *defaultparam)
   {
	int index,n;

	errorCode = "";
	if (!ini_is_open)
	   {
		return defaultparam;
	   };

// Look for the section in the lookup table
	if(sections.index_of(section,index,n))
	   {
// Search the parameter in all sections with the given name...
		for(int i=0; i<n; i++)
		   {
			sectionRow *sr   = (sectionRow*)sections[index+i];
			char*      value = sr->get(param);
			if(value!=NULL)
			   {
				return value;
			   };
		   };
	   };
	return defaultparam;
   }


/*----------------------------set-----------------------
-----------------------------------------------------------------*/
bool INIfile::set(char *section, char *param, char *value)
   {
	int index,n,res;
	sectionRow *sr;

	errorCode=0;
	if (!ini_is_open)
	   {
		errorCode = nopen;
		return false;
	   };

// Look for the section in the lookup table
	if (sections.index_of(section,index,n))
	   {
// Try to set the new value...
		for(int i=0; i<n; i++)
		   {
			sr  = (sectionRow*)sections[index+i];
			res = sr->set(param,value,false,NULL);
			if (res==-1)
			   {
				errorCode = nomem;
				return false;
			   };
			if (res==0)  { changed=true; return true; };
		   };
// Value couldn't be set, this means it doesn't already exist,
// so we have to create a new one.
		sr = (sectionRow*)sections[index];
		if (sr->set(param,value,true,&rows)==0)
		   {
			changed=true;
			return true;
		   }
		else
		   {
			errorCode = nomem;
			return false;
		   };
	   }
	else                       // Create a new section...
	   {
// Create a section definition
		char* s = (char*)malloc(strlen(section)+3);
		if (s==NULL)
		   {
			errorCode = nomem;
			return false;
		   };
		strcpy(s,"[");
		strcat(s,section);
		strcat(s,"]");

// Allocate a new sectionRow object
		sr = new sectionRow(0); // row number isn't important anymore
		if (sr==NULL)
		   {
			free(s);
			errorCode = nomem;
			return false;
		   };

// Set created definition
		if (!sr->set_string(s))
		   {
			free(s);
			delete(sr);
			errorCode = nomem;
			return false;
		   };
		free(s);

// Insert new section in row list
		rows.insert_last(sr);
// Update lookup table
		if (!sections.insert_at(sr,index)) { errorCode = nomem; return false; };

// Add parameter
		if (sr->set(param,value,true,&rows)==0)
		   {
			changed=true;
			return true;
		   }
		else { errorCode = nomem; return false; };
	   };
   }


/*------------------------get_section_name------------------------
-----------------------------------------------------------------*/
char* INIfile::get_section_name(int section_nr)
   {
	errorCode=0;
	if (!ini_is_open) return NULL;

	row* r=sections[section_nr];
	if (r!=NULL)
		return r->get_identifier();
	else
		return NULL;
   }


/*-------------------------get_name---------------------
-----------------------------------------------------------------*/
char* INIfile::get_name(int section_nr, int param_nr)
   {
	errorCode=0;
	if (!ini_is_open) return NULL;

	sectionRow *sr=(sectionRow*)sections[section_nr];
	row *r=sr->params[param_nr];
	if (r!=NULL)
		return r->get_identifier();
	else
		return NULL;
   }


/*-------------------------get--------------------------
Read an integer value with default value.
-----------------------------------------------------------------*/
int INIfile::get(char *section, char *param, const int i_default)
   {
	int i = i_default;

	char *s = get(section,param);
	if (s!=NULL)
	   {
		char *ep;
		i=strtol(s,&ep,0);
	   }
	else
	   {
		i=i_default;
	   };
	return i;
   }


/*-------------------------get--------------------------
Read a double value with default value.
-----------------------------------------------------------------*/
double INIfile::get(char *section, char *param, const double d_default)
   {
	double d = d_default;

	char *s = get(section,param);
	if (s!=NULL)
	   {
		char *ep;
		d=strtod(s,&ep);
	   }
	else
	   {
		d=d_default;
	   };
	return d;
   }


/*-------------------------get--------------------------
Read a boolean value with default value.
-----------------------------------------------------------------*/
bool INIfile::get(char *section, char *param, const bool b_default)
   {
	bool b = b_default;

	char *s = get(section,param);
	if (s!=NULL)
	   {
		if (stricmp(s,"TRUE")==0) b=true;
		else
		   {
			b=false;
			if (stricmp(s,"FALSE")!=0) errorCode = illval;
		   };
	   }
	else
	   {
		b=b_default;
	   };
	return b;
   }


/*-------------------------set--------------------------
Write an integer value.
-----------------------------------------------------------------*/
bool INIfile::set(char *section, char *param, const int i)
   {
	char s[40];
	itoa(i,s,10);
	return set(section,param,s);
   }


/*-------------------------set--------------------------
Write a double value.
-----------------------------------------------------------------*/
bool INIfile::set(char *section, char *param, const double d)
   {
	char s[40];
	sprintf(s,"%f",d);
	return set(section,param,s);
   }


/*-------------------------set--------------------------
Write a boolean value.
-----------------------------------------------------------------*/
bool INIfile::set(char *section, char *param, const bool b)
   {
	char s[10] = "False";
	if (b) strcpy(s,"True");
	return set(section,param,s);
   }


////////////////////// Internal functions /////////////////////////

/*---------------------------readRows-----------------------------
Reads an Ini file and adds the rows in the list "rows".
The list isn't cleared before reading!
The row number continues after the last row stored in "rows" or
begins with 1 if there aren't any rows.

Input:  fname - INI file name
Return: False = Error. errorCode will be set if an error occurs.
The rows read so far will be left in the list.
-----------------------------------------------------------------*/
bool INIfile::readRows()
   {
// Open the file...
	ifstream f(name);
	if (!f.good()) { errorCode = iopen; return false; }

		char         s[500];
	int          line=0;
	commentRow   comment;
	parameterRow parameter;
	sectionRow   section;
	row*         r;

// Start with a different number?
	if (rows.last!=NULL) line=((row*)rows.last)->rowNumber;

// Read INI file...
	while(!f.eof())
	   {
// Read line
		f.getline(s,sizeof(s));
		if (f.eof() && *s==0) break;
		line++;
// Is it a comment?
		if (comment.is_comment(s)) r = new commentRow(line);
		else
		   {
// Is it a parameter?
			if (parameter.is_parameter(s)) r = new parameterRow(line);
			else
			   {
// A section?
				if (section.is_section(s)) r = new sectionRow(line);
				else
// What then? Well, we treat it as comment...
					r = new commentRow(line);
			   };
		   };
		if (r==NULL) { errorCode = nomem; return false; };
		if (!r->set_string(s))
		   {
			errorCode = nomem;
			delete r;
			return false;
		   };
		rows.insert_last(r);
	   };
	return true;
   }


/*-------------------------createEmptySection---------------------
Create the dummy section for all parameters without a section
(rowNumber=0).
errorCode will be set if an error occurs (but it isn't set to 0
at the beginning!).
The section is inserted to "rows" (as first row) but not to "sections".

Return: False = Error.
-----------------------------------------------------------------*/
bool INIfile::createEmptySection()
   {
// Allocate object...
	sectionRow *srow = new sectionRow(0);
	if (srow==NULL)
	   {
		errorCode = nomem;
		return false;
	   };

// Set name ([]).
	char empty[3]="[]";
	if (!srow->set_string(empty))
	   {
		delete srow;
		errorCode = nomem;
		return false;
	   };
// Insert in list
	rows.insert_first(srow);
	return true;
   }


/*-----------------------------fexpand----------------------------
Pre: efname must be big enough to hold the expanded path (MAXPATH)
-----------------------------------------------------------------*/
bool INIfile::fexpand(char *fname, char *efname)
   {
	char drive[MAXDRIVE],dir[MAXDIR],name[MAXFILE],ext[MAXEXT];
	char cwd[MAXPATH];

// Get current working directory
	getcwd(cwd,MAXPATH);

// Decompose path "fname"
	fnsplit(fname,drive,dir,name,ext);

// Is fname an absolute path?
	if (dir[0]=='/' || dir[0]=='\\')
	   {
// Do we have to add the drive?
		if (drive[0]==0) { drive[0]=cwd[0]; drive[1]=0; };
// Merge path
		fnmerge(efname,drive,dir,name,ext);
	   }
	else                       // it was a relative path...
	   {
		char wd[MAXPATH];       // absolute path for fname

// Was there a drive in fname?
		if (drive[0]!=0)
		   {
			int disk = toupper(drive[0])-'A';
// Change disk
			setdisk(disk);
// Was the change successful?
			if (getdisk()!=disk) return false;
// Get the current directory (on the new drive)
			getcwd(wd,MAXPATH);
// Restore working directory
			chdir(cwd);
// Skip drive
			fname+=2;
		   }
		else                    // no drive, then just use the current directory
		   {
			strcpy(wd,cwd);
		   };

		int wd_len=strlen(wd);
		strcpy(efname,wd);
		if (wd[wd_len-1]!='/')
		   {
			strcat(efname,"/");
			wd_len++;
		   };
		strncat(efname,fname,MAXPATH-wd_len);
	   };
	return true;
   }


