//----------------------------------------------------------------------------
//
// DEGUI - C++ Objects for Allegro's gui 1.1 beta
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------
#ifndef DEGUI_OBJECTS_HEADER
#define DEGUI_OBJECTS_HEADER

//----------------------------------------------------------------------------
#include <values.h>
#include "allegro.h"

//----------------------------------------------------------------------------
// Version Info
#define DEGUI_VERSION          1
#define DEGUI_SUB_VERSION      2
#define DEGUI_VERSION_STR      "1.2 beta"
#define DEGUI_DATE_STR         "August 1997"

//----------------------------------------------------------------------------
// The default color RGB values
#define DEFAULT_COLOR_FORE       {32/4,32/4,32/4}
#define DEFAULT_COLOR_MID        {128/4,128/4,128/4}
#define DEFAULT_COLOR_BACK       {192/4,192/4,192/4}
#define DEFAULT_COLOR_LIGHT_SHAD {255/4,255/4,255/4}
#define DEFAULT_COLOR_DARK_SHAD  {92/4,92/4,92/4}
#define DEFAULT_COLOR_SELECT     {128/4,192/4,128/4}
#define DEFAULT_COLOR_DESELECT   {220/4,220/4,220/4}
#define DEFAULT_COLOR_DISABLE    {128/4,128/4,128/4}

//----------------------------------------------------------------------------
// Let C compilers ignore the C++ stuff
#ifndef __cplusplus
#define NO_CPP_COMPILER
#endif
#ifndef NO_CPP_COMPILER

//----------------------------------------------------------------------------
// A simple object for object color control
//----------------------------------------------------------------------------
class color_scheme
			{
			private:

   		protected:
               int _fore;
               int _mid;
               int _back;
               int _light_shad;
               int _dark_shad;
               int _select;
               int _deselect;
               int _disable;

			public:
					// Constructors and destructors
					color_scheme (void);
					virtual ~color_scheme (void);

               // Getting color info
               inline int fore       (void) const { return _fore;       };
               inline int mid        (void) const { return _mid;        };
               inline int back       (void) const { return _back;       };
               inline int light_shad (void) const { return _light_shad; };
               inline int dark_shad  (void) const { return _dark_shad;  };
               inline int select     (void) const { return _select;     };
               inline int deselect   (void) const { return _deselect;   };
               inline int disable    (void) const { return _disable;    };

               // Setting colors
               inline void set_fore       (const int x) { _fore = x;       };
               inline void set_mid        (const int x) { _mid = x;        };
               inline void set_back       (const int x) { _back = x;       };
               inline void set_light_shad (const int x) { _light_shad = x; };
               inline void set_dark_shad  (const int x) { _dark_shad = x;  };
               inline void set_select     (const int x) { _select = x;     };
               inline void set_deselect   (const int x) { _deselect = x;   };
               inline void set_disable    (const int x) { _disable = x;    };

               // Reset to the default RGB values
               // possibly forcing to those values
               void reset (const bool = false);
			};
//----------------------------------------------------------------------------
// This is the base object
//----------------------------------------------------------------------------

// Forward declarations
class dialog;
class window_object;

class dialog_object
			{
			private:
               // d pointer manipulation
               DIALOG *d;
               bool connected;
               bool in_c_dialog;

               // The parent dialog control
               dialog *_parent;

               // The controlling window
               window_object *_window;

			protected:
               // Mark an object that if will be redrawn
               bool _redraw;

               // Variables for the callback functions
               void(*all_callback)(void);
               int(*start_callback)(void);
               int(*end_callback)(void);
               int(*draw_callback)(void);
               int(*click_callback)(void);
               int(*dclick_callback)(void);
               int(*key_callback)(void);
               int(*char_callback)(int);
               int(*xchar_callback)(int);
               int(*wantfocus_callback)(void);
               int(*gotfocus_callback)(void);
               int(*lostfocus_callback)(void);
               int(*gotmouse_callback)(void);
               int(*lostmouse_callback)(void);
               int(*idle_callback)(void);
               int(*radio_callback)(int);
               int(*unknown_callback)(int);

               // Variables for handling the default behaviour for callbacks
               static int default_message;
               static dialog_object *default_object;
               static int default_data;
               static int default_answer;

               // Helper stuff for objects to be able to hold the focus
               bool demand_focus;

   		public:
					// Constructors and destructors
					dialog_object (void);
					virtual ~dialog_object (void);

               // The main dispatching function i.e. the interface between the C and C++ object
					// This function calls the right member functions when used
					// an allegro gui procedure. You probably don't have to use it yourself
               static int object_message_proc (int, DIALOG*, int);

               // Message functions
               virtual void msg_start (void);
               virtual void msg_end (void);
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual void msg_dclick (void);
               virtual void msg_key (void);
               virtual bool msg_char (const int);
               virtual bool msg_xchar (const int);
               virtual bool msg_wantfocus (void);
               virtual void msg_gotfocus (void);
               virtual void msg_lostfocus (void);
               virtual void msg_gotmouse (void);
               virtual void msg_lostmouse (void);
               virtual void msg_idle (void);
               virtual void msg_radio (const int);
               virtual void msg_unknown (const int);

               // Interface to the base message passing procedure
               int send_message (const int = MSG_IDLE, const int = 0);

               // Set the callback functions
               void set_all_callback       (void(*callback)(void)) { all_callback = callback; };
               void set_start_callback     (int(*callback)(void))  { start_callback = callback; };
               void set_end_callback       (int(*callback)(void))  { end_callback = callback; };
               void set_draw_callback      (int(*callback)(void))  { draw_callback = callback; };
               void set_click_callback     (int(*callback)(void))  { click_callback = callback; };
               void set_dclick_callback    (int(*callback)(void))  { dclick_callback = callback; };
               void set_key_callback       (int(*callback)(void))  { key_callback = callback; };
               void set_char_callback      (int(*callback)(int))   { char_callback = callback; };
               void set_xchar_callback     (int(*callback)(int))   { xchar_callback = callback; };
               void set_wantfocus_callback (int(*callback)(void))  { wantfocus_callback = callback; };
               void set_gotfocus_callback  (int(*callback)(void))  { gotfocus_callback = callback; };
               void set_lostfocus_callback (int(*callback)(void))  { lostfocus_callback = callback; };
               void set_gotmouse_callback  (int(*callback)(void))  { gotmouse_callback = callback; };
               void set_lostmouse_callback (int(*callback)(void))  { lostmouse_callback = callback; };
               void set_idle_callback      (int(*callback)(void))  { idle_callback = callback; };
               void set_radio_callback     (int(*callback)(int))   { radio_callback = callback; };
               void set_unknown_callback   (int(*callback)(int))   { unknown_callback = callback; };

               // The deafult behavious function for callbacks
               static void default_behaviour (void);

  					// Ask about the object state in a easy manner
               int  x         (void) const { return d->x; };
               int  y         (void) const { return d->y; };
               int  w         (void) const { return d->w; };
               int  h         (void) const { return d->h; };
               int  x2        (void) const { return d->x + d->w; };
               int  y2        (void) const { return d->y + d->h; };
               int  fg        (void) const { return d->fg; };
               int  bg        (void) const { return d->bg; };
               int  key       (void) const { return d->key; };
               int  flags     (void) const { return d->flags; };
					bool has_focus (void) const { if(d->flags&D_GOTFOCUS) return true; return false; }
					bool has_mouse (void) const { if(d->flags&D_GOTMOUSE) return true; return false; }
					bool hidden    (void) const { if(d->flags&D_HIDDEN  ) return true; return false; }
					bool will_exit (void) const { if(d->flags&D_EXIT    ) return true; return false; }
					bool selected  (void) const { if(d->flags&D_SELECTED) return true; return false; }
					bool disabled  (void) const { if(d->flags&D_DISABLED) return true; return false; }
               int  d1        (void) const { return d->d1; };
               int  d2        (void) const { return d->d2; };

               // Get the object parent dialog - used for redraws or such
               dialog* parent (void) const { return _parent; };

               // Change some stuff
               virtual void set_x     (const int i) { d->x = i; _redraw = true; };
               virtual void set_y     (const int i) { d->y = i; _redraw = true; };
               virtual void set_w     (const int i) { d->w = i; _redraw = true; };
               virtual void set_h     (const int i) { d->h = i; _redraw = true; };
               virtual void set_fg    (const int i) { d->fg = i; _redraw = true; };
               virtual void set_bg    (const int i) { d->bg = i; _redraw = true; };
               virtual void set_key   (const int i) { d->key = i; _redraw = true; };
               virtual void set_flag  (const int i) { d->flags |= i; _redraw = true;};
               virtual void clear_flag(const int i) { d->flags &= ~i; _redraw = true; };
               virtual void enable    (void) { d->flags &= ~D_DISABLED; _redraw = true; };
               virtual void disable   (void) { d->flags |= D_DISABLED; _redraw = true; };
               virtual void hide      (void) { d->flags |= D_HIDDEN; };
               virtual void unhide    (void) { d->flags &= ~D_HIDDEN; _redraw = true; };
               virtual void make_exit (void) { d->flags |= D_EXIT; _redraw = true; };
               virtual void dont_exit (void) { d->flags &= ~D_EXIT; _redraw = true; };
               virtual void set_d1    (const int i) { d->d1 = i; _redraw = true; };
               virtual void set_d2    (const int i) { d->d2 = i; _redraw = true; };

               virtual void redraw    (void) { _redraw = true; };

               virtual void place     (const int nx, const int ny) { set_x(nx); set_y(ny); };
               virtual void resize    (const int nw, const int nh) { set_w(nw); set_h(nh); };
               virtual void shape     (const int nx, const int ny, const int nw, const int nh) { set_x(nx); set_y(ny); set_w(nw); set_h(nh); };

               virtual void select    (void) { d->flags |= D_SELECTED; _redraw = true; };
               virtual void deselect  (void) { d->flags &= ~D_SELECTED; _redraw = true; };

               color_scheme *color;

               // Create a DIALOG so that we can easily interface with C routines
               DIALOG create_DIALOG (const int = -1, const int = -1, const int = -1, const int = -1, const int = -1, const int = -1);

               // A pointer you can use for whatever you want
               void *dataptr;

               // DIALOG *d manipulation DANGEROUS !!!
               // Make sure you know what you are doing before you use these
               virtual void connect (DIALOG *);
               virtual void disconnect (void);

         // Let the proc object see the d pointer directly
         friend class proc_object;

         // Let the dialog adjust the parent
         friend class dialog;
         friend class dialog_do;

         // Let the window adjust the base window
         friend class window_object;
			};

//----------------------------------------------------------------------------
// This is a wrapper class for classes that are based on allegro's stuff
//----------------------------------------------------------------------------
class proc_object
         :public dialog_object
			{
			private:
               // The related C procedure
               int(*dialog_procedure)(int,DIALOG*,int);

               // The address of the pointer that should be
               // passed as dp to the Allegro stuff
               // Keeping this as a pointer to a pointer allows derived
               // objects to modify the pointer and this class will
               // always point to the right place to get the pointer
               void **ptr_dp;

               // Helper function to pass messages to Allegro C functions
               // This is usefull for the C++ wrappers for Allegro C procedures
               // so that the C++ object just calls the C procedures
               int c_message (const int, const int = 0);

			protected:
               void setup (int(*pr)(int,DIALOG*,int), void **p = NULL) { dialog_procedure = pr; ptr_dp = p; };

   		public:
               // Message functions
               virtual void msg_start (void);
               virtual void msg_end (void);
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual void msg_dclick (void);
               virtual void msg_key (void);
               virtual bool msg_char (const int);
               virtual bool msg_xchar (const int);
               virtual bool msg_wantfocus (void);
               virtual void msg_gotfocus (void);
               virtual void msg_lostfocus (void);
               virtual void msg_gotmouse (void);
               virtual void msg_lostmouse (void);
               virtual void msg_idle (void);
               virtual void msg_radio (const int);
               virtual void msg_unknown (const int);

					// Constructors and destructors
					proc_object (int(*pr)(int,DIALOG*,int), void **p = NULL);
					virtual ~proc_object (void);
			};

//----------------------------------------------------------------------------
// A simple object to clear the screen
//----------------------------------------------------------------------------
class clear_screen_object
         :public dialog_object
			{
			private:

   		protected:

			public:
					// Constructors and destructors
					clear_screen_object (void);
					virtual ~clear_screen_object (void);

               // Message functions
               virtual void msg_draw (void);
			};

//----------------------------------------------------------------------------
// Base object for all the panel-like objects
//----------------------------------------------------------------------------
class panel_object
         :public dialog_object
			{
			private:

			protected:

			public:
			};

//----------------------------------------------------------------------------
// Simple object just draws a box
//----------------------------------------------------------------------------
class box_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					box_object (void);
					virtual ~box_object (void);

               // Message functions
               virtual void msg_draw (void);
			};

//----------------------------------------------------------------------------
// Simple object just draws a raised panel
//----------------------------------------------------------------------------
class panel_raised_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					panel_raised_object (void);
					virtual ~panel_raised_object (void);

               // Message functions
               virtual void msg_draw (void);
			};
//----------------------------------------------------------------------------
// Simple object just draws a sunken panel
//----------------------------------------------------------------------------
class panel_sunken_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					panel_sunken_object (void);
					virtual ~panel_sunken_object (void);

               // Message functions
               virtual void msg_draw (void);
			};
//----------------------------------------------------------------------------
// Simple object just draws a ridged panel
//----------------------------------------------------------------------------
class panel_ridge_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					panel_ridge_object (void);
					virtual ~panel_ridge_object (void);

               // Message functions
               virtual void msg_draw (void);
			};

//----------------------------------------------------------------------------
// Simple object just draws a grooved panel
//----------------------------------------------------------------------------
class panel_groove_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					panel_groove_object (void);
					virtual ~panel_groove_object (void);

               // Message functions
               virtual void msg_draw (void);
			};
//----------------------------------------------------------------------------
// Simple object to draw a shadow box
//----------------------------------------------------------------------------
class shadow_box_object
         :public panel_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					shadow_box_object (void);
					virtual ~shadow_box_object (void);

               // Message functions
               virtual void msg_draw (void);
			};
//----------------------------------------------------------------------------
// Simple object draws a bitmap
//----------------------------------------------------------------------------
class bitmap_object
			:public dialog_object
			{
			private:

			protected:
 					// The bitmap to draw
            	BITMAP *b;
               bool allocated;

			public:
					// Constructors and destructors
					bitmap_object (BITMAP* = NULL);
               bitmap_object (const char*);
					virtual ~bitmap_object (void);

               // Message functions
               virtual void msg_draw (void);

               // Change the bitmap
               void set_bitmap (BITMAP *new_b) { b = new_b; redraw(); };
			};

//----------------------------------------------------------------------------
// Simple object draws some text
//----------------------------------------------------------------------------
class text_object
			:public dialog_object
			{
			private:

			protected:
 					// The text to print
					const char *text;

			public:
					// Constructors and destructors
					text_object (const char* = NULL);
					virtual ~text_object (void);

               // Message functions
               virtual void msg_draw (void);

               // Modify the object text ont the fly
               void set_text (const char *t) { text = t; redraw(); };
			};

//----------------------------------------------------------------------------
// Simple object to draw some ceneterd text
//----------------------------------------------------------------------------
class centerd_text_object
			:public text_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					centerd_text_object (const char* = "Centerd text");
					virtual ~centerd_text_object (void);

               // Message functions
               virtual void msg_draw (void);
			};

//----------------------------------------------------------------------------
// Button object
//----------------------------------------------------------------------------
class button_object
			:public text_object
			{
			private:

			protected:
					const bool willexit;
					const char shortcut;

			public:
					// Constructors and destructors
					button_object (const char* = "Button", const char = 0, const bool = false);
					virtual ~button_object (void);

               // Message functions
               virtual void msg_draw (void);
               virtual void msg_key (void);
               virtual bool msg_wantfocus (void);
			};

//----------------------------------------------------------------------------
// Simple check box object
//----------------------------------------------------------------------------
class checkbox_object
			:public button_object
			{
			private:

			protected:

			public:
					// Constructors and destructors
					checkbox_object (const char* = "Checkbox");
					virtual ~checkbox_object (void);

               // Message functions
               virtual void msg_draw (void);
			};

//----------------------------------------------------------------------------
// Radio button object
//----------------------------------------------------------------------------
class radio_button_object
			:public button_object
			{
			private:

			protected:
               const int group;
               const bool square;

			public:
					// Constructors and destructors
					radio_button_object (const char* = "Radio button", const int = 0, const bool = false);
					virtual ~radio_button_object (void);

               // Message functions
               virtual void msg_draw (void);
               virtual void msg_key (void);
               virtual void msg_radio (const int);
			};

//----------------------------------------------------------------------------
// Keyboard object
//----------------------------------------------------------------------------
class keyboard_object
			:public proc_object
			{
			private:
					int(*proc)(void);
               int keypress;

			protected:

			public:
					// Constructors and destructors
					keyboard_object (int(*pro)(void), const int key);
					virtual ~keyboard_object (void);
			};

//----------------------------------------------------------------------------
// Editable text string object
//----------------------------------------------------------------------------
class edittext_object
         :public dialog_object
			{
			private:
               // Some internal helper functions
               void draw_text (void);

               void setup (const char*, const int);

			protected:
  					char *text;
					int max_length;

               int cursorpos;
               int offsetpos;

			public:
               // Message functions
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual bool msg_char (const int);

					// Constructors and destructors
					edittext_object (const int = 14);
					edittext_object (const char*, const int = 14);
					virtual ~edittext_object (void);

               // Set and get the text out
               void set_text (const char*, const int len = -1);
               char* get_text (void) { return text; };
			};

//----------------------------------------------------------------------------
// An object that can contain a scrollbar
//----------------------------------------------------------------------------
const int default_scrollbar_width = 10; // Change this for wider or narrow scrollbars

class scrollable_object
         :public dialog_object
			{
			private:

			protected:
               int listsize;
               int offset;
               int height;

               int width;

               // Basically lifted from Allegro's gui.c
               void scroll (int&);
               void handle_click (void);

         public:
					// Constructors and destructors
					scrollable_object (void) : listsize(0), offset(0), width(default_scrollbar_width) { };
					virtual ~scrollable_object (void) { };
         };

//----------------------------------------------------------------------------
// Listbox object
//----------------------------------------------------------------------------
// Default list procedure with some simple numbers
extern char* default_list_object_procedure (int, int*);

class list_object
      	:public scrollable_object
			{
			private:

			protected:
					// The listing function
               char*(*list_grabber_proc)(int,int*);

               int index;

               // Basically lifted from Allegro's gui.c
               void handle_click (void);

			public:
               // Message functions
               virtual void msg_start (void);
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual bool msg_char (const int);

					// Constructors and destructors
					list_object (char*(*pro)(int,int*) = default_list_object_procedure);
					virtual ~list_object (void);

					// Getting some info out
					virtual int selection (void) { return index; };
					virtual void selection (const int i) { index = i; };
			};

//----------------------------------------------------------------------------
// Partially hidden listbox object
//----------------------------------------------------------------------------
class hidden_list_object
      	:public list_object
			{
			private:

			protected:
               // Number of items to display when we display the whole list
               int nitems;

               // Are we displaying the list or not
               bool displaylist;
               BITMAP *underlist;

               bool donechoosing;

               // Size tracking
               int oldy, oldh;

			public:
               // Message functions
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual void msg_dclick (void);
               virtual void msg_key (void);
               virtual bool msg_char (const int);
               virtual bool msg_wantfocus (void) { return true; };
               virtual void msg_lostfocus (void);
               virtual void msg_lostmouse (void);

					// Constructors and destructors
					hidden_list_object (char*(*pro)(int,int*) = default_list_object_procedure, const int n = 5);
					virtual ~hidden_list_object (void);
			};

//----------------------------------------------------------------------------
// Textbox object
//----------------------------------------------------------------------------
// Default textbox text
extern char* default_textbox_object_text;

class textbox_object
         :public scrollable_object
			{
			private:
               // The initial text and info
               char *text;
               int length;
               bool self_allocated;

               // Number of lines the the text
               int _lines;

               // Word wrapping variables
               bool wrapping;
               bool word_wrapping;

               // We have to monitor the width and the font
               // to make sure that we update the screen if
               // they are changed
               int width;
               int last_width;
               FONT* last_font;

               // Internal placement of line ends
               char **places;
               int allocated;

               // Allocation jumps for internal data expansion
               static int allocation_step;

               // Helper function for determining lines in the text
               char* find_line_end (char *string);

               // Buffering for the hidden characters
               char *buffered_place;
               char buffered_char;

               // Read line out of the text - Used by the list procedure
               char* get_line (const int);
               int lines (void) const { return _lines; };

               // The list procedure to get the text lines
               static textbox_object *current_textbox;
               static char* list_procedure (int, int*);

			protected:
					// The listing function
               char*(*list_grabber_proc)(int,int*);

   		public:
               // Message functions
               virtual void msg_draw (void);
               virtual void msg_click (void);
               virtual bool msg_char (const int);
               virtual bool msg_wantfocus (void);

					// Constructors and destructors
					textbox_object (char*, const bool = false, const bool = false);
					virtual ~textbox_object (void);

               // Setup the text
               void set_text (char *tex);
			};

//----------------------------------------------------------------------------
// Simple object draws a progress bar
//----------------------------------------------------------------------------
class progress_bar_object
			:public dialog_object
			{
			private:

			protected:
               float _scale;
               float _progress;
               bool _showtext;

			public:
					// Constructors and destructors
					progress_bar_object (const bool = true, const float sc = 100.);
					virtual ~progress_bar_object (void);

               // Message functions
               virtual void msg_draw (void);

               // Modify the caharcteristics
               void scale (const float sc)    { _scale = sc; _redraw = true; };
               void progress (const float pr) { _progress = pr; if(_progress>_scale) _progress = _scale; _redraw = true; };
			};

//----------------------------------------------------------------------------
// Wrapper for MENU struct
//----------------------------------------------------------------------------
class menu
			{
			private:
               MENU *internal;
               int number;

               void _add(const char*, int(*proc)(void), const MENU*);

			protected:

			public:
					// Constructors and destructors
					menu (void);
					virtual ~menu (void);

					// Add something to the menu
					void add (const char* = "", int(*proc)(void) = NULL);
               void add (const char*, const menu&);

               // The menu object needs to be able to acces the internal menu
               friend class menu_object;
			};

//----------------------------------------------------------------------------
// Menu object
//----------------------------------------------------------------------------
struct MENU_INFO;

class menu_object
			:public dialog_object
			{
			private:
               MENU *internal;

			protected:
               // Slightly modified routines from Allegro's gui.c
               int menu_mouse_object(MENU_INFO *m);
               int mouse_in_parent_menu(MENU_INFO *m);
               void fill_menu_info(MENU_INFO *m, MENU *menu, MENU_INFO *parent, int bar, int x, int y);
               int menu_key_shortcut(int c, char *s);
               int menu_alt_key(int k, MENU *m);
               int _do_menu(MENU *menu, MENU_INFO *parent, int bar, int x, int y, int repos, int *dret);

			public:
               // Message functions
               virtual void msg_start (void);
               virtual void msg_draw (void);
               virtual bool msg_xchar (const int);
               virtual void msg_click (void);

					// Constructors and destructors
					menu_object (const menu&);
					virtual ~menu_object (void);
			};

//----------------------------------------------------------------------------
// A simple object to clear the screen and put up a grid
//----------------------------------------------------------------------------
class grid_object
			:public dialog_object
			{
			private:
               int x_gran;
               int y_gran;

   		protected:

			public:
               // Message functions
               virtual void msg_draw (void);

					// Constructors and destructors
					grid_object (const int, const int);
					virtual ~grid_object (void);

               void set_granularity (const int x, const int y) { x_gran = x; y_gran = y; redraw(); };
			};
//----------------------------------------------------------------------------
// A class for emulating Allegro's do_dialog function
//----------------------------------------------------------------------------
class dialog_do
			{
			private:

   		protected:
               DIALOG *_dialog;
					int number;
               int max_number;

               // Normal modification stuff
               bool _close;
               bool _redraw;

               // If true, an escape will close the dialog
               bool _escape_exits;
               bool _cant_exit;

               // Modified versions of Allegro's SEND_MESSAGE and MESSAGE macros
               int send_message (const int i, int msg, int c) { if(_dialog[i].proc==NULL) return D_O_K; return _dialog[i].proc(msg, &_dialog[i], c); };
               int message (const int i, const int msg, const int c, int* res, int* obj);

               // Send a message to all the objects
               int dialog_message (int msg, int c, int *obj);

               // Functions to deal with focus handling
               int focus_obj;
               int offer_focus (int obj, int force);
               int move_focus (long ch);

               // Function to deal with the mouse
               int mouse_obj;
               int find_mouse_object (void);

               // Object manipulation
               void add (int(*proc)(int,DIALOG*,int), const int = 0,  const int = 0, const int = 0, const int = 0, const int = gui_fg_color, const int = gui_bg_color, const int = 0, const int = 0, const int = 0, const int = 0, void* = NULL );

         public:
					// Constructors and destructors
					dialog_do (DIALOG*);
					dialog_do (const int);
					virtual ~dialog_do (void);

               int execute (const int);
               int popup (const int);
               void center (const int = 0);

               void close (void) { _close = true; };
               void redraw (const bool = false);
			};
//----------------------------------------------------------------------------
// Allegro dialog managing class
//----------------------------------------------------------------------------
class dialog
         :public dialog_object, public dialog_do
			{
			private:

         protected:
               // Scaling data
               int font_height;
               int font_width;

               // Private helper functions
               void find_extrema (const int, const int, int&, int&, int&, int&);

			public:
					// Constructors and destructors
					dialog (const int = 50);
					virtual ~dialog (void);

					// Add an object to the dialog
               void add (int(*proc)(int,DIALOG*,int), const int x,  const int y, const int w, const int h, const int fg, const int bg, const int key, const int flags, const int d1, const int d2, void *dp) { dialog_do::add(proc,x,y,w,h,fg,bg,key,flags,d1,d2,dp); };
					void add (dialog_object&, const int = -1, const int = -1, const int = -1, const int = -1);
					void add (window_object&, const int = -1, const int = -1, const int = -1, const int = -1);
               void add (dialog&);

               void remove (dialog_object&);
               void remove (window_object& w);
               void remove (dialog&);

               // Send a message to all the objects
               void dialog_message (int msg, int c) { int i; dialog_do::dialog_message(msg,c,&i); };

					// Do the dialog
               void scale_to_font (FONT *);
					dialog_object* popup (const dialog_object* = NULL);
					dialog_object* execute (const dialog_object* = NULL);

               void redraw (const bool b = false) { dialog_do::redraw(b); };

               // Special manipulation functions
               virtual void disable (void);
			};
//----------------------------------------------------------------------------
// Allegro window object managing class
//----------------------------------------------------------------------------
class window_object
         :public dialog
			{
			private:
               // Private and unimplemented to stop recursive windows
               void remove (window_object& w);
					void add (window_object&, const int = -1, const int = -1, const int = -1, const int = -1);

         protected:
               // The title text
               const char *text;

               // Possibly modify size to contain the internal objects
               void ensure_containment (void);

         public:
					// Constructors and destructors
					window_object (const char *t = NULL, const int = 50);
					virtual ~window_object (void);

					void add (dialog_object&, const int = -1, const int = -1, const int = -1, const int = -1);
               void add (dialog&);

               void remove (dialog_object&);
               void remove (dialog&);

               // Modify the object text ont the fly
               void set_text (const char *t) { text = t; redraw(); };

               // Message functions
               virtual void msg_draw (void);
               virtual void msg_click (void);
         };
#endif // NO_CPP_COMPILER

//----------------------------------------------------------------------------
// Here are the C support routines
//----------------------------------------------------------------------------
// Force C linkage for C routines
#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------------------
// Possibly redefine the normal Allegro objects
#define d_clear_proc d_degui_clear_proc
#define d_box_proc d_degui_box_proc
#define d_shadow_box_proc d_degui_shadow_box_proc
#define d_text_proc d_degui_text_proc
#define d_ctext_proc d_degui_ctext_proc
#define d_button_proc d_degui_button_proc
#define d_check_proc d_degui_checkbox_proc
#define d_radio_proc d_degui_radio_button_proc
#define d_edit_proc d_degui_edit_proc
#define d_list_proc d_degui_list_proc
#define d_menu_proc d_degui_menu_proc

// Replace Allegro's alerts with ours
#define alert degui_alert
#define alert3 degui_alert3

// Replace Allegro's file select with ours
#define file_select degui_file_select

// Replace Allegro's file select with ours
#define gfx_mode_select degui_gfx_mode_select

// Repacle the menu stuff with ours
#define do_menu do_degui_menu

//----------------------------------------------------------------------------
// Here are the extra colors needed
extern int degui_fore_color;
extern int degui_mid_color;
extern int degui_back_color;
extern int degui_light_shad_color;
extern int degui_dark_shad_color;
extern int degui_select_color;
extern int degui_deselect_color;
extern int degui_disable_color;

//----------------------------------------------------------------------------
// Here is a routine to set the default C object colors
void set_default_object_colors (const int, const int, const int, const int, const int, const int, const int, const int);
void reset_default_object_colors (const int force);

//----------------------------------------------------------------------------
// Replacement for Allegro's dialog functions
//----------------------------------------------------------------------------
int d_degui_clear_proc (int msg, DIALOG *d, int c);
int d_degui_box_proc (int msg, DIALOG *d, int c);
int d_degui_shadow_box_proc (int msg, DIALOG *d, int c);
int d_degui_text_proc (int msg, DIALOG *d, int c);
int d_degui_ctext_proc (int msg, DIALOG *d, int c);
int d_degui_button_proc (int msg, DIALOG *d, int c);
int d_degui_checkbox_proc (int msg, DIALOG *d, int c);
int d_degui_radio_button_proc (int msg, DIALOG *d, int c);
int d_degui_edit_proc (int msg, DIALOG *d, int c);
int d_degui_list_proc (int msg, DIALOG *d, int c);
int d_degui_menu_proc (int msg, DIALOG *d, int c);

// Some extra Allegro objects
int d_degui_progress_proc (int msg, DIALOG *d, int c);
int d_degui_panel_raised_proc (int msg, DIALOG *d, int c);
int d_degui_panel_sunken_proc (int msg, DIALOG *d, int c);
int d_degui_panel_ridge_proc (int msg, DIALOG *d, int c);
int d_degui_panel_groove_proc (int msg, DIALOG *d, int c);

//----------------------------------------------------------------------------
// A replacement for Allegro's Alert functions
//----------------------------------------------------------------------------
int degui_alert3(char *s1, char *s2, char *s3, char *b1, char *b2, char *b3, int c1, int c2, int c3);
int degui_alert(char *s1, char *s2, char *s3, char *b1, char *b2, int c1, int c2);

//----------------------------------------------------------------------------
// A replacement for Allegro's file_select function
//----------------------------------------------------------------------------
int degui_file_select(char *message, char *path, char *ext);

//----------------------------------------------------------------------------
// A replacement for Allegro's graphics mode selection function
//----------------------------------------------------------------------------
int degui_gfx_mode_select(int *card, int *resw, int *resh);

//----------------------------------------------------------------------------
// A replacement for Allegro's do_menu function
//----------------------------------------------------------------------------
int do_degui_menu(MENU *menu, int x, int y);

//----------------------------------------------------------------------------
// Force C linkage for C routines
#ifdef __cplusplus
}
#endif
//----------------------------------------------------------------------------
#endif // DEGUI_OBJECTS_HEADER

