/* lzopack.h -- LZO packer

   This file is part of the LZO real-time data compression package.

   Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library and packer is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#ifndef __LZOPACK_H
#define __LZOPACK_H

#if defined(__EMX__)
#  include <sys/emx.h>
#endif

#if defined(LZOP_HAVE_CONFIG_H)
#  include <config.h>
#endif

#undef LZO_HAVE_CONFIG_H
#include <lzoconf.h>
#if !defined(LZO_VERSION)
#  error "you need the LZO library"
#elif (LZO_VERSION < 0x1020)
#  error "please upgrade your LZO package"
#endif

/* workaround a cpp bug under hpux 10.20 */
#if !defined(LZO_0xffffffffL)
#  define LZO_0xffffffffL	4294967295ul
#endif

#include "version.h"
#include "tailor.h"

#if 0 && (UINT_MAX < LZO_0xffffffffL)
#  error "you lose"
#endif


/*************************************************************************
// system includes
**************************************************************************/

#if !defined(NO_SYS_TYPES_H)
#  include <sys/types.h>
#endif

#define NDEBUG
#undef NDEBUG
#include <assert.h>

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <errno.h>
#if !defined(NO_FCNTL_H)
#  include <fcntl.h>
#endif
#if !defined(NO_SYS_STAT_H)
#  include <sys/stat.h>
#endif
#if defined(HAVE_IO_H) && !defined(NO_IO_H)
#  include <io.h>
#endif
#if defined(HAVE_SIGNAL_H)
#  include <signal.h>
#endif
#if defined(HAVE_UNISTD_H)
#  include <unistd.h>
#endif
#if defined(TIME_WITH_SYS_TIME)
#  include <sys/time.h>
#  include <time.h>
#else
#  include <time.h>
#endif
#if defined(HAVE_UTIME_H)
#  include <utime.h>
#elif defined(HAVE_SYS_UTIME_H)
#  include <sys/utime.h>
#endif


/*************************************************************************
//
**************************************************************************/

#if !defined(__LZO_DOS16) && !defined(__LZO_WIN16)
#  define USE_MALLOC
#endif

#if defined(HAVE_SETMODE)
#  define USE_SETMODE
#endif

#if defined(__DJGPP__)
#  undef USE_SETMODE
#  define USE__READ
#  define USE__WRITE
#  if ((__DJGPP__ * 100 + __DJGPP_MINOR__) < 201)
#    error "need djgpp 2.01 or above"
#  endif
#  include <crt0.h>
#  include <dpmi.h>
#  include <sys/exceptn.h>
#  undef kbhit	/* want to be able to call kbhit from libc */
#endif

#if defined(__LZO_TOS)
#  undef HAVE_SETMODE
#  undef USE_SETMODE
#  if defined(__GNUC__)
#    define USE__READ
#    define USE__WRITE
#  elif defined(__PUREC__)
#    include <ext.h>
#    define O_EXCL_BROKEN
#    if !defined(S_IFMT)
#      if defined(S_IFREG) && defined(S_IFDIR) && defined(S_IFCHR)
#        define S_IFMT		(S_IFREG | S_IFDIR | S_IFCHR)
#      endif
#    endif
#  else
#    error
#  endif
#endif


/*************************************************************************
//
**************************************************************************/

#ifndef LZO_UTYPE_MAX
/* Maximum value of a signed/unsigned type.
   Do not use casts, avoid overflows. */
#define LZO_STYPE_MAX(b)	(((1l  << (8*(b)-2)) - 1l)  + (1l  << (8*(b)-2)))
#define LZO_UTYPE_MAX(b)	(((1ul << (8*(b)-1)) - 1ul) + (1ul << (8*(b)-1)))
#endif

#if !defined(SIZE_T_MAX)
#  if defined(SIZEOF_SIZE_T)
#    define SIZE_T_MAX		LZO_UTYPE_MAX(SIZEOF_SIZE_T)
#  else
#    define SIZE_T_MAX		UINT_MAX
#  endif
#endif

#if !defined(PATH_MAX)
#  define PATH_MAX			512
#elif (PATH_MAX < 512)
#  undef PATH_MAX
#  define PATH_MAX			512
#endif

#if defined(NO_MEMCMP)
#  undef HAVE_MEMCMP
#endif
#if !defined(HAVE_MEMCMP)
#  undef memcmp
#  define memcmp			lzo_memcmp
#endif
#if !defined(HAVE_MEMCPY)
#  undef memcpy
#  define memcpy			lzo_memcpy
#endif
#if !defined(HAVE_MEMSET)
#  undef memset
#  define memset			lzo_memset
#endif

#if !defined(HAVE_STRCHR)
#  if defined(HAVE_INDEX)
#    define strchr			index
#  endif
#endif
#if !defined(HAVE_STRCASECMP)
#  if defined(HAVE_STRICMP)
#    define strcasecmp		stricmp
#  else
#    define strcasecmp		strcmp
#  endif
#endif
#if !defined(HAVE_STRNCASECMP)
#  if defined(HAVE_STRNICMP)
#    define strncasecmp		strnicmp
#  else
#    define strncasecmp		strncmp
#  endif
#endif

#ifndef STDIN_FILENO
#  define STDIN_FILENO		(fileno(stdin))
#endif
#ifndef STDOUT_FILENO
#  define STDOUT_FILENO		(fileno(stdout))
#endif
#ifndef STDERR_FILENO
#  define STDERR_FILENO		(fileno(stderr))
#endif

#ifndef RETSIGTYPE
#  define RETSIGTYPE		void
#endif
#ifndef SIGTYPEENTRY
#  define SIGTYPEENTRY
#endif
typedef RETSIGTYPE (SIGTYPEENTRY *sig_type)(int);


#if !defined(S_IFMT) && defined(_S_IFMT)
#  define S_IFMT		_S_IFMT
#endif
#if !defined(S_IFREG) && defined(_S_IFREG)
#  define S_IFREG		_S_IFREG
#endif
#if !defined(S_IFDIR) && defined(_S_IFDIR)
#  define S_IFDIR		_S_IFDIR
#endif
#if !defined(S_IFCHR) && defined(_S_IFCHR)
#  define S_IFCHR		_S_IFCHR
#endif

#if !defined(S_ISREG)
#  if defined(S_IFMT) && defined(S_IFREG)
#    define S_ISREG(m)	(((m) & S_IFMT) == S_IFREG)
#  else
#    error S_ISREG
#  endif
#endif
#if !defined(S_ISDIR)
#  if defined(S_IFMT) && defined(S_IFDIR)
#    define S_ISDIR(m)	(((m) & S_IFMT) == S_IFDIR)
#  else
#    error S_ISDIR
#  endif
#endif
#if !defined(S_ISCHR)
#  if defined(S_IFMT) && defined(S_IFCHR)
#    define S_ISCHR(m)	(((m) & S_IFMT) == S_IFCHR)
#  endif
#endif


/*************************************************************************
//
**************************************************************************/

#if defined(HAVE_SETMODE)
#  if !defined(O_BINARY)
#    error "setmode without O_BINARY"
#  endif
#endif

#ifndef SUFFIX_MAX
#define SUFFIX_MAX		32
#endif

#define STDIN_NAME		"<stdin>"
#define STDOUT_NAME		"<stdout>"
#define STDERR_NAME		"<stderr>"
#define UNKNOWN_NAME	"<unknown>"


/*************************************************************************
//
**************************************************************************/

#undef FREE
#define FREE(ptr)		if (ptr) { free(ptr); ptr = NULL; }

#undef UNUSED
#define UNUSED(parm)	(parm = parm)

#define ALIGN_DOWN(a,b)	((a) & ~((b) - 1))
#define ALIGN_UP(a,b)	(((a) + ((b) - 1)) & ~((b) - 1))


/*************************************************************************
//
**************************************************************************/

/* exit codes of this program: 0 ok, 1 error, 2 warning */
#define EXIT_OK			0
#define EXIT_ERROR		1
#define EXIT_WARN		2

#define EXIT_USAGE		1
#define EXIT_FILE_READ	1
#define EXIT_FILE_WRITE	1
#define EXIT_MEMORY		1
#define EXIT_CHECKSUM	1
#define EXIT_LZO_ERROR	1
#define EXIT_LZO_INIT	1
#define EXIT_INTERNAL	1


/*************************************************************************
// options
**************************************************************************/

enum {
	CMD_NONE,
	CMD_COMPRESS,
	CMD_DECOMPRESS, CMD_TEST, CMD_LIST, CMD_INFO,
	CMD_SYSINFO, CMD_LICENSE, CMD_HELP, CMD_INTRO, CMD_VERSION
};


enum {
	M_LZO1X_1     =     1,
	M_LZO1X_1_15  =     2,
	M_LZO1X_999   =     3,
	M_ZLIB        =   128,

	M_UNUSED
};

extern int opt_cmd;
extern int opt_method;
extern int opt_level;

extern int opt_checksum;
extern int opt_console;
extern lzo_bool opt_decompress_safe;
extern int opt_force;
extern int opt_noheader;
extern int opt_name;
extern const char *opt_output_name;
extern lzo_bool opt_optimize;
extern lzo_bool opt_stdin;
extern lzo_bool opt_stdout;
extern char opt_suffix[];
extern lzo_bool opt_unlink;
extern int opt_verbose;


/*************************************************************************
// input and output files
**************************************************************************/

typedef struct
{
	int fd;
#if defined(USE_FOPEN)
	FILE *file;
#endif
	int open_flags;
	struct stat st;
	char name[ 2*((PATH_MAX)+1) + SUFFIX_MAX + 1 ];
	lzo_uint32 f_adler32;

	int opt_name;
	unsigned long part;
	unsigned long bytes_read;
	unsigned long bytes_written;
	lzo_bool warn_multipart;
	lzo_bool warn_unknown_suffix;

	unsigned long bytes_processed;
}
file_t;

#define f_name(ft)		((ft).name)


/*************************************************************************
// lzop file header
**************************************************************************/

/* header flags */
#define F_ADLER32_D		0x00000001L
#define F_ADLER32_C		0x00000002L
#define F_STDIN			0x00000004L
#define F_STDOUT		0x00000008L
#define F_NAME_DEFAULT	0x00000010L
#define F_DOSISH		0x00000020L
#define F_H_EXTRA_FIELD	0x00000040L
#define F_H_GMTDIFF		0x00000080L
#define F_CRC32_D		0x00000100L
#define F_CRC32_C		0x00000200L
#define F_MULTIPART		0x00000400L
#define F_H_FILTER		0x00000800L
#define F_MASK			0x00000FFFL

/* operating system that created the file [mostly unused] */
#define F_OS_FAT		0x00000000L			/* DOS, OS2, Win95 */
#define F_OS_AMIGA		0x01000000L
#define F_OS_VMS		0x02000000L
#define F_OS_UNIX		0x03000000L
#define F_OS_VM_CMS		0x04000000L
#define F_OS_ATARI		0x05000000L
#define F_OS_HPFS		0x06000000L			/* OS2 */
#define F_OS_MACOS		0x07000000L
#define F_OS_Z_SYSTEM	0x08000000L
#define F_OS_CPM		0x09000000L
#define F_OS_TOPS20		0x0a000000L
#define F_OS_NTFS		0x0b000000L			/* WinNT */
#define F_OS_QDOS		0x0c000000L
#define F_OS_ACORN		0x0d000000L
#define F_OS_VFAT		0x0e000000L			/* Win95, WinNT */
#define F_OS_MFS		0x0f000000L
#define F_OS_BEOS		0x10000000L
#define F_OS_TANDEM		0x11000000L
#define F_OS_SHIFT		24
#define F_OS_MASK		0xff000000L
#define F_OS_MAX		(F_OS_MASK >> F_OS_SHIFT)
#define F_OS_DOS		F_OS_FAT
#define F_OS_WIN16		F_OS_FAT
#define F_OS_OS2		F_OS_HPFS

/* character set for file name encoding [mostly unused] */
#define F_CS_NATIVE 	0x00000000L
#define F_CS_LATIN1 	0x00100000L
#define F_CS_DOS	 	0x00200000L
#define F_CS_WIN32	 	0x00300000L
#define F_CS_WIN16	 	0x00400000L
#define F_CS_SHIFT		20
#define F_CS_MASK		0x00f00000L
#define F_CS_MAX		(F_CS_MASK >> F_CS_SHIFT)

/* these bits must be zero */
#define F_RESERVED		((F_MASK | F_OS_MASK | F_CS_MASK) ^ 0xffffffffL)

typedef struct
{
	unsigned version;
	unsigned lib_version;
	unsigned version_needed_to_extract;
	unsigned char method;
	unsigned char level;
	lzo_uint32 flags;
	lzo_uint32 filter;
	lzo_uint32 mode;
	lzo_uint32 mtime;
	lzo_uint32 gmtdiff;
	char name[255+1];
	lzo_uint32 header_adler32;

	lzo_uint32 extra_field_len;
	lzo_uint32 extra_field_adler32;

/* info */
	const char *method_name;
}
header_t;


/*************************************************************************
// lzopack.c
**************************************************************************/

void set_err_nl(lzo_bool x);
void warn(file_t *ft, const char *m);
void error(file_t *ft, const char *m);
void fatal(file_t *ft, const char *m);
void read_error(file_t *ft);
void write_error(file_t *ft);
void e_memory(void);

lzo_int read_buf(file_t *ft, lzo_voidp buffer, lzo_int cnt);
void write_buf(file_t *ft, const lzo_voidp buffer, lzo_int cnt);
void read32(file_t *ft, lzo_uint32 *v);
void write32(file_t *ft, lzo_uint32 v);

void init_compress_header(header_t *h, const file_t *fip, const file_t *fop);
void write_header(file_t *ft, const header_t *h);

void do_info(const header_t *h, unsigned long d_len, unsigned long c_len);
void do_list(const header_t *h, unsigned long d_len, unsigned long c_len);
void do_list_total(void);


/*************************************************************************
// compress.c
**************************************************************************/

lzo_bool x_enter(const header_t *h);
void x_leave(const header_t *h);

lzo_bool x_compress(file_t *fip, file_t *fop, header_t *h);
lzo_bool x_decompress(file_t *fip, file_t *fop, const header_t *h, lzo_bool);

int x_get_method(header_t *h);
int x_set_method(int m, int l);

void x_filter(lzo_byte *p, lzo_uint l, const header_t *h);


/*************************************************************************
// p_lzo.c
**************************************************************************/

lzo_bool lzop_enter(const header_t *h);
void lzop_leave(const header_t *h);

lzo_bool lzop_compress(file_t *fip, file_t *fop, const header_t *h);
lzo_bool lzop_decompress(file_t *fip, file_t *fop, const header_t *h, lzo_bool);

int lzop_get_method(header_t *h);
int lzop_set_method(int m, int l);
void lzop_init_compress_header(header_t *h);


/*************************************************************************
// p_zlib.c
**************************************************************************/

#if defined(USE_ZLIB)

lzo_bool zlib_enter(const header_t *h);
void zlib_leave(const header_t *h);

lzo_bool zlib_compress(file_t *fip, file_t *fop, const header_t *h);
lzo_bool zlib_decompress(file_t *fip, file_t *fop, const header_t *h, lzo_bool);

int zlib_get_method(header_t *h);
int zlib_set_method(int m, int l);
void zlib_init_compress_header(header_t *h);

#endif


/*************************************************************************
// filter.c
**************************************************************************/

void t_sub1(lzo_byte *p, lzo_uint l);
void t_add1(lzo_byte *p, lzo_uint l);

void t_sub(lzo_byte *p, lzo_uint l, int n);
void t_add(lzo_byte *p, lzo_uint l, int n);

void t_mtf(lzo_byte *p, lzo_uint l);
void t_unmtf(lzo_byte *p, lzo_uint l);


/*************************************************************************
// util.c
**************************************************************************/

enum {
	SUFF_NONE,
	SUFF_LZO,
	SUFF_LZOP,
	SUFF_TZO,
	SUFF_TAR,
	SUFF_USER
};

char *fn_basename(const char *name);
void fn_addslash(char *n, lzo_bool slash);
char *fn_strlwr(char *n);
int fn_strcmp(const char *n1, const char *n2);
lzo_bool fn_is_same_file(const char *n1, const char *n2);
int fn_has_suffix(const char *name);

#if defined(HAVE_LOCALTIME)
void tm2str(char *s, const struct tm *tmp);
#endif
void time2str(char *s, const time_t *t);

lzo_bool file_exists(const char *name);
lzo_bool isafile(int fd);

lzo_uint32 fix_mode_for_header(lzo_uint32 mode);
int fix_mode_for_chmod(lzo_uint32 mode);
int fix_mode_for_ls(lzo_uint32 mode);
int fix_mode_for_open(int mode);

void mode_string(unsigned mode, char *str);
char *maybe_rename_file(const char *original_name);


/*************************************************************************
// other globals
**************************************************************************/

extern const char *progname;
extern FILE *con_term;
extern lzo_byte __lzo_copyright[];

void sysinfo(void);
void license(void);
void head(void);
void help(void);
void usage(void);
void version(void);

void sysinfo_djgpp(void);


/*************************************************************************
// LZO section
**************************************************************************/

#include <lzo1x.h>
#if !defined(LZO1X_1_MEM_COMPRESS)
#  define LZO1X_1_MEM_COMPRESS  LZO1X_MEM_COMPRESS
#endif

#define USE_LZO1X_1
#if !defined(LZO_999_UNSUPPORTED)
#  define USE_LZO1X_999
#endif
#if !defined(__LZO_DOS16)
#  define USE_LZO1X_1_15
#endif


#if !defined(LZO_EXTERN_CDECL)
#  define LZO_EXTERN_CDECL	LZO_EXTERN
#endif

#if !defined(LZO_ASM_NAME)
#  if (LZO_VERSION >= 0x1030) && defined(__GNUC__)
#    define LZO_ASM_NAME(x)	__asm__("_" ## #x)
#  else
#    define LZO_ASM_NAME(x)
#  endif
#endif


#if !defined(__LZO_NO_ASM)
#  if defined(__LZO_i386) && (UINT_MAX == LZO_0xffffffffL)
#    if defined(__GNUC__)
#      define LZO1X_HAVE_i386_DECOMPRESS_ASM
#    endif
#  endif
#endif


#if defined(LZO1X_HAVE_i386_DECOMPRESS_ASM)
LZO_EXTERN_CDECL(int)
lzo1x_decompress_asm_fast
                        ( const lzo_byte *src, lzo_uint  src_len,
                                lzo_byte *dst, lzo_uint *dst_len,
                                lzo_voidp wrkmem /* NOT USED */ )
                        LZO_ASM_NAME(lzo1x_decompress_asm_fast);
LZO_EXTERN_CDECL(int)
lzo1x_decompress_asm_fast_safe
                        ( const lzo_byte *src, lzo_uint  src_len,
                                lzo_byte *dst, lzo_uint *dst_len,
                                lzo_voidp wrkmem /* NOT USED */ )
                        LZO_ASM_NAME(lzo1x_decompress_asm_fast_safe);
#  define lzo1x_decompress			lzo1x_decompress_asm_fast
#  define lzo1x_decompress_safe		lzo1x_decompress_asm_fast_safe
#endif


/*************************************************************************
//
**************************************************************************/

#include "console.h"


#endif /* already included */


/*
vi:ts=4
*/

