//===------- JITLinkTestUtils.cpp - Utilities for JITLink unit tests ------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "JITLinkTestUtils.h"
#include "llvm/ExecutionEngine/JITLink/MachO_x86_64.h"

#include "llvm/Testing/Support/Error.h"
#include "gtest/gtest.h"

using namespace llvm;
using namespace llvm::orc;
using namespace llvm::jitlink;

static const char BlockContentBytes[] = {
    0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x6d, 0x6f,
    0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x61, 0x74, 0x20, 0x74, 0x68,
    0x65, 0x20, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x66,
    0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x6f, 0x72, 0x64, 0x20,
    0x68, 0x61, 0x64, 0x20, 0x70, 0x61, 0x73, 0x73, 0x65, 0x64, 0x20, 0x61,
    0x72, 0x6f, 0x75, 0x6e, 0x64, 0x0a, 0x54, 0x68, 0x61, 0x74, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x63, 0x6f, 0x6c, 0x74, 0x20, 0x66, 0x72, 0x6f, 0x6d,
    0x20, 0x4f, 0x6c, 0x64, 0x20, 0x52, 0x65, 0x67, 0x72, 0x65, 0x74, 0x20,
    0x68, 0x61, 0x64, 0x20, 0x67, 0x6f, 0x74, 0x20, 0x61, 0x77, 0x61, 0x79,
    0x2c, 0x0a, 0x41, 0x6e, 0x64, 0x20, 0x68, 0x61, 0x64, 0x20, 0x6a, 0x6f,
    0x69, 0x6e, 0x65, 0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x69, 0x6c,
    0x64, 0x20, 0x62, 0x75, 0x73, 0x68, 0x20, 0x68, 0x6f, 0x72, 0x73, 0x65,
    0x73, 0x20, 0x2d, 0x2d, 0x20, 0x68, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20,
    0x77, 0x6f, 0x72, 0x74, 0x68, 0x20, 0x61, 0x20, 0x74, 0x68, 0x6f, 0x75,
    0x73, 0x61, 0x6e, 0x64, 0x20, 0x70, 0x6f, 0x75, 0x6e, 0x64, 0x2c, 0x0a,
    0x53, 0x6f, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63,
    0x72, 0x61, 0x63, 0x6b, 0x73, 0x20, 0x68, 0x61, 0x64, 0x20, 0x67, 0x61,
    0x74, 0x68, 0x65, 0x72, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68,
    0x65, 0x20, 0x66, 0x72, 0x61, 0x79, 0x2e, 0x0a, 0x41, 0x6c, 0x6c, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x74, 0x72, 0x69, 0x65, 0x64, 0x20, 0x61, 0x6e,
    0x64, 0x20, 0x6e, 0x6f, 0x74, 0x65, 0x64, 0x20, 0x72, 0x69, 0x64, 0x65,
    0x72, 0x73, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x6e, 0x65, 0x61,
    0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x66, 0x61, 0x72, 0x0a, 0x48, 0x61,
    0x64, 0x20, 0x6d, 0x75, 0x73, 0x74, 0x65, 0x72, 0x65, 0x64, 0x20, 0x61,
    0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x68, 0x6f, 0x6d, 0x65, 0x73, 0x74,
    0x65, 0x61, 0x64, 0x20, 0x6f, 0x76, 0x65, 0x72, 0x6e, 0x69, 0x67, 0x68,
    0x74, 0x2c, 0x0a, 0x46, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x62,
    0x75, 0x73, 0x68, 0x6d, 0x65, 0x6e, 0x20, 0x6c, 0x6f, 0x76, 0x65, 0x20,
    0x68, 0x61, 0x72, 0x64, 0x20, 0x72, 0x69, 0x64, 0x69, 0x6e, 0x67, 0x20,
    0x77, 0x68, 0x65, 0x72, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x77, 0x69,
    0x6c, 0x64, 0x20, 0x62, 0x75, 0x73, 0x68, 0x20, 0x68, 0x6f, 0x72, 0x73,
    0x65, 0x73, 0x20, 0x61, 0x72, 0x65, 0x2c, 0x0a, 0x41, 0x6e, 0x64, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x73, 0x74, 0x6f, 0x63, 0x6b, 0x2d, 0x68, 0x6f,
    0x72, 0x73, 0x65, 0x20, 0x73, 0x6e, 0x75, 0x66, 0x66, 0x73, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x62, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x20, 0x77, 0x69,
    0x74, 0x68, 0x20, 0x64, 0x65, 0x6c, 0x69, 0x67, 0x68, 0x74, 0x2e, 0x00};

ArrayRef<char> BlockContent(BlockContentBytes);

void lookupResolveEverythingToNull(
    const llvm::jitlink::JITLinkContext::LookupMap &Symbols,
    std::unique_ptr<llvm::jitlink::JITLinkAsyncLookupContinuation> LC) {
  llvm::orc::ExecutorAddr Null;
  llvm::jitlink::AsyncLookupResult Result;
  for (auto &KV : Symbols)
    Result[KV.first] = {Null, llvm::JITSymbolFlags::Exported};
  LC->run(std::move(Result));
}

void lookupErrorOut(
    const llvm::jitlink::JITLinkContext::LookupMap &Symbols,
    std::unique_ptr<llvm::jitlink::JITLinkAsyncLookupContinuation> LC) {
  LC->run(llvm::make_error<llvm::StringError>("Lookup failed",
                                              llvm::inconvertibleErrorCode()));
}

std::unique_ptr<MockJITLinkContext> makeMockContext(
    llvm::unique_function<void(llvm::Error)> HandleFailed,
    llvm::unique_function<void(MockJITLinkMemoryManager &)> SetupMemMgr,
    llvm::unique_function<void(MockJITLinkContext &)> SetupContext) {
  auto MemMgr = std::make_unique<MockJITLinkMemoryManager>();
  SetupMemMgr(*MemMgr);
  auto Ctx = std::make_unique<MockJITLinkContext>(std::move(MemMgr),
                                                  std::move(HandleFailed));
  SetupContext(*Ctx);
  return Ctx;
}

void defaultMemMgrSetup(MockJITLinkMemoryManager &) {}
void defaultCtxSetup(MockJITLinkContext &) {}

TEST(JITLinkMocks, SmokeTest) {
  // Check that the testing infrastructure defaults can "link" a graph
  // successfully.
  auto G = std::make_unique<LinkGraph>(
      "foo", std::make_shared<orc::SymbolStringPool>(),
      Triple("x86_64-apple-darwin"), SubtargetFeatures(),
      getGenericEdgeKindName);

  ArrayRef<char> Content = "hello, world!";
  auto &Sec =
      G->createSection("__data", orc::MemProt::Read | orc::MemProt::Write);
  orc::ExecutorAddr B1Addr(0x1000);
  auto &B = G->createContentBlock(Sec, Content, B1Addr, 8, 0);
  G->addDefinedSymbol(B, 4, "S", 4, Linkage::Strong, Scope::Default, false,
                      false);

  Error Err = Error::success();
  auto Ctx =
      makeMockContext(JoinErrorsInto(Err), defaultMemMgrSetup, defaultCtxSetup);

  link_MachO_x86_64(std::move(G), std::move(Ctx));

  EXPECT_THAT_ERROR(std::move(Err), Succeeded());
}
