// This file consists of testcases taken from the MSVC STL testsuite.

// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception


// Copyright 2018 Ulf Adams
// Copyright (c) Microsoft Corporation. All rights reserved.

// Boost Software License - Version 1.0 - August 17th, 2003

// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:

// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

// { dg-do run { target c++17 } }
// { dg-require-effective-target ieee_floats }
// { dg-require-effective-target size32plus }

#include <charconv>

#include <cstring>
#include <limits>
#include <optional>

#include <testsuite_hooks.h>

using namespace std;

inline constexpr float float_inf = numeric_limits<float>::infinity();
inline constexpr float float_nan = numeric_limits<float>::quiet_NaN();

struct float_to_chars_testcase {
    float value;
    chars_format fmt;
    optional<int> precision;
    const char* correct;

    constexpr
    float_to_chars_testcase(float value, chars_format fmt, const char* correct)
      : value (value), fmt (fmt), precision (nullopt), correct (correct)
    { }

    constexpr
    float_to_chars_testcase(float value, chars_format fmt, int precision,
			    const char* correct)
      : value (value), fmt (fmt), precision (precision), correct (correct)
    { }
};

inline constexpr float_to_chars_testcase float_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::scientific, "0e+00"},
    {-0.0f, chars_format::scientific, "-0e+00"},
    {float_inf, chars_format::scientific, "inf"},
    {-float_inf, chars_format::scientific, "-inf"},
    {float_nan, chars_format::scientific, "nan"},
    {-float_nan, chars_format::scientific, "-nan"},
    {2.018f, chars_format::scientific, "2.018e+00"},
    {-2.018f, chars_format::scientific, "-2.018e+00"},

    // Ditto for fixed, which doesn't emit exponents.
    {0.0f, chars_format::fixed, "0"},
    {-0.0f, chars_format::fixed, "-0"},
    {float_inf, chars_format::fixed, "inf"},
    {-float_inf, chars_format::fixed, "-inf"},
    {float_nan, chars_format::fixed, "nan"},
    {-float_nan, chars_format::fixed, "-nan"},
    {2.018f, chars_format::fixed, "2.018"},
    {-2.018f, chars_format::fixed, "-2.018"},

    // Ditto for general, which selects fixed for the scientific exponent 0.
    {0.0f, chars_format::general, "0"},
    {-0.0f, chars_format::general, "-0"},
    {float_inf, chars_format::general, "inf"},
    {-float_inf, chars_format::general, "-inf"},
    {float_nan, chars_format::general, "nan"},
    {-float_nan, chars_format::general, "-nan"},
    {2.018f, chars_format::general, "2.018"},
    {-2.018f, chars_format::general, "-2.018"},

    // Ditto for plain, which selects fixed because it's shorter for these values.
    {0.0f, chars_format{}, "0"},
    {-0.0f, chars_format{}, "-0"},
    {float_inf, chars_format{}, "inf"},
    {-float_inf, chars_format{}, "-inf"},
    {float_nan, chars_format{}, "nan"},
    {-float_nan, chars_format{}, "-nan"},
    {2.018f, chars_format{}, "2.018"},
    {-2.018f, chars_format{}, "-2.018"},

    // Ditto for hex.
    {0.0f, chars_format::hex, "0p+0"},
    {-0.0f, chars_format::hex, "-0p+0"},
    {float_inf, chars_format::hex, "inf"},
    {-float_inf, chars_format::hex, "-inf"},
    {float_nan, chars_format::hex, "nan"},
    {-float_nan, chars_format::hex, "-nan"},
    {0x1.729p+0f, chars_format::hex, "1.729p+0"},
    {-0x1.729p+0f, chars_format::hex, "-1.729p+0"},

    // Ryu f2s_test.cc SwitchToSubnormal
    {1.1754944e-38f, chars_format::scientific, "1.1754944e-38"},

    // Ryu f2s_test.cc MinAndMax
    {0x1.fffffep+127f, chars_format::scientific, "3.4028235e+38"},
    {0x1.000000p-149f, chars_format::scientific, "1e-45"},

    // Ryu f2s_test.cc BoundaryRoundEven
    {3.355445e7f, chars_format::scientific, "3.355445e+07"},
    {8.999999e9f, chars_format::scientific, "9e+09"},
    {3.4366717e10f, chars_format::scientific, "3.436672e+10"},

    // Ryu f2s_test.cc ExactValueRoundEven
    {3.0540412e5f, chars_format::scientific, "3.0540412e+05"},
    {8.0990312e3f, chars_format::scientific, "8.0990312e+03"},

    // Ryu f2s_test.cc LotsOfTrailingZeros
    {2.4414062e-4f, chars_format::scientific, "2.4414062e-04"},
    {2.4414062e-3f, chars_format::scientific, "2.4414062e-03"},
    {4.3945312e-3f, chars_format::scientific, "4.3945312e-03"},
    {6.3476562e-3f, chars_format::scientific, "6.3476562e-03"},

    // Ryu f2s_test.cc Regression
    {4.7223665e21f, chars_format::scientific, "4.7223665e+21"},
    {8388608.0f, chars_format::scientific, "8.388608e+06"},
    {1.6777216e7f, chars_format::scientific, "1.6777216e+07"},
    {3.3554436e7f, chars_format::scientific, "3.3554436e+07"},
    {6.7131496e7f, chars_format::scientific, "6.7131496e+07"},
    {1.9310392e-38f, chars_format::scientific, "1.9310392e-38"},
    {-2.47e-43f, chars_format::scientific, "-2.47e-43"},
    {1.993244e-38f, chars_format::scientific, "1.993244e-38"},
    {4103.9003f, chars_format::scientific, "4.1039004e+03"},
    {5.3399997e9f, chars_format::scientific, "5.3399997e+09"},
    {6.0898e-39f, chars_format::scientific, "6.0898e-39"},
    {0.0010310042f, chars_format::scientific, "1.0310042e-03"},
    {2.8823261e17f, chars_format::scientific, "2.882326e+17"},
    {0x1.5c87fap-84f, chars_format::scientific, "7.038531e-26"}, // TRANSITION, VSO-629490, should be 7.038531e-26f
    {9.2234038e17f, chars_format::scientific, "9.223404e+17"},
    {6.7108872e7f, chars_format::scientific, "6.710887e+07"},
    {1.0e-44f, chars_format::scientific, "1e-44"},
    {2.816025e14f, chars_format::scientific, "2.816025e+14"},
    {9.223372e18f, chars_format::scientific, "9.223372e+18"},
    {1.5846085e29f, chars_format::scientific, "1.5846086e+29"},
    {1.1811161e19f, chars_format::scientific, "1.1811161e+19"},
    {5.368709e18f, chars_format::scientific, "5.368709e+18"},
    {4.6143165e18f, chars_format::scientific, "4.6143166e+18"},
    {0.007812537f, chars_format::scientific, "7.812537e-03"},
    {1.4e-45f, chars_format::scientific, "1e-45"},
    {1.18697724e20f, chars_format::scientific, "1.18697725e+20"},
    {1.00014165e-36f, chars_format::scientific, "1.00014165e-36"},
    {200.0f, chars_format::scientific, "2e+02"},
    {3.3554432e7f, chars_format::scientific, "3.3554432e+07"},

    // Ryu f2s_test.cc LooksLikePow5
    {0x1.2a05f2p+59f, chars_format::scientific, "6.7108864e+17"},
    {0x1.2a05f2p+60f, chars_format::scientific, "1.3421773e+18"},
    {0x1.2a05f2p+61f, chars_format::scientific, "2.6843546e+18"},

    // Ryu f2s_test.cc OutputLength
    {1.0f, chars_format::scientific, "1e+00"},
    {1.2f, chars_format::scientific, "1.2e+00"},
    {1.23f, chars_format::scientific, "1.23e+00"},
    {1.234f, chars_format::scientific, "1.234e+00"},
    {1.2345f, chars_format::scientific, "1.2345e+00"},
    {1.23456f, chars_format::scientific, "1.23456e+00"},
    {1.234567f, chars_format::scientific, "1.234567e+00"},
    {1.2345678f, chars_format::scientific, "1.2345678e+00"},
    {1.23456735e-36f, chars_format::scientific, "1.23456735e-36"},

    // Test all exponents.
    {1.729e-45f, chars_format::scientific, "1e-45"},
    {1.729e-44f, chars_format::scientific, "1.7e-44"},
    {1.729e-43f, chars_format::scientific, "1.72e-43"},
    {1.729e-42f, chars_format::scientific, "1.729e-42"},
    {1.729e-41f, chars_format::scientific, "1.729e-41"},
    {1.729e-40f, chars_format::scientific, "1.729e-40"},
    {1.729e-39f, chars_format::scientific, "1.729e-39"},
    {1.729e-38f, chars_format::scientific, "1.729e-38"},
    {1.729e-37f, chars_format::scientific, "1.729e-37"},
    {1.729e-36f, chars_format::scientific, "1.729e-36"},
    {1.729e-35f, chars_format::scientific, "1.729e-35"},
    {1.729e-34f, chars_format::scientific, "1.729e-34"},
    {1.729e-33f, chars_format::scientific, "1.729e-33"},
    {1.729e-32f, chars_format::scientific, "1.729e-32"},
    {1.729e-31f, chars_format::scientific, "1.729e-31"},
    {1.729e-30f, chars_format::scientific, "1.729e-30"},
    {1.729e-29f, chars_format::scientific, "1.729e-29"},
    {1.729e-28f, chars_format::scientific, "1.729e-28"},
    {1.729e-27f, chars_format::scientific, "1.729e-27"},
    {1.729e-26f, chars_format::scientific, "1.729e-26"},
    {1.729e-25f, chars_format::scientific, "1.729e-25"},
    {1.729e-24f, chars_format::scientific, "1.729e-24"},
    {1.729e-23f, chars_format::scientific, "1.729e-23"},
    {1.729e-22f, chars_format::scientific, "1.729e-22"},
    {1.729e-21f, chars_format::scientific, "1.729e-21"},
    {1.729e-20f, chars_format::scientific, "1.729e-20"},
    {1.729e-19f, chars_format::scientific, "1.729e-19"},
    {1.729e-18f, chars_format::scientific, "1.729e-18"},
    {1.729e-17f, chars_format::scientific, "1.729e-17"},
    {1.729e-16f, chars_format::scientific, "1.729e-16"},
    {1.729e-15f, chars_format::scientific, "1.729e-15"},
    {1.729e-14f, chars_format::scientific, "1.729e-14"},
    {1.729e-13f, chars_format::scientific, "1.729e-13"},
    {1.729e-12f, chars_format::scientific, "1.729e-12"},
    {1.729e-11f, chars_format::scientific, "1.729e-11"},
    {1.729e-10f, chars_format::scientific, "1.729e-10"},
    {1.729e-9f, chars_format::scientific, "1.729e-09"},
    {1.729e-8f, chars_format::scientific, "1.729e-08"},
    {1.729e-7f, chars_format::scientific, "1.729e-07"},
    {1.729e-6f, chars_format::scientific, "1.729e-06"},
    {1.729e-5f, chars_format::scientific, "1.729e-05"},
    {1.729e-4f, chars_format::scientific, "1.729e-04"},
    {1.729e-3f, chars_format::scientific, "1.729e-03"},
    {1.729e-2f, chars_format::scientific, "1.729e-02"},
    {1.729e-1f, chars_format::scientific, "1.729e-01"},
    {1.729e0f, chars_format::scientific, "1.729e+00"},
    {1.729e1f, chars_format::scientific, "1.729e+01"},
    {1.729e2f, chars_format::scientific, "1.729e+02"},
    {1.729e3f, chars_format::scientific, "1.729e+03"},
    {1.729e4f, chars_format::scientific, "1.729e+04"},
    {1.729e5f, chars_format::scientific, "1.729e+05"},
    {1.729e6f, chars_format::scientific, "1.729e+06"},
    {1.729e7f, chars_format::scientific, "1.729e+07"},
    {1.729e8f, chars_format::scientific, "1.729e+08"},
    {1.729e9f, chars_format::scientific, "1.729e+09"},
    {1.729e10f, chars_format::scientific, "1.729e+10"},
    {1.729e11f, chars_format::scientific, "1.729e+11"},
    {1.729e12f, chars_format::scientific, "1.729e+12"},
    {1.729e13f, chars_format::scientific, "1.729e+13"},
    {1.729e14f, chars_format::scientific, "1.729e+14"},
    {1.729e15f, chars_format::scientific, "1.729e+15"},
    {1.729e16f, chars_format::scientific, "1.729e+16"},
    {1.729e17f, chars_format::scientific, "1.729e+17"},
    {1.729e18f, chars_format::scientific, "1.729e+18"},
    {1.729e19f, chars_format::scientific, "1.729e+19"},
    {1.729e20f, chars_format::scientific, "1.729e+20"},
    {1.729e21f, chars_format::scientific, "1.729e+21"},
    {1.729e22f, chars_format::scientific, "1.729e+22"},
    {1.729e23f, chars_format::scientific, "1.729e+23"},
    {1.729e24f, chars_format::scientific, "1.729e+24"},
    {1.729e25f, chars_format::scientific, "1.729e+25"},
    {1.729e26f, chars_format::scientific, "1.729e+26"},
    {1.729e27f, chars_format::scientific, "1.729e+27"},
    {1.729e28f, chars_format::scientific, "1.729e+28"},
    {1.729e29f, chars_format::scientific, "1.729e+29"},
    {1.729e30f, chars_format::scientific, "1.729e+30"},
    {1.729e31f, chars_format::scientific, "1.729e+31"},
    {1.729e32f, chars_format::scientific, "1.729e+32"},
    {1.729e33f, chars_format::scientific, "1.729e+33"},
    {1.729e34f, chars_format::scientific, "1.729e+34"},
    {1.729e35f, chars_format::scientific, "1.729e+35"},
    {1.729e36f, chars_format::scientific, "1.729e+36"},
    {1.729e37f, chars_format::scientific, "1.729e+37"},
    {1.729e38f, chars_format::scientific, "1.729e+38"},

    // Test all of the cases for fixed notation, including the non-Ryu fallback for large integers.
    {1.729e-4f, chars_format::fixed, "0.0001729"},
    {1.729e-3f, chars_format::fixed, "0.001729"},
    {1.729e-2f, chars_format::fixed, "0.01729"},
    {1.729e-1f, chars_format::fixed, "0.1729"},
    {1.729e0f, chars_format::fixed, "1.729"},
    {1.729e1f, chars_format::fixed, "17.29"},
    {1.729e2f, chars_format::fixed, "172.9"},
    {1.729e3f, chars_format::fixed, "1729"},
    {1.729e4f, chars_format::fixed, "17290"},
    {1.729e5f, chars_format::fixed, "172900"},
    {1.729e6f, chars_format::fixed, "1729000"},
    {1.729e7f, chars_format::fixed, "17290000"},
    {1.729e8f, chars_format::fixed, "172900000"},
    {1.729e9f, chars_format::fixed, "1728999936"},
    {1.729e10f, chars_format::fixed, "17290000384"},
    {1.729e11f, chars_format::fixed, "172900007936"},
    {1.729e12f, chars_format::fixed, "1728999981056"},
    {1.729e13f, chars_format::fixed, "17290000072704"},
    {1.729e14f, chars_format::fixed, "172899998629888"},
    {1.729e15f, chars_format::fixed, "1729000019853312"},
    {1.729e16f, chars_format::fixed, "17289999661662208"},
    {1.729e17f, chars_format::fixed, "172900007354040320"},
    {1.729e18f, chars_format::fixed, "1729000039180664832"},
    {1.729e19f, chars_format::fixed, "17289999567172927488"},
    {1.729e20f, chars_format::fixed, "172899997870752530432"},
    {1.729e21f, chars_format::fixed, "1729000013891897393152"},
    {1.729e22f, chars_format::fixed, "17290000138918973931520"},
    {1.729e23f, chars_format::fixed, "172899999137389925629952"},
    {1.729e24f, chars_format::fixed, "1729000063431493294227456"},
    {1.729e25f, chars_format::fixed, "17289999481393428335427584"},
    {1.729e26f, chars_format::fixed, "172900004037306320209051648"},
    {1.729e27f, chars_format::fixed, "1729000040373063202090516480"},
    {1.729e28f, chars_format::fixed, "17290000403730632020905164800"},
    {1.729e29f, chars_format::fixed, "172900004037306320209051648000"},
    {1.729e30f, chars_format::fixed, "1728999964815199476176193060864"},
    {1.729e31f, chars_format::fixed, "17290000252614904569076517961728"},
    {1.729e32f, chars_format::fixed, "172899990436890849544473432555520"},
    {1.729e33f, chars_format::fixed, "1729000059111413406117268687945728"},
    {1.729e34f, chars_format::fixed, "17290000281629124239827618154676224"},
    {1.729e35f, chars_format::fixed, "172899995388651006685994532152016896"},
    {1.729e36f, chars_format::fixed, "1728999993500591323992114118292144128"},
    {1.729e37f, chars_format::fixed, "17289999935005913239921141182921441280"},
    {1.729e38f, chars_format::fixed, "172899996814757931942752608835808002048"},

    // Also test one-digit cases, where the decimal point can't appear between digits like "17.29".
    {7e-3f, chars_format::fixed, "0.007"},
    {7e-2f, chars_format::fixed, "0.07"},
    {7e-1f, chars_format::fixed, "0.7"},
    {7e0f, chars_format::fixed, "7"},
    {7e1f, chars_format::fixed, "70"},
    {7e2f, chars_format::fixed, "700"},
    {7e3f, chars_format::fixed, "7000"},

    // Test the maximum value in fixed notation.
    {0x1.fffffep+127f, chars_format::fixed, "340282346638528859811704183484516925440"},

    // Test highly-trimmed powers of 2.
    {0x1p118f, chars_format::fixed, "332306998946228968225951765070086144"},
    {0x1p118f, chars_format::scientific, "3.32307e+35"},
    {0x1p119f, chars_format::fixed, "664613997892457936451903530140172288"},
    {0x1p119f, chars_format::scientific, "6.64614e+35"},

    // Test powers of 10 that are exactly representable.
    {1e0f, chars_format::fixed, "1"},
    {1e1f, chars_format::fixed, "10"},
    {1e2f, chars_format::fixed, "100"},
    {1e3f, chars_format::fixed, "1000"},
    {1e4f, chars_format::fixed, "10000"},
    {1e5f, chars_format::fixed, "100000"},
    {1e6f, chars_format::fixed, "1000000"},
    {1e7f, chars_format::fixed, "10000000"},
    {1e8f, chars_format::fixed, "100000000"},
    {1e9f, chars_format::fixed, "1000000000"},
    {1e10f, chars_format::fixed, "10000000000"},

    // Test powers of 10 that aren't exactly representable.
    // This exercises the "adjustment" code.
    {1e11f, chars_format::fixed, "99999997952"},
    {1e12f, chars_format::fixed, "999999995904"},
    {1e13f, chars_format::fixed, "9999999827968"},
    {1e14f, chars_format::fixed, "100000000376832"},
    {1e15f, chars_format::fixed, "999999986991104"},
    {1e16f, chars_format::fixed, "10000000272564224"},
    {1e17f, chars_format::fixed, "99999998430674944"},
    {1e18f, chars_format::fixed, "999999984306749440"},
    {1e19f, chars_format::fixed, "9999999980506447872"},
    {1e20f, chars_format::fixed, "100000002004087734272"},
    {1e21f, chars_format::fixed, "1000000020040877342720"},
    {1e22f, chars_format::fixed, "9999999778196308361216"},
    {1e23f, chars_format::fixed, "99999997781963083612160"},
    {1e24f, chars_format::fixed, "1000000013848427855085568"},
    {1e25f, chars_format::fixed, "9999999562023526247432192"},
    {1e26f, chars_format::fixed, "100000002537764290115403776"},
    {1e27f, chars_format::fixed, "999999988484154753734934528"},
    {1e28f, chars_format::fixed, "9999999442119689768320106496"},
    {1e29f, chars_format::fixed, "100000001504746621987668885504"},
    {1e30f, chars_format::fixed, "1000000015047466219876688855040"},
    {1e31f, chars_format::fixed, "9999999848243207295109594873856"},
    {1e32f, chars_format::fixed, "100000003318135351409612647563264"},
    {1e33f, chars_format::fixed, "999999994495727286427992885035008"},
    {1e34f, chars_format::fixed, "9999999790214767953607394487959552"},
    {1e35f, chars_format::fixed, "100000004091847875962975319375216640"},
    {1e36f, chars_format::fixed, "999999961690316245365415600208216064"},
    {1e37f, chars_format::fixed, "9999999933815812510711506376257961984"},
    {1e38f, chars_format::fixed, "99999996802856924650656260769173209088"},

    // These numbers have odd mantissas (unaffected by shifting)
    // that are barely within the "max shifted mantissa" limit.
    // They're exactly-representable multiples of powers of 10, and can use Ryu with zero-filling.
    {3355443e1f, chars_format::fixed, "33554430"},
    {671087e2f, chars_format::fixed, "67108700"},
    {134217e3f, chars_format::fixed, "134217000"},
    {26843e4f, chars_format::fixed, "268430000"},
    {5367e5f, chars_format::fixed, "536700000"},
    {1073e6f, chars_format::fixed, "1073000000"},
    {213e7f, chars_format::fixed, "2130000000"},
    {41e8f, chars_format::fixed, "4100000000"},
    {7e9f, chars_format::fixed, "7000000000"},
    {1e10f, chars_format::fixed, "10000000000"},

    // These numbers have odd mantissas (unaffected by shifting)
    // that are barely above the "max shifted mantissa" limit.
    // This activates the non-Ryu fallback for large integers.
    {3355445e1f, chars_format::fixed, "33554448"},
    {671089e2f, chars_format::fixed, "67108896"},
    {134219e3f, chars_format::fixed, "134219008"},
    {26845e4f, chars_format::fixed, "268449984"},
    {5369e5f, chars_format::fixed, "536899968"},
    {1075e6f, chars_format::fixed, "1075000064"},
    {215e7f, chars_format::fixed, "2150000128"},
    {43e8f, chars_format::fixed, "4300000256"},
    {9e9f, chars_format::fixed, "8999999488"},
    {3e10f, chars_format::fixed, "30000001024"},

    // Test the mantissa shifting logic.
    {5495808e5f, chars_format::fixed, "549580800000"}, // 5367 * 2^10
    {5497856e5f, chars_format::fixed, "549785567232"}, // 5369 * 2^10

    // Inspect all of those numbers in scientific notation.
    // For the within-limit numbers, this verifies that Ryu is actually being used with zero-filling above.
    // For the above-limit numbers, this tests Ryu's trimming.
    {3355443e1f, chars_format::scientific, "3.355443e+07"},
    {671087e2f, chars_format::scientific, "6.71087e+07"},
    {134217e3f, chars_format::scientific, "1.34217e+08"},
    {26843e4f, chars_format::scientific, "2.6843e+08"},
    {5367e5f, chars_format::scientific, "5.367e+08"},
    {1073e6f, chars_format::scientific, "1.073e+09"},
    {213e7f, chars_format::scientific, "2.13e+09"},
    {41e8f, chars_format::scientific, "4.1e+09"},
    {7e9f, chars_format::scientific, "7e+09"},
    {1e10f, chars_format::scientific, "1e+10"},
    {3355445e1f, chars_format::scientific, "3.355445e+07"},
    {671089e2f, chars_format::scientific, "6.71089e+07"},
    {134219e3f, chars_format::scientific, "1.34219e+08"},
    {26845e4f, chars_format::scientific, "2.6845e+08"},
    {5369e5f, chars_format::scientific, "5.369e+08"},
    {1075e6f, chars_format::scientific, "1.075e+09"},
    {215e7f, chars_format::scientific, "2.15e+09"},
    {43e8f, chars_format::scientific, "4.3e+09"},
    {9e9f, chars_format::scientific, "9e+09"},
    {3e10f, chars_format::scientific, "3e+10"},
    {5495808e5f, chars_format::scientific, "5.495808e+11"},
    {5497856e5f, chars_format::scientific, "5.497856e+11"},

    // Test the switching logic of chars_format::general.
    // C11 7.21.6.1 "The fprintf function"/8:
    // "Let P equal [...] 6 if the precision is omitted [...].
    // Then, if a conversion with style E would have an exponent of X:
    // - if P > X >= -4, the conversion is with style f [...].
    // - otherwise, the conversion is with style e [...]."
    {1e-6f, chars_format::general, "1e-06"},
    {1e-5f, chars_format::general, "1e-05"},
    {1e-4f, chars_format::general, "0.0001"},
    {1e-3f, chars_format::general, "0.001"},
    {1e-2f, chars_format::general, "0.01"},
    {1e-1f, chars_format::general, "0.1"},
    {1e0f, chars_format::general, "1"},
    {1e1f, chars_format::general, "10"},
    {1e2f, chars_format::general, "100"},
    {1e3f, chars_format::general, "1000"},
    {1e4f, chars_format::general, "10000"},
    {1e5f, chars_format::general, "100000"},
    {1e6f, chars_format::general, "1e+06"},
    {1e7f, chars_format::general, "1e+07"},
    {1.234e-6f, chars_format::general, "1.234e-06"},
    {1.234e-5f, chars_format::general, "1.234e-05"},
    {1.234e-4f, chars_format::general, "0.0001234"},
    {1.234e-3f, chars_format::general, "0.001234"},
    {1.234e-2f, chars_format::general, "0.01234"},
    {1.234e-1f, chars_format::general, "0.1234"},
    {1.234e0f, chars_format::general, "1.234"},
    {1.234e1f, chars_format::general, "12.34"},
    {1.234e2f, chars_format::general, "123.4"},
    {1.234e3f, chars_format::general, "1234"},
    {1.234e4f, chars_format::general, "12340"},
    {1.234e5f, chars_format::general, "123400"},
    {1.234e6f, chars_format::general, "1.234e+06"},
    {1.234e7f, chars_format::general, "1.234e+07"},
    {1.234e8f, chars_format::general, "1.234e+08"},
    {1.234e9f, chars_format::general, "1.234e+09"},
    {1.234e10f, chars_format::general, "1.234e+10"},

    // Test the switching logic of the plain overload.
    // N4762 19.19.2 [charconv.to.chars]/8:
    // "The conversion specifier is f or e, chosen according to the requirement
    // for a shortest representation (see above); a tie is resolved in favor of f."
    {1e-6f, chars_format{}, "1e-06"},
    {1e-5f, chars_format{}, "1e-05"},
    {1e-4f, chars_format{}, "1e-04"},
    {1e-3f, chars_format{}, "0.001"},
    {1e-2f, chars_format{}, "0.01"},
    {1e-1f, chars_format{}, "0.1"},
    {1e0f, chars_format{}, "1"},
    {1e1f, chars_format{}, "10"},
    {1e2f, chars_format{}, "100"},
    {1e3f, chars_format{}, "1000"},
    {1e4f, chars_format{}, "10000"},
    {1e5f, chars_format{}, "1e+05"},
    {1e6f, chars_format{}, "1e+06"},
    {1e7f, chars_format{}, "1e+07"},
    {1.234e-6f, chars_format{}, "1.234e-06"},
    {1.234e-5f, chars_format{}, "1.234e-05"},
    {1.234e-4f, chars_format{}, "0.0001234"},
    {1.234e-3f, chars_format{}, "0.001234"},
    {1.234e-2f, chars_format{}, "0.01234"},
    {1.234e-1f, chars_format{}, "0.1234"},
    {1.234e0f, chars_format{}, "1.234"},
    {1.234e1f, chars_format{}, "12.34"},
    {1.234e2f, chars_format{}, "123.4"},
    {1.234e3f, chars_format{}, "1234"},
    {1.234e4f, chars_format{}, "12340"},
    {1.234e5f, chars_format{}, "123400"},
    {1.234e6f, chars_format{}, "1234000"},
    {1.234e7f, chars_format{}, "12340000"},
    {1.234e8f, chars_format{}, "123400000"},
    {1.234e9f, chars_format{}, "1.234e+09"},
    {1.234e10f, chars_format{}, "1.234e+10"},

    // GH-331 "<charconv>: Test plain shortest's large integer fallback"
    // The exactly-representable integer 123456790528 is 12 digits, but scientific shortest needs 13 characters.
    // Therefore, the plain overload must select fixed notation. Because this 12-digit number exceeds the 9-digit
    // round-trip limit, we can't use Ryu - we need to activate the large integer fallback (currently, long division for
    // float).
    {123456790528.0f, chars_format::scientific, "1.2345679e+11"},
    {123456790528.0f, chars_format{}, "123456790528"},

    // Test hexfloat corner cases.
    {0x1.728p+0f, chars_format::hex, "1.728p+0"}, // instead of "2.e5p-1"
    {0x0.000002p-126f, chars_format::hex, "0.000002p-126"}, // instead of "1p-149", min subnormal
    {0x0.fffffep-126f, chars_format::hex, "0.fffffep-126"}, // max subnormal
    {0x1p-126f, chars_format::hex, "1p-126"}, // min normal
    {0x1.fffffep+127f, chars_format::hex, "1.fffffep+127"}, // max normal

    // Test hexfloat exponents.
    {0x1p-109f, chars_format::hex, "1p-109"},
    {0x1p-99f, chars_format::hex, "1p-99"},
    {0x1p-9f, chars_format::hex, "1p-9"},
    {0x1p+0f, chars_format::hex, "1p+0"},
    {0x1p+9f, chars_format::hex, "1p+9"},
    {0x1p+99f, chars_format::hex, "1p+99"},
    {0x1p+109f, chars_format::hex, "1p+109"},

    // Test hexfloat hexits.
    {0x1.0123p+0f, chars_format::hex, "1.0123p+0"},
    {0x1.4567p+0f, chars_format::hex, "1.4567p+0"},
    {0x1.89abp+0f, chars_format::hex, "1.89abp+0"},
    {0x1.cdefp+0f, chars_format::hex, "1.cdefp+0"},

    // Test hexfloat trimming.
    {0x1.00000ap+0f, chars_format::hex, "1.00000ap+0"},
    {0x1.0000ap+0f, chars_format::hex, "1.0000ap+0"},
    {0x1.000ap+0f, chars_format::hex, "1.000ap+0"},
    {0x1.00ap+0f, chars_format::hex, "1.00ap+0"},
    {0x1.0ap+0f, chars_format::hex, "1.0ap+0"},
    {0x1.ap+0f, chars_format::hex, "1.ap+0"},
    {0x1p+0f, chars_format::hex, "1p+0"},

    // https://www.exploringbinary.com/the-shortest-decimal-string-that-round-trips-may-not-be-the-nearest/
    // This is an exhaustive list of anomalous values.
    // (See double_to_chars_test_cases.hpp for more details.)
    {0x1p90f, chars_format::scientific, "1.2379401e+27"},
    {0x1p87f, chars_format::scientific, "1.5474251e+26"},
    {0x1p-96f, chars_format::scientific, "1.2621775e-29"},
};

inline constexpr float_to_chars_testcase float_fixed_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::fixed, 4, "0.0000"},
    {-0.0f, chars_format::fixed, 4, "-0.0000"},
    {float_inf, chars_format::fixed, 4, "inf"},
    {-float_inf, chars_format::fixed, 4, "-inf"},
    {float_nan, chars_format::fixed, 4, "nan"},
    {-float_nan, chars_format::fixed, 4, "-nan"},
    {1.729f, chars_format::fixed, 4, "1.7290"},
    {-1.729f, chars_format::fixed, 4, "-1.7290"},

    // Ryu Printf Zero
    {0.0f, chars_format::fixed, 4, "0.0000"},
    {0.0f, chars_format::fixed, 3, "0.000"},
    {0.0f, chars_format::fixed, 2, "0.00"},
    {0.0f, chars_format::fixed, 1, "0.0"},
    {0.0f, chars_format::fixed, 0, "0"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::fixed, 149, // min subnormal
        "0."
        "0000000000000000000000000000000000000000000014012984643248170709237295832899161312802619418765157717570682"
        "8388979108268586060148663818836212158203125"},
    {0x0.fffffep-126f, chars_format::fixed, 149, // max subnormal
        "0."
        "0000000000000000000000000000000000000117549421069244107548702944484928734882705242874589333385717453057158"
        "8870475618904265502351336181163787841796875"},
    {0x1p-126f, chars_format::fixed, 126, // min normal
        "0."
        "0000000000000000000000000000000000000117549435082228750796873653722224567781866555677208752150875170627841"
        "72594547271728515625"},
    {0x1.fffffep+127f, chars_format::fixed, 0, // max normal
        "340282346638528859811704183484516925440"},

    // Ryu Printf AllPowersOfTen
    // These values test every power of ten that's within the range of floats.
    {1e-44f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000000000098090892502737194964661070830294129189618335931356104022994779"
        "8722853757880102421040646731853485107421875"},
    {1e-43f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000000000994921909670620120355848004135840453208985978732326197947518481"
        "5617516687069610270555131137371063232421875"},
    {1e-42f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000000010005271035279193886395429224690001177341070264998322610345467546"
        "973108330377044694614596664905548095703125"},
    {1e-41f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000000000099996658414218946181117343063568415128159492172308165472584392738"
        "37549166046301252208650112152099609375"},
    {1e-40f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000000999994610111475958152591905227349949604220526961919185041279068749"
        "432712426283842432894743978977203369140625"},
    {1e-39f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000000010000002153053332574208756001456831092687456480096866911043660970225"
        "6827159061458587530069053173065185546875"},
    {1e-38f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000099999993504564039245746141539976645128551939195729831580121174560891"
        "149363239804870318039320409297943115234375"},
    {1e-37f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000000999999991097578965450144252348949782882164643167775990861842615891642"
        "849224041356137604452669620513916015625"},
    {1e-36f, chars_format::fixed, 143,
        "0."
        "0000000000000000000000000000000000010000000359391298238442905219082964481594808441361581309103473121178279"
        "3369735600208514370024204254150390625"},
    {1e-35f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000100000001800250948048663201408455778204855436374880527489094543362735389"
        "990803014370612800121307373046875"},
    {1e-34f, chars_format::fixed, 136,
        "0."
        "0000000000000000000000000000000001000000046701102029858885626602539647826036732368569844521988439212112353"
        "970951517112553119659423828125"},
    {1e-33f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000010000000237422279903610827365881541552040508374727581888171540347443055907"
        "2061441838741302490234375"},
    {1e-32f, chars_format::fixed, 130,
        "0."
        "0000000000000000000000000000000100000002374222799036108273658815415520405083747275818881715403474430559072"
        "061441838741302490234375"},
    {1e-31f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000000999999979661189834525301187760534009369837919272799809863871978166116605279"
        "96718883514404296875"},
    {1e-30f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000010000000031710768509710513471352647538147514756461109453056224083411507308483"
        "123779296875"},
    {1e-29f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000100000000317107685097105134713526475381475147564611094530562240834115073084831"
        "23779296875"},
    {1e-28f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000001000000003171076850971051347135264753814751475646110945305622408341150730848312"
        "3779296875"},
    {1e-27f, chars_format::fixed, 112,
        "0."
        "0000000000000000000000000010000000272452011558114995103349890361263429573723815479979748488403856754302978"
        "515625"},
    {1e-26f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000099999998872660226806678244921543018442779658661034858369021094404160976409912109"
        "375"},
    {1e-25f, chars_format::fixed, 104,
        "0."
        "00000000000000000000000010000000195414813782625560981110772657866336832199749551364220678806304931640625"},
    {1e-24f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000010000000195414813782625560981110772657866336832199749551364220678806304931640625"},
    {1e-23f, chars_format::fixed, 99,
        "0.000000000000000000000009999999998199587477372609628178631337169779413898140774108469486236572265625"},
    {1e-22f, chars_format::fixed, 97,
        "0.0000000000000000000001000000031374394956577733179287005745028427128318071481771767139434814453125"},
    {1e-21f, chars_format::fixed, 88,
        "0.0000000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-20f, chars_format::fixed, 87,
        "0.000000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-19f, chars_format::fixed, 86,
        "0.00000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-18f, chars_format::fixed, 83,
        "0.00000000000000000100000004581370496574313326554328540396454627625644207000732421875"},
    {1e-17f, chars_format::fixed, 79,
        "0.0000000000000000099999998377515902426605765018763349871733225882053375244140625"},
    {1e-16f, chars_format::fixed, 77,
        "0.00000000000000010000000168623835263871646450439811815158464014530181884765625"},
    {1e-15f, chars_format::fixed, 73, "0.0000000000000010000000036274937255387218471014421083964407444000244140625"},
    {1e-14f, chars_format::fixed, 68, "0.00000000000000999999982451670044181213370393379591405391693115234375"},
    {1e-13f, chars_format::fixed, 67, "0.0000000000000999999982451670044181213370393379591405391693115234375"},
    {1e-12f, chars_format::fixed, 61, "0.0000000000009999999960041972002500187954865396022796630859375"},
    {1e-11f, chars_format::fixed, 60, "0.000000000009999999960041972002500187954865396022796630859375"},
    {1e-10f, chars_format::fixed, 57, "0.000000000100000001335143196001808973960578441619873046875"},
    {1e-09f, chars_format::fixed, 53, "0.00000000099999997171806853657471947371959686279296875"},
    {1e-08f, chars_format::fixed, 50, "0.00000000999999993922529029077850282192230224609375"},
    {1e-07f, chars_format::fixed, 47, "0.00000010000000116860974230803549289703369140625"},
    {1e-06f, chars_format::fixed, 43, "0.0000009999999974752427078783512115478515625"},
    {1e-05f, chars_format::fixed, 38, "0.00000999999974737875163555145263671875"},
    {1e-04f, chars_format::fixed, 37, "0.0000999999974737875163555145263671875"},
    {1e-03f, chars_format::fixed, 33, "0.001000000047497451305389404296875"},
    {1e-02f, chars_format::fixed, 29, "0.00999999977648258209228515625"},
    {1e-01f, chars_format::fixed, 27, "0.100000001490116119384765625"},
    {1e+00f, chars_format::fixed, 0, "1"},
    {1e+01f, chars_format::fixed, 0, "10"},
    {1e+02f, chars_format::fixed, 0, "100"},
    {1e+03f, chars_format::fixed, 0, "1000"},
    {1e+04f, chars_format::fixed, 0, "10000"},
    {1e+05f, chars_format::fixed, 0, "100000"},
    {1e+06f, chars_format::fixed, 0, "1000000"},
    {1e+07f, chars_format::fixed, 0, "10000000"},
    {1e+08f, chars_format::fixed, 0, "100000000"},
    {1e+09f, chars_format::fixed, 0, "1000000000"},
    {1e+10f, chars_format::fixed, 0, "10000000000"},
    {1e+11f, chars_format::fixed, 0, "99999997952"},
    {1e+12f, chars_format::fixed, 0, "999999995904"},
    {1e+13f, chars_format::fixed, 0, "9999999827968"},
    {1e+14f, chars_format::fixed, 0, "100000000376832"},
    {1e+15f, chars_format::fixed, 0, "999999986991104"},
    {1e+16f, chars_format::fixed, 0, "10000000272564224"},
    {1e+17f, chars_format::fixed, 0, "99999998430674944"},
    {1e+18f, chars_format::fixed, 0, "999999984306749440"},
    {1e+19f, chars_format::fixed, 0, "9999999980506447872"},
    {1e+20f, chars_format::fixed, 0, "100000002004087734272"},
    {1e+21f, chars_format::fixed, 0, "1000000020040877342720"},
    {1e+22f, chars_format::fixed, 0, "9999999778196308361216"},
    {1e+23f, chars_format::fixed, 0, "99999997781963083612160"},
    {1e+24f, chars_format::fixed, 0, "1000000013848427855085568"},
    {1e+25f, chars_format::fixed, 0, "9999999562023526247432192"},
    {1e+26f, chars_format::fixed, 0, "100000002537764290115403776"},
    {1e+27f, chars_format::fixed, 0, "999999988484154753734934528"},
    {1e+28f, chars_format::fixed, 0, "9999999442119689768320106496"},
    {1e+29f, chars_format::fixed, 0, "100000001504746621987668885504"},
    {1e+30f, chars_format::fixed, 0, "1000000015047466219876688855040"},
    {1e+31f, chars_format::fixed, 0, "9999999848243207295109594873856"},
    {1e+32f, chars_format::fixed, 0, "100000003318135351409612647563264"},
    {1e+33f, chars_format::fixed, 0, "999999994495727286427992885035008"},
    {1e+34f, chars_format::fixed, 0, "9999999790214767953607394487959552"},
    {1e+35f, chars_format::fixed, 0, "100000004091847875962975319375216640"},
    {1e+36f, chars_format::fixed, 0, "999999961690316245365415600208216064"},
    {1e+37f, chars_format::fixed, 0, "9999999933815812510711506376257961984"},
    {1e+38f, chars_format::fixed, 0, "99999996802856924650656260769173209088"},

    // Ryu Printf RoundToEven
    {0.125f, chars_format::fixed, 3, "0.125"},
    {0.125f, chars_format::fixed, 2, "0.12"},
    {0.375f, chars_format::fixed, 3, "0.375"},
    {0.375f, chars_format::fixed, 2, "0.38"},

    // Ryu Printf RoundToEvenInteger
    {2.5f, chars_format::fixed, 1, "2.5"},
    {2.5f, chars_format::fixed, 0, "2"},
    {3.5f, chars_format::fixed, 1, "3.5"},
    {3.5f, chars_format::fixed, 0, "4"},

    // Ryu Printf NonRoundToEvenScenarios
    {0.748046875f, chars_format::fixed, 3, "0.748"},
    {0.748046875f, chars_format::fixed, 2, "0.75"},
    {0.748046875f, chars_format::fixed, 1, "0.7"}, // 0.75 would round to "0.8", but this is smaller

    {0.2509765625f, chars_format::fixed, 3, "0.251"},
    {0.2509765625f, chars_format::fixed, 2, "0.25"},
    {0.2509765625f, chars_format::fixed, 1, "0.3"}, // 0.25 would round to "0.2", but this is larger

    {0x1.000002p-2f, chars_format::fixed, 25, "0.2500000298023223876953125"},
    {0x1.000002p-2f, chars_format::fixed, 3, "0.250"},
    {0x1.000002p-2f, chars_format::fixed, 2, "0.25"},
    {0x1.000002p-2f, chars_format::fixed, 1, "0.3"}, // 0.25 would round to "0.2", but this is larger (again)

    // More rounding tests.
    {9.5f, chars_format::fixed, 1, "9.5"},
    {9.5f, chars_format::fixed, 0, "10"},
    {10.5f, chars_format::fixed, 1, "10.5"},
    {10.5f, chars_format::fixed, 0, "10"},

    {1.241f, chars_format::fixed, 3, "1.241"},
    {1.241f, chars_format::fixed, 1, "1.2"},
    {1.251f, chars_format::fixed, 3, "1.251"},
    {1.251f, chars_format::fixed, 1, "1.3"},
    {1.261f, chars_format::fixed, 3, "1.261"},
    {1.261f, chars_format::fixed, 1, "1.3"},
    {1.341f, chars_format::fixed, 3, "1.341"},
    {1.341f, chars_format::fixed, 1, "1.3"},
    {1.351f, chars_format::fixed, 3, "1.351"},
    {1.351f, chars_format::fixed, 1, "1.4"},
    {1.361f, chars_format::fixed, 3, "1.361"},
    {1.361f, chars_format::fixed, 1, "1.4"},

    {2.41f, chars_format::fixed, 2, "2.41"},
    {2.41f, chars_format::fixed, 0, "2"},
    {2.51f, chars_format::fixed, 2, "2.51"},
    {2.51f, chars_format::fixed, 0, "3"},
    {2.61f, chars_format::fixed, 2, "2.61"},
    {2.61f, chars_format::fixed, 0, "3"},
    {3.41f, chars_format::fixed, 2, "3.41"},
    {3.41f, chars_format::fixed, 0, "3"},
    {3.51f, chars_format::fixed, 2, "3.51"},
    {3.51f, chars_format::fixed, 0, "4"},
    {3.61f, chars_format::fixed, 2, "3.61"},
    {3.61f, chars_format::fixed, 0, "4"},

    // Ryu Printf VaryingPrecision
    {1.142857f, chars_format::fixed, 28, "1.1428569555282592773437500000"},
    {1.142857f, chars_format::fixed, 27, "1.142856955528259277343750000"},
    {1.142857f, chars_format::fixed, 26, "1.14285695552825927734375000"},
    {1.142857f, chars_format::fixed, 25, "1.1428569555282592773437500"},
    {1.142857f, chars_format::fixed, 24, "1.142856955528259277343750"},
    {1.142857f, chars_format::fixed, 23, "1.14285695552825927734375"},
    {1.142857f, chars_format::fixed, 22, "1.1428569555282592773438"},
    {1.142857f, chars_format::fixed, 21, "1.142856955528259277344"},
    {1.142857f, chars_format::fixed, 20, "1.14285695552825927734"},
    {1.142857f, chars_format::fixed, 19, "1.1428569555282592773"},
    {1.142857f, chars_format::fixed, 18, "1.142856955528259277"},
    {1.142857f, chars_format::fixed, 17, "1.14285695552825928"},
    {1.142857f, chars_format::fixed, 16, "1.1428569555282593"},
    {1.142857f, chars_format::fixed, 15, "1.142856955528259"},
    {1.142857f, chars_format::fixed, 14, "1.14285695552826"},
    {1.142857f, chars_format::fixed, 13, "1.1428569555283"},
    {1.142857f, chars_format::fixed, 12, "1.142856955528"},
    {1.142857f, chars_format::fixed, 11, "1.14285695553"},
    {1.142857f, chars_format::fixed, 10, "1.1428569555"},
    {1.142857f, chars_format::fixed, 9, "1.142856956"},
    {1.142857f, chars_format::fixed, 8, "1.14285696"},
    {1.142857f, chars_format::fixed, 7, "1.1428570"},
    {1.142857f, chars_format::fixed, 6, "1.142857"},
    {1.142857f, chars_format::fixed, 5, "1.14286"},
    {1.142857f, chars_format::fixed, 4, "1.1429"},
    {1.142857f, chars_format::fixed, 3, "1.143"},
    {1.142857f, chars_format::fixed, 2, "1.14"},
    {1.142857f, chars_format::fixed, 1, "1.1"},
    {1.142857f, chars_format::fixed, 0, "1"},

    // Negative precision requests 6 digits of precision.
    {1.142857f, chars_format::fixed, -1, "1.142857"},
    {1.142857f, chars_format::fixed, -2, "1.142857"},
    {1.142857f, chars_format::fixed, -3, "1.142857"},

    // Ryu Printf Carrying
    {0.0009f, chars_format::fixed, 4, "0.0009"},
    {0.0009f, chars_format::fixed, 3, "0.001"},
    {0.0029f, chars_format::fixed, 4, "0.0029"},
    {0.0029f, chars_format::fixed, 3, "0.003"},
    {0.0099f, chars_format::fixed, 4, "0.0099"},
    {0.0099f, chars_format::fixed, 3, "0.010"},
    {0.0299f, chars_format::fixed, 4, "0.0299"},
    {0.0299f, chars_format::fixed, 3, "0.030"},
    {0.0999f, chars_format::fixed, 4, "0.0999"},
    {0.0999f, chars_format::fixed, 3, "0.100"},
    {0.2999f, chars_format::fixed, 4, "0.2999"},
    {0.2999f, chars_format::fixed, 3, "0.300"},
    {0.9999f, chars_format::fixed, 4, "0.9999"},
    {0.9999f, chars_format::fixed, 3, "1.000"},
    {2.9999f, chars_format::fixed, 4, "2.9999"},
    {2.9999f, chars_format::fixed, 3, "3.000"},
    {9.9999f, chars_format::fixed, 4, "9.9999"},
    {9.9999f, chars_format::fixed, 3, "10.000"},
    {29.9999f, chars_format::fixed, 4, "29.9999"},
    {29.9999f, chars_format::fixed, 3, "30.000"},
    {99.9999f, chars_format::fixed, 4, "99.9999"},
    {99.9999f, chars_format::fixed, 3, "100.000"},
    {299.9999f, chars_format::fixed, 4, "299.9999"},
    {299.9999f, chars_format::fixed, 3, "300.000"},

    {0.09f, chars_format::fixed, 2, "0.09"},
    {0.09f, chars_format::fixed, 1, "0.1"},
    {0.29f, chars_format::fixed, 2, "0.29"},
    {0.29f, chars_format::fixed, 1, "0.3"},
    {0.99f, chars_format::fixed, 2, "0.99"},
    {0.99f, chars_format::fixed, 1, "1.0"},
    {2.99f, chars_format::fixed, 2, "2.99"},
    {2.99f, chars_format::fixed, 1, "3.0"},
    {9.99f, chars_format::fixed, 2, "9.99"},
    {9.99f, chars_format::fixed, 1, "10.0"},
    {29.99f, chars_format::fixed, 2, "29.99"},
    {29.99f, chars_format::fixed, 1, "30.0"},
    {99.99f, chars_format::fixed, 2, "99.99"},
    {99.99f, chars_format::fixed, 1, "100.0"},
    {299.99f, chars_format::fixed, 2, "299.99"},
    {299.99f, chars_format::fixed, 1, "300.0"},

    {0.9f, chars_format::fixed, 1, "0.9"},
    {0.9f, chars_format::fixed, 0, "1"},
    {2.9f, chars_format::fixed, 1, "2.9"},
    {2.9f, chars_format::fixed, 0, "3"},
    {9.9f, chars_format::fixed, 1, "9.9"},
    {9.9f, chars_format::fixed, 0, "10"},
    {29.9f, chars_format::fixed, 1, "29.9"},
    {29.9f, chars_format::fixed, 0, "30"},
    {99.9f, chars_format::fixed, 1, "99.9"},
    {99.9f, chars_format::fixed, 0, "100"},
    {299.9f, chars_format::fixed, 1, "299.9"},
    {299.9f, chars_format::fixed, 0, "300"},

    // Ryu Printf RoundingResultZero
    {0.004f, chars_format::fixed, 3, "0.004"},
    {0.004f, chars_format::fixed, 2, "0.00"},
    {0.4f, chars_format::fixed, 1, "0.4"},
    {0.4f, chars_format::fixed, 0, "0"},
    {0.5f, chars_format::fixed, 1, "0.5"},
    {0.5f, chars_format::fixed, 0, "0"},

    // Ryu Printf AllBinaryExponents
    // These values test every binary exponent.
    // The mantissas were randomly generated.
    {0x0.bafab0p-126f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000000085856660078164374052571520381239855817217629811131320365461649230225"
        "8101698697601023013703525066375732421875"},
    {0x1.2c4906p-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000137884223604447257991705553959882023830165636017936204355992973751997"
        "2013648494879589634365402162075042724609375"},
    {0x1.da6c8cp-125f, chars_format::fixed, 147,
        "0."
        "0000000000000000000000000000000000000435689644606144922962341034916717454461784748180572306511365470365953"
        "68653788540314053534530103206634521484375"},
    {0x1.094fd8p-124f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000000487300980449569406486898593097235018686258578182781609022211792895361"
        "293087574949822737835347652435302734375"},
    {0x1.1fba2ap-123f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000001056942819182250793988103906000418825212997860175004035399177620979400"
        "2661102041429330711252987384796142578125"},
    {0x1.05c066p-122f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000001923046724143804860574903666495417184272104431219282642086921032167044"
        "776084452450959361158311367034912109375"},
    {0x1.aa97aep-121f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000006268213405227838203431757007863521825324812101203856469220875149766722"
        "40063493291017948649823665618896484375"},
    {0x1.dd39a8p-120f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000014024388746462477508516570165943074639858849697711014853285027082288594"
        "08023936339304782450199127197265625"},
    {0x1.d47ee4p-119f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000027535700468003209195416842287342046227746219166975594715835834154840644"
        "49429979504202492535114288330078125"},
    {0x1.3d3c36p-118f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000037290818427663765497432907740725131368928567215367442737882934305701509"
        "99159295679419301450252532958984375"},
    {0x1.1661f4p-117f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000065447441644065192772010189083715139205202243608571574700187600294454259"
        "852727773250080645084381103515625"},
    {0x1.b68df4p-116f, chars_format::fixed, 138,
        "0."
        "0000000000000000000000000000000000206207336977375818320192367186588031181149116107584831794039036470533865"
        "49714559805579483509063720703125"},
    {0x1.d99cbcp-115f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000445382813514879099167344241437088992337862296399353856123689687079858501"
        "3062620419077575206756591796875"},
    {0x1.fd046ep-114f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000957355143513621934625335631305232671680527244248210039119495722709229923"
        "4527672524563968181610107421875"},
    {0x1.89834cp-113f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000001480230929779647770398330978023641189899501899250006144284413575618053471"
        "31667076610028743743896484375"},
    {0x1.44f9f6p-112f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000002444850777614032698558632186222413041911856813461843446368257012912827974"
        "56005937419831752777099609375"},
    {0x1.610156p-111f, chars_format::fixed, 134,
        "0."
        "0000000000000000000000000000000005311432194104638958491823018983379637847618995394862767718806989547530861"
        "2731168977916240692138671875"},
    {0x1.1c4ce0p-110f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000008555350741040305433153411782350510989661057055423470539198826934068620175"
        "82170665264129638671875"},
    {0x1.c8846ap-109f, chars_format::fixed, 132,
        "0."
        "0000000000000000000000000000000027475632104005746766942313987412370292282555616009485081234833825369889837"
        "02030964195728302001953125"},
    {0x1.49aaa6p-108f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000039682172991165697309827799919973022713504068615668209476284727932338114442"
        "2455690801143646240234375"},
    {0x1.f5603cp-107f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000120701861138584576157150361758540470962569680234040676814963322094342856871"
        "66266143321990966796875"},
    {0x1.b7bbf8p-106f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000211724341322508937840915176712265363597160619557838059749677039889093066449"
        "277102947235107421875"},
    {0x1.6d305cp-105f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000351664122601460292174574136500884378151845989294218826175242309517443572985"
        "939681529998779296875"},
    {0x1.dd9944p-104f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000000919821625881433083724268012282371631728046815243377909779298740033937065163"
        "62726688385009765625"},
    {0x1.0f4254p-103f, chars_format::fixed, 125,
        "0."
        "0000000000000000000000000000001044852024561729954450605502201132293228119140390439904125807757395705266389"
        "9958133697509765625"},
    {0x1.049450p-102f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000002007430259113927348388472759172849833669167623529955984951200775867619086056"
        "9477081298828125"},
    {0x1.28d030p-101f, chars_format::fixed, 121,
        "0."
        "0000000000000000000000000000004573131937693259427041496124538667251427229422876784281637441154089174233376"
        "979827880859375"},
    {0x1.28a2bep-100f, chars_format::fixed, 123,
        "0."
        "0000000000000000000000000000009140793594875532256935908936727717060368161903718573976784789181238011224195"
        "36113739013671875"},
    {0x1.e674d2p-99f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000029980185962354845050897219390986524441297232842758378051906120731473492924124"
        "0024566650390625"},
    {0x1.227314p-98f, chars_format::fixed, 120,
        "0."
        "0000000000000000000000000000035800667869547456776694564979794758804336827210561614021067100566142471507191"
        "65802001953125"},
    {0x1.735b6cp-97f, chars_format::fixed, 119,
        "0."
        "0000000000000000000000000000091546597262378319603799081529207906703837108826262575211885064163652714341878"
        "8909912109375"},
    {0x1.ef60b4p-96f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000244240085996903849216356078751341022854748745722804070812372856380534358322620"
        "391845703125"},
    {0x1.f58d34p-95f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000494568036548015750964222103779193000963396430755616983709899159293854609131813"
        "04931640625"},
    {0x1.a9fa8ap-94f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000840094794528154864080325730507983995095850536733231295194457288744160905480384"
        "82666015625"},
    {0x1.2ebd9ap-93f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000001194101241497498690354073604120044306874891791320908440710013564967084676027297"
        "9736328125"},
    {0x1.1c25bep-92f, chars_format::fixed, 115,
        "0."
        "0000000000000000000000000002241527991772840369420073304083191365256388471842441401093992681126110255718231"
        "201171875"},
    {0x1.80d526p-91f, chars_format::fixed, 114,
        "0."
        "0000000000000000000000000006071588038765549904506806942923684386898144178454361785668425000039860606193542"
        "48046875"},
    {0x1.16cdd0p-90f, chars_format::fixed, 110,
        "0."
        "0000000000000000000000000008797501615285119946834874987404311536605839433322628906353202182799577713012695"
        "3125"},
    {0x1.be00c0p-89f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000000028146741987560362391732368583172300827762585262448169487470295280218124389648437"
        "5"},
    {0x1.dbe376p-88f, chars_format::fixed, 111,
        "0."
        "0000000000000000000000000060065575697458565955014820557088473772922162727261330417150020366534590721130371"
        "09375"},
    {0x1.75b358p-87f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000094335284238393382638914933670753282739891362740358715655020205304026603698730468"
        "75"},
    {0x1.5e56fap-86f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000176876373794073549186243776242822499413496518949617808402763330377638339996337890"
        "625"},
    {0x1.1542e6p-85f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000279962390364982552136537973532622361314990341862873890477203531190752983093261718"
        "75"},
    {0x1.37b7a6p-84f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000000629508229027247503098930994487426088983880040350626927647681441158056259155273437"
        "5"},
    {0x1.31f62ap-83f, chars_format::fixed, 106,
        "0."
        "000000000000000000000000123576897369666466985856753932114511864998745993560191891447175294160842895507812"
        "5"},
    {0x1.ac3560p-82f, chars_format::fixed, 101,
        "0.00000000000000000000000034590406845628797200186450581230516131137076030199750675819814205169677734375"},
    {0x1.a7db5cp-81f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000006847777099176331341674240101847394713956151957034990118700079619884490966796875"},
    {0x1.40189cp-80f, chars_format::fixed, 102,
        "0.000000000000000000000001034286379672715366987641733210033664035198963659922810620628297328948974609375"},
    {0x1.aad1eep-79f, chars_format::fixed, 102,
        "0.000000000000000000000002758259846499093682487211692864773559218105614121441249153576791286468505859375"},
    {0x1.49824cp-78f, chars_format::fixed, 100,
        "0.0000000000000000000000042588036474940459085811637121780459484809977510622047702781856060028076171875"},
    {0x1.955292p-77f, chars_format::fixed, 100,
        "0.0000000000000000000000104773420985315373838626628182169411331037256474019159213639795780181884765625"},
    {0x1.d8ca0cp-76f, chars_format::fixed, 98,
        "0.00000000000000000000002444263111177596802332967266437459101513507420122550684027373790740966796875"},
    {0x1.28b5aap-75f, chars_format::fixed, 98,
        "0.00000000000000000000003067905619497844072028707730390270809472941238027487997896969318389892578125"},
    {0x1.e5fda8p-74f, chars_format::fixed, 95,
        "0.00000000000000000000010050055115902033206854316793794380802129495577901252545416355133056640625"},
    {0x1.fd929cp-73f, chars_format::fixed, 95,
        "0.00000000000000000000021075432611470358337541921610390309449467594049565377645194530487060546875"},
    {0x1.c0b84cp-72f, chars_format::fixed, 94,
        "0.0000000000000000000003711724097438896357340602997067040280665395357573288492858409881591796875"},
    {0x1.5cfeaep-71f, chars_format::fixed, 94,
        "0.0000000000000000000005773635352424624465965559338086719731730767080080113373696804046630859375"},
    {0x1.bcce4ap-70f, chars_format::fixed, 93,
        "0.000000000000000000001471738991536079335112024613790339400143380998997599817812442779541015625"},
    {0x1.edf106p-69f, chars_format::fixed, 92,
        "0.00000000000000000000326863064574260634910627773444362353938430487687583081424236297607421875"},
    {0x1.30b422p-68f, chars_format::fixed, 91,
        "0.0000000000000000000040327191475944672156035895296995186232180685692583210766315460205078125"},
    {0x1.7aa8d8p-67f, chars_format::fixed, 88,
        "0.0000000000000000000100230347240102665385544432156972316505516573670320212841033935546875"},
    {0x1.4ad4e0p-66f, chars_format::fixed, 85,
        "0.0000000000000000000175140760553442509348562143578487138029231573455035686492919921875"},
    {0x1.dde636p-65f, chars_format::fixed, 88,
        "0.0000000000000000000505995524921864861016965251964971894693690046551637351512908935546875"},
    {0x1.5df870p-64f, chars_format::fixed, 84,
        "0.000000000000000000074109127331368847396687003781234892585416673682630062103271484375"},
    {0x1.c346fap-63f, chars_format::fixed, 86,
        "0.00000000000000000019112335047873604296656620434025075638828639057464897632598876953125"},
    {0x1.58d2eap-62f, chars_format::fixed, 85,
        "0.0000000000000000002920771899491385068938311721231659845443573431111872196197509765625"},
    {0x1.0d4824p-61f, chars_format::fixed, 83,
        "0.00000000000000000045618111223383324851561766710705825289551285095512866973876953125"},
    {0x1.04585cp-60f, chars_format::fixed, 82,
        "0.0000000000000000008820836917354691955064048547452415505176759324967861175537109375"},
    {0x1.55cf7ap-59f, chars_format::fixed, 82,
        "0.0000000000000000023161977389916240139687737820128887733517331071197986602783203125"},
    {0x1.1fd8ecp-58f, chars_format::fixed, 80,
        "0.00000000000000000390105904223582084021197668999292318403604440391063690185546875"},
    {0x1.0bc866p-57f, chars_format::fixed, 80,
        "0.00000000000000000725826751123333980988787395016714754092390649020671844482421875"},
    {0x1.4dfa86p-56f, chars_format::fixed, 79,
        "0.0000000000000000181050165732891247031242920595417444928898476064205169677734375"},
    {0x1.335daep-55f, chars_format::fixed, 78,
        "0.000000000000000033324681586205479543426333233213654239079914987087249755859375"},
    {0x1.5bc756p-54f, chars_format::fixed, 77,
        "0.00000000000000007541247487712833172911197632259927559061907231807708740234375"},
    {0x1.9eb052p-53f, chars_format::fixed, 76,
        "0.0000000000000001798425779915148827771409489884035792783834040164947509765625"},
    {0x1.13b6d2p-52f, chars_format::fixed, 75,
        "0.000000000000000239143897259270284301468922905087310937233269214630126953125"},
    {0x1.260438p-51f, chars_format::fixed, 72,
        "0.000000000000000510037289299151118393549353413618518970906734466552734375"},
    {0x1.9e6b44p-50f, chars_format::fixed, 72,
        "0.000000000000001437804758404521467129999479084290214814245700836181640625"},
    {0x1.89c0bcp-49f, chars_format::fixed, 71,
        "0.00000000000000273220937993773164975674916377101908437907695770263671875"},
    {0x1.e30610p-48f, chars_format::fixed, 68,
        "0.00000000000000670330015995791728133923470522859133780002593994140625"},
    {0x1.48b6e8p-47f, chars_format::fixed, 68,
        "0.00000000000000912365953728740131101204724473063834011554718017578125"},
    {0x1.41382ep-46f, chars_format::fixed, 69,
        "0.000000000000017831261573081173821275768887062440626323223114013671875"},
    {0x1.383b8ep-45f, chars_format::fixed, 68,
        "0.00000000000003466478609693256218715617933412431739270687103271484375"},
    {0x1.1e6564p-44f, chars_format::fixed, 66, "0.000000000000063592699357274684590635160930105485022068023681640625"},
    {0x1.c35e62p-43f, chars_format::fixed, 66, "0.000000000000200447961722950707130763703389675356447696685791015625"},
    {0x1.2a2f4ep-42f, chars_format::fixed, 65, "0.00000000000026484129017449731247069166784058324992656707763671875"},
    {0x1.69fae2p-41f, chars_format::fixed, 64, "0.0000000000006430056682417417679431537180789746344089508056640625"},
    {0x1.4ccefep-40f, chars_format::fixed, 63, "0.000000000001182373535017766652543969030375592410564422607421875"},
    {0x1.aa9bf6p-39f, chars_format::fixed, 62, "0.00000000000303124083993189241681420753593556582927703857421875"},
    {0x1.3b9744p-38f, chars_format::fixed, 60, "0.000000000004484816164274096905728583806194365024566650390625"},
    {0x1.b2fc6ap-37f, chars_format::fixed, 60, "0.000000000012363045483188006556929394719190895557403564453125"},
    {0x1.7bc418p-36f, chars_format::fixed, 57, "0.000000000021587197307493255493682227097451686859130859375"},
    {0x1.f4a74cp-35f, chars_format::fixed, 57, "0.000000000056917713597837149563929415307939052581787109375"},
    {0x1.89f248p-34f, chars_format::fixed, 55, "0.0000000000895730434269381703416001982986927032470703125"},
    {0x1.60ac54p-33f, chars_format::fixed, 55, "0.0000000001603771837555001411601551808416843414306640625"},
    {0x1.2f6d0ep-32f, chars_format::fixed, 55, "0.0000000002759643347172158200919511727988719940185546875"},
    {0x1.748684p-31f, chars_format::fixed, 53, "0.00000000067761984912095840627443976700305938720703125"},
    {0x1.b4fa00p-30f, chars_format::fixed, 45, "0.000000001589711473570787347853183746337890625"},
    {0x1.c204d8p-29f, chars_format::fixed, 50, "0.00000000327431859403759517590515315532684326171875"},
    {0x1.50029ep-28f, chars_format::fixed, 51, "0.000000004889592286616561978007666766643524169921875"},
    {0x1.56cf38p-27f, chars_format::fixed, 48, "0.000000009977068060607052757404744625091552734375"},
    {0x1.0b5a5cp-26f, chars_format::fixed, 48, "0.000000015561990807100301026366651058197021484375"},
    {0x1.fc8250p-25f, chars_format::fixed, 45, "0.000000059198242752245278097689151763916015625"},
    {0x1.c66674p-24f, chars_format::fixed, 46, "0.0000001057982927932243910618126392364501953125"},
    {0x1.4da57ep-23f, chars_format::fixed, 46, "0.0000001553662372089092968963086605072021484375"},
    {0x1.4fcdacp-22f, chars_format::fixed, 44, "0.00000031274129241865011863410472869873046875"},
    {0x1.5eaff4p-21f, chars_format::fixed, 43, "0.0000006532060297104180790483951568603515625"},
    {0x1.d2f696p-20f, chars_format::fixed, 43, "0.0000017395735767422593198716640472412109375"},
    {0x1.e4400cp-19f, chars_format::fixed, 41, "0.00000360794501830241642892360687255859375"},
    {0x1.03e624p-18f, chars_format::fixed, 40, "0.0000038727966966689564287662506103515625"},
    {0x1.bdb65ep-17f, chars_format::fixed, 40, "0.0000132832637973478995263576507568359375"},
    {0x1.57fb84p-16f, chars_format::fixed, 38, "0.00002050295370281673967838287353515625"},
    {0x1.fd2d62p-15f, chars_format::fixed, 38, "0.00006069866140023805201053619384765625"},
    {0x1.ca0c58p-14f, chars_format::fixed, 35, "0.00010920720524154603481292724609375"},
    {0x1.988f70p-13f, chars_format::fixed, 33, "0.000194816733710467815399169921875"},
    {0x1.032dd6p-12f, chars_format::fixed, 35, "0.00024717240012250840663909912109375"},
    {0x1.571b08p-11f, chars_format::fixed, 32, "0.00065442197956144809722900390625"},
    {0x1.53bedap-10f, chars_format::fixed, 33, "0.001296026282943785190582275390625"},
    {0x1.ab2f36p-9f, chars_format::fixed, 32, "0.00325915846042335033416748046875"},
    {0x1.7293dap-8f, chars_format::fixed, 31, "0.0056545645929872989654541015625"},
    {0x1.825eb6p-7f, chars_format::fixed, 30, "0.011791075579822063446044921875"},
    {0x1.f45aa0p-6f, chars_format::fixed, 25, "0.0305391848087310791015625"},
    {0x1.854d96p-5f, chars_format::fixed, 28, "0.0475223474204540252685546875"},
    {0x1.5650cep-4f, chars_format::fixed, 27, "0.083573155105113983154296875"},
    {0x1.03acdap-3f, chars_format::fixed, 26, "0.12679453194141387939453125"},
    {0x1.6b9416p-2f, chars_format::fixed, 25, "0.3550570905208587646484375"},
    {0x1.a8544ap-1f, chars_format::fixed, 24, "0.828768074512481689453125"},
    {0x1.0693f6p+0f, chars_format::fixed, 23, "1.02569520473480224609375"},
    {0x1.b9476ep+1f, chars_format::fixed, 22, "3.4474923610687255859375"},
    {0x1.3cb752p+2f, chars_format::fixed, 21, "4.948688983917236328125"},
    {0x1.bb8a64p+3f, chars_format::fixed, 19, "13.8606433868408203125"},
    {0x1.1de906p+4f, chars_format::fixed, 19, "17.8693904876708984375"},
    {0x1.d8e834p+5f, chars_format::fixed, 17, "59.11338043212890625"},
    {0x1.27cd38p+6f, chars_format::fixed, 15, "73.950408935546875"},
    {0x1.3cdcd6p+7f, chars_format::fixed, 16, "158.4313201904296875"},
    {0x1.392656p+8f, chars_format::fixed, 15, "313.149749755859375"},
    {0x1.c96aa8p+9f, chars_format::fixed, 12, "914.833251953125"},
    {0x1.28b6b2p+10f, chars_format::fixed, 13, "1186.8546142578125"},
    {0x1.786090p+11f, chars_format::fixed, 9, "3011.017578125"},
    {0x1.79c6f6p+12f, chars_format::fixed, 11, "6044.43505859375"},
    {0x1.ef1840p+13f, chars_format::fixed, 5, "15843.03125"},
    {0x1.539fd0p+14f, chars_format::fixed, 6, "21735.953125"},
    {0x1.b31804p+15f, chars_format::fixed, 7, "55692.0078125"},
    {0x1.ad4a9cp+16f, chars_format::fixed, 6, "109898.609375"},
    {0x1.4c43a6p+17f, chars_format::fixed, 6, "170119.296875"},
    {0x1.5598c6p+18f, chars_format::fixed, 5, "349795.09375"},
    {0x1.73695ep+19f, chars_format::fixed, 4, "760650.9375"},
    {0x1.234f2ap+20f, chars_format::fixed, 3, "1193202.625"},
    {0x1.0a4cc8p+21f, chars_format::fixed, 0, "2181529"},
    {0x1.90abd2p+22f, chars_format::fixed, 1, "6564596.5"},
    {0x1.62dde8p+23f, chars_format::fixed, 0, "11628276"},
    {0x1.9e3a8cp+24f, chars_format::fixed, 0, "27146892"},
    {0x1.53a3eap+25f, chars_format::fixed, 0, "44517332"},
    {0x1.41a1cep+26f, chars_format::fixed, 0, "84313912"},
    {0x1.8fdda4p+27f, chars_format::fixed, 0, "209644832"},
    {0x1.d0322ap+28f, chars_format::fixed, 0, "486744736"},
    {0x1.cdb764p+29f, chars_format::fixed, 0, "968289408"},
    {0x1.7620d8p+30f, chars_format::fixed, 0, "1569207808"},
    {0x1.c18df6p+31f, chars_format::fixed, 0, "3771136768"},
    {0x1.240cf8p+32f, chars_format::fixed, 0, "4899796992"},
    {0x1.81669ap+33f, chars_format::fixed, 0, "12931904512"},
    {0x1.3be30cp+34f, chars_format::fixed, 0, "21198811136"},
    {0x1.d1e6e4p+35f, chars_format::fixed, 0, "62532296704"},
    {0x1.06b274p+36f, chars_format::fixed, 0, "70517211136"},
    {0x1.a74284p+37f, chars_format::fixed, 0, "227235889152"},
    {0x1.9fd3e6p+38f, chars_format::fixed, 0, "446491623424"},
    {0x1.e2cec4p+39f, chars_format::fixed, 0, "1036821594112"},
    {0x1.3d5d32p+40f, chars_format::fixed, 0, "1363068190720"},
    {0x1.accccap+41f, chars_format::fixed, 0, "3683363586048"},
    {0x1.a120ccp+42f, chars_format::fixed, 0, "7166206410752"},
    {0x1.55a028p+43f, chars_format::fixed, 0, "11738166591488"},
    {0x1.035296p+44f, chars_format::fixed, 0, "17820513468416"},
    {0x1.22d1aap+45f, chars_format::fixed, 0, "39969859043328"},
    {0x1.eb8eaep+46f, chars_format::fixed, 0, "135118253457408"},
    {0x1.490d0ep+47f, chars_format::fixed, 0, "180897697497088"},
    {0x1.9da088p+48f, chars_format::fixed, 0, "454787778740224"},
    {0x1.e7fab4p+49f, chars_format::fixed, 0, "1073077848899584"},
    {0x1.98a534p+50f, chars_format::fixed, 0, "1797241144606720"},
    {0x1.93aeeap+51f, chars_format::fixed, 0, "3550835489374208"},
    {0x1.3df680p+52f, chars_format::fixed, 0, "5593662327095296"},
    {0x1.c763f6p+53f, chars_format::fixed, 0, "16022627827056640"},
    {0x1.8b669ep+54f, chars_format::fixed, 0, "27823861147893760"},
    {0x1.73e5b6p+55f, chars_format::fixed, 0, "52339893103230976"},
    {0x1.a13d18p+56f, chars_format::fixed, 0, "117442238576852992"},
    {0x1.a0797ep+57f, chars_format::fixed, 0, "234454344768946176"},
    {0x1.c07a80p+58f, chars_format::fixed, 0, "504941918963105792"},
    {0x1.729388p+59f, chars_format::fixed, 0, "834463629662224384"},
    {0x1.edfb70p+60f, chars_format::fixed, 0, "2224697951572197376"},
    {0x1.3d6782p+61f, chars_format::fixed, 0, "2858924021141995520"},
    {0x1.b121e8p+62f, chars_format::fixed, 0, "7802620494837972992"},
    {0x1.0efc5ap+63f, chars_format::fixed, 0, "9763290520209063936"},
    {0x1.b7dba0p+64f, chars_format::fixed, 0, "31695102724410441728"},
    {0x1.ec2306p+65f, chars_format::fixed, 0, "70924388975830368256"},
    {0x1.2e2d28p+66f, chars_format::fixed, 0, "87096415015485308928"},
    {0x1.e02208p+67f, chars_format::fixed, 0, "276777792668052750336"},
    {0x1.402636p+68f, chars_format::fixed, 0, "369106968238077509632"},
    {0x1.11f97cp+69f, chars_format::fixed, 0, "631742296991907971072"},
    {0x1.74db2ap+70f, chars_format::fixed, 0, "1719495307615820316672"},
    {0x1.94a32ap+71f, chars_format::fixed, 0, "3732120907777931476992"},
    {0x1.c272dcp+72f, chars_format::fixed, 0, "8309311323384498356224"},
    {0x1.36ca40p+73f, chars_format::fixed, 0, "11466128622488263852032"},
    {0x1.5f6fbep+74f, chars_format::fixed, 0, "25931436172223350571008"},
    {0x1.95ec4ep+75f, chars_format::fixed, 0, "59903671176748022628352"},
    {0x1.6b3912p+76f, chars_format::fixed, 0, "107204487170660958732288"},
    {0x1.10992ap+77f, chars_format::fixed, 0, "160913632700346331561984"},
    {0x1.74a25ep+78f, chars_format::fixed, 0, "439928869395322133020672"},
    {0x1.43f462p+79f, chars_format::fixed, 0, "764916220582548125777920"},
    {0x1.f12ca2p+80f, chars_format::fixed, 0, "2347839472055691035803648"},
    {0x1.2b7f18p+81f, chars_format::fixed, 0, "2828664088515283884441600"},
    {0x1.a40704p+82f, chars_format::fixed, 0, "7934093352976572433301504"},
    {0x1.35d5f8p+83f, chars_format::fixed, 0, "11705266159821935293235200"},
    {0x1.c2c9d2p+84f, chars_format::fixed, 0, "34060605519118462894473216"},
    {0x1.47bf20p+85f, chars_format::fixed, 0, "49527663163502775133798400"},
    {0x1.60b728p+86f, chars_format::fixed, 0, "106601704860119390738186240"},
    {0x1.3354c8p+87f, chars_format::fixed, 0, "185770297377533474371534848"},
    {0x1.e9e512p+88f, chars_format::fixed, 0, "592246479757524141957185536"},
    {0x1.c4b6cap+89f, chars_format::fixed, 0, "1094595334815995103451021312"},
    {0x1.799cb8p+90f, chars_format::fixed, 0, "1826020469467809704300249088"},
    {0x1.1afa36p+91f, chars_format::fixed, 0, "2736789351009782551090823168"},
    {0x1.80c214p+92f, chars_format::fixed, 0, "7442304364233212615194574848"},
    {0x1.657890p+93f, chars_format::fixed, 0, "13828987453168434783077793792"},
    {0x1.5ce17cp+94f, chars_format::fixed, 0, "26993344325171312829134798848"},
    {0x1.3f1e9ap+95f, chars_format::fixed, 0, "49381356576017938861904625664"},
    {0x1.874612p+96f, chars_format::fixed, 0, "121093348650115637567232671744"},
    {0x1.5f4d5ep+97f, chars_format::fixed, 0, "217445539275703670631001227264"},
    {0x1.45b1bep+98f, chars_format::fixed, 0, "403190021246562727728269754368"},
    {0x1.a570f4p+99f, chars_format::fixed, 0, "1043437928672039460753056464896"},
    {0x1.f5106ep+100f, chars_format::fixed, 0, "2481149635102733266542145830912"},
    {0x1.d84424p+101f, chars_format::fixed, 0, "4677097651091265616934539886592"},
    {0x1.3d6c56p+102f, chars_format::fixed, 0, "6287213966425746785671183335424"},
    {0x1.9d8cf0p+103f, chars_format::fixed, 0, "16382424580981433623378525159424"},
    {0x1.e2e73ep+104f, chars_format::fixed, 0, "38259540322544957537972440268800"},
    {0x1.2d6594p+105f, chars_format::fixed, 0, "47758227647613648865431576903680"},
    {0x1.ce43bap+106f, chars_format::fixed, 0, "146497485749802409635393442938880"},
    {0x1.b3ea00p+107f, chars_format::fixed, 0, "276293361488025452794185737306112"},
    {0x1.03a052p+108f, chars_format::fixed, 0, "329115373194929392757058784198656"},
    {0x1.6f59e0p+109f, chars_format::fixed, 0, "931345619455766569116232623063040"},
    {0x1.05adacp+110f, chars_format::fixed, 0, "1326867152522435745601434087849984"},
    {0x1.2cdef0p+111f, chars_format::fixed, 0, "3051192904788012466473218045116416"},
    {0x1.e81552p+112f, chars_format::fixed, 0, "9899505055765620068271358482579456"},
    {0x1.bfa8f4p+113f, chars_format::fixed, 0, "18159245876954178992833811110166528"},
    {0x1.a14810p+114f, chars_format::fixed, 0, "33853896736735722962455354188759040"},
    {0x1.f18b10p+115f, chars_format::fixed, 0, "80731001914916160681187088757948416"},
    {0x1.8d6e30p+116f, chars_format::fixed, 0, "128973545052908058560090358153216000"},
    {0x1.9480c2p+117f, chars_format::fixed, 0, "262537431192608192877759864086986752"},
    {0x1.60975cp+118f, chars_format::fixed, 0, "457689606761340509948952337218273280"},
    {0x1.ab1bb2p+119f, chars_format::fixed, 0, "1108836243133298765768030079592431616"},
    {0x1.6a0c80p+120f, chars_format::fixed, 0, "1879864992909653247408339011818749952"},
    {0x1.2cac2cp+121f, chars_format::fixed, 0, "3122362236102854007005843883842076672"},
    {0x1.0baaf6p+122f, chars_format::fixed, 0, "5559243043957593079267046257728684032"},
    {0x1.098282p+123f, chars_format::fixed, 0, "11028845443370647144636654644992409600"},
    {0x1.122f8ap+124f, chars_format::fixed, 0, "22778456735621461875293910785310326784"},
    {0x1.57f4c6p+125f, chars_format::fixed, 0, "57149517363101270672263900542030315520"},
    {0x1.05e028p+126f, chars_format::fixed, 0, "87023098173139747570875357950241669120"},
    {0x1.9d8424p+127f, chars_format::fixed, 0, "274828637805621292108186801756142829568"},

    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffep-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000117549421069244107548702944484928734882705242874589333385717453057158"
        "8870475618904265502351336181163787841796875"},
    {0x1.fffffep-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000235098856151472858345576598207153302664571798551798085536592623685000"
        "6129930346077117064851336181163787841796875"},
    {0x1.fffffep-125f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000470197712302945716691153196414306605329143597103596171073185247370001"
        "225986069215423412970267236232757568359375"},
    {0x1.fffffep-124f, chars_format::fixed, 147,
        "0."
        "0000000000000000000000000000000000000940395424605891433382306392828613210658287194207192342146370494740002"
        "45197213843084682594053447246551513671875"},
    {0x1.fffffep-123f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000001880790849211782866764612785657226421316574388414384684292740989480004"
        "9039442768616936518810689449310302734375"},
    {0x1.fffffep-122f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000003761581698423565733529225571314452842633148776828769368585481978960009"
        "807888553723387303762137889862060546875"},
    {0x1.fffffep-121f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000007523163396847131467058451142628905685266297553657538737170963957920019"
        "61577710744677460752427577972412109375"},
    {0x1.fffffep-120f, chars_format::fixed, 143,
        "0."
        "0000000000000000000000000000000000015046326793694262934116902285257811370532595107315077474341927915840039"
        "2315542148935492150485515594482421875"},
    {0x1.fffffep-119f, chars_format::fixed, 142,
        "0."
        "0000000000000000000000000000000000030092653587388525868233804570515622741065190214630154948683855831680078"
        "463108429787098430097103118896484375"},
    {0x1.fffffep-118f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000060185307174777051736467609141031245482130380429260309897367711663360156"
        "92621685957419686019420623779296875"},
    {0x1.fffffep-117f, chars_format::fixed, 140,
        "0."
        "0000000000000000000000000000000000120370614349554103472935218282062490964260760858520619794735423326720313"
        "8524337191483937203884124755859375"},
    {0x1.fffffep-116f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000240741228699108206945870436564124981928521521717041239589470846653440627"
        "704867438296787440776824951171875"},
    {0x1.fffffep-115f, chars_format::fixed, 138,
        "0."
        "0000000000000000000000000000000000481482457398216413891740873128249963857043043434082479178941693306881255"
        "40973487659357488155364990234375"},
    {0x1.fffffep-114f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000962964914796432827783481746256499927714086086868164958357883386613762510"
        "8194697531871497631072998046875"},
    {0x1.fffffep-113f, chars_format::fixed, 136,
        "0."
        "0000000000000000000000000000000001925929829592865655566963492512999855428172173736329916715766773227525021"
        "638939506374299526214599609375"},
    {0x1.fffffep-112f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000003851859659185731311133926985025999710856344347472659833431533546455050043"
        "27787901274859905242919921875"},
    {0x1.fffffep-111f, chars_format::fixed, 134,
        "0."
        "0000000000000000000000000000000007703719318371462622267853970051999421712688694945319666863067092910100086"
        "5557580254971981048583984375"},
    {0x1.fffffep-110f, chars_format::fixed, 133,
        "0."
        "0000000000000000000000000000000015407438636742925244535707940103998843425377389890639333726134185820200173"
        "111516050994396209716796875"},
    {0x1.fffffep-109f, chars_format::fixed, 132,
        "0."
        "0000000000000000000000000000000030814877273485850489071415880207997686850754779781278667452268371640400346"
        "22303210198879241943359375"},
    {0x1.fffffep-108f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000061629754546971700978142831760415995373701509559562557334904536743280800692"
        "4460642039775848388671875"},
    {0x1.fffffep-107f, chars_format::fixed, 130,
        "0."
        "0000000000000000000000000000000123259509093943401956285663520831990747403019119125114669809073486561601384"
        "892128407955169677734375"},
    {0x1.fffffep-106f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000246519018187886803912571327041663981494806038238250229339618146973123202769"
        "78425681591033935546875"},
    {0x1.fffffep-105f, chars_format::fixed, 128,
        "0."
        "0000000000000000000000000000000493038036375773607825142654083327962989612076476500458679236293946246405539"
        "5685136318206787109375"},
    {0x1.fffffep-104f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000986076072751547215650285308166655925979224152953000917358472587892492811079"
        "137027263641357421875"},
    {0x1.fffffep-103f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000001972152145503094431300570616333311851958448305906001834716945175784985622158"
        "27405452728271484375"},
    {0x1.fffffep-102f, chars_format::fixed, 125,
        "0."
        "0000000000000000000000000000003944304291006188862601141232666623703916896611812003669433890351569971244316"
        "5481090545654296875"},
    {0x1.fffffep-101f, chars_format::fixed, 124,
        "0."
        "0000000000000000000000000000007888608582012377725202282465333247407833793223624007338867780703139942488633"
        "096218109130859375"},
    {0x1.fffffep-100f, chars_format::fixed, 123,
        "0."
        "0000000000000000000000000000015777217164024755450404564930666494815667586447248014677735561406279884977266"
        "19243621826171875"},
    {0x1.fffffep-99f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000031554434328049510900809129861332989631335172894496029355471122812559769954532"
        "3848724365234375"},
    {0x1.fffffep-98f, chars_format::fixed, 121,
        "0."
        "0000000000000000000000000000063108868656099021801618259722665979262670345788992058710942245625119539909064"
        "769744873046875"},
    {0x1.fffffep-97f, chars_format::fixed, 120,
        "0."
        "0000000000000000000000000000126217737312198043603236519445331958525340691577984117421884491250239079818129"
        "53948974609375"},
    {0x1.fffffep-96f, chars_format::fixed, 119,
        "0."
        "0000000000000000000000000000252435474624396087206473038890663917050681383155968234843768982500478159636259"
        "0789794921875"},
    {0x1.fffffep-95f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000504870949248792174412946077781327834101362766311936469687537965000956319272518"
        "157958984375"},
    {0x1.fffffep-94f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000001009741898497584348825892155562655668202725532623872939375075930001912638545036"
        "31591796875"},
    {0x1.fffffep-93f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000002019483796995168697651784311125311336405451065247745878750151860003825277090072"
        "6318359375"},
    {0x1.fffffep-92f, chars_format::fixed, 115,
        "0."
        "0000000000000000000000000004038967593990337395303568622250622672810902130495491757500303720007650554180145"
        "263671875"},
    {0x1.fffffep-91f, chars_format::fixed, 114,
        "0."
        "0000000000000000000000000008077935187980674790607137244501245345621804260990983515000607440015301108360290"
        "52734375"},
    {0x1.fffffep-90f, chars_format::fixed, 113,
        "0."
        "0000000000000000000000000016155870375961349581214274489002490691243608521981967030001214880030602216720581"
        "0546875"},
    {0x1.fffffep-89f, chars_format::fixed, 112,
        "0."
        "0000000000000000000000000032311740751922699162428548978004981382487217043963934060002429760061204433441162"
        "109375"},
    {0x1.fffffep-88f, chars_format::fixed, 111,
        "0."
        "0000000000000000000000000064623481503845398324857097956009962764974434087927868120004859520122408866882324"
        "21875"},
    {0x1.fffffep-87f, chars_format::fixed, 110,
        "0."
        "0000000000000000000000000129246963007690796649714195912019925529948868175855736240009719040244817733764648"
        "4375"},
    {0x1.fffffep-86f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000258493926015381593299428391824039851059897736351711472480019438080489635467529296"
        "875"},
    {0x1.fffffep-85f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000516987852030763186598856783648079702119795472703422944960038876160979270935058593"
        "75"},
    {0x1.fffffep-84f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000001033975704061526373197713567296159404239590945406845889920077752321958541870117187"
        "5"},
    {0x1.fffffep-83f, chars_format::fixed, 106,
        "0."
        "000000000000000000000000206795140812305274639542713459231880847918189081369177984015550464391708374023437"
        "5"},
    {0x1.fffffep-82f, chars_format::fixed, 105,
        "0."
        "00000000000000000000000041359028162461054927908542691846376169583637816273835596803110092878341674804687"
        "5"},
    {0x1.fffffep-81f, chars_format::fixed, 104,
        "0."
        "00000000000000000000000082718056324922109855817085383692752339167275632547671193606220185756683349609375"},
    {0x1.fffffep-80f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000016543611264984421971163417076738550467833455126509534238721244037151336669921875"},
    {0x1.fffffep-79f, chars_format::fixed, 102,
        "0.000000000000000000000003308722252996884394232683415347710093566691025301906847744248807430267333984375"},
    {0x1.fffffep-78f, chars_format::fixed, 101,
        "0.00000000000000000000000661744450599376878846536683069542018713338205060381369548849761486053466796875"},
    {0x1.fffffep-77f, chars_format::fixed, 100,
        "0.0000000000000000000000132348890119875375769307336613908403742667641012076273909769952297210693359375"},
    {0x1.fffffep-76f, chars_format::fixed, 99,
        "0.000000000000000000000026469778023975075153861467322781680748533528202415254781953990459442138671875"},
    {0x1.fffffep-75f, chars_format::fixed, 98,
        "0.00000000000000000000005293955604795015030772293464556336149706705640483050956390798091888427734375"},
    {0x1.fffffep-74f, chars_format::fixed, 97,
        "0.0000000000000000000001058791120959003006154458692911267229941341128096610191278159618377685546875"},
    {0x1.fffffep-73f, chars_format::fixed, 96,
        "0.000000000000000000000211758224191800601230891738582253445988268225619322038255631923675537109375"},
    {0x1.fffffep-72f, chars_format::fixed, 95,
        "0.00000000000000000000042351644838360120246178347716450689197653645123864407651126384735107421875"},
    {0x1.fffffep-71f, chars_format::fixed, 94,
        "0.0000000000000000000008470328967672024049235669543290137839530729024772881530225276947021484375"},
    {0x1.fffffep-70f, chars_format::fixed, 93,
        "0.000000000000000000001694065793534404809847133908658027567906145804954576306045055389404296875"},
    {0x1.fffffep-69f, chars_format::fixed, 92,
        "0.00000000000000000000338813158706880961969426781731605513581229160990915261209011077880859375"},
    {0x1.fffffep-68f, chars_format::fixed, 91,
        "0.0000000000000000000067762631741376192393885356346321102716245832198183052241802215576171875"},
    {0x1.fffffep-67f, chars_format::fixed, 90,
        "0.000000000000000000013552526348275238478777071269264220543249166439636610448360443115234375"},
    {0x1.fffffep-66f, chars_format::fixed, 89,
        "0.00000000000000000002710505269655047695755414253852844108649833287927322089672088623046875"},
    {0x1.fffffep-65f, chars_format::fixed, 88,
        "0.0000000000000000000542101053931009539151082850770568821729966657585464417934417724609375"},
    {0x1.fffffep-64f, chars_format::fixed, 87,
        "0.000000000000000000108420210786201907830216570154113764345993331517092883586883544921875"},
    {0x1.fffffep-63f, chars_format::fixed, 86,
        "0.00000000000000000021684042157240381566043314030822752869198666303418576717376708984375"},
    {0x1.fffffep-62f, chars_format::fixed, 85,
        "0.0000000000000000004336808431448076313208662806164550573839733260683715343475341796875"},
    {0x1.fffffep-61f, chars_format::fixed, 84,
        "0.000000000000000000867361686289615262641732561232910114767946652136743068695068359375"},
    {0x1.fffffep-60f, chars_format::fixed, 83,
        "0.00000000000000000173472337257923052528346512246582022953589330427348613739013671875"},
    {0x1.fffffep-59f, chars_format::fixed, 82,
        "0.0000000000000000034694467451584610505669302449316404590717866085469722747802734375"},
    {0x1.fffffep-58f, chars_format::fixed, 81,
        "0.000000000000000006938893490316922101133860489863280918143573217093944549560546875"},
    {0x1.fffffep-57f, chars_format::fixed, 80,
        "0.00000000000000001387778698063384420226772097972656183628714643418788909912109375"},
    {0x1.fffffep-56f, chars_format::fixed, 79,
        "0.0000000000000000277555739612676884045354419594531236725742928683757781982421875"},
    {0x1.fffffep-55f, chars_format::fixed, 78,
        "0.000000000000000055511147922535376809070883918906247345148585736751556396484375"},
    {0x1.fffffep-54f, chars_format::fixed, 77,
        "0.00000000000000011102229584507075361814176783781249469029717147350311279296875"},
    {0x1.fffffep-53f, chars_format::fixed, 76,
        "0.0000000000000002220445916901415072362835356756249893805943429470062255859375"},
    {0x1.fffffep-52f, chars_format::fixed, 75,
        "0.000000000000000444089183380283014472567071351249978761188685894012451171875"},
    {0x1.fffffep-51f, chars_format::fixed, 74,
        "0.00000000000000088817836676056602894513414270249995752237737178802490234375"},
    {0x1.fffffep-50f, chars_format::fixed, 73,
        "0.0000000000000017763567335211320578902682854049999150447547435760498046875"},
    {0x1.fffffep-49f, chars_format::fixed, 72,
        "0.000000000000003552713467042264115780536570809999830089509487152099609375"},
    {0x1.fffffep-48f, chars_format::fixed, 71,
        "0.00000000000000710542693408452823156107314161999966017901897430419921875"},
    {0x1.fffffep-47f, chars_format::fixed, 70,
        "0.0000000000000142108538681690564631221462832399993203580379486083984375"},
    {0x1.fffffep-46f, chars_format::fixed, 69,
        "0.000000000000028421707736338112926244292566479998640716075897216796875"},
    {0x1.fffffep-45f, chars_format::fixed, 68,
        "0.00000000000005684341547267622585248858513295999728143215179443359375"},
    {0x1.fffffep-44f, chars_format::fixed, 67, "0.0000000000001136868309453524517049771702659199945628643035888671875"},
    {0x1.fffffep-43f, chars_format::fixed, 66, "0.000000000000227373661890704903409954340531839989125728607177734375"},
    {0x1.fffffep-42f, chars_format::fixed, 65, "0.00000000000045474732378140980681990868106367997825145721435546875"},
    {0x1.fffffep-41f, chars_format::fixed, 64, "0.0000000000009094946475628196136398173621273599565029144287109375"},
    {0x1.fffffep-40f, chars_format::fixed, 63, "0.000000000001818989295125639227279634724254719913005828857421875"},
    {0x1.fffffep-39f, chars_format::fixed, 62, "0.00000000000363797859025127845455926944850943982601165771484375"},
    {0x1.fffffep-38f, chars_format::fixed, 61, "0.0000000000072759571805025569091185388970188796520233154296875"},
    {0x1.fffffep-37f, chars_format::fixed, 60, "0.000000000014551914361005113818237077794037759304046630859375"},
    {0x1.fffffep-36f, chars_format::fixed, 59, "0.00000000002910382872201022763647415558807551860809326171875"},
    {0x1.fffffep-35f, chars_format::fixed, 58, "0.0000000000582076574440204552729483111761510372161865234375"},
    {0x1.fffffep-34f, chars_format::fixed, 57, "0.000000000116415314888040910545896622352302074432373046875"},
    {0x1.fffffep-33f, chars_format::fixed, 56, "0.00000000023283062977608182109179324470460414886474609375"},
    {0x1.fffffep-32f, chars_format::fixed, 55, "0.0000000004656612595521636421835864894092082977294921875"},
    {0x1.fffffep-31f, chars_format::fixed, 54, "0.000000000931322519104327284367172978818416595458984375"},
    {0x1.fffffep-30f, chars_format::fixed, 53, "0.00000000186264503820865456873434595763683319091796875"},
    {0x1.fffffep-29f, chars_format::fixed, 52, "0.0000000037252900764173091374686919152736663818359375"},
    {0x1.fffffep-28f, chars_format::fixed, 51, "0.000000007450580152834618274937383830547332763671875"},
    {0x1.fffffep-27f, chars_format::fixed, 50, "0.00000001490116030566923654987476766109466552734375"},
    {0x1.fffffep-26f, chars_format::fixed, 49, "0.0000000298023206113384730997495353221893310546875"},
    {0x1.fffffep-25f, chars_format::fixed, 48, "0.000000059604641222676946199499070644378662109375"},
    {0x1.fffffep-24f, chars_format::fixed, 47, "0.00000011920928244535389239899814128875732421875"},
    {0x1.fffffep-23f, chars_format::fixed, 46, "0.0000002384185648907077847979962825775146484375"},
    {0x1.fffffep-22f, chars_format::fixed, 45, "0.000000476837129781415569595992565155029296875"},
    {0x1.fffffep-21f, chars_format::fixed, 44, "0.00000095367425956283113919198513031005859375"},
    {0x1.fffffep-20f, chars_format::fixed, 43, "0.0000019073485191256622783839702606201171875"},
    {0x1.fffffep-19f, chars_format::fixed, 42, "0.000003814697038251324556767940521240234375"},
    {0x1.fffffep-18f, chars_format::fixed, 41, "0.00000762939407650264911353588104248046875"},
    {0x1.fffffep-17f, chars_format::fixed, 40, "0.0000152587881530052982270717620849609375"},
    {0x1.fffffep-16f, chars_format::fixed, 39, "0.000030517576306010596454143524169921875"},
    {0x1.fffffep-15f, chars_format::fixed, 38, "0.00006103515261202119290828704833984375"},
    {0x1.fffffep-14f, chars_format::fixed, 37, "0.0001220703052240423858165740966796875"},
    {0x1.fffffep-13f, chars_format::fixed, 36, "0.000244140610448084771633148193359375"},
    {0x1.fffffep-12f, chars_format::fixed, 35, "0.00048828122089616954326629638671875"},
    {0x1.fffffep-11f, chars_format::fixed, 34, "0.0009765624417923390865325927734375"},
    {0x1.fffffep-10f, chars_format::fixed, 33, "0.001953124883584678173065185546875"},
    {0x1.fffffep-9f, chars_format::fixed, 32, "0.00390624976716935634613037109375"},
    {0x1.fffffep-8f, chars_format::fixed, 31, "0.0078124995343387126922607421875"},
    {0x1.fffffep-7f, chars_format::fixed, 30, "0.015624999068677425384521484375"},
    {0x1.fffffep-6f, chars_format::fixed, 29, "0.03124999813735485076904296875"},
    {0x1.fffffep-5f, chars_format::fixed, 28, "0.0624999962747097015380859375"},
    {0x1.fffffep-4f, chars_format::fixed, 27, "0.124999992549419403076171875"},
    {0x1.fffffep-3f, chars_format::fixed, 26, "0.24999998509883880615234375"},
    {0x1.fffffep-2f, chars_format::fixed, 25, "0.4999999701976776123046875"},
    {0x1.fffffep-1f, chars_format::fixed, 24, "0.999999940395355224609375"},
    {0x1.fffffep+0f, chars_format::fixed, 23, "1.99999988079071044921875"},
    {0x1.fffffep+1f, chars_format::fixed, 22, "3.9999997615814208984375"},
    {0x1.fffffep+2f, chars_format::fixed, 21, "7.999999523162841796875"},
    {0x1.fffffep+3f, chars_format::fixed, 20, "15.99999904632568359375"},
    {0x1.fffffep+4f, chars_format::fixed, 19, "31.9999980926513671875"},
    {0x1.fffffep+5f, chars_format::fixed, 18, "63.999996185302734375"},
    {0x1.fffffep+6f, chars_format::fixed, 17, "127.99999237060546875"},
    {0x1.fffffep+7f, chars_format::fixed, 16, "255.9999847412109375"},
    {0x1.fffffep+8f, chars_format::fixed, 15, "511.999969482421875"},
    {0x1.fffffep+9f, chars_format::fixed, 14, "1023.99993896484375"},
    {0x1.fffffep+10f, chars_format::fixed, 13, "2047.9998779296875"},
    {0x1.fffffep+11f, chars_format::fixed, 12, "4095.999755859375"},
    {0x1.fffffep+12f, chars_format::fixed, 11, "8191.99951171875"},
    {0x1.fffffep+13f, chars_format::fixed, 10, "16383.9990234375"},
    {0x1.fffffep+14f, chars_format::fixed, 9, "32767.998046875"},
    {0x1.fffffep+15f, chars_format::fixed, 8, "65535.99609375"},
    {0x1.fffffep+16f, chars_format::fixed, 7, "131071.9921875"},
    {0x1.fffffep+17f, chars_format::fixed, 6, "262143.984375"},
    {0x1.fffffep+18f, chars_format::fixed, 5, "524287.96875"},
    {0x1.fffffep+19f, chars_format::fixed, 4, "1048575.9375"},
    {0x1.fffffep+20f, chars_format::fixed, 3, "2097151.875"},
    {0x1.fffffep+21f, chars_format::fixed, 2, "4194303.75"},
    {0x1.fffffep+22f, chars_format::fixed, 1, "8388607.5"},
    {0x1.fffffep+23f, chars_format::fixed, 0, "16777215"},
    {0x1.fffffep+24f, chars_format::fixed, 0, "33554430"},
    {0x1.fffffep+25f, chars_format::fixed, 0, "67108860"},
    {0x1.fffffep+26f, chars_format::fixed, 0, "134217720"},
    {0x1.fffffep+27f, chars_format::fixed, 0, "268435440"},
    {0x1.fffffep+28f, chars_format::fixed, 0, "536870880"},
    {0x1.fffffep+29f, chars_format::fixed, 0, "1073741760"},
    {0x1.fffffep+30f, chars_format::fixed, 0, "2147483520"},
    {0x1.fffffep+31f, chars_format::fixed, 0, "4294967040"},
    {0x1.fffffep+32f, chars_format::fixed, 0, "8589934080"},
    {0x1.fffffep+33f, chars_format::fixed, 0, "17179868160"},
    {0x1.fffffep+34f, chars_format::fixed, 0, "34359736320"},
    {0x1.fffffep+35f, chars_format::fixed, 0, "68719472640"},
    {0x1.fffffep+36f, chars_format::fixed, 0, "137438945280"},
    {0x1.fffffep+37f, chars_format::fixed, 0, "274877890560"},
    {0x1.fffffep+38f, chars_format::fixed, 0, "549755781120"},
    {0x1.fffffep+39f, chars_format::fixed, 0, "1099511562240"},
    {0x1.fffffep+40f, chars_format::fixed, 0, "2199023124480"},
    {0x1.fffffep+41f, chars_format::fixed, 0, "4398046248960"},
    {0x1.fffffep+42f, chars_format::fixed, 0, "8796092497920"},
    {0x1.fffffep+43f, chars_format::fixed, 0, "17592184995840"},
    {0x1.fffffep+44f, chars_format::fixed, 0, "35184369991680"},
    {0x1.fffffep+45f, chars_format::fixed, 0, "70368739983360"},
    {0x1.fffffep+46f, chars_format::fixed, 0, "140737479966720"},
    {0x1.fffffep+47f, chars_format::fixed, 0, "281474959933440"},
    {0x1.fffffep+48f, chars_format::fixed, 0, "562949919866880"},
    {0x1.fffffep+49f, chars_format::fixed, 0, "1125899839733760"},
    {0x1.fffffep+50f, chars_format::fixed, 0, "2251799679467520"},
    {0x1.fffffep+51f, chars_format::fixed, 0, "4503599358935040"},
    {0x1.fffffep+52f, chars_format::fixed, 0, "9007198717870080"},
    {0x1.fffffep+53f, chars_format::fixed, 0, "18014397435740160"},
    {0x1.fffffep+54f, chars_format::fixed, 0, "36028794871480320"},
    {0x1.fffffep+55f, chars_format::fixed, 0, "72057589742960640"},
    {0x1.fffffep+56f, chars_format::fixed, 0, "144115179485921280"},
    {0x1.fffffep+57f, chars_format::fixed, 0, "288230358971842560"},
    {0x1.fffffep+58f, chars_format::fixed, 0, "576460717943685120"},
    {0x1.fffffep+59f, chars_format::fixed, 0, "1152921435887370240"},
    {0x1.fffffep+60f, chars_format::fixed, 0, "2305842871774740480"},
    {0x1.fffffep+61f, chars_format::fixed, 0, "4611685743549480960"},
    {0x1.fffffep+62f, chars_format::fixed, 0, "9223371487098961920"},
    {0x1.fffffep+63f, chars_format::fixed, 0, "18446742974197923840"},
    {0x1.fffffep+64f, chars_format::fixed, 0, "36893485948395847680"},
    {0x1.fffffep+65f, chars_format::fixed, 0, "73786971896791695360"},
    {0x1.fffffep+66f, chars_format::fixed, 0, "147573943793583390720"},
    {0x1.fffffep+67f, chars_format::fixed, 0, "295147887587166781440"},
    {0x1.fffffep+68f, chars_format::fixed, 0, "590295775174333562880"},
    {0x1.fffffep+69f, chars_format::fixed, 0, "1180591550348667125760"},
    {0x1.fffffep+70f, chars_format::fixed, 0, "2361183100697334251520"},
    {0x1.fffffep+71f, chars_format::fixed, 0, "4722366201394668503040"},
    {0x1.fffffep+72f, chars_format::fixed, 0, "9444732402789337006080"},
    {0x1.fffffep+73f, chars_format::fixed, 0, "18889464805578674012160"},
    {0x1.fffffep+74f, chars_format::fixed, 0, "37778929611157348024320"},
    {0x1.fffffep+75f, chars_format::fixed, 0, "75557859222314696048640"},
    {0x1.fffffep+76f, chars_format::fixed, 0, "151115718444629392097280"},
    {0x1.fffffep+77f, chars_format::fixed, 0, "302231436889258784194560"},
    {0x1.fffffep+78f, chars_format::fixed, 0, "604462873778517568389120"},
    {0x1.fffffep+79f, chars_format::fixed, 0, "1208925747557035136778240"},
    {0x1.fffffep+80f, chars_format::fixed, 0, "2417851495114070273556480"},
    {0x1.fffffep+81f, chars_format::fixed, 0, "4835702990228140547112960"},
    {0x1.fffffep+82f, chars_format::fixed, 0, "9671405980456281094225920"},
    {0x1.fffffep+83f, chars_format::fixed, 0, "19342811960912562188451840"},
    {0x1.fffffep+84f, chars_format::fixed, 0, "38685623921825124376903680"},
    {0x1.fffffep+85f, chars_format::fixed, 0, "77371247843650248753807360"},
    {0x1.fffffep+86f, chars_format::fixed, 0, "154742495687300497507614720"},
    {0x1.fffffep+87f, chars_format::fixed, 0, "309484991374600995015229440"},
    {0x1.fffffep+88f, chars_format::fixed, 0, "618969982749201990030458880"},
    {0x1.fffffep+89f, chars_format::fixed, 0, "1237939965498403980060917760"},
    {0x1.fffffep+90f, chars_format::fixed, 0, "2475879930996807960121835520"},
    {0x1.fffffep+91f, chars_format::fixed, 0, "4951759861993615920243671040"},
    {0x1.fffffep+92f, chars_format::fixed, 0, "9903519723987231840487342080"},
    {0x1.fffffep+93f, chars_format::fixed, 0, "19807039447974463680974684160"},
    {0x1.fffffep+94f, chars_format::fixed, 0, "39614078895948927361949368320"},
    {0x1.fffffep+95f, chars_format::fixed, 0, "79228157791897854723898736640"},
    {0x1.fffffep+96f, chars_format::fixed, 0, "158456315583795709447797473280"},
    {0x1.fffffep+97f, chars_format::fixed, 0, "316912631167591418895594946560"},
    {0x1.fffffep+98f, chars_format::fixed, 0, "633825262335182837791189893120"},
    {0x1.fffffep+99f, chars_format::fixed, 0, "1267650524670365675582379786240"},
    {0x1.fffffep+100f, chars_format::fixed, 0, "2535301049340731351164759572480"},
    {0x1.fffffep+101f, chars_format::fixed, 0, "5070602098681462702329519144960"},
    {0x1.fffffep+102f, chars_format::fixed, 0, "10141204197362925404659038289920"},
    {0x1.fffffep+103f, chars_format::fixed, 0, "20282408394725850809318076579840"},
    {0x1.fffffep+104f, chars_format::fixed, 0, "40564816789451701618636153159680"},
    {0x1.fffffep+105f, chars_format::fixed, 0, "81129633578903403237272306319360"},
    {0x1.fffffep+106f, chars_format::fixed, 0, "162259267157806806474544612638720"},
    {0x1.fffffep+107f, chars_format::fixed, 0, "324518534315613612949089225277440"},
    {0x1.fffffep+108f, chars_format::fixed, 0, "649037068631227225898178450554880"},
    {0x1.fffffep+109f, chars_format::fixed, 0, "1298074137262454451796356901109760"},
    {0x1.fffffep+110f, chars_format::fixed, 0, "2596148274524908903592713802219520"},
    {0x1.fffffep+111f, chars_format::fixed, 0, "5192296549049817807185427604439040"},
    {0x1.fffffep+112f, chars_format::fixed, 0, "10384593098099635614370855208878080"},
    {0x1.fffffep+113f, chars_format::fixed, 0, "20769186196199271228741710417756160"},
    {0x1.fffffep+114f, chars_format::fixed, 0, "41538372392398542457483420835512320"},
    {0x1.fffffep+115f, chars_format::fixed, 0, "83076744784797084914966841671024640"},
    {0x1.fffffep+116f, chars_format::fixed, 0, "166153489569594169829933683342049280"},
    {0x1.fffffep+117f, chars_format::fixed, 0, "332306979139188339659867366684098560"},
    {0x1.fffffep+118f, chars_format::fixed, 0, "664613958278376679319734733368197120"},
    {0x1.fffffep+119f, chars_format::fixed, 0, "1329227916556753358639469466736394240"},
    {0x1.fffffep+120f, chars_format::fixed, 0, "2658455833113506717278938933472788480"},
    {0x1.fffffep+121f, chars_format::fixed, 0, "5316911666227013434557877866945576960"},
    {0x1.fffffep+122f, chars_format::fixed, 0, "10633823332454026869115755733891153920"},
    {0x1.fffffep+123f, chars_format::fixed, 0, "21267646664908053738231511467782307840"},
    {0x1.fffffep+124f, chars_format::fixed, 0, "42535293329816107476463022935564615680"},
    {0x1.fffffep+125f, chars_format::fixed, 0, "85070586659632214952926045871129231360"},
    {0x1.fffffep+126f, chars_format::fixed, 0, "170141173319264429905852091742258462720"},
    {0x1.fffffep+127f, chars_format::fixed, 0, "340282346638528859811704183484516925440"},
};

inline constexpr float_to_chars_testcase float_scientific_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::scientific, 4, "0.0000e+00"},
    {-0.0f, chars_format::scientific, 4, "-0.0000e+00"},
    {float_inf, chars_format::scientific, 4, "inf"},
    {-float_inf, chars_format::scientific, 4, "-inf"},
    {float_nan, chars_format::scientific, 4, "nan"},
    {-float_nan, chars_format::scientific, 4, "-nan"},
    {1.729f, chars_format::scientific, 4, "1.7290e+00"},
    {-1.729f, chars_format::scientific, 4, "-1.7290e+00"},

    // Ryu Printf Zero
    {0.0f, chars_format::scientific, 4, "0.0000e+00"},
    {0.0f, chars_format::scientific, 3, "0.000e+00"},
    {0.0f, chars_format::scientific, 2, "0.00e+00"},
    {0.0f, chars_format::scientific, 1, "0.0e+00"},
    {0.0f, chars_format::scientific, 0, "0e+00"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::scientific, 104, // min subnormal
        "1."
        "40129846432481707092372958328991613128026194187651577175706828388979108268586060148663818836212158203125e-"
        "45"},
    {0x0.fffffep-126f, chars_format::scientific, 111, // max subnormal
        "1."
        "1754942106924410754870294448492873488270524287458933338571745305715888704756189042655023513361811637878417"
        "96875e-38"},
    {0x1p-126f, chars_format::scientific, 88, // min normal
        "1.1754943508222875079687365372222456778186655567720875215087517062784172594547271728515625e-38"},
    {0x1.fffffep+127f, chars_format::scientific, 37, // max normal
        "3.4028234663852885981170418348451692544e+38"},

    // Ryu Printf AllPowersOfTen
    // These values test every power of ten that's within the range of floats.
    {1e-44f, chars_format::scientific, 104,
        "9."
        "80908925027371949646610708302941291896183359313561040229947798722853757880102421040646731853485107421875e-"
        "45"},
    {1e-43f, chars_format::scientific, 105,
        "9."
        "949219096706201203558480041358404532089859787323261979475184815617516687069610270555131137371063232421875e"
        "-44"},
    {1e-42f, chars_format::scientific, 106,
        "1."
        "0005271035279193886395429224690001177341070264998322610345467546973108330377044694614596664905548095703125"
        "e-42"},
    {1e-41f, chars_format::scientific, 102,
        "9.999665841421894618111734306356841512815949217230816547258439273837549166046301252208650112152099609375e-"
        "42"},
    {1e-40f, chars_format::scientific, 107,
        "9."
        "9999461011147595815259190522734994960422052696191918504127906874943271242628384243289474397897720336914062"
        "5e-41"},
    {1e-39f, chars_format::scientific, 107,
        "1."
        "0000002153053332574208756001456831092687456480096866911043660970225682715906145858753006905317306518554687"
        "5e-39"},
    {1e-38f, chars_format::scientific, 109,
        "9."
        "9999993504564039245746141539976645128551939195729831580121174560891149363239804870318039320409297943115234"
        "375e-39"},
    {1e-37f, chars_format::scientific, 107,
        "9."
        "9999999109757896545014425234894978288216464316777599086184261589164284922404135613760445266962051391601562"
        "5e-38"},
    {1e-36f, chars_format::scientific, 107,
        "1."
        "0000000359391298238442905219082964481594808441361581309103473121178279336973560020851437002420425415039062"
        "5e-36"},
    {1e-35f, chars_format::scientific, 104,
        "1."
        "00000001800250948048663201408455778204855436374880527489094543362735389990803014370612800121307373046875e-"
        "35"},
    {1e-34f, chars_format::scientific, 102,
        "1.000000046701102029858885626602539647826036732368569844521988439212112353970951517112553119659423828125e-"
        "34"},
    {1e-33f, chars_format::scientific, 98,
        "1.00000002374222799036108273658815415520405083747275818881715403474430559072061441838741302490234375e-33"},
    {1e-32f, chars_format::scientific, 98,
        "1.00000002374222799036108273658815415520405083747275818881715403474430559072061441838741302490234375e-32"},
    {1e-31f, chars_format::scientific, 94,
        "9.9999997966118983452530118776053400936983791927279980986387197816611660527996718883514404296875e-32"},
    {1e-30f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-30"},
    {1e-29f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-29"},
    {1e-28f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-28"},
    {1e-27f, chars_format::scientific, 85,
        "1.0000000272452011558114995103349890361263429573723815479979748488403856754302978515625e-27"},
    {1e-26f, chars_format::scientific, 82,
        "9.9999998872660226806678244921543018442779658661034858369021094404160976409912109375e-27"},
    {1e-25f, chars_format::scientific, 79,
        "1.0000000195414813782625560981110772657866336832199749551364220678806304931640625e-25"},
    {1e-24f, chars_format::scientific, 79,
        "1.0000000195414813782625560981110772657866336832199749551364220678806304931640625e-24"},
    {1e-23f, chars_format::scientific, 75,
        "9.999999998199587477372609628178631337169779413898140774108469486236572265625e-24"},
    {1e-22f, chars_format::scientific, 75,
        "1.000000031374394956577733179287005745028427128318071481771767139434814453125e-22"},
    {1e-21f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-22"},
    {1e-20f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-21"},
    {1e-19f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-20"},
    {1e-18f, chars_format::scientific, 65, "1.00000004581370496574313326554328540396454627625644207000732421875e-18"},
    {1e-17f, chars_format::scientific, 61, "9.9999998377515902426605765018763349871733225882053375244140625e-18"},
    {1e-16f, chars_format::scientific, 61, "1.0000000168623835263871646450439811815158464014530181884765625e-16"},
    {1e-15f, chars_format::scientific, 58, "1.0000000036274937255387218471014421083964407444000244140625e-15"},
    {1e-14f, chars_format::scientific, 53, "9.99999982451670044181213370393379591405391693115234375e-15"},
    {1e-13f, chars_format::scientific, 53, "9.99999982451670044181213370393379591405391693115234375e-14"},
    {1e-12f, chars_format::scientific, 48, "9.999999960041972002500187954865396022796630859375e-13"},
    {1e-11f, chars_format::scientific, 48, "9.999999960041972002500187954865396022796630859375e-12"},
    {1e-10f, chars_format::scientific, 47, "1.00000001335143196001808973960578441619873046875e-10"},
    {1e-09f, chars_format::scientific, 43, "9.9999997171806853657471947371959686279296875e-10"},
    {1e-08f, chars_format::scientific, 41, "9.99999993922529029077850282192230224609375e-09"},
    {1e-07f, chars_format::scientific, 40, "1.0000000116860974230803549289703369140625e-07"},
    {1e-06f, chars_format::scientific, 36, "9.999999974752427078783512115478515625e-07"},
    {1e-05f, chars_format::scientific, 32, "9.99999974737875163555145263671875e-06"},
    {1e-04f, chars_format::scientific, 32, "9.99999974737875163555145263671875e-05"},
    {1e-03f, chars_format::scientific, 30, "1.000000047497451305389404296875e-03"},
    {1e-02f, chars_format::scientific, 26, "9.99999977648258209228515625e-03"},
    {1e-01f, chars_format::scientific, 26, "1.00000001490116119384765625e-01"},
    {1e+00f, chars_format::scientific, 0, "1e+00"},
    {1e+01f, chars_format::scientific, 0, "1e+01"},
    {1e+02f, chars_format::scientific, 0, "1e+02"},
    {1e+03f, chars_format::scientific, 0, "1e+03"},
    {1e+04f, chars_format::scientific, 0, "1e+04"},
    {1e+05f, chars_format::scientific, 0, "1e+05"},
    {1e+06f, chars_format::scientific, 0, "1e+06"},
    {1e+07f, chars_format::scientific, 0, "1e+07"},
    {1e+08f, chars_format::scientific, 0, "1e+08"},
    {1e+09f, chars_format::scientific, 0, "1e+09"},
    {1e+10f, chars_format::scientific, 0, "1e+10"},
    {1e+11f, chars_format::scientific, 10, "9.9999997952e+10"},
    {1e+12f, chars_format::scientific, 11, "9.99999995904e+11"},
    {1e+13f, chars_format::scientific, 12, "9.999999827968e+12"},
    {1e+14f, chars_format::scientific, 14, "1.00000000376832e+14"},
    {1e+15f, chars_format::scientific, 14, "9.99999986991104e+14"},
    {1e+16f, chars_format::scientific, 16, "1.0000000272564224e+16"},
    {1e+17f, chars_format::scientific, 16, "9.9999998430674944e+16"},
    {1e+18f, chars_format::scientific, 16, "9.9999998430674944e+17"},
    {1e+19f, chars_format::scientific, 18, "9.999999980506447872e+18"},
    {1e+20f, chars_format::scientific, 20, "1.00000002004087734272e+20"},
    {1e+21f, chars_format::scientific, 20, "1.00000002004087734272e+21"},
    {1e+22f, chars_format::scientific, 21, "9.999999778196308361216e+21"},
    {1e+23f, chars_format::scientific, 21, "9.999999778196308361216e+22"},
    {1e+24f, chars_format::scientific, 24, "1.000000013848427855085568e+24"},
    {1e+25f, chars_format::scientific, 24, "9.999999562023526247432192e+24"},
    {1e+26f, chars_format::scientific, 26, "1.00000002537764290115403776e+26"},
    {1e+27f, chars_format::scientific, 26, "9.99999988484154753734934528e+26"},
    {1e+28f, chars_format::scientific, 27, "9.999999442119689768320106496e+27"},
    {1e+29f, chars_format::scientific, 29, "1.00000001504746621987668885504e+29"},
    {1e+30f, chars_format::scientific, 29, "1.00000001504746621987668885504e+30"},
    {1e+31f, chars_format::scientific, 30, "9.999999848243207295109594873856e+30"},
    {1e+32f, chars_format::scientific, 32, "1.00000003318135351409612647563264e+32"},
    {1e+33f, chars_format::scientific, 32, "9.99999994495727286427992885035008e+32"},
    {1e+34f, chars_format::scientific, 33, "9.999999790214767953607394487959552e+33"},
    {1e+35f, chars_format::scientific, 34, "1.0000000409184787596297531937521664e+35"},
    {1e+36f, chars_format::scientific, 35, "9.99999961690316245365415600208216064e+35"},
    {1e+37f, chars_format::scientific, 36, "9.999999933815812510711506376257961984e+36"},
    {1e+38f, chars_format::scientific, 37, "9.9999996802856924650656260769173209088e+37"},

    // Ryu Printf RoundToEven
    {0.125f, chars_format::scientific, 2, "1.25e-01"},
    {0.125f, chars_format::scientific, 1, "1.2e-01"},
    {0.375f, chars_format::scientific, 2, "3.75e-01"},
    {0.375f, chars_format::scientific, 1, "3.8e-01"},

    // Ryu Printf RoundToEvenInteger
    {2.5f, chars_format::scientific, 1, "2.5e+00"},
    {2.5f, chars_format::scientific, 0, "2e+00"},
    {3.5f, chars_format::scientific, 1, "3.5e+00"},
    {3.5f, chars_format::scientific, 0, "4e+00"},

    // Ryu Printf NonRoundToEvenScenarios
    {0.748046875f, chars_format::scientific, 2, "7.48e-01"},
    {0.748046875f, chars_format::scientific, 1, "7.5e-01"},
    {0.748046875f, chars_format::scientific, 0, "7e-01"}, // 0.75 would round to "8e-01", but this is smaller

    {0.2509765625f, chars_format::scientific, 2, "2.51e-01"},
    {0.2509765625f, chars_format::scientific, 1, "2.5e-01"},
    {0.2509765625f, chars_format::scientific, 0, "3e-01"}, // 0.25 would round to "2e-01", but this is larger

    {0x1.000002p-2f, chars_format::scientific, 24, "2.500000298023223876953125e-01"},
    {0x1.000002p-2f, chars_format::scientific, 2, "2.50e-01"},
    {0x1.000002p-2f, chars_format::scientific, 1, "2.5e-01"},
    {0x1.000002p-2f, chars_format::scientific, 0, "3e-01"}, // 0.25 would round to "2e-01", but this is larger (again)

    // More rounding tests.
    {9.5f, chars_format::scientific, 1, "9.5e+00"},
    {9.5f, chars_format::scientific, 0, "1e+01"},
    {10.5f, chars_format::scientific, 2, "1.05e+01"},
    {10.5f, chars_format::scientific, 1, "1.0e+01"},

    {1.241f, chars_format::scientific, 3, "1.241e+00"},
    {1.241f, chars_format::scientific, 1, "1.2e+00"},
    {1.251f, chars_format::scientific, 3, "1.251e+00"},
    {1.251f, chars_format::scientific, 1, "1.3e+00"},
    {1.261f, chars_format::scientific, 3, "1.261e+00"},
    {1.261f, chars_format::scientific, 1, "1.3e+00"},
    {1.341f, chars_format::scientific, 3, "1.341e+00"},
    {1.341f, chars_format::scientific, 1, "1.3e+00"},
    {1.351f, chars_format::scientific, 3, "1.351e+00"},
    {1.351f, chars_format::scientific, 1, "1.4e+00"},
    {1.361f, chars_format::scientific, 3, "1.361e+00"},
    {1.361f, chars_format::scientific, 1, "1.4e+00"},

    {2.41f, chars_format::scientific, 2, "2.41e+00"},
    {2.41f, chars_format::scientific, 0, "2e+00"},
    {2.51f, chars_format::scientific, 2, "2.51e+00"},
    {2.51f, chars_format::scientific, 0, "3e+00"},
    {2.61f, chars_format::scientific, 2, "2.61e+00"},
    {2.61f, chars_format::scientific, 0, "3e+00"},
    {3.41f, chars_format::scientific, 2, "3.41e+00"},
    {3.41f, chars_format::scientific, 0, "3e+00"},
    {3.51f, chars_format::scientific, 2, "3.51e+00"},
    {3.51f, chars_format::scientific, 0, "4e+00"},
    {3.61f, chars_format::scientific, 2, "3.61e+00"},
    {3.61f, chars_format::scientific, 0, "4e+00"},

    // Ryu Printf VaryingPrecision
    {1.142857f, chars_format::scientific, 28, "1.1428569555282592773437500000e+00"},
    {1.142857f, chars_format::scientific, 27, "1.142856955528259277343750000e+00"},
    {1.142857f, chars_format::scientific, 26, "1.14285695552825927734375000e+00"},
    {1.142857f, chars_format::scientific, 25, "1.1428569555282592773437500e+00"},
    {1.142857f, chars_format::scientific, 24, "1.142856955528259277343750e+00"},
    {1.142857f, chars_format::scientific, 23, "1.14285695552825927734375e+00"},
    {1.142857f, chars_format::scientific, 22, "1.1428569555282592773438e+00"},
    {1.142857f, chars_format::scientific, 21, "1.142856955528259277344e+00"},
    {1.142857f, chars_format::scientific, 20, "1.14285695552825927734e+00"},
    {1.142857f, chars_format::scientific, 19, "1.1428569555282592773e+00"},
    {1.142857f, chars_format::scientific, 18, "1.142856955528259277e+00"},
    {1.142857f, chars_format::scientific, 17, "1.14285695552825928e+00"},
    {1.142857f, chars_format::scientific, 16, "1.1428569555282593e+00"},
    {1.142857f, chars_format::scientific, 15, "1.142856955528259e+00"},
    {1.142857f, chars_format::scientific, 14, "1.14285695552826e+00"},
    {1.142857f, chars_format::scientific, 13, "1.1428569555283e+00"},
    {1.142857f, chars_format::scientific, 12, "1.142856955528e+00"},
    {1.142857f, chars_format::scientific, 11, "1.14285695553e+00"},
    {1.142857f, chars_format::scientific, 10, "1.1428569555e+00"},
    {1.142857f, chars_format::scientific, 9, "1.142856956e+00"},
    {1.142857f, chars_format::scientific, 8, "1.14285696e+00"},
    {1.142857f, chars_format::scientific, 7, "1.1428570e+00"},
    {1.142857f, chars_format::scientific, 6, "1.142857e+00"},
    {1.142857f, chars_format::scientific, 5, "1.14286e+00"},
    {1.142857f, chars_format::scientific, 4, "1.1429e+00"},
    {1.142857f, chars_format::scientific, 3, "1.143e+00"},
    {1.142857f, chars_format::scientific, 2, "1.14e+00"},
    {1.142857f, chars_format::scientific, 1, "1.1e+00"},
    {1.142857f, chars_format::scientific, 0, "1e+00"},

    // Negative precision requests 6 digits of precision.
    {1.142857f, chars_format::scientific, -1, "1.142857e+00"},
    {1.142857f, chars_format::scientific, -2, "1.142857e+00"},
    {1.142857f, chars_format::scientific, -3, "1.142857e+00"},

    // Ryu Printf Carrying
    {2.0009f, chars_format::scientific, 4, "2.0009e+00"},
    {2.0009f, chars_format::scientific, 3, "2.001e+00"},
    {2.0029f, chars_format::scientific, 4, "2.0029e+00"},
    {2.0029f, chars_format::scientific, 3, "2.003e+00"},
    {2.0099f, chars_format::scientific, 4, "2.0099e+00"},
    {2.0099f, chars_format::scientific, 3, "2.010e+00"},
    {2.0299f, chars_format::scientific, 4, "2.0299e+00"},
    {2.0299f, chars_format::scientific, 3, "2.030e+00"},
    {2.0999f, chars_format::scientific, 4, "2.0999e+00"},
    {2.0999f, chars_format::scientific, 3, "2.100e+00"},
    {2.2999f, chars_format::scientific, 4, "2.2999e+00"},
    {2.2999f, chars_format::scientific, 3, "2.300e+00"},
    {2.9999f, chars_format::scientific, 4, "2.9999e+00"},
    {2.9999f, chars_format::scientific, 3, "3.000e+00"},
    {9.9999f, chars_format::scientific, 4, "9.9999e+00"},
    {9.9999f, chars_format::scientific, 3, "1.000e+01"},

    {2.09f, chars_format::scientific, 2, "2.09e+00"},
    {2.09f, chars_format::scientific, 1, "2.1e+00"},
    {2.29f, chars_format::scientific, 2, "2.29e+00"},
    {2.29f, chars_format::scientific, 1, "2.3e+00"},
    {2.99f, chars_format::scientific, 2, "2.99e+00"},
    {2.99f, chars_format::scientific, 1, "3.0e+00"},
    {9.99f, chars_format::scientific, 2, "9.99e+00"},
    {9.99f, chars_format::scientific, 1, "1.0e+01"},

    {2.9f, chars_format::scientific, 1, "2.9e+00"},
    {2.9f, chars_format::scientific, 0, "3e+00"},
    {9.9f, chars_format::scientific, 1, "9.9e+00"},
    {9.9f, chars_format::scientific, 0, "1e+01"},

    // Ryu Printf Exponents
    {9.99e-10f, chars_format::scientific, 2, "9.99e-10"},
    {9.99e-09f, chars_format::scientific, 2, "9.99e-09"},
    {9.99e-01f, chars_format::scientific, 2, "9.99e-01"},
    {9.99e+00f, chars_format::scientific, 2, "9.99e+00"},
    {9.99e+01f, chars_format::scientific, 2, "9.99e+01"},
    {9.99e+09f, chars_format::scientific, 2, "9.99e+09"},
    {9.99e+10f, chars_format::scientific, 2, "9.99e+10"},

    {9.99e-10f, chars_format::scientific, 1, "1.0e-09"},
    {9.99e-09f, chars_format::scientific, 1, "1.0e-08"},
    {9.99e-01f, chars_format::scientific, 1, "1.0e+00"},
    {9.99e+00f, chars_format::scientific, 1, "1.0e+01"},
    {9.99e+01f, chars_format::scientific, 1, "1.0e+02"},
    {9.99e+09f, chars_format::scientific, 1, "1.0e+10"},
    {9.99e+10f, chars_format::scientific, 1, "1.0e+11"},

    // Ryu Printf AllBinaryExponents
    // These values test every binary exponent.
    // The mantissas were randomly generated.
    {0x0.bafab0p-126f, chars_format::scientific, 107,
        "8."
        "5856660078164374052571520381239855817217629811131320365461649230225810169869760102301370352506637573242187"
        "5e-39"},
    {0x1.2c4906p-126f, chars_format::scientific, 111,
        "1."
        "3788422360444725799170555395988202383016563601793620435599297375199720136484948795896343654021620750427246"
        "09375e-38"},
    {0x1.da6c8cp-125f, chars_format::scientific, 109,
        "4."
        "3568964460614492296234103491671745446178474818057230651136547036595368653788540314053534530103206634521484"
        "375e-38"},
    {0x1.094fd8p-124f, chars_format::scientific, 107,
        "4."
        "8730098044956940648689859309723501868625857818278160902221179289536129308757494982273783534765243530273437"
        "5e-38"},
    {0x1.1fba2ap-123f, chars_format::scientific, 109,
        "1."
        "0569428191822507939881039060004188252129978601750040353991776209794002661102041429330711252987384796142578"
        "125e-37"},
    {0x1.05c066p-122f, chars_format::scientific, 108,
        "1."
        "9230467241438048605749036664954171842721044312192826420869210321670447760844524509593611583113670349121093"
        "75e-37"},
    {0x1.aa97aep-121f, chars_format::scientific, 107,
        "6."
        "2682134052278382034317570078635218253248121012038564692208751497667224006349329101794864982366561889648437"
        "5e-37"},
    {0x1.dd39a8p-120f, chars_format::scientific, 105,
        "1."
        "402438874646247750851657016594307463985884969771101485328502708228859408023936339304782450199127197265625e"
        "-36"},
    {0x1.d47ee4p-119f, chars_format::scientific, 105,
        "2."
        "753570046800320919541684228734204622774621916697559471583583415484064449429979504202492535114288330078125e"
        "-36"},
    {0x1.3d3c36p-118f, chars_format::scientific, 105,
        "3."
        "729081842766376549743290774072513136892856721536744273788293430570150999159295679419301450252532958984375e"
        "-36"},
    {0x1.1661f4p-117f, chars_format::scientific, 103,
        "6."
        "5447441644065192772010189083715139205202243608571574700187600294454259852727773250080645084381103515625e-"
        "36"},
    {0x1.b68df4p-116f, chars_format::scientific, 103,
        "2."
        "0620733697737581832019236718658803118114911610758483179403903647053386549714559805579483509063720703125e-"
        "35"},
    {0x1.d99cbcp-115f, chars_format::scientific, 102,
        "4.453828135148790991673442414370889923378622963993538561236896870798585013062620419077575206756591796875e-"
        "35"},
    {0x1.fd046ep-114f, chars_format::scientific, 102,
        "9.573551435136219346253356313052326716805272442482100391194957227092299234527672524563968181610107421875e-"
        "35"},
    {0x1.89834cp-113f, chars_format::scientific, 101,
        "1.48023092977964777039833097802364118989950189925000614428441357561805347131667076610028743743896484375e-"
        "34"},
    {0x1.44f9f6p-112f, chars_format::scientific, 101,
        "2.44485077761403269855863218622241304191185681346184344636825701291282797456005937419831752777099609375e-"
        "34"},
    {0x1.610156p-111f, chars_format::scientific, 100,
        "5.3114321941046389584918230189833796378476189953948627677188069895475308612731168977916240692138671875e-"
        "34"},
    {0x1.1c4ce0p-110f, chars_format::scientific, 95,
        "8.55535074104030543315341178235051098966105705542347053919882693406862017582170665264129638671875e-34"},
    {0x1.c8846ap-109f, chars_format::scientific, 99,
        "2.747563210400574676694231398741237029228255561600948508123483382536988983702030964195728302001953125e-"
        "33"},
    {0x1.49aaa6p-108f, chars_format::scientific, 98,
        "3.96821729911656973098277999199730227135040686156682094762847279323381144422455690801143646240234375e-33"},
    {0x1.f5603cp-107f, chars_format::scientific, 97,
        "1.2070186113858457615715036175854047096256968023404067681496332209434285687166266143321990966796875e-32"},
    {0x1.b7bbf8p-106f, chars_format::scientific, 95,
        "2.11724341322508937840915176712265363597160619557838059749677039889093066449277102947235107421875e-32"},
    {0x1.6d305cp-105f, chars_format::scientific, 95,
        "3.51664122601460292174574136500884378151845989294218826175242309517443572985939681529998779296875e-32"},
    {0x1.dd9944p-104f, chars_format::scientific, 94,
        "9.1982162588143308372426801228237163172804681524337790977929874003393706516362726688385009765625e-32"},
    {0x1.0f4254p-103f, chars_format::scientific, 94,
        "1.0448520245617299544506055022011322932281191403904399041258077573957052663899958133697509765625e-31"},
    {0x1.049450p-102f, chars_format::scientific, 91,
        "2.0074302591139273483884727591728498336691676235299559849512007758676190860569477081298828125e-31"},
    {0x1.28d030p-101f, chars_format::scientific, 90,
        "4.573131937693259427041496124538667251427229422876784281637441154089174233376979827880859375e-31"},
    {0x1.28a2bep-100f, chars_format::scientific, 92,
        "9.14079359487553225693590893672771706036816190371857397678478918123801122419536113739013671875e-31"},
    {0x1.e674d2p-99f, chars_format::scientific, 92,
        "2.99801859623548450508972193909865244412972328427583780519061207314734929241240024566650390625e-30"},
    {0x1.227314p-98f, chars_format::scientific, 90,
        "3.580066786954745677669456497979475880433682721056161402106710056614247150719165802001953125e-30"},
    {0x1.735b6cp-97f, chars_format::scientific, 89,
        "9.15465972623783196037990815292079067038371088262625752118850641636527143418788909912109375e-30"},
    {0x1.ef60b4p-96f, chars_format::scientific, 89,
        "2.44240085996903849216356078751341022854748745722804070812372856380534358322620391845703125e-29"},
    {0x1.f58d34p-95f, chars_format::scientific, 88,
        "4.9456803654801575096422210377919300096339643075561698370989915929385460913181304931640625e-29"},
    {0x1.a9fa8ap-94f, chars_format::scientific, 88,
        "8.4009479452815486408032573050798399509585053673323129519445728874416090548038482666015625e-29"},
    {0x1.2ebd9ap-93f, chars_format::scientific, 88,
        "1.1941012414974986903540736041200443068748917913209084407100135649670846760272979736328125e-28"},
    {0x1.1c25bep-92f, chars_format::scientific, 87,
        "2.241527991772840369420073304083191365256388471842441401093992681126110255718231201171875e-28"},
    {0x1.80d526p-91f, chars_format::scientific, 86,
        "6.07158803876554990450680694292368438689814417845436178566842500003986060619354248046875e-28"},
    {0x1.16cdd0p-90f, chars_format::scientific, 82,
        "8.7975016152851199468348749874043115366058394333226289063532021827995777130126953125e-28"},
    {0x1.be00c0p-89f, chars_format::scientific, 80,
        "2.81467419875603623917323685831723008277625852624481694874702952802181243896484375e-27"},
    {0x1.dbe376p-88f, chars_format::scientific, 84,
        "6.006557569745856595501482055708847377292216272726133041715002036653459072113037109375e-27"},
    {0x1.75b358p-87f, chars_format::scientific, 81,
        "9.433528423839338263891493367075328273989136274035871565502020530402660369873046875e-27"},
    {0x1.5e56fap-86f, chars_format::scientific, 83,
        "1.76876373794073549186243776242822499413496518949617808402763330377638339996337890625e-26"},
    {0x1.1542e6p-85f, chars_format::scientific, 82,
        "2.7996239036498255213653797353262236131499034186287389047720353119075298309326171875e-26"},
    {0x1.37b7a6p-84f, chars_format::scientific, 81,
        "6.295082290272475030989309944874260889838800403506269276476814411580562591552734375e-26"},
    {0x1.31f62ap-83f, chars_format::scientific, 81,
        "1.235768973696664669858567539321145118649987459935601918914471752941608428955078125e-25"},
    {0x1.ac3560p-82f, chars_format::scientific, 76,
        "3.4590406845628797200186450581230516131137076030199750675819814205169677734375e-25"},
    {0x1.a7db5cp-81f, chars_format::scientific, 78,
        "6.847777099176331341674240101847394713956151957034990118700079619884490966796875e-25"},
    {0x1.40189cp-80f, chars_format::scientific, 78,
        "1.034286379672715366987641733210033664035198963659922810620628297328948974609375e-24"},
    {0x1.aad1eep-79f, chars_format::scientific, 78,
        "2.758259846499093682487211692864773559218105614121441249153576791286468505859375e-24"},
    {0x1.49824cp-78f, chars_format::scientific, 76,
        "4.2588036474940459085811637121780459484809977510622047702781856060028076171875e-24"},
    {0x1.955292p-77f, chars_format::scientific, 77,
        "1.04773420985315373838626628182169411331037256474019159213639795780181884765625e-23"},
    {0x1.d8ca0cp-76f, chars_format::scientific, 75,
        "2.444263111177596802332967266437459101513507420122550684027373790740966796875e-23"},
    {0x1.28b5aap-75f, chars_format::scientific, 75,
        "3.067905619497844072028707730390270809472941238027487997896969318389892578125e-23"},
    {0x1.e5fda8p-74f, chars_format::scientific, 73,
        "1.0050055115902033206854316793794380802129495577901252545416355133056640625e-22"},
    {0x1.fd929cp-73f, chars_format::scientific, 73,
        "2.1075432611470358337541921610390309449467594049565377645194530487060546875e-22"},
    {0x1.c0b84cp-72f, chars_format::scientific, 72,
        "3.711724097438896357340602997067040280665395357573288492858409881591796875e-22"},
    {0x1.5cfeaep-71f, chars_format::scientific, 72,
        "5.773635352424624465965559338086719731730767080080113373696804046630859375e-22"},
    {0x1.bcce4ap-70f, chars_format::scientific, 72,
        "1.471738991536079335112024613790339400143380998997599817812442779541015625e-21"},
    {0x1.edf106p-69f, chars_format::scientific, 71,
        "3.26863064574260634910627773444362353938430487687583081424236297607421875e-21"},
    {0x1.30b422p-68f, chars_format::scientific, 70,
        "4.0327191475944672156035895296995186232180685692583210766315460205078125e-21"},
    {0x1.7aa8d8p-67f, chars_format::scientific, 68,
        "1.00230347240102665385544432156972316505516573670320212841033935546875e-20"},
    {0x1.4ad4e0p-66f, chars_format::scientific, 65,
        "1.75140760553442509348562143578487138029231573455035686492919921875e-20"},
    {0x1.dde636p-65f, chars_format::scientific, 68,
        "5.05995524921864861016965251964971894693690046551637351512908935546875e-20"},
    {0x1.5df870p-64f, chars_format::scientific, 64,
        "7.4109127331368847396687003781234892585416673682630062103271484375e-20"},
    {0x1.c346fap-63f, chars_format::scientific, 67,
        "1.9112335047873604296656620434025075638828639057464897632598876953125e-19"},
    {0x1.58d2eap-62f, chars_format::scientific, 66,
        "2.920771899491385068938311721231659845443573431111872196197509765625e-19"},
    {0x1.0d4824p-61f, chars_format::scientific, 64,
        "4.5618111223383324851561766710705825289551285095512866973876953125e-19"},
    {0x1.04585cp-60f, chars_format::scientific, 63,
        "8.820836917354691955064048547452415505176759324967861175537109375e-19"},
    {0x1.55cf7ap-59f, chars_format::scientific, 64,
        "2.3161977389916240139687737820128887733517331071197986602783203125e-18"},
    {0x1.1fd8ecp-58f, chars_format::scientific, 62,
        "3.90105904223582084021197668999292318403604440391063690185546875e-18"},
    {0x1.0bc866p-57f, chars_format::scientific, 62,
        "7.25826751123333980988787395016714754092390649020671844482421875e-18"},
    {0x1.4dfa86p-56f, chars_format::scientific, 62,
        "1.81050165732891247031242920595417444928898476064205169677734375e-17"},
    {0x1.335daep-55f, chars_format::scientific, 61,
        "3.3324681586205479543426333233213654239079914987087249755859375e-17"},
    {0x1.5bc756p-54f, chars_format::scientific, 60,
        "7.541247487712833172911197632259927559061907231807708740234375e-17"},
    {0x1.9eb052p-53f, chars_format::scientific, 60,
        "1.798425779915148827771409489884035792783834040164947509765625e-16"},
    {0x1.13b6d2p-52f, chars_format::scientific, 59,
        "2.39143897259270284301468922905087310937233269214630126953125e-16"},
    {0x1.260438p-51f, chars_format::scientific, 56, "5.10037289299151118393549353413618518970906734466552734375e-16"},
    {0x1.9e6b44p-50f, chars_format::scientific, 57, "1.437804758404521467129999479084290214814245700836181640625e-15"},
    {0x1.89c0bcp-49f, chars_format::scientific, 56, "2.73220937993773164975674916377101908437907695770263671875e-15"},
    {0x1.e30610p-48f, chars_format::scientific, 53, "6.70330015995791728133923470522859133780002593994140625e-15"},
    {0x1.48b6e8p-47f, chars_format::scientific, 53, "9.12365953728740131101204724473063834011554718017578125e-15"},
    {0x1.41382ep-46f, chars_format::scientific, 55, "1.7831261573081173821275768887062440626323223114013671875e-14"},
    {0x1.383b8ep-45f, chars_format::scientific, 54, "3.466478609693256218715617933412431739270687103271484375e-14"},
    {0x1.1e6564p-44f, chars_format::scientific, 52, "6.3592699357274684590635160930105485022068023681640625e-14"},
    {0x1.c35e62p-43f, chars_format::scientific, 53, "2.00447961722950707130763703389675356447696685791015625e-13"},
    {0x1.2a2f4ep-42f, chars_format::scientific, 52, "2.6484129017449731247069166784058324992656707763671875e-13"},
    {0x1.69fae2p-41f, chars_format::scientific, 51, "6.430056682417417679431537180789746344089508056640625e-13"},
    {0x1.4ccefep-40f, chars_format::scientific, 51, "1.182373535017766652543969030375592410564422607421875e-12"},
    {0x1.aa9bf6p-39f, chars_format::scientific, 50, "3.03124083993189241681420753593556582927703857421875e-12"},
    {0x1.3b9744p-38f, chars_format::scientific, 48, "4.484816164274096905728583806194365024566650390625e-12"},
    {0x1.b2fc6ap-37f, chars_format::scientific, 49, "1.2363045483188006556929394719190895557403564453125e-11"},
    {0x1.7bc418p-36f, chars_format::scientific, 46, "2.1587197307493255493682227097451686859130859375e-11"},
    {0x1.f4a74cp-35f, chars_format::scientific, 46, "5.6917713597837149563929415307939052581787109375e-11"},
    {0x1.89f248p-34f, chars_format::scientific, 44, "8.95730434269381703416001982986927032470703125e-11"},
    {0x1.60ac54p-33f, chars_format::scientific, 45, "1.603771837555001411601551808416843414306640625e-10"},
    {0x1.2f6d0ep-32f, chars_format::scientific, 45, "2.759643347172158200919511727988719940185546875e-10"},
    {0x1.748684p-31f, chars_format::scientific, 43, "6.7761984912095840627443976700305938720703125e-10"},
    {0x1.b4fa00p-30f, chars_format::scientific, 36, "1.589711473570787347853183746337890625e-09"},
    {0x1.c204d8p-29f, chars_format::scientific, 41, "3.27431859403759517590515315532684326171875e-09"},
    {0x1.50029ep-28f, chars_format::scientific, 42, "4.889592286616561978007666766643524169921875e-09"},
    {0x1.56cf38p-27f, chars_format::scientific, 39, "9.977068060607052757404744625091552734375e-09"},
    {0x1.0b5a5cp-26f, chars_format::scientific, 40, "1.5561990807100301026366651058197021484375e-08"},
    {0x1.fc8250p-25f, chars_format::scientific, 37, "5.9198242752245278097689151763916015625e-08"},
    {0x1.c66674p-24f, chars_format::scientific, 39, "1.057982927932243910618126392364501953125e-07"},
    {0x1.4da57ep-23f, chars_format::scientific, 39, "1.553662372089092968963086605072021484375e-07"},
    {0x1.4fcdacp-22f, chars_format::scientific, 37, "3.1274129241865011863410472869873046875e-07"},
    {0x1.5eaff4p-21f, chars_format::scientific, 36, "6.532060297104180790483951568603515625e-07"},
    {0x1.d2f696p-20f, chars_format::scientific, 37, "1.7395735767422593198716640472412109375e-06"},
    {0x1.e4400cp-19f, chars_format::scientific, 35, "3.60794501830241642892360687255859375e-06"},
    {0x1.03e624p-18f, chars_format::scientific, 34, "3.8727966966689564287662506103515625e-06"},
    {0x1.bdb65ep-17f, chars_format::scientific, 35, "1.32832637973478995263576507568359375e-05"},
    {0x1.57fb84p-16f, chars_format::scientific, 33, "2.050295370281673967838287353515625e-05"},
    {0x1.fd2d62p-15f, chars_format::scientific, 33, "6.069866140023805201053619384765625e-05"},
    {0x1.ca0c58p-14f, chars_format::scientific, 31, "1.0920720524154603481292724609375e-04"},
    {0x1.988f70p-13f, chars_format::scientific, 29, "1.94816733710467815399169921875e-04"},
    {0x1.032dd6p-12f, chars_format::scientific, 31, "2.4717240012250840663909912109375e-04"},
    {0x1.571b08p-11f, chars_format::scientific, 28, "6.5442197956144809722900390625e-04"},
    {0x1.53bedap-10f, chars_format::scientific, 30, "1.296026282943785190582275390625e-03"},
    {0x1.ab2f36p-9f, chars_format::scientific, 29, "3.25915846042335033416748046875e-03"},
    {0x1.7293dap-8f, chars_format::scientific, 28, "5.6545645929872989654541015625e-03"},
    {0x1.825eb6p-7f, chars_format::scientific, 28, "1.1791075579822063446044921875e-02"},
    {0x1.f45aa0p-6f, chars_format::scientific, 23, "3.05391848087310791015625e-02"},
    {0x1.854d96p-5f, chars_format::scientific, 26, "4.75223474204540252685546875e-02"},
    {0x1.5650cep-4f, chars_format::scientific, 25, "8.3573155105113983154296875e-02"},
    {0x1.03acdap-3f, chars_format::scientific, 25, "1.2679453194141387939453125e-01"},
    {0x1.6b9416p-2f, chars_format::scientific, 24, "3.550570905208587646484375e-01"},
    {0x1.a8544ap-1f, chars_format::scientific, 23, "8.28768074512481689453125e-01"},
    {0x1.0693f6p+0f, chars_format::scientific, 23, "1.02569520473480224609375e+00"},
    {0x1.b9476ep+1f, chars_format::scientific, 22, "3.4474923610687255859375e+00"},
    {0x1.3cb752p+2f, chars_format::scientific, 21, "4.948688983917236328125e+00"},
    {0x1.bb8a64p+3f, chars_format::scientific, 20, "1.38606433868408203125e+01"},
    {0x1.1de906p+4f, chars_format::scientific, 20, "1.78693904876708984375e+01"},
    {0x1.d8e834p+5f, chars_format::scientific, 18, "5.911338043212890625e+01"},
    {0x1.27cd38p+6f, chars_format::scientific, 16, "7.3950408935546875e+01"},
    {0x1.3cdcd6p+7f, chars_format::scientific, 18, "1.584313201904296875e+02"},
    {0x1.392656p+8f, chars_format::scientific, 17, "3.13149749755859375e+02"},
    {0x1.c96aa8p+9f, chars_format::scientific, 14, "9.14833251953125e+02"},
    {0x1.28b6b2p+10f, chars_format::scientific, 16, "1.1868546142578125e+03"},
    {0x1.786090p+11f, chars_format::scientific, 12, "3.011017578125e+03"},
    {0x1.79c6f6p+12f, chars_format::scientific, 14, "6.04443505859375e+03"},
    {0x1.ef1840p+13f, chars_format::scientific, 9, "1.584303125e+04"},
    {0x1.539fd0p+14f, chars_format::scientific, 10, "2.1735953125e+04"},
    {0x1.b31804p+15f, chars_format::scientific, 11, "5.56920078125e+04"},
    {0x1.ad4a9cp+16f, chars_format::scientific, 11, "1.09898609375e+05"},
    {0x1.4c43a6p+17f, chars_format::scientific, 11, "1.70119296875e+05"},
    {0x1.5598c6p+18f, chars_format::scientific, 10, "3.4979509375e+05"},
    {0x1.73695ep+19f, chars_format::scientific, 9, "7.606509375e+05"},
    {0x1.234f2ap+20f, chars_format::scientific, 9, "1.193202625e+06"},
    {0x1.0a4cc8p+21f, chars_format::scientific, 6, "2.181529e+06"},
    {0x1.90abd2p+22f, chars_format::scientific, 7, "6.5645965e+06"},
    {0x1.62dde8p+23f, chars_format::scientific, 7, "1.1628276e+07"},
    {0x1.9e3a8cp+24f, chars_format::scientific, 7, "2.7146892e+07"},
    {0x1.53a3eap+25f, chars_format::scientific, 7, "4.4517332e+07"},
    {0x1.41a1cep+26f, chars_format::scientific, 7, "8.4313912e+07"},
    {0x1.8fdda4p+27f, chars_format::scientific, 8, "2.09644832e+08"},
    {0x1.d0322ap+28f, chars_format::scientific, 8, "4.86744736e+08"},
    {0x1.cdb764p+29f, chars_format::scientific, 8, "9.68289408e+08"},
    {0x1.7620d8p+30f, chars_format::scientific, 9, "1.569207808e+09"},
    {0x1.c18df6p+31f, chars_format::scientific, 9, "3.771136768e+09"},
    {0x1.240cf8p+32f, chars_format::scientific, 9, "4.899796992e+09"},
    {0x1.81669ap+33f, chars_format::scientific, 10, "1.2931904512e+10"},
    {0x1.3be30cp+34f, chars_format::scientific, 10, "2.1198811136e+10"},
    {0x1.d1e6e4p+35f, chars_format::scientific, 10, "6.2532296704e+10"},
    {0x1.06b274p+36f, chars_format::scientific, 10, "7.0517211136e+10"},
    {0x1.a74284p+37f, chars_format::scientific, 11, "2.27235889152e+11"},
    {0x1.9fd3e6p+38f, chars_format::scientific, 11, "4.46491623424e+11"},
    {0x1.e2cec4p+39f, chars_format::scientific, 12, "1.036821594112e+12"},
    {0x1.3d5d32p+40f, chars_format::scientific, 11, "1.36306819072e+12"},
    {0x1.accccap+41f, chars_format::scientific, 12, "3.683363586048e+12"},
    {0x1.a120ccp+42f, chars_format::scientific, 12, "7.166206410752e+12"},
    {0x1.55a028p+43f, chars_format::scientific, 13, "1.1738166591488e+13"},
    {0x1.035296p+44f, chars_format::scientific, 13, "1.7820513468416e+13"},
    {0x1.22d1aap+45f, chars_format::scientific, 13, "3.9969859043328e+13"},
    {0x1.eb8eaep+46f, chars_format::scientific, 14, "1.35118253457408e+14"},
    {0x1.490d0ep+47f, chars_format::scientific, 14, "1.80897697497088e+14"},
    {0x1.9da088p+48f, chars_format::scientific, 14, "4.54787778740224e+14"},
    {0x1.e7fab4p+49f, chars_format::scientific, 15, "1.073077848899584e+15"},
    {0x1.98a534p+50f, chars_format::scientific, 14, "1.79724114460672e+15"},
    {0x1.93aeeap+51f, chars_format::scientific, 15, "3.550835489374208e+15"},
    {0x1.3df680p+52f, chars_format::scientific, 15, "5.593662327095296e+15"},
    {0x1.c763f6p+53f, chars_format::scientific, 15, "1.602262782705664e+16"},
    {0x1.8b669ep+54f, chars_format::scientific, 15, "2.782386114789376e+16"},
    {0x1.73e5b6p+55f, chars_format::scientific, 16, "5.2339893103230976e+16"},
    {0x1.a13d18p+56f, chars_format::scientific, 17, "1.17442238576852992e+17"},
    {0x1.a0797ep+57f, chars_format::scientific, 17, "2.34454344768946176e+17"},
    {0x1.c07a80p+58f, chars_format::scientific, 17, "5.04941918963105792e+17"},
    {0x1.729388p+59f, chars_format::scientific, 17, "8.34463629662224384e+17"},
    {0x1.edfb70p+60f, chars_format::scientific, 18, "2.224697951572197376e+18"},
    {0x1.3d6782p+61f, chars_format::scientific, 17, "2.85892402114199552e+18"},
    {0x1.b121e8p+62f, chars_format::scientific, 18, "7.802620494837972992e+18"},
    {0x1.0efc5ap+63f, chars_format::scientific, 18, "9.763290520209063936e+18"},
    {0x1.b7dba0p+64f, chars_format::scientific, 19, "3.1695102724410441728e+19"},
    {0x1.ec2306p+65f, chars_format::scientific, 19, "7.0924388975830368256e+19"},
    {0x1.2e2d28p+66f, chars_format::scientific, 19, "8.7096415015485308928e+19"},
    {0x1.e02208p+67f, chars_format::scientific, 20, "2.76777792668052750336e+20"},
    {0x1.402636p+68f, chars_format::scientific, 20, "3.69106968238077509632e+20"},
    {0x1.11f97cp+69f, chars_format::scientific, 20, "6.31742296991907971072e+20"},
    {0x1.74db2ap+70f, chars_format::scientific, 21, "1.719495307615820316672e+21"},
    {0x1.94a32ap+71f, chars_format::scientific, 21, "3.732120907777931476992e+21"},
    {0x1.c272dcp+72f, chars_format::scientific, 21, "8.309311323384498356224e+21"},
    {0x1.36ca40p+73f, chars_format::scientific, 22, "1.1466128622488263852032e+22"},
    {0x1.5f6fbep+74f, chars_format::scientific, 22, "2.5931436172223350571008e+22"},
    {0x1.95ec4ep+75f, chars_format::scientific, 22, "5.9903671176748022628352e+22"},
    {0x1.6b3912p+76f, chars_format::scientific, 23, "1.07204487170660958732288e+23"},
    {0x1.10992ap+77f, chars_format::scientific, 23, "1.60913632700346331561984e+23"},
    {0x1.74a25ep+78f, chars_format::scientific, 23, "4.39928869395322133020672e+23"},
    {0x1.43f462p+79f, chars_format::scientific, 22, "7.6491622058254812577792e+23"},
    {0x1.f12ca2p+80f, chars_format::scientific, 24, "2.347839472055691035803648e+24"},
    {0x1.2b7f18p+81f, chars_format::scientific, 22, "2.8286640885152838844416e+24"},
    {0x1.a40704p+82f, chars_format::scientific, 24, "7.934093352976572433301504e+24"},
    {0x1.35d5f8p+83f, chars_format::scientific, 23, "1.17052661598219352932352e+25"},
    {0x1.c2c9d2p+84f, chars_format::scientific, 25, "3.4060605519118462894473216e+25"},
    {0x1.47bf20p+85f, chars_format::scientific, 23, "4.95276631635027751337984e+25"},
    {0x1.60b728p+86f, chars_format::scientific, 25, "1.0660170486011939073818624e+26"},
    {0x1.3354c8p+87f, chars_format::scientific, 26, "1.85770297377533474371534848e+26"},
    {0x1.e9e512p+88f, chars_format::scientific, 26, "5.92246479757524141957185536e+26"},
    {0x1.c4b6cap+89f, chars_format::scientific, 27, "1.094595334815995103451021312e+27"},
    {0x1.799cb8p+90f, chars_format::scientific, 27, "1.826020469467809704300249088e+27"},
    {0x1.1afa36p+91f, chars_format::scientific, 27, "2.736789351009782551090823168e+27"},
    {0x1.80c214p+92f, chars_format::scientific, 27, "7.442304364233212615194574848e+27"},
    {0x1.657890p+93f, chars_format::scientific, 28, "1.3828987453168434783077793792e+28"},
    {0x1.5ce17cp+94f, chars_format::scientific, 28, "2.6993344325171312829134798848e+28"},
    {0x1.3f1e9ap+95f, chars_format::scientific, 28, "4.9381356576017938861904625664e+28"},
    {0x1.874612p+96f, chars_format::scientific, 29, "1.21093348650115637567232671744e+29"},
    {0x1.5f4d5ep+97f, chars_format::scientific, 29, "2.17445539275703670631001227264e+29"},
    {0x1.45b1bep+98f, chars_format::scientific, 29, "4.03190021246562727728269754368e+29"},
    {0x1.a570f4p+99f, chars_format::scientific, 30, "1.043437928672039460753056464896e+30"},
    {0x1.f5106ep+100f, chars_format::scientific, 30, "2.481149635102733266542145830912e+30"},
    {0x1.d84424p+101f, chars_format::scientific, 30, "4.677097651091265616934539886592e+30"},
    {0x1.3d6c56p+102f, chars_format::scientific, 30, "6.287213966425746785671183335424e+30"},
    {0x1.9d8cf0p+103f, chars_format::scientific, 31, "1.6382424580981433623378525159424e+31"},
    {0x1.e2e73ep+104f, chars_format::scientific, 29, "3.82595403225449575379724402688e+31"},
    {0x1.2d6594p+105f, chars_format::scientific, 30, "4.775822764761364886543157690368e+31"},
    {0x1.ce43bap+106f, chars_format::scientific, 31, "1.4649748574980240963539344293888e+32"},
    {0x1.b3ea00p+107f, chars_format::scientific, 32, "2.76293361488025452794185737306112e+32"},
    {0x1.03a052p+108f, chars_format::scientific, 32, "3.29115373194929392757058784198656e+32"},
    {0x1.6f59e0p+109f, chars_format::scientific, 31, "9.3134561945576656911623262306304e+32"},
    {0x1.05adacp+110f, chars_format::scientific, 33, "1.326867152522435745601434087849984e+33"},
    {0x1.2cdef0p+111f, chars_format::scientific, 33, "3.051192904788012466473218045116416e+33"},
    {0x1.e81552p+112f, chars_format::scientific, 33, "9.899505055765620068271358482579456e+33"},
    {0x1.bfa8f4p+113f, chars_format::scientific, 34, "1.8159245876954178992833811110166528e+34"},
    {0x1.a14810p+114f, chars_format::scientific, 33, "3.385389673673572296245535418875904e+34"},
    {0x1.f18b10p+115f, chars_format::scientific, 34, "8.0731001914916160681187088757948416e+34"},
    {0x1.8d6e30p+116f, chars_format::scientific, 32, "1.28973545052908058560090358153216e+35"},
    {0x1.9480c2p+117f, chars_format::scientific, 35, "2.62537431192608192877759864086986752e+35"},
    {0x1.60975cp+118f, chars_format::scientific, 34, "4.5768960676134050994895233721827328e+35"},
    {0x1.ab1bb2p+119f, chars_format::scientific, 36, "1.108836243133298765768030079592431616e+36"},
    {0x1.6a0c80p+120f, chars_format::scientific, 36, "1.879864992909653247408339011818749952e+36"},
    {0x1.2cac2cp+121f, chars_format::scientific, 36, "3.122362236102854007005843883842076672e+36"},
    {0x1.0baaf6p+122f, chars_format::scientific, 36, "5.559243043957593079267046257728684032e+36"},
    {0x1.098282p+123f, chars_format::scientific, 35, "1.10288454433706471446366546449924096e+37"},
    {0x1.122f8ap+124f, chars_format::scientific, 37, "2.2778456735621461875293910785310326784e+37"},
    {0x1.57f4c6p+125f, chars_format::scientific, 36, "5.714951736310127067226390054203031552e+37"},
    {0x1.05e028p+126f, chars_format::scientific, 36, "8.702309817313974757087535795024166912e+37"},
    {0x1.9d8424p+127f, chars_format::scientific, 38, "2.74828637805621292108186801756142829568e+38"},

    // Ryu Printf PrintDecimalPoint
    // These values exercise each codepath.
    {0x1.59bc8cp+92f, chars_format::scientific, 0, "7e+27"},
    {0x1.59bc8cp+92f, chars_format::scientific, 1, "6.7e+27"},
    {0x1.37da7cp-30f, chars_format::scientific, 0, "1e-09"},
    {0x1.37da7cp-30f, chars_format::scientific, 1, "1.1e-09"},
    {0x1.834c98p+29f, chars_format::scientific, 0, "8e+08"},
    {0x1.834c98p+29f, chars_format::scientific, 1, "8.1e+08"},

    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffep-126f, chars_format::scientific, 111,
        "1."
        "1754942106924410754870294448492873488270524287458933338571745305715888704756189042655023513361811637878417"
        "96875e-38"},
    {0x1.fffffep-126f, chars_format::scientific, 111,
        "2."
        "3509885615147285834557659820715330266457179855179808553659262368500061299303460771170648513361811637878417"
        "96875e-38"},
    {0x1.fffffep-125f, chars_format::scientific, 110,
        "4."
        "7019771230294571669115319641430660532914359710359617107318524737000122598606921542341297026723623275756835"
        "9375e-38"},
    {0x1.fffffep-124f, chars_format::scientific, 109,
        "9."
        "4039542460589143338230639282861321065828719420719234214637049474000245197213843084682594053447246551513671"
        "875e-38"},
    {0x1.fffffep-123f, chars_format::scientific, 109,
        "1."
        "8807908492117828667646127856572264213165743884143846842927409894800049039442768616936518810689449310302734"
        "375e-37"},
    {0x1.fffffep-122f, chars_format::scientific, 108,
        "3."
        "7615816984235657335292255713144528426331487768287693685854819789600098078885537233873037621378898620605468"
        "75e-37"},
    {0x1.fffffep-121f, chars_format::scientific, 107,
        "7."
        "5231633968471314670584511426289056852662975536575387371709639579200196157771074467746075242757797241210937"
        "5e-37"},
    {0x1.fffffep-120f, chars_format::scientific, 107,
        "1."
        "5046326793694262934116902285257811370532595107315077474341927915840039231554214893549215048551559448242187"
        "5e-36"},
    {0x1.fffffep-119f, chars_format::scientific, 106,
        "3."
        "0092653587388525868233804570515622741065190214630154948683855831680078463108429787098430097103118896484375"
        "e-36"},
    {0x1.fffffep-118f, chars_format::scientific, 105,
        "6."
        "018530717477705173646760914103124548213038042926030989736771166336015692621685957419686019420623779296875e"
        "-36"},
    {0x1.fffffep-117f, chars_format::scientific, 105,
        "1."
        "203706143495541034729352182820624909642607608585206197947354233267203138524337191483937203884124755859375e"
        "-35"},
    {0x1.fffffep-116f, chars_format::scientific, 104,
        "2."
        "40741228699108206945870436564124981928521521717041239589470846653440627704867438296787440776824951171875e-"
        "35"},
    {0x1.fffffep-115f, chars_format::scientific, 103,
        "4."
        "8148245739821641389174087312824996385704304343408247917894169330688125540973487659357488155364990234375e-"
        "35"},
    {0x1.fffffep-114f, chars_format::scientific, 102,
        "9.629649147964328277834817462564999277140860868681649583578833866137625108194697531871497631072998046875e-"
        "35"},
    {0x1.fffffep-113f, chars_format::scientific, 102,
        "1.925929829592865655566963492512999855428172173736329916715766773227525021638939506374299526214599609375e-"
        "34"},
    {0x1.fffffep-112f, chars_format::scientific, 101,
        "3.85185965918573131113392698502599971085634434747265983343153354645505004327787901274859905242919921875e-"
        "34"},
    {0x1.fffffep-111f, chars_format::scientific, 100,
        "7.7037193183714626222678539700519994217126886949453196668630670929101000865557580254971981048583984375e-"
        "34"},
    {0x1.fffffep-110f, chars_format::scientific, 100,
        "1.5407438636742925244535707940103998843425377389890639333726134185820200173111516050994396209716796875e-"
        "33"},
    {0x1.fffffep-109f, chars_format::scientific, 99,
        "3.081487727348585048907141588020799768685075477978127866745226837164040034622303210198879241943359375e-"
        "33"},
    {0x1.fffffep-108f, chars_format::scientific, 98,
        "6.16297545469717009781428317604159953737015095595625573349045367432808006924460642039775848388671875e-33"},
    {0x1.fffffep-107f, chars_format::scientific, 98,
        "1.23259509093943401956285663520831990747403019119125114669809073486561601384892128407955169677734375e-32"},
    {0x1.fffffep-106f, chars_format::scientific, 97,
        "2.4651901818788680391257132704166398149480603823825022933961814697312320276978425681591033935546875e-32"},
    {0x1.fffffep-105f, chars_format::scientific, 96,
        "4.930380363757736078251426540833279629896120764765004586792362939462464055395685136318206787109375e-32"},
    {0x1.fffffep-104f, chars_format::scientific, 95,
        "9.86076072751547215650285308166655925979224152953000917358472587892492811079137027263641357421875e-32"},
    {0x1.fffffep-103f, chars_format::scientific, 95,
        "1.97215214550309443130057061633331185195844830590600183471694517578498562215827405452728271484375e-31"},
    {0x1.fffffep-102f, chars_format::scientific, 94,
        "3.9443042910061888626011412326666237039168966118120036694338903515699712443165481090545654296875e-31"},
    {0x1.fffffep-101f, chars_format::scientific, 93,
        "7.888608582012377725202282465333247407833793223624007338867780703139942488633096218109130859375e-31"},
    {0x1.fffffep-100f, chars_format::scientific, 93,
        "1.577721716402475545040456493066649481566758644724801467773556140627988497726619243621826171875e-30"},
    {0x1.fffffep-99f, chars_format::scientific, 92,
        "3.15544343280495109008091298613329896313351728944960293554711228125597699545323848724365234375e-30"},
    {0x1.fffffep-98f, chars_format::scientific, 91,
        "6.3108868656099021801618259722665979262670345788992058710942245625119539909064769744873046875e-30"},
    {0x1.fffffep-97f, chars_format::scientific, 91,
        "1.2621773731219804360323651944533195852534069157798411742188449125023907981812953948974609375e-29"},
    {0x1.fffffep-96f, chars_format::scientific, 90,
        "2.524354746243960872064730388906639170506813831559682348437689825004781596362590789794921875e-29"},
    {0x1.fffffep-95f, chars_format::scientific, 89,
        "5.04870949248792174412946077781327834101362766311936469687537965000956319272518157958984375e-29"},
    {0x1.fffffep-94f, chars_format::scientific, 89,
        "1.00974189849758434882589215556265566820272553262387293937507593000191263854503631591796875e-28"},
    {0x1.fffffep-93f, chars_format::scientific, 88,
        "2.0194837969951686976517843111253113364054510652477458787501518600038252770900726318359375e-28"},
    {0x1.fffffep-92f, chars_format::scientific, 87,
        "4.038967593990337395303568622250622672810902130495491757500303720007650554180145263671875e-28"},
    {0x1.fffffep-91f, chars_format::scientific, 86,
        "8.07793518798067479060713724450124534562180426099098351500060744001530110836029052734375e-28"},
    {0x1.fffffep-90f, chars_format::scientific, 86,
        "1.61558703759613495812142744890024906912436085219819670300012148800306022167205810546875e-27"},
    {0x1.fffffep-89f, chars_format::scientific, 85,
        "3.2311740751922699162428548978004981382487217043963934060002429760061204433441162109375e-27"},
    {0x1.fffffep-88f, chars_format::scientific, 84,
        "6.462348150384539832485709795600996276497443408792786812000485952012240886688232421875e-27"},
    {0x1.fffffep-87f, chars_format::scientific, 84,
        "1.292469630076907966497141959120199255299488681758557362400097190402448177337646484375e-26"},
    {0x1.fffffep-86f, chars_format::scientific, 83,
        "2.58493926015381593299428391824039851059897736351711472480019438080489635467529296875e-26"},
    {0x1.fffffep-85f, chars_format::scientific, 82,
        "5.1698785203076318659885678364807970211979547270342294496003887616097927093505859375e-26"},
    {0x1.fffffep-84f, chars_format::scientific, 82,
        "1.0339757040615263731977135672961594042395909454068458899200777523219585418701171875e-25"},
    {0x1.fffffep-83f, chars_format::scientific, 81,
        "2.067951408123052746395427134592318808479181890813691779840155504643917083740234375e-25"},
    {0x1.fffffep-82f, chars_format::scientific, 80,
        "4.13590281624610549279085426918463761695836378162738355968031100928783416748046875e-25"},
    {0x1.fffffep-81f, chars_format::scientific, 79,
        "8.2718056324922109855817085383692752339167275632547671193606220185756683349609375e-25"},
    {0x1.fffffep-80f, chars_format::scientific, 79,
        "1.6543611264984421971163417076738550467833455126509534238721244037151336669921875e-24"},
    {0x1.fffffep-79f, chars_format::scientific, 78,
        "3.308722252996884394232683415347710093566691025301906847744248807430267333984375e-24"},
    {0x1.fffffep-78f, chars_format::scientific, 77,
        "6.61744450599376878846536683069542018713338205060381369548849761486053466796875e-24"},
    {0x1.fffffep-77f, chars_format::scientific, 77,
        "1.32348890119875375769307336613908403742667641012076273909769952297210693359375e-23"},
    {0x1.fffffep-76f, chars_format::scientific, 76,
        "2.6469778023975075153861467322781680748533528202415254781953990459442138671875e-23"},
    {0x1.fffffep-75f, chars_format::scientific, 75,
        "5.293955604795015030772293464556336149706705640483050956390798091888427734375e-23"},
    {0x1.fffffep-74f, chars_format::scientific, 75,
        "1.058791120959003006154458692911267229941341128096610191278159618377685546875e-22"},
    {0x1.fffffep-73f, chars_format::scientific, 74,
        "2.11758224191800601230891738582253445988268225619322038255631923675537109375e-22"},
    {0x1.fffffep-72f, chars_format::scientific, 73,
        "4.2351644838360120246178347716450689197653645123864407651126384735107421875e-22"},
    {0x1.fffffep-71f, chars_format::scientific, 72,
        "8.470328967672024049235669543290137839530729024772881530225276947021484375e-22"},
    {0x1.fffffep-70f, chars_format::scientific, 72,
        "1.694065793534404809847133908658027567906145804954576306045055389404296875e-21"},
    {0x1.fffffep-69f, chars_format::scientific, 71,
        "3.38813158706880961969426781731605513581229160990915261209011077880859375e-21"},
    {0x1.fffffep-68f, chars_format::scientific, 70,
        "6.7762631741376192393885356346321102716245832198183052241802215576171875e-21"},
    {0x1.fffffep-67f, chars_format::scientific, 70,
        "1.3552526348275238478777071269264220543249166439636610448360443115234375e-20"},
    {0x1.fffffep-66f, chars_format::scientific, 69,
        "2.710505269655047695755414253852844108649833287927322089672088623046875e-20"},
    {0x1.fffffep-65f, chars_format::scientific, 68,
        "5.42101053931009539151082850770568821729966657585464417934417724609375e-20"},
    {0x1.fffffep-64f, chars_format::scientific, 68,
        "1.08420210786201907830216570154113764345993331517092883586883544921875e-19"},
    {0x1.fffffep-63f, chars_format::scientific, 67,
        "2.1684042157240381566043314030822752869198666303418576717376708984375e-19"},
    {0x1.fffffep-62f, chars_format::scientific, 66,
        "4.336808431448076313208662806164550573839733260683715343475341796875e-19"},
    {0x1.fffffep-61f, chars_format::scientific, 65,
        "8.67361686289615262641732561232910114767946652136743068695068359375e-19"},
    {0x1.fffffep-60f, chars_format::scientific, 65,
        "1.73472337257923052528346512246582022953589330427348613739013671875e-18"},
    {0x1.fffffep-59f, chars_format::scientific, 64,
        "3.4694467451584610505669302449316404590717866085469722747802734375e-18"},
    {0x1.fffffep-58f, chars_format::scientific, 63,
        "6.938893490316922101133860489863280918143573217093944549560546875e-18"},
    {0x1.fffffep-57f, chars_format::scientific, 63,
        "1.387778698063384420226772097972656183628714643418788909912109375e-17"},
    {0x1.fffffep-56f, chars_format::scientific, 62,
        "2.77555739612676884045354419594531236725742928683757781982421875e-17"},
    {0x1.fffffep-55f, chars_format::scientific, 61,
        "5.5511147922535376809070883918906247345148585736751556396484375e-17"},
    {0x1.fffffep-54f, chars_format::scientific, 61,
        "1.1102229584507075361814176783781249469029717147350311279296875e-16"},
    {0x1.fffffep-53f, chars_format::scientific, 60,
        "2.220445916901415072362835356756249893805943429470062255859375e-16"},
    {0x1.fffffep-52f, chars_format::scientific, 59,
        "4.44089183380283014472567071351249978761188685894012451171875e-16"},
    {0x1.fffffep-51f, chars_format::scientific, 58, "8.8817836676056602894513414270249995752237737178802490234375e-16"},
    {0x1.fffffep-50f, chars_format::scientific, 58, "1.7763567335211320578902682854049999150447547435760498046875e-15"},
    {0x1.fffffep-49f, chars_format::scientific, 57, "3.552713467042264115780536570809999830089509487152099609375e-15"},
    {0x1.fffffep-48f, chars_format::scientific, 56, "7.10542693408452823156107314161999966017901897430419921875e-15"},
    {0x1.fffffep-47f, chars_format::scientific, 56, "1.42108538681690564631221462832399993203580379486083984375e-14"},
    {0x1.fffffep-46f, chars_format::scientific, 55, "2.8421707736338112926244292566479998640716075897216796875e-14"},
    {0x1.fffffep-45f, chars_format::scientific, 54, "5.684341547267622585248858513295999728143215179443359375e-14"},
    {0x1.fffffep-44f, chars_format::scientific, 54, "1.136868309453524517049771702659199945628643035888671875e-13"},
    {0x1.fffffep-43f, chars_format::scientific, 53, "2.27373661890704903409954340531839989125728607177734375e-13"},
    {0x1.fffffep-42f, chars_format::scientific, 52, "4.5474732378140980681990868106367997825145721435546875e-13"},
    {0x1.fffffep-41f, chars_format::scientific, 51, "9.094946475628196136398173621273599565029144287109375e-13"},
    {0x1.fffffep-40f, chars_format::scientific, 51, "1.818989295125639227279634724254719913005828857421875e-12"},
    {0x1.fffffep-39f, chars_format::scientific, 50, "3.63797859025127845455926944850943982601165771484375e-12"},
    {0x1.fffffep-38f, chars_format::scientific, 49, "7.2759571805025569091185388970188796520233154296875e-12"},
    {0x1.fffffep-37f, chars_format::scientific, 49, "1.4551914361005113818237077794037759304046630859375e-11"},
    {0x1.fffffep-36f, chars_format::scientific, 48, "2.910382872201022763647415558807551860809326171875e-11"},
    {0x1.fffffep-35f, chars_format::scientific, 47, "5.82076574440204552729483111761510372161865234375e-11"},
    {0x1.fffffep-34f, chars_format::scientific, 47, "1.16415314888040910545896622352302074432373046875e-10"},
    {0x1.fffffep-33f, chars_format::scientific, 46, "2.3283062977608182109179324470460414886474609375e-10"},
    {0x1.fffffep-32f, chars_format::scientific, 45, "4.656612595521636421835864894092082977294921875e-10"},
    {0x1.fffffep-31f, chars_format::scientific, 44, "9.31322519104327284367172978818416595458984375e-10"},
    {0x1.fffffep-30f, chars_format::scientific, 44, "1.86264503820865456873434595763683319091796875e-09"},
    {0x1.fffffep-29f, chars_format::scientific, 43, "3.7252900764173091374686919152736663818359375e-09"},
    {0x1.fffffep-28f, chars_format::scientific, 42, "7.450580152834618274937383830547332763671875e-09"},
    {0x1.fffffep-27f, chars_format::scientific, 42, "1.490116030566923654987476766109466552734375e-08"},
    {0x1.fffffep-26f, chars_format::scientific, 41, "2.98023206113384730997495353221893310546875e-08"},
    {0x1.fffffep-25f, chars_format::scientific, 40, "5.9604641222676946199499070644378662109375e-08"},
    {0x1.fffffep-24f, chars_format::scientific, 40, "1.1920928244535389239899814128875732421875e-07"},
    {0x1.fffffep-23f, chars_format::scientific, 39, "2.384185648907077847979962825775146484375e-07"},
    {0x1.fffffep-22f, chars_format::scientific, 38, "4.76837129781415569595992565155029296875e-07"},
    {0x1.fffffep-21f, chars_format::scientific, 37, "9.5367425956283113919198513031005859375e-07"},
    {0x1.fffffep-20f, chars_format::scientific, 37, "1.9073485191256622783839702606201171875e-06"},
    {0x1.fffffep-19f, chars_format::scientific, 36, "3.814697038251324556767940521240234375e-06"},
    {0x1.fffffep-18f, chars_format::scientific, 35, "7.62939407650264911353588104248046875e-06"},
    {0x1.fffffep-17f, chars_format::scientific, 35, "1.52587881530052982270717620849609375e-05"},
    {0x1.fffffep-16f, chars_format::scientific, 34, "3.0517576306010596454143524169921875e-05"},
    {0x1.fffffep-15f, chars_format::scientific, 33, "6.103515261202119290828704833984375e-05"},
    {0x1.fffffep-14f, chars_format::scientific, 33, "1.220703052240423858165740966796875e-04"},
    {0x1.fffffep-13f, chars_format::scientific, 32, "2.44140610448084771633148193359375e-04"},
    {0x1.fffffep-12f, chars_format::scientific, 31, "4.8828122089616954326629638671875e-04"},
    {0x1.fffffep-11f, chars_format::scientific, 30, "9.765624417923390865325927734375e-04"},
    {0x1.fffffep-10f, chars_format::scientific, 30, "1.953124883584678173065185546875e-03"},
    {0x1.fffffep-9f, chars_format::scientific, 29, "3.90624976716935634613037109375e-03"},
    {0x1.fffffep-8f, chars_format::scientific, 28, "7.8124995343387126922607421875e-03"},
    {0x1.fffffep-7f, chars_format::scientific, 28, "1.5624999068677425384521484375e-02"},
    {0x1.fffffep-6f, chars_format::scientific, 27, "3.124999813735485076904296875e-02"},
    {0x1.fffffep-5f, chars_format::scientific, 26, "6.24999962747097015380859375e-02"},
    {0x1.fffffep-4f, chars_format::scientific, 26, "1.24999992549419403076171875e-01"},
    {0x1.fffffep-3f, chars_format::scientific, 25, "2.4999998509883880615234375e-01"},
    {0x1.fffffep-2f, chars_format::scientific, 24, "4.999999701976776123046875e-01"},
    {0x1.fffffep-1f, chars_format::scientific, 23, "9.99999940395355224609375e-01"},
    {0x1.fffffep+0f, chars_format::scientific, 23, "1.99999988079071044921875e+00"},
    {0x1.fffffep+1f, chars_format::scientific, 22, "3.9999997615814208984375e+00"},
    {0x1.fffffep+2f, chars_format::scientific, 21, "7.999999523162841796875e+00"},
    {0x1.fffffep+3f, chars_format::scientific, 21, "1.599999904632568359375e+01"},
    {0x1.fffffep+4f, chars_format::scientific, 20, "3.19999980926513671875e+01"},
    {0x1.fffffep+5f, chars_format::scientific, 19, "6.3999996185302734375e+01"},
    {0x1.fffffep+6f, chars_format::scientific, 19, "1.2799999237060546875e+02"},
    {0x1.fffffep+7f, chars_format::scientific, 18, "2.559999847412109375e+02"},
    {0x1.fffffep+8f, chars_format::scientific, 17, "5.11999969482421875e+02"},
    {0x1.fffffep+9f, chars_format::scientific, 17, "1.02399993896484375e+03"},
    {0x1.fffffep+10f, chars_format::scientific, 16, "2.0479998779296875e+03"},
    {0x1.fffffep+11f, chars_format::scientific, 15, "4.095999755859375e+03"},
    {0x1.fffffep+12f, chars_format::scientific, 14, "8.19199951171875e+03"},
    {0x1.fffffep+13f, chars_format::scientific, 14, "1.63839990234375e+04"},
    {0x1.fffffep+14f, chars_format::scientific, 13, "3.2767998046875e+04"},
    {0x1.fffffep+15f, chars_format::scientific, 12, "6.553599609375e+04"},
    {0x1.fffffep+16f, chars_format::scientific, 12, "1.310719921875e+05"},
    {0x1.fffffep+17f, chars_format::scientific, 11, "2.62143984375e+05"},
    {0x1.fffffep+18f, chars_format::scientific, 10, "5.2428796875e+05"},
    {0x1.fffffep+19f, chars_format::scientific, 10, "1.0485759375e+06"},
    {0x1.fffffep+20f, chars_format::scientific, 9, "2.097151875e+06"},
    {0x1.fffffep+21f, chars_format::scientific, 8, "4.19430375e+06"},
    {0x1.fffffep+22f, chars_format::scientific, 7, "8.3886075e+06"},
    {0x1.fffffep+23f, chars_format::scientific, 7, "1.6777215e+07"},
    {0x1.fffffep+24f, chars_format::scientific, 6, "3.355443e+07"},
    {0x1.fffffep+25f, chars_format::scientific, 6, "6.710886e+07"},
    {0x1.fffffep+26f, chars_format::scientific, 7, "1.3421772e+08"},
    {0x1.fffffep+27f, chars_format::scientific, 7, "2.6843544e+08"},
    {0x1.fffffep+28f, chars_format::scientific, 7, "5.3687088e+08"},
    {0x1.fffffep+29f, chars_format::scientific, 8, "1.07374176e+09"},
    {0x1.fffffep+30f, chars_format::scientific, 8, "2.14748352e+09"},
    {0x1.fffffep+31f, chars_format::scientific, 8, "4.29496704e+09"},
    {0x1.fffffep+32f, chars_format::scientific, 8, "8.58993408e+09"},
    {0x1.fffffep+33f, chars_format::scientific, 9, "1.717986816e+10"},
    {0x1.fffffep+34f, chars_format::scientific, 9, "3.435973632e+10"},
    {0x1.fffffep+35f, chars_format::scientific, 9, "6.871947264e+10"},
    {0x1.fffffep+36f, chars_format::scientific, 10, "1.3743894528e+11"},
    {0x1.fffffep+37f, chars_format::scientific, 10, "2.7487789056e+11"},
    {0x1.fffffep+38f, chars_format::scientific, 10, "5.4975578112e+11"},
    {0x1.fffffep+39f, chars_format::scientific, 11, "1.09951156224e+12"},
    {0x1.fffffep+40f, chars_format::scientific, 11, "2.19902312448e+12"},
    {0x1.fffffep+41f, chars_format::scientific, 11, "4.39804624896e+12"},
    {0x1.fffffep+42f, chars_format::scientific, 11, "8.79609249792e+12"},
    {0x1.fffffep+43f, chars_format::scientific, 12, "1.759218499584e+13"},
    {0x1.fffffep+44f, chars_format::scientific, 12, "3.518436999168e+13"},
    {0x1.fffffep+45f, chars_format::scientific, 12, "7.036873998336e+13"},
    {0x1.fffffep+46f, chars_format::scientific, 13, "1.4073747996672e+14"},
    {0x1.fffffep+47f, chars_format::scientific, 13, "2.8147495993344e+14"},
    {0x1.fffffep+48f, chars_format::scientific, 13, "5.6294991986688e+14"},
    {0x1.fffffep+49f, chars_format::scientific, 14, "1.12589983973376e+15"},
    {0x1.fffffep+50f, chars_format::scientific, 14, "2.25179967946752e+15"},
    {0x1.fffffep+51f, chars_format::scientific, 14, "4.50359935893504e+15"},
    {0x1.fffffep+52f, chars_format::scientific, 14, "9.00719871787008e+15"},
    {0x1.fffffep+53f, chars_format::scientific, 15, "1.801439743574016e+16"},
    {0x1.fffffep+54f, chars_format::scientific, 15, "3.602879487148032e+16"},
    {0x1.fffffep+55f, chars_format::scientific, 15, "7.205758974296064e+16"},
    {0x1.fffffep+56f, chars_format::scientific, 16, "1.4411517948592128e+17"},
    {0x1.fffffep+57f, chars_format::scientific, 16, "2.8823035897184256e+17"},
    {0x1.fffffep+58f, chars_format::scientific, 16, "5.7646071794368512e+17"},
    {0x1.fffffep+59f, chars_format::scientific, 17, "1.15292143588737024e+18"},
    {0x1.fffffep+60f, chars_format::scientific, 17, "2.30584287177474048e+18"},
    {0x1.fffffep+61f, chars_format::scientific, 17, "4.61168574354948096e+18"},
    {0x1.fffffep+62f, chars_format::scientific, 17, "9.22337148709896192e+18"},
    {0x1.fffffep+63f, chars_format::scientific, 18, "1.844674297419792384e+19"},
    {0x1.fffffep+64f, chars_format::scientific, 18, "3.689348594839584768e+19"},
    {0x1.fffffep+65f, chars_format::scientific, 18, "7.378697189679169536e+19"},
    {0x1.fffffep+66f, chars_format::scientific, 19, "1.4757394379358339072e+20"},
    {0x1.fffffep+67f, chars_format::scientific, 19, "2.9514788758716678144e+20"},
    {0x1.fffffep+68f, chars_format::scientific, 19, "5.9029577517433356288e+20"},
    {0x1.fffffep+69f, chars_format::scientific, 20, "1.18059155034866712576e+21"},
    {0x1.fffffep+70f, chars_format::scientific, 20, "2.36118310069733425152e+21"},
    {0x1.fffffep+71f, chars_format::scientific, 20, "4.72236620139466850304e+21"},
    {0x1.fffffep+72f, chars_format::scientific, 20, "9.44473240278933700608e+21"},
    {0x1.fffffep+73f, chars_format::scientific, 21, "1.888946480557867401216e+22"},
    {0x1.fffffep+74f, chars_format::scientific, 21, "3.777892961115734802432e+22"},
    {0x1.fffffep+75f, chars_format::scientific, 21, "7.555785922231469604864e+22"},
    {0x1.fffffep+76f, chars_format::scientific, 22, "1.5111571844462939209728e+23"},
    {0x1.fffffep+77f, chars_format::scientific, 22, "3.0223143688925878419456e+23"},
    {0x1.fffffep+78f, chars_format::scientific, 22, "6.0446287377851756838912e+23"},
    {0x1.fffffep+79f, chars_format::scientific, 23, "1.20892574755703513677824e+24"},
    {0x1.fffffep+80f, chars_format::scientific, 23, "2.41785149511407027355648e+24"},
    {0x1.fffffep+81f, chars_format::scientific, 23, "4.83570299022814054711296e+24"},
    {0x1.fffffep+82f, chars_format::scientific, 23, "9.67140598045628109422592e+24"},
    {0x1.fffffep+83f, chars_format::scientific, 24, "1.934281196091256218845184e+25"},
    {0x1.fffffep+84f, chars_format::scientific, 24, "3.868562392182512437690368e+25"},
    {0x1.fffffep+85f, chars_format::scientific, 24, "7.737124784365024875380736e+25"},
    {0x1.fffffep+86f, chars_format::scientific, 25, "1.5474249568730049750761472e+26"},
    {0x1.fffffep+87f, chars_format::scientific, 25, "3.0948499137460099501522944e+26"},
    {0x1.fffffep+88f, chars_format::scientific, 25, "6.1896998274920199003045888e+26"},
    {0x1.fffffep+89f, chars_format::scientific, 26, "1.23793996549840398006091776e+27"},
    {0x1.fffffep+90f, chars_format::scientific, 26, "2.47587993099680796012183552e+27"},
    {0x1.fffffep+91f, chars_format::scientific, 26, "4.95175986199361592024367104e+27"},
    {0x1.fffffep+92f, chars_format::scientific, 26, "9.90351972398723184048734208e+27"},
    {0x1.fffffep+93f, chars_format::scientific, 27, "1.980703944797446368097468416e+28"},
    {0x1.fffffep+94f, chars_format::scientific, 27, "3.961407889594892736194936832e+28"},
    {0x1.fffffep+95f, chars_format::scientific, 27, "7.922815779189785472389873664e+28"},
    {0x1.fffffep+96f, chars_format::scientific, 28, "1.5845631558379570944779747328e+29"},
    {0x1.fffffep+97f, chars_format::scientific, 28, "3.1691263116759141889559494656e+29"},
    {0x1.fffffep+98f, chars_format::scientific, 28, "6.3382526233518283779118989312e+29"},
    {0x1.fffffep+99f, chars_format::scientific, 29, "1.26765052467036567558237978624e+30"},
    {0x1.fffffep+100f, chars_format::scientific, 29, "2.53530104934073135116475957248e+30"},
    {0x1.fffffep+101f, chars_format::scientific, 29, "5.07060209868146270232951914496e+30"},
    {0x1.fffffep+102f, chars_format::scientific, 30, "1.014120419736292540465903828992e+31"},
    {0x1.fffffep+103f, chars_format::scientific, 30, "2.028240839472585080931807657984e+31"},
    {0x1.fffffep+104f, chars_format::scientific, 30, "4.056481678945170161863615315968e+31"},
    {0x1.fffffep+105f, chars_format::scientific, 30, "8.112963357890340323727230631936e+31"},
    {0x1.fffffep+106f, chars_format::scientific, 31, "1.6225926715780680647454461263872e+32"},
    {0x1.fffffep+107f, chars_format::scientific, 31, "3.2451853431561361294908922527744e+32"},
    {0x1.fffffep+108f, chars_format::scientific, 31, "6.4903706863122722589817845055488e+32"},
    {0x1.fffffep+109f, chars_format::scientific, 32, "1.29807413726245445179635690110976e+33"},
    {0x1.fffffep+110f, chars_format::scientific, 32, "2.59614827452490890359271380221952e+33"},
    {0x1.fffffep+111f, chars_format::scientific, 32, "5.19229654904981780718542760443904e+33"},
    {0x1.fffffep+112f, chars_format::scientific, 33, "1.038459309809963561437085520887808e+34"},
    {0x1.fffffep+113f, chars_format::scientific, 33, "2.076918619619927122874171041775616e+34"},
    {0x1.fffffep+114f, chars_format::scientific, 33, "4.153837239239854245748342083551232e+34"},
    {0x1.fffffep+115f, chars_format::scientific, 33, "8.307674478479708491496684167102464e+34"},
    {0x1.fffffep+116f, chars_format::scientific, 34, "1.6615348956959416982993368334204928e+35"},
    {0x1.fffffep+117f, chars_format::scientific, 34, "3.3230697913918833965986736668409856e+35"},
    {0x1.fffffep+118f, chars_format::scientific, 34, "6.6461395827837667931973473336819712e+35"},
    {0x1.fffffep+119f, chars_format::scientific, 35, "1.32922791655675335863946946673639424e+36"},
    {0x1.fffffep+120f, chars_format::scientific, 35, "2.65845583311350671727893893347278848e+36"},
    {0x1.fffffep+121f, chars_format::scientific, 35, "5.31691166622701343455787786694557696e+36"},
    {0x1.fffffep+122f, chars_format::scientific, 36, "1.063382333245402686911575573389115392e+37"},
    {0x1.fffffep+123f, chars_format::scientific, 36, "2.126764666490805373823151146778230784e+37"},
    {0x1.fffffep+124f, chars_format::scientific, 36, "4.253529332981610747646302293556461568e+37"},
    {0x1.fffffep+125f, chars_format::scientific, 36, "8.507058665963221495292604587112923136e+37"},
    {0x1.fffffep+126f, chars_format::scientific, 37, "1.7014117331926442990585209174225846272e+38"},
    {0x1.fffffep+127f, chars_format::scientific, 37, "3.4028234663852885981170418348451692544e+38"},
};

inline constexpr float_to_chars_testcase float_general_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::general, 4, "0"},
    {-0.0f, chars_format::general, 4, "-0"},
    {float_inf, chars_format::general, 4, "inf"},
    {-float_inf, chars_format::general, 4, "-inf"},
    {float_nan, chars_format::general, 4, "nan"},
    {-float_nan, chars_format::general, 4, "-nan"},
    {1.729f, chars_format::general, 4, "1.729"},
    {-1.729f, chars_format::general, 4, "-1.729"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::general, 1000,
        "1.40129846432481707092372958328991613128026194187651577175706828388979108268586060148663818836212158203125"
        "e-45"}, // min subnormal
    {0x0.fffffep-126f, chars_format::general, 1000,
        "1."
        "17549421069244107548702944484928734882705242874589333385717453057158887047561890426550235133618116378784179687"
        "5e-38"}, // max subnormal
    {0x1p-126f, chars_format::general, 1000,
        "1.1754943508222875079687365372222456778186655567720875215087517062784172594547271728515625e-38"}, // min normal
    {0x1.fffffep+127f, chars_format::general, 1000, "340282346638528859811704183484516925440"}, // max normal

    {0x0.000002p-126f, chars_format::general, 6, "1.4013e-45"}, // min subnormal
    {0x0.fffffep-126f, chars_format::general, 6, "1.17549e-38"}, // max subnormal
    {0x1p-126f, chars_format::general, 6, "1.17549e-38"}, // min normal
    {0x1.fffffep+127f, chars_format::general, 6, "3.40282e+38"}, // max normal

    // Test maximum-length output (excluding minus signs).
    {0x1.fffffep-126f, chars_format::general, 1000,
        "2."
        "35098856151472858345576598207153302664571798551798085536592623685000612993034607711706485133618116378784179687"
        "5e-38"}, // scientific, happens to be the same length as max subnormal
    {0x1.fffffep-14f, chars_format::general, 1000, "0.0001220703052240423858165740966796875"}, // fixed

    // Test varying precision. Negative precision requests P == 6. Zero precision requests P == 1.
    // Here, the scientific exponent X is 0.
    // Therefore, fixed notation is always chosen with precision P - (X + 1) == P - 1.
    {0x1.b04p0f, chars_format::general, -2, "1.68848"},
    {0x1.b04p0f, chars_format::general, -1, "1.68848"},
    {0x1.b04p0f, chars_format::general, 0, "2"},
    {0x1.b04p0f, chars_format::general, 1, "2"}, // fixed notation trims decimal point
    {0x1.b04p0f, chars_format::general, 2, "1.7"},
    {0x1.b04p0f, chars_format::general, 3, "1.69"},
    {0x1.b04p0f, chars_format::general, 4, "1.688"},
    {0x1.b04p0f, chars_format::general, 5, "1.6885"},
    {0x1.b04p0f, chars_format::general, 6, "1.68848"},
    {0x1.b04p0f, chars_format::general, 7, "1.688477"},
    {0x1.b04p0f, chars_format::general, 8, "1.6884766"},
    {0x1.b04p0f, chars_format::general, 9, "1.68847656"},
    {0x1.b04p0f, chars_format::general, 10, "1.688476562"}, // round to even
    {0x1.b04p0f, chars_format::general, 11, "1.6884765625"}, // exact
    {0x1.b04p0f, chars_format::general, 12, "1.6884765625"}, // trim trailing zeros
    {0x1.b04p0f, chars_format::general, 13, "1.6884765625"},

    // Here, the scientific exponent X is -5.
    // Therefore, scientific notation is always chosen with precision P - 1.
    {0x1.8p-15f, chars_format::general, -2, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, -1, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, 0, "5e-05"},
    {0x1.8p-15f, chars_format::general, 1, "5e-05"}, // scientific notation trims decimal point
    {0x1.8p-15f, chars_format::general, 2, "4.6e-05"},
    {0x1.8p-15f, chars_format::general, 3, "4.58e-05"},
    {0x1.8p-15f, chars_format::general, 4, "4.578e-05"},
    {0x1.8p-15f, chars_format::general, 5, "4.5776e-05"},
    {0x1.8p-15f, chars_format::general, 6, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, 7, "4.577637e-05"},
    {0x1.8p-15f, chars_format::general, 8, "4.5776367e-05"},
    {0x1.8p-15f, chars_format::general, 9, "4.57763672e-05"},
    {0x1.8p-15f, chars_format::general, 10, "4.577636719e-05"},
    {0x1.8p-15f, chars_format::general, 11, "4.5776367188e-05"}, // round to even
    {0x1.8p-15f, chars_format::general, 12, "4.57763671875e-05"}, // exact
    {0x1.8p-15f, chars_format::general, 13, "4.57763671875e-05"}, // trim trailing zeros
    {0x1.8p-15f, chars_format::general, 14, "4.57763671875e-05"},

    // Trim trailing zeros.
    {0x1.80015p0f, chars_format::general, 1, "2"}, // fixed notation trims decimal point
    {0x1.80015p0f, chars_format::general, 2, "1.5"},
    {0x1.80015p0f, chars_format::general, 3, "1.5"}, // general trims trailing zeros
    {0x1.80015p0f, chars_format::general, 4, "1.5"},
    {0x1.80015p0f, chars_format::general, 5, "1.5"},
    {0x1.80015p0f, chars_format::general, 6, "1.50002"},
    {0x1.80015p0f, chars_format::general, 7, "1.50002"},
    {0x1.80015p0f, chars_format::general, 8, "1.50002"},
    {0x1.80015p0f, chars_format::general, 9, "1.50002003"},
    {0x1.80015p0f, chars_format::general, 10, "1.500020027"},
    {0x1.80015p0f, chars_format::general, 11, "1.5000200272"},
    {0x1.80015p0f, chars_format::general, 12, "1.50002002716"},
    {0x1.80015p0f, chars_format::general, 13, "1.500020027161"},
    {0x1.80015p0f, chars_format::general, 14, "1.5000200271606"},
    {0x1.80015p0f, chars_format::general, 15, "1.50002002716064"},
    {0x1.80015p0f, chars_format::general, 16, "1.500020027160645"},
    {0x1.80015p0f, chars_format::general, 17, "1.5000200271606445"},
    {0x1.80015p0f, chars_format::general, 18, "1.50002002716064453"},
    {0x1.80015p0f, chars_format::general, 19, "1.500020027160644531"},
    {0x1.80015p0f, chars_format::general, 20, "1.5000200271606445312"}, // round to even
    {0x1.80015p0f, chars_format::general, 21, "1.50002002716064453125"}, // exact

    // Trim trailing zeros and decimal point.
    {0x1.00015p0f, chars_format::general, 1, "1"}, // fixed notation trims decimal point
    {0x1.00015p0f, chars_format::general, 2, "1"}, // general trims decimal point and trailing zeros
    {0x1.00015p0f, chars_format::general, 3, "1"},
    {0x1.00015p0f, chars_format::general, 4, "1"},
    {0x1.00015p0f, chars_format::general, 5, "1"},
    {0x1.00015p0f, chars_format::general, 6, "1.00002"},
    {0x1.00015p0f, chars_format::general, 7, "1.00002"},
    {0x1.00015p0f, chars_format::general, 8, "1.00002"},
    {0x1.00015p0f, chars_format::general, 9, "1.00002003"},
    {0x1.00015p0f, chars_format::general, 10, "1.000020027"},
    {0x1.00015p0f, chars_format::general, 11, "1.0000200272"},
    {0x1.00015p0f, chars_format::general, 12, "1.00002002716"},
    {0x1.00015p0f, chars_format::general, 13, "1.000020027161"},
    {0x1.00015p0f, chars_format::general, 14, "1.0000200271606"},
    {0x1.00015p0f, chars_format::general, 15, "1.00002002716064"},
    {0x1.00015p0f, chars_format::general, 16, "1.000020027160645"},
    {0x1.00015p0f, chars_format::general, 17, "1.0000200271606445"},
    {0x1.00015p0f, chars_format::general, 18, "1.00002002716064453"},
    {0x1.00015p0f, chars_format::general, 19, "1.000020027160644531"},
    {0x1.00015p0f, chars_format::general, 20, "1.0000200271606445312"}, // round to even
    {0x1.00015p0f, chars_format::general, 21, "1.00002002716064453125"}, // exact

    // Trim trailing zeros, scientific notation.
    {0x1.5cf752p-20f, chars_format::general, 1, "1e-06"}, // scientific notation trims decimal point
    {0x1.5cf752p-20f, chars_format::general, 2, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 3, "1.3e-06"}, // general trims trailing zeros
    {0x1.5cf752p-20f, chars_format::general, 4, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 5, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 6, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 7, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 8, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 9, "1.30000001e-06"},
    {0x1.5cf752p-20f, chars_format::general, 10, "1.300000008e-06"},
    {0x1.5cf752p-20f, chars_format::general, 11, "1.3000000081e-06"},
    {0x1.5cf752p-20f, chars_format::general, 12, "1.30000000809e-06"},
    {0x1.5cf752p-20f, chars_format::general, 13, "1.300000008086e-06"},
    {0x1.5cf752p-20f, chars_format::general, 14, "1.3000000080865e-06"},
    {0x1.5cf752p-20f, chars_format::general, 15, "1.3000000080865e-06"},
    {0x1.5cf752p-20f, chars_format::general, 16, "1.300000008086499e-06"},
    {0x1.5cf752p-20f, chars_format::general, 17, "1.3000000080864993e-06"},
    {0x1.5cf752p-20f, chars_format::general, 18, "1.30000000808649929e-06"},
    {0x1.5cf752p-20f, chars_format::general, 19, "1.300000008086499292e-06"},
    {0x1.5cf752p-20f, chars_format::general, 20, "1.3000000080864992924e-06"},
    {0x1.5cf752p-20f, chars_format::general, 21, "1.3000000080864992924e-06"},
    {0x1.5cf752p-20f, chars_format::general, 22, "1.300000008086499292403e-06"},
    {0x1.5cf752p-20f, chars_format::general, 23, "1.3000000080864992924035e-06"},
    {0x1.5cf752p-20f, chars_format::general, 24, "1.30000000808649929240346e-06"},
    {0x1.5cf752p-20f, chars_format::general, 25, "1.30000000808649929240346e-06"},
    {0x1.5cf752p-20f, chars_format::general, 26, "1.3000000080864992924034595e-06"},
    {0x1.5cf752p-20f, chars_format::general, 27, "1.30000000808649929240345955e-06"},
    {0x1.5cf752p-20f, chars_format::general, 28, "1.300000008086499292403459549e-06"},
    {0x1.5cf752p-20f, chars_format::general, 29, "1.300000008086499292403459549e-06"},
    {0x1.5cf752p-20f, chars_format::general, 30, "1.30000000808649929240345954895e-06"},
    {0x1.5cf752p-20f, chars_format::general, 31, "1.30000000808649929240345954895e-06"},
    {0x1.5cf752p-20f, chars_format::general, 32, "1.3000000080864992924034595489502e-06"},
    {0x1.5cf752p-20f, chars_format::general, 33, "1.3000000080864992924034595489502e-06"},
    {0x1.5cf752p-20f, chars_format::general, 34, "1.300000008086499292403459548950195e-06"},
    {0x1.5cf752p-20f, chars_format::general, 35, "1.3000000080864992924034595489501953e-06"},
    {0x1.5cf752p-20f, chars_format::general, 36, "1.30000000808649929240345954895019531e-06"},
    {0x1.5cf752p-20f, chars_format::general, 37, "1.300000008086499292403459548950195312e-06"}, // round to even
    {0x1.5cf752p-20f, chars_format::general, 38, "1.3000000080864992924034595489501953125e-06"}, // exact

    // Trim trailing zeros and decimal point, scientific notation.
    {0x1.92a738p-19f, chars_format::general, 1, "3e-06"}, // scientific notation trims decimal point
    {0x1.92a738p-19f, chars_format::general, 2, "3e-06"}, // general trims decimal point and trailing zeros
    {0x1.92a738p-19f, chars_format::general, 3, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 4, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 5, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 6, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 7, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 8, "3.0000001e-06"},
    {0x1.92a738p-19f, chars_format::general, 9, "3.00000011e-06"},
    {0x1.92a738p-19f, chars_format::general, 10, "3.000000106e-06"},
    {0x1.92a738p-19f, chars_format::general, 11, "3.0000001061e-06"},
    {0x1.92a738p-19f, chars_format::general, 12, "3.00000010611e-06"},
    {0x1.92a738p-19f, chars_format::general, 13, "3.000000106113e-06"},
    {0x1.92a738p-19f, chars_format::general, 14, "3.0000001061126e-06"},
    {0x1.92a738p-19f, chars_format::general, 15, "3.00000010611257e-06"},
    {0x1.92a738p-19f, chars_format::general, 16, "3.000000106112566e-06"},
    {0x1.92a738p-19f, chars_format::general, 17, "3.0000001061125658e-06"},
    {0x1.92a738p-19f, chars_format::general, 18, "3.00000010611256585e-06"},
    {0x1.92a738p-19f, chars_format::general, 19, "3.000000106112565845e-06"},
    {0x1.92a738p-19f, chars_format::general, 20, "3.0000001061125658453e-06"},
    {0x1.92a738p-19f, chars_format::general, 21, "3.00000010611256584525e-06"},
    {0x1.92a738p-19f, chars_format::general, 22, "3.000000106112565845251e-06"},
    {0x1.92a738p-19f, chars_format::general, 23, "3.0000001061125658452511e-06"},
    {0x1.92a738p-19f, chars_format::general, 24, "3.00000010611256584525108e-06"},
    {0x1.92a738p-19f, chars_format::general, 25, "3.000000106112565845251083e-06"},
    {0x1.92a738p-19f, chars_format::general, 26, "3.0000001061125658452510834e-06"},
    {0x1.92a738p-19f, chars_format::general, 27, "3.00000010611256584525108337e-06"},
    {0x1.92a738p-19f, chars_format::general, 28, "3.000000106112565845251083374e-06"},
    {0x1.92a738p-19f, chars_format::general, 29, "3.000000106112565845251083374e-06"},
    {0x1.92a738p-19f, chars_format::general, 30, "3.00000010611256584525108337402e-06"},
    {0x1.92a738p-19f, chars_format::general, 31, "3.000000106112565845251083374023e-06"},
    {0x1.92a738p-19f, chars_format::general, 32, "3.0000001061125658452510833740234e-06"},
    {0x1.92a738p-19f, chars_format::general, 33, "3.00000010611256584525108337402344e-06"},
    {0x1.92a738p-19f, chars_format::general, 34, "3.000000106112565845251083374023438e-06"}, // round to even
    {0x1.92a738p-19f, chars_format::general, 35, "3.0000001061125658452510833740234375e-06"}, // exact

    // Test a large precision with fixed notation and scientific notation,
    // verifying that we remain within the bounds of any lookup tables.
    {0x1.ba9fbep+0f, chars_format::general, 5000, "1.72899997234344482421875"},
    {0x1.d01ffap-20f, chars_format::general, 5000, "1.7290000187131226994097232818603515625e-06"},

    // Test the transitions between fixed notation and scientific notation.
    {5555555.0f, chars_format::general, 1, "6e+06"},
    {555555.0f, chars_format::general, 1, "6e+05"},
    {55555.0f, chars_format::general, 1, "6e+04"},
    {5555.0f, chars_format::general, 1, "6e+03"},
    {555.0f, chars_format::general, 1, "6e+02"},
    {55.0f, chars_format::general, 1, "6e+01"}, // round to even
    {5.0f, chars_format::general, 1, "5"},
    {0x1p-3f, chars_format::general, 1, "0.1"}, // 0.125
    {0x1p-6f, chars_format::general, 1, "0.02"}, // 0.015625
    {0x1p-9f, chars_format::general, 1, "0.002"}, // 0.001953125
    {0x1p-13f, chars_format::general, 1, "0.0001"}, // 0.0001220703125
    {0x1p-16f, chars_format::general, 1, "2e-05"}, // 1.52587890625e-05
    {0x1p-19f, chars_format::general, 1, "2e-06"}, // 1.9073486328125e-06

    {5555555.0f, chars_format::general, 2, "5.6e+06"},
    {555555.0f, chars_format::general, 2, "5.6e+05"},
    {55555.0f, chars_format::general, 2, "5.6e+04"},
    {5555.0f, chars_format::general, 2, "5.6e+03"},
    {555.0f, chars_format::general, 2, "5.6e+02"}, // round to even
    {55.0f, chars_format::general, 2, "55"},
    {5.0f, chars_format::general, 2, "5"},
    {0x1p-3f, chars_format::general, 2, "0.12"}, // round to even
    {0x1p-6f, chars_format::general, 2, "0.016"},
    {0x1p-9f, chars_format::general, 2, "0.002"},
    {0x1p-13f, chars_format::general, 2, "0.00012"},
    {0x1p-16f, chars_format::general, 2, "1.5e-05"},
    {0x1p-19f, chars_format::general, 2, "1.9e-06"},

    {5555555.0f, chars_format::general, 3, "5.56e+06"},
    {555555.0f, chars_format::general, 3, "5.56e+05"},
    {55555.0f, chars_format::general, 3, "5.56e+04"},
    {5555.0f, chars_format::general, 3, "5.56e+03"}, // round to even
    {555.0f, chars_format::general, 3, "555"},
    {55.0f, chars_format::general, 3, "55"},
    {5.0f, chars_format::general, 3, "5"},
    {0x1p-3f, chars_format::general, 3, "0.125"},
    {0x1p-6f, chars_format::general, 3, "0.0156"},
    {0x1p-9f, chars_format::general, 3, "0.00195"},
    {0x1p-13f, chars_format::general, 3, "0.000122"},
    {0x1p-16f, chars_format::general, 3, "1.53e-05"},
    {0x1p-19f, chars_format::general, 3, "1.91e-06"},

    {5555555.0f, chars_format::general, 4, "5.556e+06"},
    {555555.0f, chars_format::general, 4, "5.556e+05"},
    {55555.0f, chars_format::general, 4, "5.556e+04"}, // round to even
    {5555.0f, chars_format::general, 4, "5555"},
    {555.0f, chars_format::general, 4, "555"},
    {55.0f, chars_format::general, 4, "55"},
    {5.0f, chars_format::general, 4, "5"},
    {0x1p-3f, chars_format::general, 4, "0.125"},
    {0x1p-6f, chars_format::general, 4, "0.01562"}, // round to even
    {0x1p-9f, chars_format::general, 4, "0.001953"},
    {0x1p-13f, chars_format::general, 4, "0.0001221"},
    {0x1p-16f, chars_format::general, 4, "1.526e-05"},
    {0x1p-19f, chars_format::general, 4, "1.907e-06"},

    {5555555.0f, chars_format::general, 5, "5.5556e+06"},
    {555555.0f, chars_format::general, 5, "5.5556e+05"}, // round to even
    {55555.0f, chars_format::general, 5, "55555"},
    {5555.0f, chars_format::general, 5, "5555"},
    {555.0f, chars_format::general, 5, "555"},
    {55.0f, chars_format::general, 5, "55"},
    {5.0f, chars_format::general, 5, "5"},
    {0x1p-3f, chars_format::general, 5, "0.125"},
    {0x1p-6f, chars_format::general, 5, "0.015625"},
    {0x1p-9f, chars_format::general, 5, "0.0019531"},
    {0x1p-13f, chars_format::general, 5, "0.00012207"},
    {0x1p-16f, chars_format::general, 5, "1.5259e-05"},
    {0x1p-19f, chars_format::general, 5, "1.9073e-06"},

    // Tricky corner cases.
    // In these scenarios, rounding can adjust the scientific exponent X,
    // which affects the transition between fixed notation and scientific notation.
    {999.999f, chars_format::general, 1, "1e+03"}, // "%.0e" is "1e+03"; X == 3
    {999.999f, chars_format::general, 2, "1e+03"}, // "%.1e" is "1.0e+03"; X == 3
    {999.999f, chars_format::general, 3, "1e+03"}, // "%.2e" is "1.00e+03"; X == 3
    {999.999f, chars_format::general, 4, "1000"}, // "%.3e" is "1.000e+03"; X == 3
    {999.999f, chars_format::general, 5, "1000"}, // "%.4e" is "1.0000e+03"; X == 3
    {999.999f, chars_format::general, 6, "999.999"}, // "%.5e" is "9.99999e+02"; X == 2

    {999.99f, chars_format::general, 1, "1e+03"},
    {999.99f, chars_format::general, 2, "1e+03"},
    {999.99f, chars_format::general, 3, "1e+03"},
    {999.99f, chars_format::general, 4, "1000"},
    {999.99f, chars_format::general, 5, "999.99"},
    {999.99f, chars_format::general, 6, "999.99"},

    // C11's Standardese is slightly vague about how to perform the trial formatting in scientific notation,
    // but the intention is to use precision P - 1, which is what's used when scientific notation is actually chosen.
    // This example verifies this behavior. Here, P == 3 performs trial formatting with "%.2e", triggering rounding.
    // That increases X to 3, forcing scientific notation to be chosen.
    // If P == 3 performed trial formatting with "%.3e", rounding wouldn't happen,
    // X would be 2, and fixed notation would be chosen.
    {999.9f, chars_format::general, 1, "1e+03"}, // "%.0e" is "1e+03"; X == 3
    {999.9f, chars_format::general, 2, "1e+03"}, // "%.1e" is "1.0e+03"; X == 3
    {999.9f, chars_format::general, 3, "1e+03"}, // "%.2e" is "1.00e+03"; X == 3; SPECIAL CORNER CASE
    {999.9f, chars_format::general, 4, "999.9"}, // "%.3e" is "9.999e+02"; X == 2
    {999.9f, chars_format::general, 5, "999.9"}, // "%.4e" is "9.9990e+02"; X == 2
    {999.9f, chars_format::general, 6, "999.9"}, // "%.5e" is "9.99900e+02"; X == 2

    {999.0f, chars_format::general, 1, "1e+03"},
    {999.0f, chars_format::general, 2, "1e+03"},
    {999.0f, chars_format::general, 3, "999"},
    {999.0f, chars_format::general, 4, "999"},
    {999.0f, chars_format::general, 5, "999"},
    {999.0f, chars_format::general, 6, "999"},

    {99.9999f, chars_format::general, 1, "1e+02"},
    {99.9999f, chars_format::general, 2, "1e+02"},
    {99.9999f, chars_format::general, 3, "100"},
    {99.9999f, chars_format::general, 4, "100"},
    {99.9999f, chars_format::general, 5, "100"},
    {99.9999f, chars_format::general, 6, "99.9999"},

    {99.999f, chars_format::general, 1, "1e+02"},
    {99.999f, chars_format::general, 2, "1e+02"},
    {99.999f, chars_format::general, 3, "100"},
    {99.999f, chars_format::general, 4, "100"},
    {99.999f, chars_format::general, 5, "99.999"},
    {99.999f, chars_format::general, 6, "99.999"},

    {99.99f, chars_format::general, 1, "1e+02"},
    {99.99f, chars_format::general, 2, "1e+02"},
    {99.99f, chars_format::general, 3, "100"},
    {99.99f, chars_format::general, 4, "99.99"},
    {99.99f, chars_format::general, 5, "99.99"},
    {99.99f, chars_format::general, 6, "99.99"},

    {99.9f, chars_format::general, 1, "1e+02"},
    {99.9f, chars_format::general, 2, "1e+02"},
    {99.9f, chars_format::general, 3, "99.9"},
    {99.9f, chars_format::general, 4, "99.9"},
    {99.9f, chars_format::general, 5, "99.9"},
    {99.9f, chars_format::general, 6, "99.9"},

    {99.0f, chars_format::general, 1, "1e+02"},
    {99.0f, chars_format::general, 2, "99"},
    {99.0f, chars_format::general, 3, "99"},
    {99.0f, chars_format::general, 4, "99"},
    {99.0f, chars_format::general, 5, "99"},
    {99.0f, chars_format::general, 6, "99"},

    {9.99999f, chars_format::general, 1, "1e+01"},
    {9.99999f, chars_format::general, 2, "10"},
    {9.99999f, chars_format::general, 3, "10"},
    {9.99999f, chars_format::general, 4, "10"},
    {9.99999f, chars_format::general, 5, "10"},
    {9.99999f, chars_format::general, 6, "9.99999"},

    {9.9999f, chars_format::general, 1, "1e+01"},
    {9.9999f, chars_format::general, 2, "10"},
    {9.9999f, chars_format::general, 3, "10"},
    {9.9999f, chars_format::general, 4, "10"},
    {9.9999f, chars_format::general, 5, "9.9999"},
    {9.9999f, chars_format::general, 6, "9.9999"},

    {9.999f, chars_format::general, 1, "1e+01"},
    {9.999f, chars_format::general, 2, "10"},
    {9.999f, chars_format::general, 3, "10"},
    {9.999f, chars_format::general, 4, "9.999"},
    {9.999f, chars_format::general, 5, "9.999"},
    {9.999f, chars_format::general, 6, "9.999"},

    {9.99f, chars_format::general, 1, "1e+01"},
    {9.99f, chars_format::general, 2, "10"},
    {9.99f, chars_format::general, 3, "9.99"},
    {9.99f, chars_format::general, 4, "9.99"},
    {9.99f, chars_format::general, 5, "9.99"},
    {9.99f, chars_format::general, 6, "9.99"},

    {9.9f, chars_format::general, 1, "1e+01"},
    {9.9f, chars_format::general, 2, "9.9"},
    {9.9f, chars_format::general, 3, "9.9"},
    {9.9f, chars_format::general, 4, "9.9"},
    {9.9f, chars_format::general, 5, "9.9"},
    {9.9f, chars_format::general, 6, "9.9"},

    {9.0f, chars_format::general, 1, "9"},
    {9.0f, chars_format::general, 2, "9"},
    {9.0f, chars_format::general, 3, "9"},
    {9.0f, chars_format::general, 4, "9"},
    {9.0f, chars_format::general, 5, "9"},
    {9.0f, chars_format::general, 6, "9"},

    {0.999999f, chars_format::general, 1, "1"},
    {0.999999f, chars_format::general, 2, "1"},
    {0.999999f, chars_format::general, 3, "1"},
    {0.999999f, chars_format::general, 4, "1"},
    {0.999999f, chars_format::general, 5, "1"},
    {0.999999f, chars_format::general, 6, "0.999999"},

    {0.99999f, chars_format::general, 1, "1"},
    {0.99999f, chars_format::general, 2, "1"},
    {0.99999f, chars_format::general, 3, "1"},
    {0.99999f, chars_format::general, 4, "1"},
    {0.99999f, chars_format::general, 5, "0.99999"},
    {0.99999f, chars_format::general, 6, "0.99999"},

    {0.9999f, chars_format::general, 1, "1"},
    {0.9999f, chars_format::general, 2, "1"},
    {0.9999f, chars_format::general, 3, "1"},
    {0.9999f, chars_format::general, 4, "0.9999"},
    {0.9999f, chars_format::general, 5, "0.9999"},
    {0.9999f, chars_format::general, 6, "0.9999"},

    {0.999f, chars_format::general, 1, "1"},
    {0.999f, chars_format::general, 2, "1"},
    {0.999f, chars_format::general, 3, "0.999"},
    {0.999f, chars_format::general, 4, "0.999"},
    {0.999f, chars_format::general, 5, "0.999"},
    {0.999f, chars_format::general, 6, "0.999"},

    {0.99f, chars_format::general, 1, "1"},
    {0.99f, chars_format::general, 2, "0.99"},
    {0.99f, chars_format::general, 3, "0.99"},
    {0.99f, chars_format::general, 4, "0.99"},
    {0.99f, chars_format::general, 5, "0.99"},
    {0.99f, chars_format::general, 6, "0.99"},

    {0.9f, chars_format::general, 1, "0.9"},
    {0.9f, chars_format::general, 2, "0.9"},
    {0.9f, chars_format::general, 3, "0.9"},
    {0.9f, chars_format::general, 4, "0.9"},
    {0.9f, chars_format::general, 5, "0.9"},
    {0.9f, chars_format::general, 6, "0.9"},

    {0.0999999f, chars_format::general, 1, "0.1"},
    {0.0999999f, chars_format::general, 2, "0.1"},
    {0.0999999f, chars_format::general, 3, "0.1"},
    {0.0999999f, chars_format::general, 4, "0.1"},
    {0.0999999f, chars_format::general, 5, "0.1"},
    {0.0999999f, chars_format::general, 6, "0.0999999"},

    {0.099999f, chars_format::general, 1, "0.1"},
    {0.099999f, chars_format::general, 2, "0.1"},
    {0.099999f, chars_format::general, 3, "0.1"},
    {0.099999f, chars_format::general, 4, "0.1"},
    {0.099999f, chars_format::general, 5, "0.099999"},
    {0.099999f, chars_format::general, 6, "0.099999"},

    {0.09999f, chars_format::general, 1, "0.1"},
    {0.09999f, chars_format::general, 2, "0.1"},
    {0.09999f, chars_format::general, 3, "0.1"},
    {0.09999f, chars_format::general, 4, "0.09999"},
    {0.09999f, chars_format::general, 5, "0.09999"},
    {0.09999f, chars_format::general, 6, "0.09999"},

    {0.0999f, chars_format::general, 1, "0.1"},
    {0.0999f, chars_format::general, 2, "0.1"},
    {0.0999f, chars_format::general, 3, "0.0999"},
    {0.0999f, chars_format::general, 4, "0.0999"},
    {0.0999f, chars_format::general, 5, "0.0999"},
    {0.0999f, chars_format::general, 6, "0.0999"},

    {0.099f, chars_format::general, 1, "0.1"},
    {0.099f, chars_format::general, 2, "0.099"},
    {0.099f, chars_format::general, 3, "0.099"},
    {0.099f, chars_format::general, 4, "0.099"},
    {0.099f, chars_format::general, 5, "0.099"},
    {0.099f, chars_format::general, 6, "0.099"},

    {0.09f, chars_format::general, 1, "0.09"},
    {0.09f, chars_format::general, 2, "0.09"},
    {0.09f, chars_format::general, 3, "0.09"},
    {0.09f, chars_format::general, 4, "0.09"},
    {0.09f, chars_format::general, 5, "0.09"},
    {0.09f, chars_format::general, 6, "0.09"},

    {0.00999999f, chars_format::general, 1, "0.01"},
    {0.00999999f, chars_format::general, 2, "0.01"},
    {0.00999999f, chars_format::general, 3, "0.01"},
    {0.00999999f, chars_format::general, 4, "0.01"},
    {0.00999999f, chars_format::general, 5, "0.01"},
    {0.00999999f, chars_format::general, 6, "0.00999999"},

    {0.0099999f, chars_format::general, 1, "0.01"},
    {0.0099999f, chars_format::general, 2, "0.01"},
    {0.0099999f, chars_format::general, 3, "0.01"},
    {0.0099999f, chars_format::general, 4, "0.01"},
    {0.0099999f, chars_format::general, 5, "0.0099999"},
    {0.0099999f, chars_format::general, 6, "0.0099999"},

    {0.009999f, chars_format::general, 1, "0.01"},
    {0.009999f, chars_format::general, 2, "0.01"},
    {0.009999f, chars_format::general, 3, "0.01"},
    {0.009999f, chars_format::general, 4, "0.009999"},
    {0.009999f, chars_format::general, 5, "0.009999"},
    {0.009999f, chars_format::general, 6, "0.009999"},

    {0.00999f, chars_format::general, 1, "0.01"},
    {0.00999f, chars_format::general, 2, "0.01"},
    {0.00999f, chars_format::general, 3, "0.00999"},
    {0.00999f, chars_format::general, 4, "0.00999"},
    {0.00999f, chars_format::general, 5, "0.00999"},
    {0.00999f, chars_format::general, 6, "0.00999"},

    {0.0099f, chars_format::general, 1, "0.01"},
    {0.0099f, chars_format::general, 2, "0.0099"},
    {0.0099f, chars_format::general, 3, "0.0099"},
    {0.0099f, chars_format::general, 4, "0.0099"},
    {0.0099f, chars_format::general, 5, "0.0099"},
    {0.0099f, chars_format::general, 6, "0.0099"},

    {0.009f, chars_format::general, 1, "0.009"},
    {0.009f, chars_format::general, 2, "0.009"},
    {0.009f, chars_format::general, 3, "0.009"},
    {0.009f, chars_format::general, 4, "0.009"},
    {0.009f, chars_format::general, 5, "0.009"},
    {0.009f, chars_format::general, 6, "0.009"},

    {0.000999999f, chars_format::general, 1, "0.001"},
    {0.000999999f, chars_format::general, 2, "0.001"},
    {0.000999999f, chars_format::general, 3, "0.001"},
    {0.000999999f, chars_format::general, 4, "0.001"},
    {0.000999999f, chars_format::general, 5, "0.001"},
    {0.000999999f, chars_format::general, 6, "0.000999999"},

    {0.00099999f, chars_format::general, 1, "0.001"},
    {0.00099999f, chars_format::general, 2, "0.001"},
    {0.00099999f, chars_format::general, 3, "0.001"},
    {0.00099999f, chars_format::general, 4, "0.001"},
    {0.00099999f, chars_format::general, 5, "0.00099999"},
    {0.00099999f, chars_format::general, 6, "0.00099999"},

    {0.0009999f, chars_format::general, 1, "0.001"},
    {0.0009999f, chars_format::general, 2, "0.001"},
    {0.0009999f, chars_format::general, 3, "0.001"},
    {0.0009999f, chars_format::general, 4, "0.0009999"},
    {0.0009999f, chars_format::general, 5, "0.0009999"},
    {0.0009999f, chars_format::general, 6, "0.0009999"},

    {0.000999f, chars_format::general, 1, "0.001"},
    {0.000999f, chars_format::general, 2, "0.001"},
    {0.000999f, chars_format::general, 3, "0.000999"},
    {0.000999f, chars_format::general, 4, "0.000999"},
    {0.000999f, chars_format::general, 5, "0.000999"},
    {0.000999f, chars_format::general, 6, "0.000999"},

    {0.00099f, chars_format::general, 1, "0.001"},
    {0.00099f, chars_format::general, 2, "0.00099"},
    {0.00099f, chars_format::general, 3, "0.00099"},
    {0.00099f, chars_format::general, 4, "0.00099"},
    {0.00099f, chars_format::general, 5, "0.00099"},
    {0.00099f, chars_format::general, 6, "0.00099"},

    {0.0009f, chars_format::general, 1, "0.0009"},
    {0.0009f, chars_format::general, 2, "0.0009"},
    {0.0009f, chars_format::general, 3, "0.0009"},
    {0.0009f, chars_format::general, 4, "0.0009"},
    {0.0009f, chars_format::general, 5, "0.0009"},
    {0.0009f, chars_format::general, 6, "0.0009"},

    // Having a scientific exponent X == -5 triggers scientific notation.
    // If rounding adjusts this to X == -4, then fixed notation will be selected.
    {0.0000999999f, chars_format::general, 1, "0.0001"},
    {0.0000999999f, chars_format::general, 2, "0.0001"},
    {0.0000999999f, chars_format::general, 3, "0.0001"},
    {0.0000999999f, chars_format::general, 4, "0.0001"},
    {0.0000999999f, chars_format::general, 5, "0.0001"},
    {0.0000999999f, chars_format::general, 6, "9.99999e-05"},

    {0.000099999f, chars_format::general, 1, "0.0001"},
    {0.000099999f, chars_format::general, 2, "0.0001"},
    {0.000099999f, chars_format::general, 3, "0.0001"},
    {0.000099999f, chars_format::general, 4, "0.0001"},
    {0.000099999f, chars_format::general, 5, "9.9999e-05"},
    {0.000099999f, chars_format::general, 6, "9.9999e-05"},

    {0.00009999f, chars_format::general, 1, "0.0001"},
    {0.00009999f, chars_format::general, 2, "0.0001"},
    {0.00009999f, chars_format::general, 3, "0.0001"},
    {0.00009999f, chars_format::general, 4, "9.999e-05"},
    {0.00009999f, chars_format::general, 5, "9.999e-05"},
    {0.00009999f, chars_format::general, 6, "9.999e-05"},

    {0.0000999f, chars_format::general, 1, "0.0001"},
    {0.0000999f, chars_format::general, 2, "0.0001"},
    {0.0000999f, chars_format::general, 3, "9.99e-05"},
    {0.0000999f, chars_format::general, 4, "9.99e-05"},
    {0.0000999f, chars_format::general, 5, "9.99e-05"},
    {0.0000999f, chars_format::general, 6, "9.99e-05"},

    {0.000099f, chars_format::general, 1, "0.0001"},
    {0.000099f, chars_format::general, 2, "9.9e-05"},
    {0.000099f, chars_format::general, 3, "9.9e-05"},
    {0.000099f, chars_format::general, 4, "9.9e-05"},
    {0.000099f, chars_format::general, 5, "9.9e-05"},
    {0.000099f, chars_format::general, 6, "9.9e-05"},

    {0.00009f, chars_format::general, 1, "9e-05"},
    {0.00009f, chars_format::general, 2, "9e-05"},
    {0.00009f, chars_format::general, 3, "9e-05"},
    {0.00009f, chars_format::general, 4, "9e-05"},
    {0.00009f, chars_format::general, 5, "9e-05"},
    {0.00009f, chars_format::general, 6, "9e-05"},

    // Rounding test cases without exponent-adjusting behavior.
    {2999.999f, chars_format::general, 1, "3e+03"},
    {2999.999f, chars_format::general, 2, "3e+03"},
    {2999.999f, chars_format::general, 3, "3e+03"},
    {2999.999f, chars_format::general, 4, "3000"},
    {2999.999f, chars_format::general, 5, "3000"},
    {2999.999f, chars_format::general, 6, "3000"},

    {2999.99f, chars_format::general, 1, "3e+03"},
    {2999.99f, chars_format::general, 2, "3e+03"},
    {2999.99f, chars_format::general, 3, "3e+03"},
    {2999.99f, chars_format::general, 4, "3000"},
    {2999.99f, chars_format::general, 5, "3000"},
    {2999.99f, chars_format::general, 6, "2999.99"},

    {2999.9f, chars_format::general, 1, "3e+03"},
    {2999.9f, chars_format::general, 2, "3e+03"},
    {2999.9f, chars_format::general, 3, "3e+03"},
    {2999.9f, chars_format::general, 4, "3000"},
    {2999.9f, chars_format::general, 5, "2999.9"},
    {2999.9f, chars_format::general, 6, "2999.9"},

    {2999.0f, chars_format::general, 1, "3e+03"},
    {2999.0f, chars_format::general, 2, "3e+03"},
    {2999.0f, chars_format::general, 3, "3e+03"},
    {2999.0f, chars_format::general, 4, "2999"},
    {2999.0f, chars_format::general, 5, "2999"},
    {2999.0f, chars_format::general, 6, "2999"},

    {299.999f, chars_format::general, 1, "3e+02"},
    {299.999f, chars_format::general, 2, "3e+02"},
    {299.999f, chars_format::general, 3, "300"},
    {299.999f, chars_format::general, 4, "300"},
    {299.999f, chars_format::general, 5, "300"},
    {299.999f, chars_format::general, 6, "299.999"},

    {299.99f, chars_format::general, 1, "3e+02"},
    {299.99f, chars_format::general, 2, "3e+02"},
    {299.99f, chars_format::general, 3, "300"},
    {299.99f, chars_format::general, 4, "300"},
    {299.99f, chars_format::general, 5, "299.99"},
    {299.99f, chars_format::general, 6, "299.99"},

    {299.9f, chars_format::general, 1, "3e+02"},
    {299.9f, chars_format::general, 2, "3e+02"},
    {299.9f, chars_format::general, 3, "300"},
    {299.9f, chars_format::general, 4, "299.9"},
    {299.9f, chars_format::general, 5, "299.9"},
    {299.9f, chars_format::general, 6, "299.9"},

    {299.0f, chars_format::general, 1, "3e+02"},
    {299.0f, chars_format::general, 2, "3e+02"},
    {299.0f, chars_format::general, 3, "299"},
    {299.0f, chars_format::general, 4, "299"},
    {299.0f, chars_format::general, 5, "299"},
    {299.0f, chars_format::general, 6, "299"},

    {29.999f, chars_format::general, 1, "3e+01"},
    {29.999f, chars_format::general, 2, "30"},
    {29.999f, chars_format::general, 3, "30"},
    {29.999f, chars_format::general, 4, "30"},
    {29.999f, chars_format::general, 5, "29.999"},
    {29.999f, chars_format::general, 6, "29.999"},

    {29.99f, chars_format::general, 1, "3e+01"},
    {29.99f, chars_format::general, 2, "30"},
    {29.99f, chars_format::general, 3, "30"},
    {29.99f, chars_format::general, 4, "29.99"},
    {29.99f, chars_format::general, 5, "29.99"},
    {29.99f, chars_format::general, 6, "29.99"},

    {29.9f, chars_format::general, 1, "3e+01"},
    {29.9f, chars_format::general, 2, "30"},
    {29.9f, chars_format::general, 3, "29.9"},
    {29.9f, chars_format::general, 4, "29.9"},
    {29.9f, chars_format::general, 5, "29.9"},
    {29.9f, chars_format::general, 6, "29.9"},

    {29.0f, chars_format::general, 1, "3e+01"},
    {29.0f, chars_format::general, 2, "29"},
    {29.0f, chars_format::general, 3, "29"},
    {29.0f, chars_format::general, 4, "29"},
    {29.0f, chars_format::general, 5, "29"},
    {29.0f, chars_format::general, 6, "29"},

    {2.999f, chars_format::general, 1, "3"},
    {2.999f, chars_format::general, 2, "3"},
    {2.999f, chars_format::general, 3, "3"},
    {2.999f, chars_format::general, 4, "2.999"},
    {2.999f, chars_format::general, 5, "2.999"},
    {2.999f, chars_format::general, 6, "2.999"},

    {2.99f, chars_format::general, 1, "3"},
    {2.99f, chars_format::general, 2, "3"},
    {2.99f, chars_format::general, 3, "2.99"},
    {2.99f, chars_format::general, 4, "2.99"},
    {2.99f, chars_format::general, 5, "2.99"},
    {2.99f, chars_format::general, 6, "2.99"},

    {2.9f, chars_format::general, 1, "3"},
    {2.9f, chars_format::general, 2, "2.9"},
    {2.9f, chars_format::general, 3, "2.9"},
    {2.9f, chars_format::general, 4, "2.9"},
    {2.9f, chars_format::general, 5, "2.9"},
    {2.9f, chars_format::general, 6, "2.9"},

    {2.0f, chars_format::general, 1, "2"},
    {2.0f, chars_format::general, 2, "2"},
    {2.0f, chars_format::general, 3, "2"},
    {2.0f, chars_format::general, 4, "2"},
    {2.0f, chars_format::general, 5, "2"},
    {2.0f, chars_format::general, 6, "2"},

    {0.2999f, chars_format::general, 1, "0.3"},
    {0.2999f, chars_format::general, 2, "0.3"},
    {0.2999f, chars_format::general, 3, "0.3"},
    {0.2999f, chars_format::general, 4, "0.2999"},
    {0.2999f, chars_format::general, 5, "0.2999"},
    {0.2999f, chars_format::general, 6, "0.2999"},

    {0.299f, chars_format::general, 1, "0.3"},
    {0.299f, chars_format::general, 2, "0.3"},
    {0.299f, chars_format::general, 3, "0.299"},
    {0.299f, chars_format::general, 4, "0.299"},
    {0.299f, chars_format::general, 5, "0.299"},
    {0.299f, chars_format::general, 6, "0.299"},

    {0.29f, chars_format::general, 1, "0.3"},
    {0.29f, chars_format::general, 2, "0.29"},
    {0.29f, chars_format::general, 3, "0.29"},
    {0.29f, chars_format::general, 4, "0.29"},
    {0.29f, chars_format::general, 5, "0.29"},
    {0.29f, chars_format::general, 6, "0.29"},

    {0.2f, chars_format::general, 1, "0.2"},
    {0.2f, chars_format::general, 2, "0.2"},
    {0.2f, chars_format::general, 3, "0.2"},
    {0.2f, chars_format::general, 4, "0.2"},
    {0.2f, chars_format::general, 5, "0.2"},
    {0.2f, chars_format::general, 6, "0.2"},

    {0.02999f, chars_format::general, 1, "0.03"},
    {0.02999f, chars_format::general, 2, "0.03"},
    {0.02999f, chars_format::general, 3, "0.03"},
    {0.02999f, chars_format::general, 4, "0.02999"},
    {0.02999f, chars_format::general, 5, "0.02999"},
    {0.02999f, chars_format::general, 6, "0.02999"},

    {0.0299f, chars_format::general, 1, "0.03"},
    {0.0299f, chars_format::general, 2, "0.03"},
    {0.0299f, chars_format::general, 3, "0.0299"},
    {0.0299f, chars_format::general, 4, "0.0299"},
    {0.0299f, chars_format::general, 5, "0.0299"},
    {0.0299f, chars_format::general, 6, "0.0299"},

    {0.029f, chars_format::general, 1, "0.03"},
    {0.029f, chars_format::general, 2, "0.029"},
    {0.029f, chars_format::general, 3, "0.029"},
    {0.029f, chars_format::general, 4, "0.029"},
    {0.029f, chars_format::general, 5, "0.029"},
    {0.029f, chars_format::general, 6, "0.029"},

    {0.02f, chars_format::general, 1, "0.02"},
    {0.02f, chars_format::general, 2, "0.02"},
    {0.02f, chars_format::general, 3, "0.02"},
    {0.02f, chars_format::general, 4, "0.02"},
    {0.02f, chars_format::general, 5, "0.02"},
    {0.02f, chars_format::general, 6, "0.02"},

    {0.002999f, chars_format::general, 1, "0.003"},
    {0.002999f, chars_format::general, 2, "0.003"},
    {0.002999f, chars_format::general, 3, "0.003"},
    {0.002999f, chars_format::general, 4, "0.002999"},
    {0.002999f, chars_format::general, 5, "0.002999"},
    {0.002999f, chars_format::general, 6, "0.002999"},

    {0.00299f, chars_format::general, 1, "0.003"},
    {0.00299f, chars_format::general, 2, "0.003"},
    {0.00299f, chars_format::general, 3, "0.00299"},
    {0.00299f, chars_format::general, 4, "0.00299"},
    {0.00299f, chars_format::general, 5, "0.00299"},
    {0.00299f, chars_format::general, 6, "0.00299"},

    {0.0029f, chars_format::general, 1, "0.003"},
    {0.0029f, chars_format::general, 2, "0.0029"},
    {0.0029f, chars_format::general, 3, "0.0029"},
    {0.0029f, chars_format::general, 4, "0.0029"},
    {0.0029f, chars_format::general, 5, "0.0029"},
    {0.0029f, chars_format::general, 6, "0.0029"},

    {0.002f, chars_format::general, 1, "0.002"},
    {0.002f, chars_format::general, 2, "0.002"},
    {0.002f, chars_format::general, 3, "0.002"},
    {0.002f, chars_format::general, 4, "0.002"},
    {0.002f, chars_format::general, 5, "0.002"},
    {0.002f, chars_format::general, 6, "0.002"},

    {0.0002999f, chars_format::general, 1, "0.0003"},
    {0.0002999f, chars_format::general, 2, "0.0003"},
    {0.0002999f, chars_format::general, 3, "0.0003"},
    {0.0002999f, chars_format::general, 4, "0.0002999"},
    {0.0002999f, chars_format::general, 5, "0.0002999"},
    {0.0002999f, chars_format::general, 6, "0.0002999"},

    {0.000299f, chars_format::general, 1, "0.0003"},
    {0.000299f, chars_format::general, 2, "0.0003"},
    {0.000299f, chars_format::general, 3, "0.000299"},
    {0.000299f, chars_format::general, 4, "0.000299"},
    {0.000299f, chars_format::general, 5, "0.000299"},
    {0.000299f, chars_format::general, 6, "0.000299"},

    {0.00029f, chars_format::general, 1, "0.0003"},
    {0.00029f, chars_format::general, 2, "0.00029"},
    {0.00029f, chars_format::general, 3, "0.00029"},
    {0.00029f, chars_format::general, 4, "0.00029"},
    {0.00029f, chars_format::general, 5, "0.00029"},
    {0.00029f, chars_format::general, 6, "0.00029"},

    {0.0002f, chars_format::general, 1, "0.0002"},
    {0.0002f, chars_format::general, 2, "0.0002"},
    {0.0002f, chars_format::general, 3, "0.0002"},
    {0.0002f, chars_format::general, 4, "0.0002"},
    {0.0002f, chars_format::general, 5, "0.0002"},
    {0.0002f, chars_format::general, 6, "0.0002"},

    {0.00002999f, chars_format::general, 1, "3e-05"},
    {0.00002999f, chars_format::general, 2, "3e-05"},
    {0.00002999f, chars_format::general, 3, "3e-05"},
    {0.00002999f, chars_format::general, 4, "2.999e-05"},
    {0.00002999f, chars_format::general, 5, "2.999e-05"},
    {0.00002999f, chars_format::general, 6, "2.999e-05"},

    {0.0000299f, chars_format::general, 1, "3e-05"},
    {0.0000299f, chars_format::general, 2, "3e-05"},
    {0.0000299f, chars_format::general, 3, "2.99e-05"},
    {0.0000299f, chars_format::general, 4, "2.99e-05"},
    {0.0000299f, chars_format::general, 5, "2.99e-05"},
    {0.0000299f, chars_format::general, 6, "2.99e-05"},

    {0.000029f, chars_format::general, 1, "3e-05"},
    {0.000029f, chars_format::general, 2, "2.9e-05"},
    {0.000029f, chars_format::general, 3, "2.9e-05"},
    {0.000029f, chars_format::general, 4, "2.9e-05"},
    {0.000029f, chars_format::general, 5, "2.9e-05"},
    {0.000029f, chars_format::general, 6, "2.9e-05"},

    {0.00002f, chars_format::general, 1, "2e-05"},
    {0.00002f, chars_format::general, 2, "2e-05"},
    {0.00002f, chars_format::general, 3, "2e-05"},
    {0.00002f, chars_format::general, 4, "2e-05"},
    {0.00002f, chars_format::general, 5, "2e-05"},
    {0.00002f, chars_format::general, 6, "2e-05"},

    // Test the transitions between values of the scientific exponent X.
    // For brevity, we avoid testing all possible combinations of P and X. Instead, we test:
    // * All values of P where some X can be affected by rounding. (For float, this is [1, 7].)
    // * P == 25, which is arbitrary.
    // * P == numeric_limits::max_exponent10 + 1. This selects fixed notation for numeric_limits::max(),
    //   so it's the largest interesting value of P.
    // * Finally, we test the transitions around X == P - 1, ensuring that we can recognize every value of X.
    {0x1.8e7578p-14f, chars_format::general, 1, "9e-05"},
    {0x1.8e757ap-14f, chars_format::general, 1, "0.0001"},
    {0x1.f212d6p-11f, chars_format::general, 1, "0.0009"},
    {0x1.f212d8p-11f, chars_format::general, 1, "0.001"},
    {0x1.374bc6p-7f, chars_format::general, 1, "0.009"},
    {0x1.374bc8p-7f, chars_format::general, 1, "0.01"},
    {0x1.851eb8p-4f, chars_format::general, 1, "0.09"},
    {0x1.851ebap-4f, chars_format::general, 1, "0.1"},
    {0x1.e66666p-1f, chars_format::general, 1, "0.9"},
    {0x1.e66668p-1f, chars_format::general, 1, "1"},
    {0x1.2ffffep+3f, chars_format::general, 1, "9"},
    {0x1.300000p+3f, chars_format::general, 1, "1e+01"},
    {0x1.a1554ep-14f, chars_format::general, 2, "9.9e-05"},
    {0x1.a15550p-14f, chars_format::general, 2, "0.0001"},
    {0x1.04d550p-10f, chars_format::general, 2, "0.00099"},
    {0x1.04d552p-10f, chars_format::general, 2, "0.001"},
    {0x1.460aa6p-7f, chars_format::general, 2, "0.0099"},
    {0x1.460aa8p-7f, chars_format::general, 2, "0.01"},
    {0x1.978d4ep-4f, chars_format::general, 2, "0.099"},
    {0x1.978d50p-4f, chars_format::general, 2, "0.1"},
    {0x1.fd70a2p-1f, chars_format::general, 2, "0.99"},
    {0x1.fd70a4p-1f, chars_format::general, 2, "1"},
    {0x1.3e6666p+3f, chars_format::general, 2, "9.9"},
    {0x1.3e6668p+3f, chars_format::general, 2, "10"},
    {0x1.8dfffep+6f, chars_format::general, 2, "99"},
    {0x1.8e0000p+6f, chars_format::general, 2, "1e+02"},
    {0x1.a3387ep-14f, chars_format::general, 3, "9.99e-05"},
    {0x1.a33880p-14f, chars_format::general, 3, "0.0001"},
    {0x1.06034ep-10f, chars_format::general, 3, "0.000999"},
    {0x1.060350p-10f, chars_format::general, 3, "0.001"},
    {0x1.478422p-7f, chars_format::general, 3, "0.00999"},
    {0x1.478424p-7f, chars_format::general, 3, "0.01"},
    {0x1.99652ap-4f, chars_format::general, 3, "0.0999"},
    {0x1.99652cp-4f, chars_format::general, 3, "0.1"},
    {0x1.ffbe76p-1f, chars_format::general, 3, "0.999"},
    {0x1.ffbe78p-1f, chars_format::general, 3, "1"},
    {0x1.3fd70ap+3f, chars_format::general, 3, "9.99"},
    {0x1.3fd70cp+3f, chars_format::general, 3, "10"},
    {0x1.8fccccp+6f, chars_format::general, 3, "99.9"},
    {0x1.8fcccep+6f, chars_format::general, 3, "100"},
    {0x1.f3bffep+9f, chars_format::general, 3, "999"},
    {0x1.f3c000p+9f, chars_format::general, 3, "1e+03"},
    {0x1.a368d0p-14f, chars_format::general, 4, "9.999e-05"},
    {0x1.a368d2p-14f, chars_format::general, 4, "0.0001"},
    {0x1.062182p-10f, chars_format::general, 4, "0.0009999"},
    {0x1.062184p-10f, chars_format::general, 4, "0.001"},
    {0x1.47a9e2p-7f, chars_format::general, 4, "0.009999"},
    {0x1.47a9e4p-7f, chars_format::general, 4, "0.01"},
    {0x1.99945ap-4f, chars_format::general, 4, "0.09999"},
    {0x1.99945cp-4f, chars_format::general, 4, "0.1"},
    {0x1.fff972p-1f, chars_format::general, 4, "0.9999"},
    {0x1.fff974p-1f, chars_format::general, 4, "1"},
    {0x1.3ffbe6p+3f, chars_format::general, 4, "9.999"},
    {0x1.3ffbe8p+3f, chars_format::general, 4, "10"},
    {0x1.8ffae0p+6f, chars_format::general, 4, "99.99"},
    {0x1.8ffae2p+6f, chars_format::general, 4, "100"},
    {0x1.f3f998p+9f, chars_format::general, 4, "999.9"},
    {0x1.f3f99ap+9f, chars_format::general, 4, "1000"},
    {0x1.387bfep+13f, chars_format::general, 4, "9999"},
    {0x1.387c00p+13f, chars_format::general, 4, "1e+04"},
    {0x1.a36da4p-14f, chars_format::general, 5, "9.9999e-05"},
    {0x1.a36da6p-14f, chars_format::general, 5, "0.0001"},
    {0x1.062486p-10f, chars_format::general, 5, "0.00099999"},
    {0x1.062488p-10f, chars_format::general, 5, "0.001"},
    {0x1.47ada8p-7f, chars_format::general, 5, "0.0099999"},
    {0x1.47adaap-7f, chars_format::general, 5, "0.01"},
    {0x1.999912p-4f, chars_format::general, 5, "0.099999"},
    {0x1.999914p-4f, chars_format::general, 5, "0.1"},
    {0x1.ffff58p-1f, chars_format::general, 5, "0.99999"},
    {0x1.ffff5ap-1f, chars_format::general, 5, "1"},
    {0x1.3fff96p+3f, chars_format::general, 5, "9.9999"},
    {0x1.3fff98p+3f, chars_format::general, 5, "10"},
    {0x1.8fff7cp+6f, chars_format::general, 5, "99.999"},
    {0x1.8fff7ep+6f, chars_format::general, 5, "100"},
    {0x1.f3ff5cp+9f, chars_format::general, 5, "999.99"},
    {0x1.f3ff5ep+9f, chars_format::general, 5, "1000"},
    {0x1.387f98p+13f, chars_format::general, 5, "9999.9"},
    {0x1.387f9ap+13f, chars_format::general, 5, "10000"},
    {0x1.869f7ep+16f, chars_format::general, 5, "99999"},
    {0x1.869f80p+16f, chars_format::general, 5, "1e+05"},
    {0x1.a36e20p-14f, chars_format::general, 6, "9.99999e-05"},
    {0x1.a36e22p-14f, chars_format::general, 6, "0.0001"},
    {0x1.0624d4p-10f, chars_format::general, 6, "0.000999999"},
    {0x1.0624d6p-10f, chars_format::general, 6, "0.001"},
    {0x1.47ae08p-7f, chars_format::general, 6, "0.00999999"},
    {0x1.47ae0ap-7f, chars_format::general, 6, "0.01"},
    {0x1.99998cp-4f, chars_format::general, 6, "0.0999999"},
    {0x1.99998ep-4f, chars_format::general, 6, "0.1"},
    {0x1.ffffeep-1f, chars_format::general, 6, "0.999999"},
    {0x1.fffff0p-1f, chars_format::general, 6, "1"},
    {0x1.3ffff4p+3f, chars_format::general, 6, "9.99999"},
    {0x1.3ffff6p+3f, chars_format::general, 6, "10"},
    {0x1.8ffff2p+6f, chars_format::general, 6, "99.9999"},
    {0x1.8ffff4p+6f, chars_format::general, 6, "100"},
    {0x1.f3ffeep+9f, chars_format::general, 6, "999.999"},
    {0x1.f3fff0p+9f, chars_format::general, 6, "1000"},
    {0x1.387ff4p+13f, chars_format::general, 6, "9999.99"},
    {0x1.387ff6p+13f, chars_format::general, 6, "10000"},
    {0x1.869ff2p+16f, chars_format::general, 6, "99999.9"},
    {0x1.869ff4p+16f, chars_format::general, 6, "100000"},
    {0x1.e847eep+19f, chars_format::general, 6, "999999"},
    {0x1.e847f0p+19f, chars_format::general, 6, "1e+06"},
    {0x1.a36e2cp-14f, chars_format::general, 7, "9.999999e-05"},
    {0x1.a36e2ep-14f, chars_format::general, 7, "0.0001"},
    {0x1.0624dcp-10f, chars_format::general, 7, "0.0009999999"},
    {0x1.0624dep-10f, chars_format::general, 7, "0.001"},
    {0x1.47ae12p-7f, chars_format::general, 7, "0.009999999"},
    {0x1.47ae14p-7f, chars_format::general, 7, "0.01"},
    {0x1.999998p-4f, chars_format::general, 7, "0.09999999"},
    {0x1.99999ap-4f, chars_format::general, 7, "0.1"},
    {0x1.fffffep-1f, chars_format::general, 7, "0.9999999"},
    {0x1.000000p+0f, chars_format::general, 7, "1"},
    {0x1.3ffffep+3f, chars_format::general, 7, "9.999999"},
    {0x1.400000p+3f, chars_format::general, 7, "10"},
    {0x1.8ffffep+6f, chars_format::general, 7, "99.99999"},
    {0x1.900000p+6f, chars_format::general, 7, "100"},
    {0x1.f3fffep+9f, chars_format::general, 7, "999.9999"},
    {0x1.f40000p+9f, chars_format::general, 7, "1000"},
    {0x1.387ffep+13f, chars_format::general, 7, "9999.999"},
    {0x1.388000p+13f, chars_format::general, 7, "10000"},
    {0x1.869ffep+16f, chars_format::general, 7, "99999.99"},
    {0x1.86a000p+16f, chars_format::general, 7, "100000"},
    {0x1.e847fep+19f, chars_format::general, 7, "999999.9"},
    {0x1.e84800p+19f, chars_format::general, 7, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 7, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 7, "1e+07"},
    {0x1.7d783ep+26f, chars_format::general, 8, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 8, "1e+08"},
    {0x1.dcd64ep+29f, chars_format::general, 9, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 9, "1e+09"},
    {0x1.2a05f0p+33f, chars_format::general, 10, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 10, "1e+10"},
    {0x1.74876ep+36f, chars_format::general, 11, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 11, "1.0000000614e+11"},
    {0x1.d1a94ap+39f, chars_format::general, 12, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 12, "1.00000006144e+12"},
    {0x1.2309cep+43f, chars_format::general, 13, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 13, "1.000000087654e+13"},
    {0x1.6bcc40p+46f, chars_format::general, 14, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 14, "1.0000000037683e+14"},
    {0x1.c6bf52p+49f, chars_format::general, 15, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 15, "1.00000005409997e+15"},
    {0x1.1c3792p+53f, chars_format::general, 16, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 16, "1.000000027256422e+16"},
    {0x1.634578p+56f, chars_format::general, 17, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 17, "1.0000000702060954e+17"},
    {0x1.bc16d6p+59f, chars_format::general, 18, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 18, "1.00000005302622618e+18"},
    {0x1.158e46p+63f, chars_format::general, 19, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 19, "1.000000108001807565e+19"},
    {0x1.5af1d6p+66f, chars_format::general, 20, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 20, "1.0000000200408773427e+20"},
    {0x1.b1ae4cp+69f, chars_format::general, 21, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 21, "1.00000002004087734272e+21"},
    {0x1.0f0cf0p+73f, chars_format::general, 22, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 22, "1.000000090409621520384e+22"},
    {0x1.52d02cp+76f, chars_format::general, 23, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 23, "1.0000000678916233835315e+23"},
    {0x1.a78436p+79f, chars_format::general, 24, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 24, "1.00000001384842785508557e+24"},
    {0x1.a36e2ep-14f, chars_format::general, 25, "9.999999747378751635551453e-05"},
    {0x1.a36e30p-14f, chars_format::general, 25, "0.0001000000047497451305389404"},
    {0x1.0624dcp-10f, chars_format::general, 25, "0.0009999999310821294784545898"},
    {0x1.0624dep-10f, chars_format::general, 25, "0.001000000047497451305389404"},
    {0x1.47ae14p-7f, chars_format::general, 25, "0.009999999776482582092285156"},
    {0x1.47ae16p-7f, chars_format::general, 25, "0.01000000070780515670776367"},
    {0x1.999998p-4f, chars_format::general, 25, "0.0999999940395355224609375"},
    {0x1.99999ap-4f, chars_format::general, 25, "0.1000000014901161193847656"},
    {0x1.fffffep-1f, chars_format::general, 25, "0.999999940395355224609375"},
    {0x1.000000p+0f, chars_format::general, 25, "1"},
    {0x1.3ffffep+3f, chars_format::general, 25, "9.99999904632568359375"},
    {0x1.400000p+3f, chars_format::general, 25, "10"},
    {0x1.8ffffep+6f, chars_format::general, 25, "99.99999237060546875"},
    {0x1.900000p+6f, chars_format::general, 25, "100"},
    {0x1.f3fffep+9f, chars_format::general, 25, "999.99993896484375"},
    {0x1.f40000p+9f, chars_format::general, 25, "1000"},
    {0x1.387ffep+13f, chars_format::general, 25, "9999.9990234375"},
    {0x1.388000p+13f, chars_format::general, 25, "10000"},
    {0x1.869ffep+16f, chars_format::general, 25, "99999.9921875"},
    {0x1.86a000p+16f, chars_format::general, 25, "100000"},
    {0x1.e847fep+19f, chars_format::general, 25, "999999.9375"},
    {0x1.e84800p+19f, chars_format::general, 25, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 25, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 25, "10000000"},
    {0x1.7d783ep+26f, chars_format::general, 25, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 25, "100000000"},
    {0x1.dcd64ep+29f, chars_format::general, 25, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 25, "1000000000"},
    {0x1.2a05f0p+33f, chars_format::general, 25, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 25, "10000000000"},
    {0x1.74876ep+36f, chars_format::general, 25, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 25, "100000006144"},
    {0x1.d1a94ap+39f, chars_format::general, 25, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 25, "1000000061440"},
    {0x1.2309cep+43f, chars_format::general, 25, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 25, "10000000876544"},
    {0x1.6bcc40p+46f, chars_format::general, 25, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 25, "100000000376832"},
    {0x1.c6bf52p+49f, chars_format::general, 25, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 25, "1000000054099968"},
    {0x1.1c3792p+53f, chars_format::general, 25, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 25, "10000000272564224"},
    {0x1.634578p+56f, chars_format::general, 25, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 25, "100000007020609536"},
    {0x1.bc16d6p+59f, chars_format::general, 25, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 25, "1000000053026226176"},
    {0x1.158e46p+63f, chars_format::general, 25, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 25, "10000001080018075648"},
    {0x1.5af1d6p+66f, chars_format::general, 25, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 25, "100000002004087734272"},
    {0x1.b1ae4cp+69f, chars_format::general, 25, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 25, "1000000020040877342720"},
    {0x1.0f0cf0p+73f, chars_format::general, 25, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 25, "10000000904096215203840"},
    {0x1.52d02cp+76f, chars_format::general, 25, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 25, "100000006789162338353152"},
    {0x1.a78436p+79f, chars_format::general, 25, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 25, "1000000013848427855085568"},
    {0x1.08b2a2p+83f, chars_format::general, 25, "9999999562023526247432192"},
    {0x1.08b2a4p+83f, chars_format::general, 25, "1.000000071494503085427917e+25"},
    {0x1.4adf4ap+86f, chars_format::general, 26, "99999993314392253260627968"},
    {0x1.4adf4cp+86f, chars_format::general, 26, "1.0000000253776429011540378e+26"},
    {0x1.9d971ep+89f, chars_format::general, 27, "999999988484154753734934528"},
    {0x1.9d9720p+89f, chars_format::general, 27, "1.00000006227113104857314099e+27"},
    {0x1.027e72p+93f, chars_format::general, 28, "9999999442119689768320106496"},
    {0x1.027e74p+93f, chars_format::general, 28, "1.000000062271131048573140992e+28"},
    {0x1.431e0ep+96f, chars_format::general, 29, "99999992060013656248378458112"},
    {0x1.431e10p+96f, chars_format::general, 29, "1.000000015047466219876688855e+29"},
    {0x1.93e592p+99f, chars_format::general, 30, "999999939489602493962365435904"},
    {0x1.93e594p+99f, chars_format::general, 30, "1.00000001504746621987668885504e+30"},
    {0x1.f8def8p+102f, chars_format::general, 31, "9999999848243207295109594873856"},
    {0x1.f8defap+102f, chars_format::general, 31, "1.000000045270611710242418222694e+31"},
    {0x1.3b8b5ap+106f, chars_format::general, 32, "99999993646728794492579249913856"},
    {0x1.3b8b5cp+106f, chars_format::general, 32, "1.0000000331813535140961264756326e+32"},
    {0x1.8a6e32p+109f, chars_format::general, 33, "999999994495727286427992885035008"},
    {0x1.8a6e34p+109f, chars_format::general, 33, "1.00000007186697974176426006623027e+33"},
    {0x1.ed09bep+112f, chars_format::general, 34, "9999999790214767953607394487959552"},
    {0x1.ed09c0p+112f, chars_format::general, 34, "1.000000040918478759629753193752166e+34"},
    {0x1.342616p+116f, chars_format::general, 35, "99999994188327561679933120182222848"},
    {0x1.342618p+116f, chars_format::general, 35, "1.0000000409184787596297531937521664e+35"},
    {0x1.812f9cp+119f, chars_format::general, 36, "999999961690316245365415600208216064"},
    {0x1.812f9ep+119f, chars_format::general, 36, "1.0000000409184787596297531937521664e+36"},
    {0x1.e17b84p+122f, chars_format::general, 37, "9999999933815812510711506376257961984"},
    {0x1.e17b86p+122f, chars_format::general, 37, "1.000000056764111262482620712460956467e+37"},
    {0x1.2ced32p+126f, chars_format::general, 38, "99999996802856924650656260769173209088"},
    {0x1.2ced34p+126f, chars_format::general, 38, "1.000000069440617264764914727427988521e+38"},
    {0x1.a36e2ep-14f, chars_format::general, 39, "9.99999974737875163555145263671875e-05"},
    {0x1.a36e30p-14f, chars_format::general, 39, "0.0001000000047497451305389404296875"},
    {0x1.0624dcp-10f, chars_format::general, 39, "0.00099999993108212947845458984375"},
    {0x1.0624dep-10f, chars_format::general, 39, "0.001000000047497451305389404296875"},
    {0x1.47ae14p-7f, chars_format::general, 39, "0.00999999977648258209228515625"},
    {0x1.47ae16p-7f, chars_format::general, 39, "0.010000000707805156707763671875"},
    {0x1.999998p-4f, chars_format::general, 39, "0.0999999940395355224609375"},
    {0x1.99999ap-4f, chars_format::general, 39, "0.100000001490116119384765625"},
    {0x1.fffffep-1f, chars_format::general, 39, "0.999999940395355224609375"},
    {0x1.000000p+0f, chars_format::general, 39, "1"},
    {0x1.3ffffep+3f, chars_format::general, 39, "9.99999904632568359375"},
    {0x1.400000p+3f, chars_format::general, 39, "10"},
    {0x1.8ffffep+6f, chars_format::general, 39, "99.99999237060546875"},
    {0x1.900000p+6f, chars_format::general, 39, "100"},
    {0x1.f3fffep+9f, chars_format::general, 39, "999.99993896484375"},
    {0x1.f40000p+9f, chars_format::general, 39, "1000"},
    {0x1.387ffep+13f, chars_format::general, 39, "9999.9990234375"},
    {0x1.388000p+13f, chars_format::general, 39, "10000"},
    {0x1.869ffep+16f, chars_format::general, 39, "99999.9921875"},
    {0x1.86a000p+16f, chars_format::general, 39, "100000"},
    {0x1.e847fep+19f, chars_format::general, 39, "999999.9375"},
    {0x1.e84800p+19f, chars_format::general, 39, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 39, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 39, "10000000"},
    {0x1.7d783ep+26f, chars_format::general, 39, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 39, "100000000"},
    {0x1.dcd64ep+29f, chars_format::general, 39, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 39, "1000000000"},
    {0x1.2a05f0p+33f, chars_format::general, 39, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 39, "10000000000"},
    {0x1.74876ep+36f, chars_format::general, 39, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 39, "100000006144"},
    {0x1.d1a94ap+39f, chars_format::general, 39, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 39, "1000000061440"},
    {0x1.2309cep+43f, chars_format::general, 39, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 39, "10000000876544"},
    {0x1.6bcc40p+46f, chars_format::general, 39, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 39, "100000000376832"},
    {0x1.c6bf52p+49f, chars_format::general, 39, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 39, "1000000054099968"},
    {0x1.1c3792p+53f, chars_format::general, 39, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 39, "10000000272564224"},
    {0x1.634578p+56f, chars_format::general, 39, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 39, "100000007020609536"},
    {0x1.bc16d6p+59f, chars_format::general, 39, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 39, "1000000053026226176"},
    {0x1.158e46p+63f, chars_format::general, 39, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 39, "10000001080018075648"},
    {0x1.5af1d6p+66f, chars_format::general, 39, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 39, "100000002004087734272"},
    {0x1.b1ae4cp+69f, chars_format::general, 39, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 39, "1000000020040877342720"},
    {0x1.0f0cf0p+73f, chars_format::general, 39, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 39, "10000000904096215203840"},
    {0x1.52d02cp+76f, chars_format::general, 39, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 39, "100000006789162338353152"},
    {0x1.a78436p+79f, chars_format::general, 39, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 39, "1000000013848427855085568"},
    {0x1.08b2a2p+83f, chars_format::general, 39, "9999999562023526247432192"},
    {0x1.08b2a4p+83f, chars_format::general, 39, "10000000714945030854279168"},
    {0x1.4adf4ap+86f, chars_format::general, 39, "99999993314392253260627968"},
    {0x1.4adf4cp+86f, chars_format::general, 39, "100000002537764290115403776"},
    {0x1.9d971ep+89f, chars_format::general, 39, "999999988484154753734934528"},
    {0x1.9d9720p+89f, chars_format::general, 39, "1000000062271131048573140992"},
    {0x1.027e72p+93f, chars_format::general, 39, "9999999442119689768320106496"},
    {0x1.027e74p+93f, chars_format::general, 39, "10000000622711310485731409920"},
    {0x1.431e0ep+96f, chars_format::general, 39, "99999992060013656248378458112"},
    {0x1.431e10p+96f, chars_format::general, 39, "100000001504746621987668885504"},
    {0x1.93e592p+99f, chars_format::general, 39, "999999939489602493962365435904"},
    {0x1.93e594p+99f, chars_format::general, 39, "1000000015047466219876688855040"},
    {0x1.f8def8p+102f, chars_format::general, 39, "9999999848243207295109594873856"},
    {0x1.f8defap+102f, chars_format::general, 39, "10000000452706117102424182226944"},
    {0x1.3b8b5ap+106f, chars_format::general, 39, "99999993646728794492579249913856"},
    {0x1.3b8b5cp+106f, chars_format::general, 39, "100000003318135351409612647563264"},
    {0x1.8a6e32p+109f, chars_format::general, 39, "999999994495727286427992885035008"},
    {0x1.8a6e34p+109f, chars_format::general, 39, "1000000071866979741764260066230272"},
    {0x1.ed09bep+112f, chars_format::general, 39, "9999999790214767953607394487959552"},
    {0x1.ed09c0p+112f, chars_format::general, 39, "10000000409184787596297531937521664"},
    {0x1.342616p+116f, chars_format::general, 39, "99999994188327561679933120182222848"},
    {0x1.342618p+116f, chars_format::general, 39, "100000004091847875962975319375216640"},
    {0x1.812f9cp+119f, chars_format::general, 39, "999999961690316245365415600208216064"},
    {0x1.812f9ep+119f, chars_format::general, 39, "1000000040918478759629753193752166400"},
    {0x1.e17b84p+122f, chars_format::general, 39, "9999999933815812510711506376257961984"},
    {0x1.e17b86p+122f, chars_format::general, 39, "10000000567641112624826207124609564672"},
    {0x1.2ced32p+126f, chars_format::general, 39, "99999996802856924650656260769173209088"},
    {0x1.2ced34p+126f, chars_format::general, 39, "100000006944061726476491472742798852096"},
    {0x1.fffffep+127f, chars_format::general, 39, "340282346638528859811704183484516925440"},
};

inline constexpr float_to_chars_testcase float_hex_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::hex, 4, "0.0000p+0"},
    {-0.0f, chars_format::hex, 4, "-0.0000p+0"},
    {float_inf, chars_format::hex, 4, "inf"},
    {-float_inf, chars_format::hex, 4, "-inf"},
    {float_nan, chars_format::hex, 4, "nan"},
    {-float_nan, chars_format::hex, 4, "-nan"},
    {0x1.729p+0f, chars_format::hex, 4, "1.7290p+0"},
    {-0x1.729p+0f, chars_format::hex, 4, "-1.7290p+0"},

    // Test hexfloat corner cases.
    {0x1.728p+0f, chars_format::hex, 6, "1.728000p+0"},
    {0x0.000002p-126f, chars_format::hex, 6, "0.000002p-126"}, // min subnormal
    {0x0.fffffep-126f, chars_format::hex, 6, "0.fffffep-126"}, // max subnormal
    {0x1p-126f, chars_format::hex, 6, "1.000000p-126"}, // min normal
    {0x1.fffffep+127f, chars_format::hex, 6, "1.fffffep+127"}, // max normal

    // Test hexfloat exponents.
    {0x1p-109f, chars_format::hex, 0, "1p-109"},
    {0x1p-99f, chars_format::hex, 0, "1p-99"},
    {0x1p-9f, chars_format::hex, 0, "1p-9"},
    {0x1p+0f, chars_format::hex, 0, "1p+0"},
    {0x1p+9f, chars_format::hex, 0, "1p+9"},
    {0x1p+99f, chars_format::hex, 0, "1p+99"},
    {0x1p+109f, chars_format::hex, 0, "1p+109"},

    // Test hexfloat hexits.
    {0x1.0123p+0f, chars_format::hex, 4, "1.0123p+0"},
    {0x1.4567p+0f, chars_format::hex, 4, "1.4567p+0"},
    {0x1.89abp+0f, chars_format::hex, 4, "1.89abp+0"},
    {0x1.cdefp+0f, chars_format::hex, 4, "1.cdefp+0"},

    // Test varying precision. Negative precision requests full precision, not shortest round-trip.
    {0x1.123456p+0f, chars_format::hex, -2, "1.123456p+0"},
    {0x1.123456p+0f, chars_format::hex, -1, "1.123456p+0"},
    {0x1.123456p+0f, chars_format::hex, 0, "1p+0"},
    {0x1.123456p+0f, chars_format::hex, 1, "1.1p+0"},
    {0x1.123456p+0f, chars_format::hex, 2, "1.12p+0"},
    {0x1.123456p+0f, chars_format::hex, 3, "1.123p+0"},
    {0x1.123456p+0f, chars_format::hex, 4, "1.1234p+0"},
    {0x1.123456p+0f, chars_format::hex, 5, "1.12345p+0"},
    {0x1.123456p+0f, chars_format::hex, 6, "1.123456p+0"},
    {0x1.123456p+0f, chars_format::hex, 7, "1.1234560p+0"},
    {0x1.123456p+0f, chars_format::hex, 8, "1.12345600p+0"},
    {0x1.123456p+0f, chars_format::hex, 9, "1.123456000p+0"},

    // Test rounding at every position.
    {0x1.ccccccp+0f, chars_format::hex, 0, "2p+0"},
    {0x1.ccccccp+0f, chars_format::hex, 1, "1.dp+0"},
    {0x1.ccccccp+0f, chars_format::hex, 2, "1.cdp+0"},
    {0x1.ccccccp+0f, chars_format::hex, 3, "1.ccdp+0"},
    {0x1.ccccccp+0f, chars_format::hex, 4, "1.cccdp+0"},
    {0x1.ccccccp+0f, chars_format::hex, 5, "1.ccccdp+0"},
    {0x1.ccccccp+0f, chars_format::hex, 6, "1.ccccccp+0"},

    // Test all combinations of least significant bit, round bit, and trailing bits.
    {0x1.04000p+0f, chars_format::hex, 2, "1.04p+0"}, // Lsb 0, Round 0, Trailing 0
    {0x1.04001p+0f, chars_format::hex, 2, "1.04p+0"}, // Lsb 0, Round 0 and Trailing 1 in different hexits
    {0x1.04200p+0f, chars_format::hex, 2, "1.04p+0"}, // Lsb 0, Round 0 and Trailing 1 in same hexit
    {0x1.04800p+0f, chars_format::hex, 2, "1.04p+0"}, // Lsb 0, Round 1, Trailing 0
    {0x1.04801p+0f, chars_format::hex, 2, "1.05p+0"}, // Lsb 0, Round 1 and Trailing 1 in different hexits
    {0x1.04900p+0f, chars_format::hex, 2, "1.05p+0"}, // Lsb 0, Round 1 and Trailing 1 in same hexit
    {0x1.05000p+0f, chars_format::hex, 2, "1.05p+0"}, // Lsb 1, Round 0, Trailing 0
    {0x1.05001p+0f, chars_format::hex, 2, "1.05p+0"}, // Lsb 1, Round 0 and Trailing 1 in different hexits
    {0x1.05200p+0f, chars_format::hex, 2, "1.05p+0"}, // Lsb 1, Round 0 and Trailing 1 in same hexit
    {0x1.05800p+0f, chars_format::hex, 2, "1.06p+0"}, // Lsb 1, Round 1, Trailing 0
    {0x1.05801p+0f, chars_format::hex, 2, "1.06p+0"}, // Lsb 1, Round 1 and Trailing 1 in different hexits
    {0x1.05900p+0f, chars_format::hex, 2, "1.06p+0"}, // Lsb 1, Round 1 and Trailing 1 in same hexit

    // Test carry propagation.
    {0x1.0afffep+0f, chars_format::hex, 5, "1.0b000p+0"},

    // Test carry propagation into the leading hexit.
    {0x0.fffffep-126f, chars_format::hex, 5, "1.00000p-126"},
    {0x1.fffffep+127f, chars_format::hex, 5, "2.00000p+127"},

    // Test how the leading hexit participates in the rounding decision.
    {0x0.000p+0f, chars_format::hex, 0, "0p+0"},
    {0x0.001p-126f, chars_format::hex, 0, "0p-126"},
    {0x0.200p-126f, chars_format::hex, 0, "0p-126"},
    {0x0.800p-126f, chars_format::hex, 0, "0p-126"},
    {0x0.801p-126f, chars_format::hex, 0, "1p-126"},
    {0x0.900p-126f, chars_format::hex, 0, "1p-126"},
    {0x1.000p+0f, chars_format::hex, 0, "1p+0"},
    {0x1.001p+0f, chars_format::hex, 0, "1p+0"},
    {0x1.200p+0f, chars_format::hex, 0, "1p+0"},
    {0x1.800p+0f, chars_format::hex, 0, "2p+0"},
    {0x1.801p+0f, chars_format::hex, 0, "2p+0"},
    {0x1.900p+0f, chars_format::hex, 0, "2p+0"},
};

void
test01()
{
  auto handle_testcases = [] (const auto& testcases) {
    for (const auto [value,fmt,precision,correct] : testcases)
      {
	const int correct_len = strlen(correct);
	char buffer[correct_len];
	memset(buffer, '\0', sizeof(buffer));

	to_chars_result result
	  = (precision
	     ? to_chars(buffer, buffer+correct_len, value, fmt, *precision)
	     : to_chars(buffer, buffer+correct_len, value, fmt));
	VERIFY( result.ec == errc{} );
	VERIFY( result.ptr - buffer == correct_len );
	VERIFY( !memcmp(buffer, correct, correct_len) );

	result
	  = (precision
	     ? to_chars(buffer, buffer+correct_len-1, value, fmt, *precision)
	     : to_chars(buffer, buffer+correct_len-1, value, fmt));
	VERIFY( result.ec == errc::value_too_large );
	VERIFY( result.ptr == buffer+correct_len-1 );
      }
  };

  handle_testcases(float_to_chars_test_cases);
  handle_testcases(float_fixed_precision_to_chars_test_cases);
  handle_testcases(float_general_precision_to_chars_test_cases);
  handle_testcases(float_hex_precision_to_chars_test_cases);
}

int
main()
{
  test01();
}
